/**
 * <h1>AWS AppConfig Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * For a high level overview of what AWS AppConfig is and how it works, please take a look here:
 * <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/what-is-appconfig.html">What is AWS AppConfig?</a>
 * <p>
 * <h2>Basic Hosted Configuration Use Case</h2>
 * <p>
 * <blockquote>
 * <p>
 * The main way most AWS AppConfig users utilize the service is through hosted configuration, which involves storing
 * configuration data directly within AWS AppConfig.
 * <p>
 * </blockquote>
 * <p>
 * An example use case:
 * <p>
 * <blockquote><pre>
 * Application app = new Application(this, "MyApp");
 * Environment env = Environment.Builder.create(this, "MyEnv")
 *         .application(app)
 *         .build();
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfig")
 *         .application(app)
 *         .deployTo(List.of(env))
 *         .content(ConfigurationContent.fromInlineText("This is my configuration content."))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will create the application and environment for your configuration and then deploy your configuration to the
 * specified environment.
 * <p>
 * For more information about what these resources are: <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/creating-feature-flags-and-configuration-data.html">Creating feature flags and free form configuration data in AWS AppConfig</a>.
 * <p>
 * For more information about deploying configuration: <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/deploying-feature-flags.html">Deploying feature flags and configuration data in AWS AppConfig</a>
 * <p>
 * <hr>
 * <p>
 * For an in-depth walkthrough of specific resources and how to use them, please take a look at the following sections.
 * <p>
 * <h2>Application</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-namespace.html">AWS AppConfig Application Documentation</a>
 * <p>
 * In AWS AppConfig, an application is simply an organizational
 * construct like a folder. Configurations and environments are
 * associated with the application.
 * <p>
 * When creating an application through CDK, the name and
 * description of an application are optional.
 * <p>
 * Create a simple application:
 * <p>
 * <blockquote><pre>
 * new Application(this, "MyApplication");
 * </pre></blockquote>
 * <p>
 * <h2>Environment</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-environment.html">AWS AppConfig Environment Documentation</a>
 * <p>
 * Basic environment with monitors:
 * <p>
 * <blockquote><pre>
 * Application application;
 * Alarm alarm;
 * CompositeAlarm compositeAlarm;
 * 
 * 
 * Environment.Builder.create(this, "MyEnvironment")
 *         .application(application)
 *         .monitors(List.of(Monitor.fromCloudWatchAlarm(alarm), Monitor.fromCloudWatchAlarm(compositeAlarm)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Environment monitors also support L1 <code>CfnEnvironment.MonitorsProperty</code> constructs through the <code>fromCfnMonitorsProperty</code> method.
 * However, this is not the recommended approach for CloudWatch alarms because a role will not be auto-generated if not provided.
 * <p>
 * See <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/about-data-plane.html">About the AWS AppConfig data plane service</a> for more information.
 * <p>
 * <h3>Permissions</h3>
 * <p>
 * You can grant read permission on the environment's configurations with the grantReadConfig method as follows:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * 
 * Application app = new Application(this, "MyAppConfig");
 * Environment env = Environment.Builder.create(this, "MyEnvironment")
 *         .application(app)
 *         .build();
 * 
 * User user = new User(this, "MyUser");
 * env.grantReadConfig(user);
 * </pre></blockquote>
 * <p>
 * <h3>Deletion Protection Check</h3>
 * <p>
 * You can enable <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/deletion-protection.html">deletion protection</a> on the environment by setting the <code>deletionProtectionCheck</code> property.
 * <p>
 * <ul>
 * <li>ACCOUNT_DEFAULT: The default setting, which uses account-level deletion protection. To configure account-level deletion protection, use the UpdateAccountSettings API.</li>
 * <li>APPLY: Instructs the deletion protection check to run, even if deletion protection is disabled at the account level. APPLY also forces the deletion protection check to run against resources created in the past hour, which are normally excluded from deletion protection checks.</li>
 * <li>BYPASS: Instructs AWS AppConfig to bypass the deletion protection check and delete an environment even if deletion protection would have otherwise prevented it.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Application application;
 * Alarm alarm;
 * CompositeAlarm compositeAlarm;
 * 
 * 
 * Environment.Builder.create(this, "MyEnvironment")
 *         .application(application)
 *         .deletionProtectionCheck(DeletionProtectionCheck.APPLY)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Deployment Strategy</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-deployment-strategy.html">AWS AppConfig Deployment Strategy Documentation</a>
 * <p>
 * A deployment strategy defines how a configuration will roll out. The roll out is defined by four parameters: deployment type,
 * growth factor, deployment duration, and final bake time.
 * <p>
 * Deployment strategy with predefined values:
 * <p>
 * <blockquote><pre>
 * DeploymentStrategy.Builder.create(this, "MyDeploymentStrategy")
 *         .rolloutStrategy(RolloutStrategy.CANARY_10_PERCENT_20_MINUTES)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Deployment strategy with custom values:
 * <p>
 * <blockquote><pre>
 * DeploymentStrategy.Builder.create(this, "MyDeploymentStrategy")
 *         .rolloutStrategy(RolloutStrategy.linear(RolloutStrategyProps.builder()
 *                 .growthFactor(20)
 *                 .deploymentDuration(Duration.minutes(30))
 *                 .finalBakeTime(Duration.minutes(30))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Referencing a deployment strategy by ID:
 * <p>
 * <blockquote><pre>
 * DeploymentStrategy.fromDeploymentStrategyId(this, "MyImportedDeploymentStrategy", DeploymentStrategyId.fromString("abc123"));
 * </pre></blockquote>
 * <p>
 * Referencing an AWS AppConfig predefined deployment strategy by ID:
 * <p>
 * <blockquote><pre>
 * DeploymentStrategy.fromDeploymentStrategyId(this, "MyImportedPredefinedDeploymentStrategy", DeploymentStrategyId.CANARY_10_PERCENT_20_MINUTES);
 * </pre></blockquote>
 * <p>
 * <h2>Configuration</h2>
 * <p>
 * A configuration is a higher-level construct that can either be a <code>HostedConfiguration</code> (stored internally through AWS
 * AppConfig) or a <code>SourcedConfiguration</code> (stored in an Amazon S3 bucket, AWS Secrets Manager secrets, Systems Manager (SSM)
 * Parameter Store parameters, SSM documents, or AWS CodePipeline). This construct manages deployments on creation.
 * <p>
 * <h3>HostedConfiguration</h3>
 * <p>
 * A hosted configuration represents configuration stored in the AWS AppConfig hosted configuration store. A hosted configuration
 * takes in the configuration content and associated AWS AppConfig application. On construction of a hosted configuration, the
 * configuration is deployed.
 * <p>
 * You can define hosted configuration content using any of the following ConfigurationContent methods:
 * <p>
 * <ul>
 * <li><code>fromFile</code> - Defines the hosted configuration content from a file (you can specify a relative path). The content type will
 * be determined by the file extension unless specified.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromFile("config.json"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>fromInlineText</code> - Defines the hosted configuration from inline text. The content type will be set as <code>text/plain</code>.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromInlineText("This is my configuration content."))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>fromInlineJson</code> - Defines the hosted configuration from inline JSON. The content type will be set as <code>application/json</code> unless specified.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromInlineJson("{}"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>fromInlineYaml</code> - Defines the hosted configuration from inline YAML. The content type will be set as <code>application/x-yaml</code>.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromInlineYaml("MyConfig: This is my content."))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <ul>
 * <li><code>fromInline</code> - Defines the hosted configuration from user-specified content types. The content type will be set as <code>application/octet-stream</code> unless specified.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromInline("This is my configuration content."))
 *         .build();
 * </pre></blockquote>
 * <p>
 * AWS AppConfig supports the following types of configuration profiles.
 * <p>
 * <ul>
 * <li><strong><a href="https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-configuration-and-profile-feature-flags.html">Feature flag</a></strong>: Use a feature flag configuration to turn on new features that require a timely deployment, such as a product launch or announcement.</li>
 * <li><strong><a href="https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-free-form-configurations-creating.html">Freeform</a></strong>: Use a freeform configuration to carefully introduce changes to your application.</li>
 * </ul>
 * <p>
 * A hosted configuration with type:
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromInlineText("This is my configuration content."))
 *         .type(ConfigurationType.FEATURE_FLAGS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When you create a configuration and configuration profile, you can specify up to two validators. A validator ensures that your
 * configuration data is syntactically and semantically correct. You can create validators in either JSON Schema or as an AWS
 * Lambda function.
 * See <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-configuration-and-profile.html#appconfig-creating-configuration-and-profile-validators">About validators</a> for more information.
 * <p>
 * When you import a JSON Schema validator from a file, you can pass in a relative path.
 * <p>
 * A hosted configuration with validators:
 * <p>
 * <blockquote><pre>
 * Application application;
 * Function fn;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromInlineText("This is my configuration content."))
 *         .validators(List.of(JsonSchemaValidator.fromFile("schema.json"), LambdaValidator.fromFunction(fn)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can attach a deployment strategy (as described in the previous section) to your configuration to specify how you want your
 * configuration to roll out.
 * <p>
 * A hosted configuration with a deployment strategy:
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromInlineText("This is my configuration content."))
 *         .deploymentStrategy(DeploymentStrategy.Builder.create(this, "MyDeploymentStrategy")
 *                 .rolloutStrategy(RolloutStrategy.linear(RolloutStrategyProps.builder()
 *                         .growthFactor(15)
 *                         .deploymentDuration(Duration.minutes(30))
 *                         .finalBakeTime(Duration.minutes(15))
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>deployTo</code> parameter is used to specify which environments to deploy the configuration to.
 * <p>
 * A hosted configuration with <code>deployTo</code>:
 * <p>
 * <blockquote><pre>
 * Application application;
 * Environment env;
 * 
 * 
 * HostedConfiguration.Builder.create(this, "MyHostedConfiguration")
 *         .application(application)
 *         .content(ConfigurationContent.fromInlineText("This is my configuration content."))
 *         .deployTo(List.of(env))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When more than one configuration is set to deploy to the same environment, the
 * deployments will occur one at a time. This is done to satisfy
 * <a href="https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-deploying.html">AppConfig's constraint</a>:
 * <p>
 * <blockquote>
 * <p>
 * [!NOTE]
 * You can only deploy one configuration at a time to an environment.
 * However, you can deploy one configuration each to different environments at the same time.
 * <p>
 * </blockquote>
 * <p>
 * The deployment order matches the order in which the configurations are declared.
 * <p>
 * <blockquote><pre>
 * Application app = new Application(this, "MyApp");
 * Environment env = Environment.Builder.create(this, "MyEnv")
 *         .application(app)
 *         .build();
 * 
 * HostedConfiguration.Builder.create(this, "MyFirstHostedConfig")
 *         .application(app)
 *         .deployTo(List.of(env))
 *         .content(ConfigurationContent.fromInlineText("This is my first configuration content."))
 *         .build();
 * 
 * HostedConfiguration.Builder.create(this, "MySecondHostedConfig")
 *         .application(app)
 *         .deployTo(List.of(env))
 *         .content(ConfigurationContent.fromInlineText("This is my second configuration content."))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If an application would benefit from a deployment order that differs from the
 * declared order, you can defer the decision by using <code>IEnvironment.addDeployment</code>
 * rather than the <code>deployTo</code> property.
 * In this example, <code>firstConfig</code> will be deployed before <code>secondConfig</code>.
 * <p>
 * <blockquote><pre>
 * Application app = new Application(this, "MyApp");
 * Environment env = Environment.Builder.create(this, "MyEnv")
 *         .application(app)
 *         .build();
 * 
 * HostedConfiguration secondConfig = HostedConfiguration.Builder.create(this, "MySecondHostedConfig")
 *         .application(app)
 *         .content(ConfigurationContent.fromInlineText("This is my second configuration content."))
 *         .build();
 * 
 * HostedConfiguration firstConfig = HostedConfiguration.Builder.create(this, "MyFirstHostedConfig")
 *         .application(app)
 *         .deployTo(List.of(env))
 *         .content(ConfigurationContent.fromInlineText("This is my first configuration content."))
 *         .build();
 * 
 * env.addDeployment(secondConfig);
 * </pre></blockquote>
 * <p>
 * Alternatively, you can defer multiple deployments in favor of
 * <code>IEnvironment.addDeployments</code>, which allows you to declare multiple
 * configurations in the order they will be deployed.
 * In this example the deployment order will be
 * <code>firstConfig</code>, then <code>secondConfig</code>, and finally <code>thirdConfig</code>.
 * <p>
 * <blockquote><pre>
 * Application app = new Application(this, "MyApp");
 * Environment env = Environment.Builder.create(this, "MyEnv")
 *         .application(app)
 *         .build();
 * 
 * HostedConfiguration secondConfig = HostedConfiguration.Builder.create(this, "MySecondHostedConfig")
 *         .application(app)
 *         .content(ConfigurationContent.fromInlineText("This is my second configuration content."))
 *         .build();
 * 
 * HostedConfiguration thirdConfig = HostedConfiguration.Builder.create(this, "MyThirdHostedConfig")
 *         .application(app)
 *         .content(ConfigurationContent.fromInlineText("This is my third configuration content."))
 *         .build();
 * 
 * HostedConfiguration firstConfig = HostedConfiguration.Builder.create(this, "MyFirstHostedConfig")
 *         .application(app)
 *         .content(ConfigurationContent.fromInlineText("This is my first configuration content."))
 *         .build();
 * 
 * env.addDeployments(firstConfig, secondConfig, thirdConfig);
 * </pre></blockquote>
 * <p>
 * Any mix of <code>deployTo</code>, <code>addDeployment</code>, and <code>addDeployments</code> is permitted.
 * The declaration order will be respected regardless of the approach used.
 * <p>
 * <blockquote>
 * <p>
 * [!IMPORTANT]
 * If none of these options are utilized, there will not be any deployments.
 * <p>
 * </blockquote>
 * <p>
 * <h3>SourcedConfiguration</h3>
 * <p>
 * A sourced configuration represents configuration stored in any of the following:
 * <p>
 * <ul>
 * <li>Amazon S3 bucket</li>
 * <li>AWS Secrets Manager secret</li>
 * <li>Systems Manager</li>
 * <li>(SSM) Parameter Store parameter</li>
 * <li>SSM document</li>
 * <li>AWS CodePipeline.</li>
 * </ul>
 * <p>
 * A sourced configuration takes in the location source
 * construct and optionally a version number to deploy. On construction of a sourced configuration, the configuration is deployed
 * only if a version number is specified.
 * <p>
 * <h3>S3</h3>
 * <p>
 * Use an Amazon S3 bucket to store a configuration.
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .versioned(true)
 *         .build();
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromBucket(bucket, "path/to/file.json"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use an encrypted bucket:
 * <p>
 * <blockquote><pre>
 * Application application;
 * 
 * 
 * Bucket bucket = Bucket.Builder.create(this, "MyBucket")
 *         .versioned(true)
 *         .encryption(BucketEncryption.KMS)
 *         .build();
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromBucket(bucket, "path/to/file.json"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>AWS Secrets Manager secret</h3>
 * <p>
 * Use a Secrets Manager secret to store a configuration.
 * <p>
 * <blockquote><pre>
 * Application application;
 * Secret secret;
 * 
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromSecret(secret))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>SSM Parameter Store parameter</h3>
 * <p>
 * Use an SSM parameter to store a configuration.
 * <p>
 * <blockquote><pre>
 * Application application;
 * StringParameter parameter;
 * 
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromParameter(parameter))
 *         .versionNumber("1")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>SSM document</h3>
 * <p>
 * Use an SSM document to store a configuration.
 * <p>
 * <blockquote><pre>
 * Application application;
 * CfnDocument document;
 * 
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromCfnDocument(document))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>AWS CodePipeline</h3>
 * <p>
 * Use an AWS CodePipeline pipeline to store a configuration.
 * <p>
 * <blockquote><pre>
 * Application application;
 * Pipeline pipeline;
 * 
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromPipeline(pipeline))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Similar to a hosted configuration, a sourced configuration can optionally take in a type, validators, a <code>deployTo</code> parameter, and a deployment strategy.
 * <p>
 * A sourced configuration with type:
 * <p>
 * <blockquote><pre>
 * Application application;
 * Bucket bucket;
 * 
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromBucket(bucket, "path/to/file.json"))
 *         .type(ConfigurationType.FEATURE_FLAGS)
 *         .name("MyConfig")
 *         .description("This is my sourced configuration from CDK.")
 *         .build();
 * </pre></blockquote>
 * <p>
 * A sourced configuration with validators:
 * <p>
 * <blockquote><pre>
 * Application application;
 * Bucket bucket;
 * Function fn;
 * 
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromBucket(bucket, "path/to/file.json"))
 *         .validators(List.of(JsonSchemaValidator.fromFile("schema.json"), LambdaValidator.fromFunction(fn)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * A sourced configuration with a deployment strategy:
 * <p>
 * <blockquote><pre>
 * Application application;
 * Bucket bucket;
 * 
 * 
 * SourcedConfiguration.Builder.create(this, "MySourcedConfiguration")
 *         .application(application)
 *         .location(ConfigurationSource.fromBucket(bucket, "path/to/file.json"))
 *         .deploymentStrategy(DeploymentStrategy.Builder.create(this, "MyDeploymentStrategy")
 *                 .rolloutStrategy(RolloutStrategy.linear(RolloutStrategyProps.builder()
 *                         .growthFactor(15)
 *                         .deploymentDuration(Duration.minutes(30))
 *                         .finalBakeTime(Duration.minutes(15))
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Extension</h2>
 * <p>
 * An extension augments your ability to inject logic or behavior at different points during the AWS AppConfig workflow of
 * creating or deploying a configuration. You can associate these types of tasks with AWS AppConfig applications, environments, and configuration profiles.
 * See: https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html
 * <p>
 * An extension defines one or more actions, that it performs during an AWS AppConfig workflow. Each action is invoked either when you interact with AWS AppConfig or when AWS AppConfig is performing a process on your behalf. These invocation points are called action points. AWS AppConfig extensions support the following action points:
 * <p>
 * <ul>
 * <li>PRE_START_DEPLOYMENT</li>
 * <li>PRE_CREATE_HOSTED_CONFIGURATION_VERSION</li>
 * <li>ON_DEPLOYMENT_START</li>
 * <li>ON_DEPLOYMENT_STEP</li>
 * <li>ON_DEPLOYMENT_BAKING</li>
 * <li>ON_DEPLOYMENT_COMPLETE</li>
 * <li>ON_DEPLOYMENT_ROLLED_BACK</li>
 * <li>AT_DEPLOYMENT_TICK</li>
 * </ul>
 * <p>
 * See: https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions-about.html
 * <p>
 * <h3>AWS Lambda destination</h3>
 * <p>
 * Use an AWS Lambda as the event destination for an extension.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Extension.Builder.create(this, "MyExtension")
 *         .actions(List.of(
 *             Action.Builder.create()
 *                     .actionPoints(List.of(ActionPoint.ON_DEPLOYMENT_START))
 *                     .eventDestination(new LambdaDestination(fn))
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Lambda extension with parameters:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Extension.Builder.create(this, "MyExtension")
 *         .actions(List.of(
 *             Action.Builder.create()
 *                     .actionPoints(List.of(ActionPoint.ON_DEPLOYMENT_START))
 *                     .eventDestination(new LambdaDestination(fn))
 *                     .build()))
 *         .parameters(List.of(Parameter.required("testParam", "true"), Parameter.notRequired("testNotRequiredParam")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Amazon Simple Queue Service (SQS) destination</h3>
 * <p>
 * Use a queue as the event destination for an extension.
 * <p>
 * <blockquote><pre>
 * Queue queue;
 * 
 * 
 * Extension.Builder.create(this, "MyExtension")
 *         .actions(List.of(
 *             Action.Builder.create()
 *                     .actionPoints(List.of(ActionPoint.ON_DEPLOYMENT_START))
 *                     .eventDestination(new SqsDestination(queue))
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Amazon Simple Notification Service (SNS) destination</h3>
 * <p>
 * Use an SNS topic as the event destination for an extension.
 * <p>
 * <blockquote><pre>
 * Topic topic;
 * 
 * 
 * Extension.Builder.create(this, "MyExtension")
 *         .actions(List.of(
 *             Action.Builder.create()
 *                     .actionPoints(List.of(ActionPoint.ON_DEPLOYMENT_START))
 *                     .eventDestination(new SnsDestination(topic))
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Amazon EventBridge destination</h3>
 * <p>
 * Use the default event bus as the event destination for an extension.
 * <p>
 * <blockquote><pre>
 * IEventBus bus = EventBus.fromEventBusName(this, "MyEventBus", "default");
 * 
 * Extension.Builder.create(this, "MyExtension")
 *         .actions(List.of(
 *             Action.Builder.create()
 *                     .actionPoints(List.of(ActionPoint.ON_DEPLOYMENT_START))
 *                     .eventDestination(new EventBridgeDestination(bus))
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also add extensions and their associations directly by calling <code>onDeploymentComplete()</code> or any other action point
 * method on the AWS AppConfig application, configuration, or environment resource. To add an association to an existing
 * extension, you can call <code>addExtension()</code> on the resource.
 * <p>
 * Adding an association to an AWS AppConfig application:
 * <p>
 * <blockquote><pre>
 * Application application;
 * Extension extension;
 * LambdaDestination lambdaDestination;
 * 
 * 
 * application.addExtension(extension);
 * application.onDeploymentComplete(lambdaDestination);
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.appconfig;
