(ns gorilla-notes.state
  (:require [gorilla-notes.util :refer [next-id vec-remove]]
            [gorilla-notes.defaults :as defaults]))

(def *state (atom {}))

(defn init! []
  (reset! *state {:options        defaults/options
                  :mode->ids      {}
                  :id->content    {}
                  :inputs         {}
                  :input-handlers []}))

(init!)

(defn restrict-modes! [modes]
  (swap! *state
         update :mode->ids #(select-keys % modes)))

(defn ->timestamp []
  (pr-str (java.util.Date.)))

(defn vconj [v x]
  (if v
    (conj v x)
    [x]))

(defn cleanup-content [state]
  (let [existing-ids (->> state
                          :mode->ids
                          vals
                          (apply concat)
                          set)
        obsolete-ids (->> state
                          :id->content
                          keys
                          (filter (complement existing-ids)))]
    (-> state
        (update :id->content #(apply dissoc % obsolete-ids)))))

(defn add-note!
  ([mode extended-hiccup]
   (let [id (next-id)]
     (swap! *state
            (fn [state]
              (-> state
                  (assoc-in [:id->content id] extended-hiccup)
                  (update-in [:mode->ids mode] vconj id)))))))

(defn assoc-note!
  ([mode idx extended-hiccup]
   (let [id (next-id)]
     (swap! *state
            (fn [state]
              (-> state
                  (update :id->content
                          (fn [id->content]
                            (-> id->content
                                (assoc id extended-hiccup))))
                  (update-in [:mode->ids mode] assoc idx id)
                  cleanup-content))))))

(defn remove-note!
  ([mode idx]
   (swap! *state
          (fn [state]
            (-> state
                (update-in [:mode->ids mode] vec-remove idx)
                cleanup-content)))))

(defn drop-tail! [mode n]
  (let [n-remaining (-> @*state :mode->ids (get mode) count (- n) (max 0))]
    (println n-remaining)
    (swap! *state
           (fn [state]
             (-> state
                 (update-in [:mode->ids mode] #(subvec % 0 n-remaining))
                 cleanup-content)))))

(defn reset-notes!
  ([mode]
   (reset-notes! mode {}))
  ([mode
    {:keys [ids-and-content
            str-ids-and-content
            ids
            id->content]
     :or   {ids-and-content []
            str-ids-and-content (map (juxt (comp str first) second)
                                     ids-and-content)
            ids (mapv first str-ids-and-content)
            id->content (into {} str-ids-and-content)}}]
   (->> ids-and-content
        (map first)
        frequencies
        vals
        (every? (partial = 1))
        assert)
   #_(clojure.pprint/pprint [:id->content id->content])
   (swap! *state
          (fn [state]
            (-> state
                (assoc-in [:mode->ids mode] ids)
                (update :id->content merge id->content)
                cleanup-content)))))


;; https://dnaeon.github.io/recursively-merging-maps-in-clojure/
(defn deep-merge
  "Recursively merges maps."
  [& maps]
  (letfn [(m [& xs]
            (if (some #(and (map? %) (not (record? %))) xs)
              (apply merge-with m xs)
              (last xs)))]
    (reduce m maps)))

(defn merge-new-options! [new-options]
  (swap! *state
         update
         :options
         #(deep-merge % new-options)))

(defn toggle-option! [k]
  (swap! *state
         update-in
         [:options k]
         not))

(defn content-ids []
  (:mode->ids @*state))

(defn options []
  (:options @*state))

(defn page-options []
  (:page (options)))

(defn watch-inputs! [handler]
  (swap! *state update :input-handlers #(conj % handler)))

(defn assoc-input! [symbol value]
  (swap! *state assoc-in [:inputs symbol] value)
  (doseq [handler (:input-handlers @*state)]
    (handler symbol value)))

(defn inputs []
  (:inputs @*state))

(defn port []
  (-> @*state :options :port))
