package pl.decerto.hyperon.persistence.service;

import pl.decerto.hyperon.persistence.cache.GmoCacheManager;
import pl.decerto.hyperon.persistence.exception.HyperonPersistenceException;
import pl.decerto.hyperon.persistence.model.def.BundleDef;
import pl.decerto.hyperon.persistence.model.value.Bundle;
import pl.decerto.hyperon.persistence.sync.diff.BundleDiff;

/**
 * This interface acts as main facade service to work with Hyperon Persistence engine functionality. Before it can be used, it is really important to setup
 * configuration with {@link pl.decerto.hyperon.persistence.factory.HyperonPersistenceFactory}.
 *
 * @author przemek hertel
 */
public interface BundleService {

	/**
	 * Saves bundle in tables using database connection defined with {@link pl.decerto.hyperon.persistence.factory.HyperonPersistenceFactory}.
	 * <br><span>There are few cases of persisting bundle and how it will be stored:</span>
	 * <ol>
	 *   <li>if bundle is saved for the first time, it will store root bundle with revision set to 1 and update date to null.</li>
	 *   <li>if bundle was already persisted in the past, then same bundle will be updated with incremented revision number and update date will be set to
	 *   time of saving.</li>
	 * </ol>
	 * All inserts, updates and deletes actions will be generated based on bundle and it's definition. Then used by engine, to invoke them on database.
	 *
	 * @param bundle to be persisted, can't be null
	 */
	void persist(Bundle bundle);

	/**
	 * Retrieves a bundle based on given id for default profile. Default profile might be setup using
	 * {@link pl.decerto.hyperon.persistence.factory.HyperonPersistenceFactory#setDefaultProfile(String)}.
	 * Bundle definition will be used from current Hyperon Studio environment, to which Hyperon Persistence is connected.
	 * Rest is the same as in {@link #load(long, BundleDef)}. Calling this method multiple times for the same id, will always return new Bundle
	 * instance.
	 *
	 * <br>Example:
	 * <pre>{@code
	 *	BundleService service = ..;
	 *	Bundle firstBundle = service.load(12);
	 *	Bundle secondBundle = service.load(12);
	 *
	 *	firstBundle != secondBundle;  // true
	 *	Objects.equals(firstBundle, secondBundle); // true
	 * }</pre>
	 *
	 * @param id bundle id to load
	 * @return loaded bundle instance
	 * @throws HyperonPersistenceException if there was no bundle definition for default profile
	 * @see #load(long, BundleDef)
	 * @see #load(long, String)
	 */
	Bundle load(long id);

	/**
	 * Retrieves a bundle based on given id and bundle definition. This definition is used to:
	 * <ul>
	 * 	<li>find root table name. If table name is not defined in
	 * 		bundle definition, then default table name is used: <i>bundle</i>.
	 *  </li>
	 *  <li>
	 *		fetching bundle from database, based on bundle definition.
	 *  </li>
	 * 	</ul>
	 * Calling this method multiple times for the same id, will always return new Bundle instance.
	 *
	 * <br>Example:
	 * <pre>{@code
	 *	BundleService service = ..;
	 *	Bundle firstBundle = service.load(12);
	 *	Bundle secondBundle = service.load(12);
	 *
	 *	firstBundle != secondBundle;  // true
	 *	Objects.equals(firstBundle, secondBundle); // true
	 * }</pre>
	 *
	 * @param id bundle id to load
	 * @param def bundle definition, can't be null
	 * @return loaded bundle instance
	 * @see pl.decerto.hyperon.persistence.dao.DaoConfig#setBundleTable(String)
	 * @see Bundle#deepcopy()
	 * @see #load(long)
	 * @see #load(long, String)
	 */
	Bundle load(long id, BundleDef def);

	/**
	 * Retrieves a bundle based on given id and profile. Based on profile, bundle definition will be fetched from current Hyperon Studio environment, to
	 * which Hyperon Persistence is connected. Rest is the same as in {@link #load(long, BundleDef)}. Calling this method multiple times for the same id,
	 * will always return new Bundle instance.
	 *
	 * <br>Example:
	 * <pre>{@code
	 * 	BundleService service = ..;
	 * 	Bundle firstBundle = service.load(12);
	 *	Bundle secondBundle = service.load(12);
	 *
	 *	firstBundle != secondBundle;  // true
	 *	Objects.equals(firstBundle, secondBundle); // true
	 * }</pre>
	 *
	 * @param id id bundle id to load
	 * @param profile allows to get current domain definition
	 * @return loaded bundle instance
	 * @throws HyperonPersistenceException if there was no bundle definition for default profile
	 * @see #load(long)
	 * @see #load(long, BundleDef)
	 */
	Bundle load(long id, String profile);

	/**
	 * Based on given profile, bundle definition will be fetched from current Hyperon Studio environment, to which Hyperon Persistence is connected.
	 *
	 * @param profile allows to get current domain definition
	 * @return loaded bundle definition
	 * @throws HyperonPersistenceException if there was no bundle definition for default profile
	 * @see BundleDef
	 */
	BundleDef getDefinition(String profile);

	/**
	 * Creates bundle for the first time, based on default profile. There must be bundle definition on Hyperon Studio environment, to which persistence
	 * engine is connected. Calling this multiple times, each time this will create new bundle instance.
	 *
	 * @return new bundle instance
	 * @throws HyperonPersistenceException if there was no bundle definition for default profile
	 * @see pl.decerto.hyperon.persistence.dao.DaoConfig#setBundleTable(String)
	 */
	Bundle create();

	/**
	 * Creates bundle for the first time, based on given profile. There must be profile and bundle definition on Hyperon Studio environment, to which
	 * persistence engine is connected. Calling this multiple times, each time this will create new bundle instance.
	 *
	 * @param profile allows to get current domain definition
	 * @return new bundle instance
	 * @throws HyperonPersistenceException if there was no bundle definition for default profile
	 * @see pl.decerto.hyperon.persistence.dao.DaoConfig - for default table name
	 */
	Bundle create(String profile);

	/**
	 * Returns cache manager, that is used within persistence engine as first level cache. It stores information about bundles and bundles definitions.
	 * This manager is exposed for custom management control of caches - for more advanced and complicated cases. It allows also to invalidate exposed
	 * caches.
	 * <br>
	 * <b>For most cases, there shouldn't be need to use this directly at all.</b>
	 *
	 * @return instance of cache manager
	 */
	GmoCacheManager getCacheManager();

	/**
	 * <p>
	 * Compares two different bundles and stores the result of comparison in {@link BundleDiff} object. The result will be in form of separated collections of:
	 * elements to insert, to update, to delete.
	 * </p>
	 * Example:
	 * <pre>{@code
	 *  BundleService service = ..;
	 *
	 *  Bundle firstBundle = service.load(11);
	 *  Bundle secondBundle = service.load(22);
	 *
	 *  BundleDiff result = service.diff(firstBundle, secondBundle); // difference between these two bundles
	 *
	 *  --------------
	 *  Bundle firstBundle = service.load(33);
	 *  Bundle secondBundle = service.load(33);
	 *
	 *  BundleDiff result = service.diff(firstBundle, secondBundle); // same bundles, then no differences should be there
	 * }</pre>
	 *
	 * @param prev first bundle, not null
	 * @param next second bundle, not null
	 * @return difference between prev and next bundle
	 * @see BundleDiff
	 */
	BundleDiff diff(Bundle prev, Bundle next);

	/**
	 * <p>Compares given bundle with initial version of the same bundle(by id). Initial version means that it looks for bundle (with the same id) in:</p>
	 * <ul>
	 *     <li>cache - if found, then it is used for comparision</li>
	 *     <li>database - gets object, stores it in cache for later usage and uses it for comparision</li>
	 * </ul>
	 * Result of comparison is stored in {@link BundleDiff} object. It will
	 * be in form of separated collections of: elements to insert, to update, to delete.
	 *
	 * Example:
	 * <pre>{@code
	 *  BundleService service = ..;
	 *
	 *  Bundle bundle = service.load(12);
	 *  bundle.get("policy").remove("product");
	 *
	 *  BundleDiff result = service.diff(bundle); // there will be information about deleted product from policy
	 * }</pre>
	 * <p>
	 *     <b>This method doesn't look for previous versions of provided bundle!</b>
	 * </p>
	 *
	 * @param next bundle, not null
	 * @return difference between given bundle and previous version of this bundle
	 * @see BundleDiff
	 */
	BundleDiff diff(Bundle next);

}
