package pl.decerto.hyperon.persistence.model.value;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;

import org.smartparam.engine.core.type.ValueHolder;
import org.smartparam.engine.types.string.StringHolder;

import pl.decerto.hyperon.persistence.exception.HyperonPersistenceUsageException;
import pl.decerto.hyperon.persistence.model.def.TypeDef;
import pl.decerto.hyperon.runtime.exception.HyperonRuntimeException;

/**
 * This class represents value property, which is a simple wrapper for any value of <i>simple type</i>.
 * These values should be type of any allowed types by {@link pl.decerto.hyperon.runtime.helper.TypeConverter}.
 *
 * @author przemek hertel
 * @see pl.decerto.hyperon.runtime.helper.TypeConverter
 * @see ValueHolder
 * @implNote instead of providing simple types as String, there should be enum of allowed types and/or proper helper static methods.
 */
public class ValueProperty extends Property {

	private ValueHolder value;

	/**
	 * Creates {@link ValueProperty} based on simple type name.
	 *
	 * @param simpleType supported simple type
	 *
	 */
	public ValueProperty(String simpleType) {
		super(simpleType);
	}

	/**
	 * Creates {@link ValueProperty} based on provided {@code type} type definition and prepared {@link ValueHolder} with value.
	 *
	 * @param type definition type
	 * @param holder holder with value
	 */
	public ValueProperty(TypeDef type, ValueHolder holder) {
		super(type);
		setValue(holder);
	}

	/**
	 * Creates {@link ValueProperty} based on simple type name and prepared {@link ValueHolder} with value.
	 *
	 * @param simpleType supported simple type
	 * @param holder holder with value
	 */
	public ValueProperty(String simpleType, ValueHolder holder) {
		this(simpleType);
		setValue(holder);
	}

	/**
	 * Creates {@link ValueProperty} based on simple type and given {@code text} for {@link StringHolder}.
	 *
	 * @param simpleType supported simple type
	 * @param text value for {@link StringHolder}
	 */
	public ValueProperty(String simpleType, String text) {
		this(simpleType);
		setValue(new StringHolder(text));
	}

	/**
	 * @return value holder, that might have null value.
	 */
	public ValueHolder getValue() {
		if (value == null) {
			value = converter.toHolder(null, getSimpleType());
		}
		return value;
	}

	/**
	 * @return direct value from holder or null if there is no holder
	 */
	public Object getObject() {
		return value != null ? value.getValue() : null;
	}

	/**
	 * Sets value holder and state of this property to {@code EntityState#PERSISTENT}.
	 *
	 * @param value for this property
	 */
	public final void setValue(ValueHolder value) {
		this.value = value;
		this.state = EntityState.PERSISTENT;
	}

	/**
	 * @return simple type name from definition
	 * @see TypeDef
	 */
	public String getSimpleType() {
		return getType().getSimpleType();
	}


	/*
	 *  ===========================  abstract methods implementation  =============================
	 */

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Property deepcopy(boolean resetIds) {
		return deepcopyInternal(resetIds);
	}

	/**
	 * @return new instance of {@link ValueProperty} with referenced type and value to previous {@link ValueProperty}
	 */
	@Override
	protected Property deepcopyInternal() {
		return new ValueProperty(type, value);
	}

	/**
	 * Same as {@link #deepcopyInternal()}
	 * @param resetIds does nothing here
	 */
	@Override
	protected Property deepcopyInternal(boolean resetIds) {
		return new ValueProperty(type, value);
	}

	/**
	 * Adds to {@code sb} formatted String, specific for {@link ValueProperty}, which looks like:
	 * <pre>{@code
	 *         * code = CA  (string)
	 *         * premium = 100  (number)
	 *         * su = 10000  (number)
	 * }</pre>
	 *
	 * @param sb StringBuilder containing data to print
	 * @param level level used to indentation
	 * @param prefix value to used as prefix
	 * @param suffix value to used as suffix
	 * @see #toString()
	 */
	@Override
	public void print(StringBuilder sb, int level, String prefix, String suffix) {
		write(sb, level, dot(), prefix, " = ", printValue(), "  (", type, ")");
	}

	/**
	 * @throws HyperonPersistenceUsageException since this method is not supported on
	 */
	@Override
	protected Property removeChild(Property child) {
		throw new HyperonPersistenceUsageException("removing child element is not available for this type", this);
	}


	/*
	 *  ==================================  API implementation   ==================================
	 */

	/**
	 * Sets new <i>simple</i> value, which must be supported by {@link pl.decerto.hyperon.runtime.helper.TypeConverter}.
	 *
	 * @param value new simple value
	 * @throws HyperonRuntimeException if value is not supported
	 */
	@Override
	public void set(Object value) {
		setValue(converter.toHolder(value, getSimpleType()));
	}

	/**
	 * @return value converted to String
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#getString(Object)
	 */
	@Override
	public String getString() {
		return converter.getString(value);
	}

	/**
	 * @return value converted to Integer
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#getInteger(Object)
	 */
	@Override
	public Integer getInteger() {
		return converter.getInteger(value);
	}

	/**
	 * @return value converted to BigDecimal
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#getDecimal(Object)
	 */
	@Override
	public BigDecimal getDecimal() {
		return converter.getDecimal(value);
	}

	/**
	 * @return value converted to boolean
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#getBoolean(Object)
	 */
	@Override
	public boolean booleanValue() {
		return converter.getBoolean(value);
	}

	/**
	 * @return value converted to Boolean
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#toBooleanHolder(Object)
	 */
	@Override
	public Boolean getBoolean() {
		return converter.toBooleanHolder(value).getBoolean();
	}

	/**
	 * @return value converted to Date
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#getDate(Object)
	 */
	@Override
	public Date getDate() {
		return converter.getDate(value);
	}

	/**
	 * @return value converted to Date with time
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#getDatetime(Object)
	 */
	@Override
	public Date getDatetime() {
		return converter.getDatetime(value);
	}

	/**
	 * @return value as {@link ValueHolder}
	 */
	@Override
	public ValueHolder getHolder() {
		return value;
	}

	/**
	 * @return value converted to Date with time
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#getDatetime(Object)
	 */
	@Override
	public Long getLong() {
		return converter.toIntegerHolder(value).getLong();
	}

	/**
	 * @return value converted to double
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#getNumber(Object)
	 */
	@Override
	public double getNumber() {
		return converter.getNumber(value);
	}

	/**
	 * @return value converted to int
	 * @see pl.decerto.hyperon.runtime.helper.TypeConverter#toIntegerHolder(Object)
	 */
	@Override
	public int intValue() {
		return converter.toIntegerHolder(value).intValue();
	}

	/**
	 * @return {@link ElementType#VALUE} element type
	 */
	@Override
	public ElementType getElementType() {
		return ElementType.VALUE;
	}

	/*
	 *  ===================================  ordinary methods   ===================================
	 */

	/**
	 * @return formatted {@code String}
	 * @see #print(StringBuilder, int, String, String)
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		print(sb, 0, name, getPath());
		return sb.toString();
	}

	private String printValue() {
		return value == null ? null : value.getString();
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof ValueProperty)) {
			return false;
		}

		ValueProperty that = (ValueProperty) o;
		return equalValue(value, that.value) && Objects.equals(type, that.type);
	}

	private boolean equalValue(ValueHolder v1, ValueHolder v2) {
		if (v1 == null || v2 == null) {
			return v1 == v2;
		}

		return v1.compareTo(v2) == 0;
	}

	@Override
	public int hashCode() {
		return super.hashCode();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public long getId() {
		return getOwnerId();
	}

	/**
	 * @return {@code true}
	 */
	@Override
	public boolean isValue() {
		return true;
	}

}
