package pl.decerto.hyperon.persistence.model.value;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import pl.decerto.hyperon.persistence.exception.HyperonPersistenceUsageException;
import pl.decerto.hyperon.persistence.model.def.EntityType;
import pl.decerto.hyperon.persistence.model.def.TypeDef;

/**
 * This class represents collection of properties, that should store other properties with the same type {@link TypeDef}.
 * All methods, which are accessing nested list, must follow this rule:
 * {@code 0 <= index < list.size()}.
 * Two collections of type {@link CollectionProperty} are equal in the sense of
 * {@code equals and hashCode}, if their nested {@code list} are equals.
 *
 * @author przemek hertel
 * @see TypeDef
 * @see Property
 * @implNote on each add, set, methods there should be extra check, if added property is the same {@link TypeDef} as collection property type.
 */
public class CollectionProperty extends Property {

	private static final int END_INDEX = -1;

	private final List<Property> list = new ArrayList<>();

	/**
	 * Creates {@link CollectionProperty} for provided {@code type} definition.
	 *
	 * @param type definition
	 * @see TypeDef
	 */
	public CollectionProperty(TypeDef type) {
		super(type);
	}

	/**
	 * Creates {@link CollectionProperty} for provided compound {@code entityType}.
	 *
	 * @param entityType compound type
	 */
	public CollectionProperty(EntityType entityType) {
		super(entityType);
	}

	/**
	 * Attach given {@code property} at the end.
	 * Check if given {@code property} is already in tree, then creates reference to {@code property} and adds it to collection.
	 *
	 * @param property to add
	 * @return collection property on which this method is invoked
	 * @see #add(Property...)
	 * @see #add(int, Property)
	 */
	public CollectionProperty add(Property property) {
		return add(END_INDEX, property);
	}

	/**
	 * Attach given {@code property} at the end.
	 * Check if given {@code property} is already in tree and {@code checkRef == true}, then creates reference to {@code property} and adds it to collection.
	 *
	 * @param property to add
	 * @param checkRef indicates whether add mechanism should check if given property is already in bundle. If yes, then ref property is created
	 * @return collection property on which this method is invoked
	 */
	public CollectionProperty add(Property property, boolean checkRef) {
		return add(END_INDEX, property, checkRef);
	}

	/**
	 * Attach given {@code property} at provided {@code index}.
	 * Check if given {@code property} is already in tree, then creates reference to {@code property} and adds it to collection.
	 *
	 * @param index in which given property should be added
	 * @param property to add
	 * @return collection property on which this method is invoked
	 * @see #add(Property)
	 * @see #add(Property...)
	 */
	public CollectionProperty add(int index, Property property) {
		return add(index, property, true);
	}

	/**
	 * Attach given {@code property} at provided {@code index}.
	 * Check if given {@code property} is already in tree and {@code checkRef == true}, then creates reference to {@code property} and adds it to collection.
	 *
	 * @param index index in which given property should be added
	 * @param property to add
	 * @param checkRef indicates whether add mechanism should check if given property is already in bundle. If yes, then ref property is created
	 * @return collection property on which this method is invoked
	 */
	public CollectionProperty add(int index, Property property, boolean checkRef) {

		Property element = property;

		// use ref proxy if prop is already used in bundle
		if (checkRef && alreadyInTree(property)) {
			element = new RefProperty(property);
		}

		// standard prepare
		element.setName(name);
		element.setParent(getParent());
		element.setOwnerId(ownerId);
		element.setContainer(this);
		element.setBundle(bundle);

		// add property to internal collection (at index or at the end)
		if (index == END_INDEX) {
			list.add(element);

		} else {
			list.add(index, element);
		}

		// scan property
		scan(element);

		return this;
	}

	/**
	 * Attach given {@code property} at provided {@code index}.
	 * Previous node will be removed from collections at {@code index} position.
	 *
	 * @param index index at which property should be replaced
	 * @param property property to put at {@code index}
	 * @return collection property on which this method is invoked
	 * @throws IndexOutOfBoundsException if index is greater then size of collection
	 * @see #add(int, Property)
	 */
	public CollectionProperty set(int index, Property property) {
		replace(index, property);
		return this;
	}

	/**
	 * Removes {@code src} element and set {@code dst} instead of it.
	 *
	 * @param src element to be removed
	 * @param dst element to be added
	 */
	void replace(Property src, Property dst) {

		int srcIx = list.indexOf(src);
		replace(srcIx, dst);
	}

	private void replace(int index, Property prop) {

		// remove previous element from [index]
		Property prev = list.get(index);
		prev.remove();

		// add new element at [index]
		add(index, prop);
	}

	/**
	 * @return list of properties in kept order
	 */
	public List<Property> getList() {
		return list;
	}

	/**
	 * {@inheritDoc}
	 * <br>It also setup bundle for all properties within collection.
	 *
	 * @param bundle root for parameter
	 */
	@Override
	protected void setBundle(Bundle bundle) {
		super.setBundle(bundle);

		// cascade bundle ref to whole subtree
		for (Property prop : list) {
			prop.setBundle(bundle);
		}
	}

	/**
	 * Marks collection property {@code state} to {@link EntityState#PERSISTENT}.
	 */
	public void mark() {
		state = EntityState.PERSISTENT;
	}

	/*
	 *  ==================================  API implementation   ==================================
	 */

	/**
	 * Adds multiple properties in given order using {@link #add(Property)}.
	 *
	 * @param properties to add
	 * @return collection property on which this method is invoked
	 * @see #add(Property)
	 */
	@Override
	public CollectionProperty add(Property... properties) {
		for (Property p : properties) {
			add(p);
		}
		return this;
	}

	/**
	 * @return size of collection
	 */
	@Override
	public int size() {
		return list.size();
	}

	/**
	 * @return new iterator instance
	 */
	@Override
	public Iterator<Property> iterator() {
		return new Itr();
	}

	/**
	 * {@inheritDoc}
	 * <br>
	 * If {@code index} is smaller then 0 or exceeds list size, then exception will be throw.
	 *
	 * @param index index to retrieve element from
	 * @return property from collection
	 * @throws HyperonPersistenceUsageException if index is not available for list
	 */
	@Override
	public Property at(int index) {
		if (index >= 0 && index < list.size()) {
			return list.get(index);
		}

		throw new HyperonPersistenceUsageException("Getting element from index " + index + " but collection size = " + size(), this);
	}

	/**
	 * {@inheritDoc}
	 * <br>
	 * Removes each element and all metadata behind.
	 *
	 * @return {@code this}
	 */
	@Override
	public Property clear() {
		for (Property e : toArray()) {
			e.remove();
		}
		return this;
	}

	/**
	 * {@code value} must be of type {@link CollectionProperty}, then all elements of this value will be added to {@code this} {@link CollectionProperty}
	 * instance. Otherwise exception will be thrown.
	 *
	 * @param value to be used
	 * @throws HyperonPersistenceUsageException if {@code value} is not instance of {@link CollectionProperty}
	 */
	@Override
	public void set(Object value) {

		if (value instanceof CollectionProperty) {
			CollectionProperty coll = (CollectionProperty) value;

			clear();
			for (Property e : coll.list) {
				add(e);
			}

		} else {
			super.set(value);
		}
	}

	/**
	 * Creates new property of {@link EntityProperty} with compound type matching collection supported type.
	 *
	 * @return new property
	 */
	@Override
	public Property create() {
		return new EntityProperty(type.getCompoundType());
	}

	/**
	 * Search for element with id within collection and it's subtree.
	 *
	 * @param id child id
	 * @return property if found or null
	 */
	@Override
	public Property find(long id) {

		// check all elements in collection
		for (Property element : list) {
			Property result = element.find(id);
			if (result != null) {
				return result;
			}
		}

		return null;
	}

	/**
	 * @return {@link ElementType#COLLECTION} element type
	 */
	@Override
	public ElementType getElementType() {
		return ElementType.COLLECTION;
	}


	/*
	 *  ===========================  abstract methods implementation  =============================
	 */

	/**
	 * Creates defensive copies of collection property and it's property elements.
	 * There is possibility to control id's of elements with given {@code resetIds}.
	 *
	 * @param resetIds whether to reset ids on copy
	 * @return new instance of {@link CollectionProperty}
	 */
	@Override
	public Property deepcopy(boolean resetIds) {
		return deepcopyInternal(resetIds);
	}

	/**
	 * Creates defensive copies of collection property and it's property elements.
	 *
	 * @return new instance of {@link CollectionProperty}
	 */
	@Override
	protected Property deepcopyInternal() {
		return deepcopyInternal(false);
	}

	/**
	 * Creates defensive copies of collection property and it's property elements.
	 * There is possibility to control id's of elements with given {@code resetIds}.
	 *
	 * @param resetIds control flag for resetting ids of properties
	 * @return new instance of {@link CollectionProperty}
	 */
	@Override
	protected Property deepcopyInternal(boolean resetIds) {
		CollectionProperty coll = new CollectionProperty(type);
		coll.setName(name);
		coll.setOwnerId(resetIds ? 0 : ownerId);
		coll.setState(state);

		for (Property element : list) {
			if (!element.isRef()) {
				coll.add(element.deepcopyInternal(resetIds), false);
			}
		}

		return coll;
	}

	/**
	 * Resets owner id and reset ids on each property element from {@code list}.
	 */
	@Override
	public void resetIds() {

		// reset this collection
		setOwnerId(0);

		// reset all elements
		for (Property p : list) {
			p.resetIds();
		}
	}

	/**
	 * {@inheritDoc}
	 * Child is searched within {@code list} of this property.
	 *
	 * @param child to be removed
	 * @return removed child or null if not found
	 */
	@Override
	protected Property removeChild(Property child) {
		for (Iterator<Property> it = list.iterator(); it.hasNext(); ) {
			Property p = it.next();
			if (p == child) {
				it.remove();
				return p;
			}
		}

		return null;
	}

	/**
	 * Adds to {@code sb} formatted String, specific for {@link ValueProperty}, which looks like:
	 * <pre>{@code
	 *   * covers  collection(Cover)  size=2
	 *     * covers[0]  (Cover#1003  @21133) refc=1
	 *       * code = CA  (string)
	 *       * premium = 100  (number)
	 *       * su = 10000  (number)
	 *     * covers[1]  (Cover#1004  @45743) refc=1
	 *       * code = CB  (string)
	 *       * premium = 200  (number)
	 *       * su = 20000  (number)
	 * }</pre>
	 *
	 * @param sb StringBuilder containing data to print
	 * @param level level used to indentation
	 * @param prefix value to used as prefix
	 * @param suffix value to used as suffix
	 * @see #toString()
	 */
	@Override
	public void print(StringBuilder sb, int level, String prefix, String suffix) {
		write(sb, level, dot(), prefix, "  collection(", type, ")  size=", list.size());

		for (int i = 0; i < list.size(); i++) {
			Property prop = list.get(i);
			prop.print(sb, level + 1, prefix + "[" + i + "]", suffix);
		}
	}


	/*
	 *  ===================================  ordinary methods   ===================================
	 */

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof CollectionProperty)) {
			return false;
		}
		if (!super.equals(o)) {
			return false;
		}

		CollectionProperty that = (CollectionProperty) o;
		return Objects.equals(list, that.list);
	}

	@Override
	public int hashCode() {
		return 31 * super.hashCode() + Objects.hashCode(list);
	}

	/**
	 * @return owner id
	 */
	@Override
	public long getId() {
		return getOwnerId();
	}

	/**
	 * @return {@code true}
	 */
	@Override
	public boolean isCollection() {
		return true;
	}

	/**
	 * Formatted String as follows:
	 * <pre>{@code
	 * 	CollectionProperty[collection1 (MyType)  size=1]
	 * }</pre>
	 *
	 * @return formatted String
	 */
	@Override
	public String toString() {
		return "CollectionProperty[" + getName() + " (" + type + ")  size=" + list.size() + "]";
	}

	/**
	 * Creates snapshot array
	 * @return array of properties
	 */
	public Property[] toArray() {
		return list.toArray(new Property[0]);
	}

	private void scan(Property p) {
		if (bundle != null) {
			p.traverse(new IdentityScanner(bundle), false);
		}
	}

	/**
	 * Nested iterator implementation using {@code list} of collection property.
	 */
	private class Itr implements Iterator<Property> {

		private final Iterator<Property> it;
		private Property prop;

		private Itr() {
			this.it = list.iterator();
		}

		@Override
		public boolean hasNext() {
			return it.hasNext();
		}

		@Override
		public Property next() {
			prop = it.next();
			return prop;
		}

		@Override
		public void remove() {

			// remove from underlying collection
			it.remove();

			// deep remove, clear identity set
			prop.remove();
		}
	}

}
