package pl.decerto.hyperon.persistence.model.def;

import static pl.decerto.hyperon.persistence.model.def.EntityTypeState.PERSISTENT;
import static pl.decerto.hyperon.persistence.model.def.EntityTypeState.TRANSIENT;

import org.apache.commons.lang3.StringUtils;

import pl.decerto.hyperon.persistence.exception.HyperonPersistenceUsageException;
import pl.decerto.hyperon.persistence.model.value.CollectionProperty;
import pl.decerto.hyperon.persistence.model.value.EntityProperty;
import pl.decerto.hyperon.persistence.model.value.Property;
import pl.decerto.hyperon.persistence.model.value.ValueProperty;

/**
 * @author przemek hertel
 */
public class PropertyDef {

	private TypeDef type;
	private boolean collection;
	private String description;

	private EntityTypeState persistenceState = PERSISTENT;

	public PropertyDef(TypeDef type, boolean collection) {
		this.type = type;
		this.collection = collection;
	}

	public PropertyDef(TypeDef type) {
		this(type, false);
	}

	public PropertyDef(String simpleType) {
		this(TypeDef.buildSimpleType(simpleType));
	}

	public PropertyDef(EntityType entityType, boolean collection) {
		this(TypeDef.buildCompoundType(entityType), collection);
	}

	public PropertyDef(EntityType entityType) {
		this(entityType, false);
	}

	public void setDescription(String description) {
		if (StringUtils.isNotBlank(description)) {
			this.description = description;
		} else {
			this.description = null;
		}
	}

	public void setPersistenceState(EntityTypeState state) {
		this.persistenceState = state != null ? state : PERSISTENT;
	}

	public boolean isTransient() {
		return persistenceState == TRANSIENT;
	}

	public String getDescription() {
		return description;
	}

	public boolean isCollection() {
		return collection;
	}

	public boolean isEntityType() {
		return type.isCompound();
	}

	public boolean isSimpleType() {
		return type.isSimple();
	}

	public TypeDef getType() {
		return type;
	}

	public EntityType getEntityType() {
		if (type.isCompound()) {
			return type.getCompoundType();
		}

		throw new HyperonPersistenceUsageException("getting EntityType from simple-type property", this);
	}

	public String getSimpleType() {
		if (type.isSimple()) {
			return type.getSimpleType();
		}

		throw new HyperonPersistenceUsageException("getting SimpleType from entity-type property", this);
	}

	public PropertyDef getComponentDef() {
		if (isCollection()) {
			return new PropertyDef(type);
		}

		throw new HyperonPersistenceUsageException("getting collection component type from non-collection property", this);
	}

	public Property createProperty() {

		if (isCollection()) {
			return new CollectionProperty(type);
		}

		if (isEntityType()) {
			return new EntityProperty(getEntityType());
		}

		return new ValueProperty(getSimpleType());
	}

	@Override
	public String toString() {
		return "Prop[" + (collection ? "collection/" : "") + type + "]";
	}

	public boolean isCompoundType() {
		return type.isCompound();
	}
}
