package pl.decerto.hyperon.persistence.config;

import static pl.decerto.hyperon.persistence.config.DefinitionKeyword.SPLIT_PATTERN;
import static pl.decerto.hyperon.persistence.config.DefinitionKeyword.TOKEN_COLUMN;
import static pl.decerto.hyperon.persistence.config.DefinitionKeyword.TOKEN_DEF;
import static pl.decerto.hyperon.persistence.config.DefinitionKeyword.TOKEN_DESC;
import static pl.decerto.hyperon.persistence.config.DefinitionKeyword.TOKEN_MULTI;
import static pl.decerto.hyperon.persistence.config.DefinitionKeyword.TOKEN_TABLE;
import static pl.decerto.hyperon.persistence.config.DefinitionKeyword.TOKEN_TRANSIENT;
import static pl.decerto.hyperon.persistence.model.def.EntityTypeState.PERSISTENT;
import static pl.decerto.hyperon.persistence.model.def.EntityTypeState.TRANSIENT;

import java.util.LinkedList;
import java.util.Queue;
import java.util.Scanner;

import org.apache.commons.lang3.StringUtils;

import pl.decerto.hyperon.persistence.model.def.BundleDef;
import pl.decerto.hyperon.persistence.model.def.EntityTypeState;

/**
 * @author przemek hertel
 */
public class DefinitionParser {

	private final DefinitionBuilder builder = new DefinitionBuilder();

	// full text
	private final String text;

	// current line
	private String line;

	// tokenized current line
	private Queue<String> tokens;

	// description found in current line
	private String description;

	// persistence state for current entity
	private EntityTypeState persistenceState = PERSISTENT;


	public DefinitionParser(String text) {
		this.text = text;
	}

	public BundleDef parse() {

		try (Scanner scanner = new Scanner(text)) {

			while (scanner.hasNextLine()) {

				// reset state
				reset();

				// current line
				line = scanner.nextLine();

				// cut description if found
				if (line.contains(TOKEN_DESC)) {
					int ix = line.indexOf(TOKEN_DESC);
					description = line.substring(ix + TOKEN_DESC.length()).trim();
					line = line.substring(0, ix);
				}

				// remove comment if found
				if (line.contains("#")) {
					line = line.substring(0, line.indexOf('#'));
				}

				line = line.trim();

				// remove transient modifier
				if (line.startsWith(TOKEN_TRANSIENT)) {
					line = line.substring(TOKEN_TRANSIENT.length());
					persistenceState = TRANSIENT;
				}

				// def line or field line
				if (!StringUtils.isBlank(line)) {
					parseLine();
				}
			}
		}

		return builder.build();
	}

	private void reset() {
		line = null;
		description = null;
	}

	private void resetPersistenceState() {
		persistenceState = PERSISTENT;
	}

	private void parseLine() {

		// final all tokens in this line
		parseTokens();

		// def or property line
		if (TOKEN_DEF.equals(peek())) {
			startDef();

		} else {
			readProperty();
		}
	}

	/**
	 * current line defines new entity
	 */
	private void startDef() {

		// skip def token
		skip();

		// read name of the entity (type)
		String name = next();

		// optional table name
		String table = null;

		// read tokens until queue is empty
		while (true) {

			String token = next();
			if (token == null) {
				break;
			}

			if (TOKEN_TABLE.equals(token)) {
				table = next();
			}

		}

		// open entity in builder
		if (isNameOfRoot(name)) {
			builder.startEntity(getNameOfRoot(name), table, true);

		} else {
			builder.startEntity(name, table, false);
		}

		if (description != null) {
			builder.setEntityDescription(description);
		}

		// entity persistence state
		builder.setEntityPersistence(persistenceState);
		resetPersistenceState();
	}

	private boolean isNameOfRoot(String name) {
		return name.startsWith("/") || BundleDef.BUNDLE_DEF_NAME.equals(name);
	}

	private String getNameOfRoot(String name) {
		return name.startsWith("/") ? name.substring(1) : name;
	}

	/**
	 * current line defines new property in owning entity
	 */
	private void readProperty() {

		// property name
		String name = next();

		// property type
		boolean collection = false;
		String type = next();

		if (type.endsWith(TOKEN_MULTI)) {
			type = type.substring(0, type.length() - TOKEN_MULTI.length());
			collection = true;
		}

		// open attribute in builder
		builder.setAttr(name, collection, type);

		// read tokens until queue is empty
		while (true) {

			String token = next();
			if (token == null) {
				break;
			}

			if (TOKEN_COLUMN.equals(token)) {
				String column = getColumnName(name);
				builder.setColumnMapping(column, type, name);

			}

		}

		// attach description (may be null)
		builder.setAttrDescription(name, description);

		// set transient state (if configured)
		builder.setAttrPersistence(name, persistenceState);
		resetPersistenceState();
	}

	private String getColumnName(String propertyName) {
		String column = next();
		return column != null ? column : propertyName;
	}

	private void parseTokens() {
		tokens = new LinkedList<>();

		for (String token : SPLIT_PATTERN.split(line)) {
			if (StringUtils.isNotBlank(token)) {
				this.tokens.add(token.trim());
			}
		}
	}

	private String next() {
		return tokens.poll();
	}

	private String peek() {
		return tokens.peek();
	}

	private void skip() {
		tokens.poll();
	}

}
