/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2018
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 *
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.downloader

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import org.gradle.wrapper.PathAssembler
import org.gradle.wrapper.WrapperConfiguration
import org.ysb33r.grolifant.api.ExclusiveFileAccess
import org.ysb33r.grolifant.api.errors.ChecksumFailedException
import org.ysb33r.grolifant.api.errors.DistributionFailedException
import org.ysb33r.grolifant.internal.LegacyLevel

import java.util.concurrent.Callable

import static org.ysb33r.grolifant.api.FileUtils.listDirs
import static org.ysb33r.grolifant.api.UriUtils.safeUri

/**
 * @since 0.8
 */
@CompileStatic
class ArtifactDownloader {

    /** Verifies a checksum
     *
     */
    static interface CheckSumVerification {

        /** Verifies a file against a checksum
         *
         * @param downloadedTarget File that was downloaded
         * @throw Throws {@link ChecksumFailedException} if verification failed
         */
        void verify(final File downloadedTarget)

        /** Returns the checksum in question.
         *
         * @return Checksum. Can be {@code null}.
         */
        String getChecksum()
    }

    /** Verifies that the unpacked artifact root is sane.
     *
     */
    static interface ArtifactRootVerification {
        /** Verifies a artifact root
         *
         * @param unpackedRoot Directory where unpacked package is unpakced (or in case of a single file the parent directory).
         * @return The correctly verified root. (Could be a child of the unpacked root).
         * @throw Throws {@link DistributionFailedException} if verification failed
         */
        File verify(final File unpackedRoot)
    }

    /** Unpacks a downloaded artifact
     *
     */
    static interface ArtifactUnpacker {
        /** Unpacks the source archive
         *
         * @param source Source archive to unpack (Can also be a single non-archive file).
         * @param destDir Destination directory.
         */
        void unpack(File source, File destDir)
    }

    /** Creates an instance which takes care of the actual downloading and caching.
     *
     * @param downloadURI URI to download package from.
     * @param downloadRoot Base directory where to download to.
     * @param basePath Relative path to the downloadRoot.
     * @param verifyArtifactRoot Callback to verify the unpacked artifact. Never {@code null}.
     * @param verifyDownloadChecksum Callback to verify the checksum of the downloaded target.
     *   Can be {@code null}.
     */
    ArtifactDownloader(
        final URI downloadURI,
        final File downloadRoot,
        final String basePath,
        final ArtifactRootVerification verifyArtifactRoot,
        final ArtifactUnpacker unpacker,
        final CheckSumVerification verifyDownloadChecksum
    ) {
        this.downloadURI = downloadURI
        this.downloadRoot = downloadRoot
        this.basePath = basePath
        this.verifyDownloadChecksum = verifyDownloadChecksum
        this.verifyArtifactRoot = verifyArtifactRoot
        this.unpacker = unpacker
    }

    /** Creates a distribution it it does not exist already.
     *
     * @param description Name of the downloaded entity.
     * @param offlineMode Whether to operate in download mode.
     * @param downloadInstance Download & logger instances to use
     *
     * @return Location of distribution
     */
    File getFromCache(final String description, boolean offlineMode, final Downloader downloadInstance) {
        final WrapperConfiguration configuration = getNewWrapperConfiguration()

        final URI distributionUrl = configuration.getDistribution()

        final PathAssembler pathAssembler = new PathAssembler(downloadRoot)
        final PathAssembler.LocalDistribution localDistribution = pathAssembler.getDistribution(configuration)
        final File distDir = localDistribution.distributionDir
        final File localCompressedFile = localDistribution.zipFile

        return exclusiveFileAccessManager.access(localCompressedFile, new Callable<File>() {
            File call() throws Exception {
                final File markerFile = new File(localCompressedFile.getParentFile(), localCompressedFile.getName() + ".ok")
                if (distDir.isDirectory() && markerFile.isFile()) {
                    return verifyArtifactRoot.verify(distDir)
                }

                boolean needsDownload = !localCompressedFile.isFile()
                URI safeDistributionUrl = safeUri(distributionUrl)

                if (needsDownload) {

                    if (offlineMode && distributionUrl.scheme != 'file') {
                        throw new DistributionFailedException("Cannot download ${description} as currently offline")
                    }

                    File tmpCompressedFile = new File(localCompressedFile.getParentFile(), localCompressedFile.getName() + ".part")
                    tmpCompressedFile.delete()
                    downloadInstance.progressLogger.log("Downloading ${safeDistributionUrl}")
                    downloadInstance.downloader.download(distributionUrl, tmpCompressedFile)
                    tmpCompressedFile.renameTo(localCompressedFile)
                }

                List<File> topLevelDirs = listDirs(distDir)
                for (File dir : topLevelDirs) {
                    downloadInstance.progressLogger.log("Deleting directory " + dir.getAbsolutePath())
                    dir.deleteDir()
                }

                if (verifyDownloadChecksum) {
                    verifyDownloadChecksum.verify(localCompressedFile)
                }

                downloadInstance.progressLogger.log("Unpacking " + localCompressedFile.getAbsolutePath() + " to " + distDir.getAbsolutePath())
                unpacker.unpack(localCompressedFile, distDir)


                File root = verifyArtifactRoot.verify(distDir)
                markerFile.createNewFile()

                return root
            }
        })
    }

    private WrapperConfiguration getNewWrapperConfiguration() {
        final WrapperConfiguration configuration = new WrapperConfiguration()
        configuration.distribution = this.downloadURI
        configuration.distributionPath = configuration.zipPath = basePath

        return setConfigChecksum(configuration)
    }

    @CompileDynamic
    private WrapperConfiguration setConfigChecksum(WrapperConfiguration configuration) {
        if (!LegacyLevel.PRE_2_6 && verifyDownloadChecksum) {
            configuration.distributionSha256Sum = verifyDownloadChecksum.checksum
        }

        configuration
    }

    private final ExclusiveFileAccess exclusiveFileAccessManager = new ExclusiveFileAccess(120000, 200)
    private final URI downloadURI
    private final File downloadRoot
    private final String basePath
    private final CheckSumVerification verifyDownloadChecksum
    private final ArtifactUnpacker unpacker
    private final ArtifactRootVerification verifyArtifactRoot


}
