package org.xelevra.architecture.util;

import android.annotation.SuppressLint;
import android.content.SharedPreferences;

import com.google.gson.Gson;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ConcurrentModificationException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public class PreferenceDataProviderFactory{
    private final SharedPreferences preferences;
    private final Gson gson;

    public PreferenceDataProviderFactory(SharedPreferences preferences, Gson gson) {
        this.preferences = preferences;
        this.gson = gson;
    }

    @SuppressWarnings("unchecked")
    public <T> T create(Class<T> targetInterface, String path) {
        Invocation invocation = new Invocation(path, targetInterface);
        T result = (T) Proxy.newProxyInstance(
                targetInterface.getClassLoader(),
                new Class<?>[]{targetInterface},
                invocation
        );
        invocation.realisation = result;
        return result;
    }

    private class Invocation implements InvocationHandler {
        private final String path;
        private final Class targetClass;
        private Object realisation;

        private Lock editorLock = new ReentrantLock(true);
        private SharedPreferences.Editor editor;

        private Invocation(String path, Class targetClass) {
            this.path = path;
            this.targetClass = targetClass;
        }

        @Override
        public Object invoke(Object o, Method method, Object[] objects) throws Throwable {
            if(method.getName().startsWith("get") || method.getName().startsWith("is")){
                if(objects != null && objects.length > 2){
                    throw new IllegalArgumentException("Count of parameters must be 0 or 1 (default value)");
                }
                Class returnType = method.getReturnType();
                String key;
                if(method.getName().startsWith("is")){
                    if(!returnType.equals(Boolean.class) && !returnType.equals(boolean.class)){
                        throw new IllegalArgumentException("Returning type of method " + method.getName() + " must be boolean or " + Boolean.class.getName());
                    }
                    key = path + method.getName().substring(2).toLowerCase();
                } else {
                    key = path + method.getName().substring(3).toLowerCase();
                }
                Object def = null;
                if(objects != null && objects.length == 1){
                    if(!returnType.equals(method.getParameterTypes()[0])){
                        throw new IllegalArgumentException("Type of default value must be equal to returning type." +
                                " Default value type is " + method.getParameterTypes()[0].getName() +
                                " Returning type is " + returnType.getName()
                        );
                    }
                    def = objects[0];
                } else if(objects != null && objects.length == 2){
                    if(!String.class.equals(method.getParameterTypes()[0])){
                        throw new IllegalArgumentException("Suffix must be String");
                    }
                    if(objects[0] == null){
                        throw new IllegalArgumentException("Suffix must not be null");
                    }
                    if(!returnType.equals(method.getParameterTypes()[1])){
                        throw new IllegalArgumentException("Type of default value must be equal to returning type." +
                                " Default value type is " + method.getParameterTypes()[1].getName() +
                                " Returning type is " + returnType.getName()
                        );
                    }
                    def = objects[1];
                    key += objects[0];
                }
                return get(returnType, key, def);
            } else if(method.getName().startsWith("set")){
                if(objects == null || objects.length == 0 || objects.length > 2 ){
                    throw new IllegalArgumentException("Count of parameters must be 1 (value) or 2 (suffix and value)");
                }

                boolean isChained = checkTypeOrThrow(method);
                String key = path + method.getName().substring(3).toLowerCase();
                Object value;

                if(objects.length == 1){
                    value = objects[0];
                } else {
                    if(!String.class.equals(method.getParameterTypes()[0])){
                        throw new IllegalArgumentException("Suffix must be String");
                    }
                    if(objects[0] == null){
                        throw new IllegalArgumentException("Suffix must not be null");
                    }
                    key += objects[0];
                    value = objects[1];
                }
                set(method.getParameterTypes()[0], key, value, method.getAnnotation(Synchronously.class) != null);
                if(isChained) return realisation;
            } else if(method.getName().equals("edit")){
                if(objects != null && objects.length != 0){
                    throw new IllegalArgumentException("Method can't has any parameters");
                }
                boolean isChained = checkTypeOrThrow(method);
                try {
                    editorLock.lock();
                    if(editor != null) throw new ConcurrentModificationException();
                    else editor = preferences.edit();
                } finally {
                    editorLock.unlock();
                }
                if(isChained) return realisation;
            } else if(method.getName().equals("commit")){
                if(objects != null && objects.length > 0){
                    throw new IllegalArgumentException("Method can't has any parameters");
                }
                if(!method.getReturnType().equals(void.class)){
                    throw new IllegalArgumentException("Method must returns void");
                }
                try {
                    editorLock.lock();
                    if(editor == null) throw new IllegalStateException("The object is not in editor mode");
                    editor.commit();
                    editor = null;
                } finally {
                    editorLock.unlock();
                }
            } else if(method.getName().equals("apply")){
                if(objects != null && objects.length > 0){
                    throw new IllegalArgumentException("Method can't has any parameters");
                }
                if(!method.getReturnType().equals(void.class)){
                    throw new IllegalArgumentException("Method must returns void");
                }
                try {
                    editorLock.lock();
                    if(editor == null) throw new IllegalStateException("The object is not in editor mode");
                    editor.apply();
                    editor = null;
                } finally {
                    editorLock.unlock();
                }
            } else if(method.getName().equals("clear")){
                // todo checking
                // todo tests
                SharedPreferences.Editor editor = preferences.edit();
                for (String key : preferences.getAll().keySet()){
                    if(key.startsWith(path)) editor.remove(key);
                }
                editor.commit();
            } else {
                throw new UnsupportedOperationException();
            }
            return null;
        }

        private Object get(Class type, String key, Object def){
            switch (type.getName()){
                case "java.lang.Integer":
                case "int":
                    return preferences.getInt(key, def == null ? 0 : (int) def);
                case "java.lang.Float":
                case "float":
                    return preferences.getFloat(key, def == null ? 0 : (int) def);
                case "java.lang.Long":
                case "long":
                    return preferences.getLong(key, def == null ? 0 : (long) def);
                case "java.lang.Boolean":
                case "boolean":
                    return preferences.getBoolean(key, def == null ? false : (boolean) def);
                case "java.lang.String":
                    return preferences.getString(key, (String) def);
                default:
                    return gson.fromJson(preferences.getString(key, null), type);
            }
        }

        @SuppressLint("CommitPrefEdits")
        private void set(Class type, String key, Object value, boolean synchronously){
            editorLock.lock();
            boolean editorMode = editor != null;
            if(!editorMode) editorLock.unlock();

            SharedPreferences.Editor localEditor = editorMode ? editor : preferences.edit();

            switch (type.getName()){
                case "java.lang.Integer":
                case "int":
                    localEditor.putInt(key, (int) value);
                    break;
                case "java.lang.Float":
                case "float":
                    localEditor.putFloat(key, (float) value);
                    break;
                case "java.lang.Long":
                case "long":
                    localEditor.putLong(key, (long) value);
                    break;
                case "java.lang.Boolean":
                case "boolean":
                    localEditor.putBoolean(key, (boolean) value);
                    break;
                case "java.lang.String":
                    localEditor.putString(key, (String) value);
                    break;
                default:
                    localEditor.putString(key, gson.toJson(value));
            }

            if (editorMode) {
                editorLock.unlock();
            } else {
                if (synchronously) localEditor.commit();
                else localEditor.apply();
            }
        }

        private boolean checkTypeOrThrow(Method method){
            boolean isChained = method.getReturnType().equals(targetClass);
            if(!isChained && !method.getReturnType().equals(void.class)){
                throw new IllegalArgumentException("Method must returns void or " + targetClass.getName());
            }
            return isChained;
        }
    }
}
