package org.xelevra.architecture.base;

import android.os.Bundle;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentTransaction;

import org.xelevra.architecture.util.Easy;

import java.util.HashSet;
import java.util.Random;
import java.util.Set;

/**
 * This is an example fragment stack handler.
 */
public class FragmentStack {
    private FragmentManager manager;
    private int containerId;
    private Random random;
    private FragmentManager.OnBackStackChangedListener listener;

    public static final String SAVE_TAG_FOR_ARGUMENTS = "_SAVE_TAG_FOR_ARGUMENTS";

    public FragmentStack(FragmentManager manager, int containerId) {
        this.manager = manager;
        this.containerId = containerId;
        random = new Random(System.currentTimeMillis());
    }

    public void setOnBackStackChangedListener(FragmentManager.OnBackStackChangedListener listener) {
        if (this.listener != null) manager.removeOnBackStackChangedListener(this.listener);
        this.listener = listener;
        manager.addOnBackStackChangedListener(listener);
    }

    /**
     * Pushes a fragment to the top of the stack.
     */
    public void push(Fragment fragment) {
        FragmentTransaction transaction = manager.beginTransaction();
        String tag = setTag(fragment);
        if (peek() != null)
            fragment.getArguments().putString("_before", peek().getArguments().getString("_tag"));
        transaction.replace(containerId, fragment, tag);
        transaction.addToBackStack(null);
        transaction.commit();
        executePendingTransactions();
    }

    /**
     * Replaces entire stack contents with just one fragment.
     */
    public void replace(Fragment fragment) {
        popAll();
        manager.beginTransaction()
                .replace(containerId, fragment, setTag(fragment))
                .commit();
        executePendingTransactions();
        if (listener != null) listener.onBackStackChanged();
    }

    /**
     * Pops the topmost fragment from the stack.
     */
    public boolean pop() {
        if (manager.getBackStackEntryCount() == 0)
            return false;
        manager.popBackStack();
        return true;
    }


    public void popReturnArgs(Bundle args) {
        String tag = peek().getArguments().getString("_before", null);
        if (tag == null) throw new RuntimeException("Can not find link to previous fragment");
        Fragment pre = manager.findFragmentByTag(tag);
        if (pre == null) throw new RuntimeException("Previous fragment not found");
        Bundle arguments = pre.getArguments();
        if (arguments == null) throw new RuntimeException("Can not set arguments");
        removeSystemTags(args);
        args.putBoolean(SAVE_TAG_FOR_ARGUMENTS, true);
        arguments.putAll(args);
        manager.popBackStack();
    }

    public void popAll() {
        for (int i = manager.getBackStackEntryCount(); i > 0; i--) manager.popBackStackImmediate();
        if(peek() != null) manager.beginTransaction().remove(peek()).commit();
        executePendingTransactions();
        if(listener != null) listener.onBackStackChanged();
    }

    /**
     * Returns the topmost fragment in the stack.
     */
    public Fragment peek() {
        return manager.findFragmentById(containerId);
    }

    private String setTag(Fragment fragment) {
        String tag = random.nextLong() + "" + System.nanoTime();
        if (fragment.getArguments() == null) {
            fragment.setArguments(new Bundle());
        }
        fragment.getArguments().putString("_tag", tag);
        return tag;
    }

    public static void removeSystemTags(Bundle bundle) {
        if (bundle == null) return;
        Set<String> keys = new HashSet<>(bundle.keySet());
        for (String key : keys) if (key.startsWith("_")) bundle.remove(key);
    }

    private void executePendingTransactions(){
        try {
            manager.executePendingTransactions();
        } catch (IllegalStateException e){
            Easy.logE(e.toString());
        }
    }

}
