package org.xelevra.architecture.base;

import android.databinding.DataBindingUtil;
import android.databinding.ViewDataBinding;
import android.os.Bundle;
import android.support.v4.app.Fragment;
import android.support.v4.view.ViewPager;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AlertDialog;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.GridView;
import android.widget.ListView;
import android.widget.TextView;

import org.xelevra.architecture.util.Easy;
import org.xelevra.architecture.R;
import org.xelevra.architecture.util.Toaster;

import java.util.HashMap;
import java.util.Map;

import icepick.Icepick;
import icepick.State;
import rx.Scheduler;
import rx.Subscriber;
import rx.Subscription;
import rx.internal.util.SubscriptionList;
import rx.schedulers.Schedulers;

public abstract class BaseFragment<T extends ViewDataBinding> extends Fragment {
    Toaster toaster;

    protected Scheduler uiScheduler;
    protected int layoutId;
    protected Map<String,View> savable;
    @State
    protected int titleId;
    @State
    protected String title;

    protected int sidebarId;
    private Bundle transactionSavedState;

    private SubscriptionList viewSubscriptions, fragmentSubscriptions;
    private AlertDialog alert;
    private UIExecutor uiExecutor;
    protected ErrorHandler errorHandler;

    private T binding;

    {
        uiExecutor = new UIExecutor();
        uiScheduler = Schedulers.from(uiExecutor);
        super.setArguments(new Bundle());
        savable = new HashMap<>();
        transactionSavedState = new Bundle();
        setRetainInstance(true);
    }

    protected T getBinding(){
        return binding;
    }


    @Override
    public void setArguments(Bundle args) {
        Bundle clone = new Bundle(args);
        FragmentStack.removeSystemTags(clone);
        getArguments().putAll(args);
    }

    public void addFragmentSubscription(Subscription s) {
        fragmentSubscriptions.add(s);
    }

    public void addViewSubscription(Subscription s) {
        viewSubscriptions.add(s);
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        if(layoutId == 0) {
            String resName = "screen_" + this.getClass().getSimpleName().replace("Fragment", "").toLowerCase();
            Easy.logD("Creating fragment view. Name", resName);
            layoutId = getResources().getIdentifier(resName, "layout", getActivity().getPackageName());
            if (layoutId <= 0)
                throw new IllegalStateException("screen name not follow Name Convention or resource " + resName + " not exists");
        }
        binding = DataBindingUtil.inflate(inflater, layoutId, container, false);
        return binding.getRoot();
    }

    protected void showAlert(int message) {
//        dismissAlert();
        alert = new AlertDialog.Builder(getActivity()).setMessage(message).setPositiveButton(android.R.string.ok, null).show();
    }

    protected void showAlert(String message) {
//        dismissAlert();
        alert = new AlertDialog.Builder(getActivity()).setMessage(message).setPositiveButton(android.R.string.ok, null).create();
        alert.show();
    }

    protected void showLargeAlert(CharSequence message){
        ViewGroup view = (ViewGroup) LayoutInflater.from(getActivity()).inflate(R.layout.dialog_message, null, false);
        TextView message_TV = (TextView) view.getChildAt(0);
        alert = new AlertDialog.Builder(getActivity()).setView(view).setPositiveButton(android.R.string.ok, null).create();
        message_TV.setText(message);
        alert.show();
    }

    private void dismissAlert() {
        if (alert != null) {
            alert.dismiss();
            alert = null;
        }
    }

    protected void showToast(String message){
        if(toaster == null) toaster = new Toaster(getActivity().getApplicationContext());
        toaster.toast(message);
    }

    protected void showToast(int resId){
        if(toaster == null) toaster = new Toaster(getActivity().getApplicationContext());
        toaster.toast(resId);
    }

    public FragmentStack getFragmentStack(){
        return ((BaseActivity) getActivity()).getFragmentStack();
    }

    public void onFragmentResult() {
    }

    public void post(Runnable command){
        uiExecutor.executeNextFrame(command);
    }

    public int getSidebarId(){
        return sidebarId;
    }

    @Override
    public void onCreate(Bundle bundle) {
        super.onCreate(bundle);
        fragmentSubscriptions = new SubscriptionList();
    }


    @Override
    public void onResume() {
        super.onResume();
        uiExecutor.ready();
        if (getArguments().getBoolean(FragmentStack.SAVE_TAG_FOR_ARGUMENTS)){
            getArguments().remove(FragmentStack.SAVE_TAG_FOR_ARGUMENTS);
            onFragmentResult();
        }
    }

    private ActionBar getActionBar() {
        return ((BaseActivity) getActivity()).getSupportActionBar();
    }

    @Override
    public void onPause() {
        uiExecutor.pause();
        super.onPause();
    }

    @Override
    public void onSaveInstanceState(Bundle bundle) {
        super.onSaveInstanceState(bundle);
        Icepick.saveInstanceState(this, bundle);
    }

    @Override
    public void onViewCreated(View view, Bundle savedInstanceState) {
        Icepick.restoreInstanceState(this, savedInstanceState);
        viewSubscriptions = new SubscriptionList();
        ActionBar actionBar = ((AppCompatActivity) getActivity()).getSupportActionBar();
        if(title != null) actionBar.setTitle(title);
        else if (titleId != 0) actionBar.setTitle(titleId);
    }

    @Override
    public void onViewStateRestored(Bundle bundle) {
        super.onViewStateRestored(bundle);
        if(!transactionSavedState.isEmpty()){
            restoreViewsStates(transactionSavedState);
        } else if(bundle != null) {
            restoreViewsStates(bundle);
        }
    }

    @Override
    public void onDestroyView() {
        saveViewsStates(transactionSavedState);
        savable.clear();
        super.onDestroyView();
        viewSubscriptions.unsubscribe();
        viewSubscriptions.clear();
        binding = null;
    }

    @Override
    public void onDestroy() {
        fragmentSubscriptions.unsubscribe();
        fragmentSubscriptions.clear();
        super.onDestroy();
    }

    protected void saveViewsStates(Bundle bundle){
        for (Map.Entry<String, View> entry : savable.entrySet()) {
            if (entry.getValue() instanceof ListView) {
                bundle.putInt(entry.getKey(), ((ListView) entry.getValue()).getFirstVisiblePosition());
            } else if(entry.getValue() instanceof RecyclerView){
                bundle.putParcelable(entry.getKey(), ((RecyclerView) entry.getValue()).getLayoutManager().onSaveInstanceState());
            } else if (entry.getValue() instanceof EditText) {
                EditText et = (EditText) entry.getValue();
                bundle.putString(entry.getKey(), et.getText().toString());
                if (et.getError() != null) {
                    String error = et.getError().toString();
                    if (!error.isEmpty()) {
                        bundle.putString(entry.getKey() + "_error", error);
                    }
                }
            } else if(entry.getValue() instanceof CheckBox) {
                bundle.putBoolean(entry.getKey(), ((CheckBox) entry.getValue()).isChecked());
            } else if (entry.getValue() instanceof TextView) {
                bundle.putString(entry.getKey(), ((TextView) entry.getValue()).getText().toString());
            } else if (entry.getValue() instanceof ViewPager){
                bundle.putInt(entry.getKey(), ((ViewPager) entry.getValue()).getCurrentItem());
            } else if (entry.getValue() instanceof GridView) {
                bundle.putInt(entry.getKey(), ((GridView) entry.getValue()).getFirstVisiblePosition());
            }
        }
    }

    protected void restoreViewsStates(Bundle bundle){
        if(getBinding() != null) getBinding().executePendingBindings();

        for (Map.Entry<String, View> entry : savable.entrySet()){
            if(!bundle.containsKey(entry.getKey())) continue;
            if(entry.getValue() instanceof ListView){
                ((ListView) entry.getValue()).setSelection(bundle.getInt(entry.getKey(), 0));
            } else if(entry.getValue() instanceof RecyclerView){
                ((RecyclerView) entry.getValue()).getLayoutManager().onRestoreInstanceState(bundle.getParcelable(entry.getKey()));
            } else if (entry.getValue() instanceof EditText) {
                EditText et = (EditText) entry.getValue();
                et.setText(bundle.getString(entry.getKey()));
                String error = bundle.getString(entry.getKey() + "_error", null);
                if (error != null) et.setError(error);
            } else if(entry.getValue() instanceof CheckBox) {
                ((CheckBox) entry.getValue()).setChecked(bundle.getBoolean(entry.getKey(), false));
            } else if (entry.getValue() instanceof TextView) {
                ((TextView) entry.getValue()).setText(bundle.getString(entry.getKey(), ""));
            } else if (entry.getValue() instanceof ViewPager){
                ((ViewPager) entry.getValue()).setCurrentItem(bundle.getInt(entry.getKey()));
            } else if (entry.getValue() instanceof GridView){
                ((GridView) entry.getValue()).setSelection(bundle.getInt(entry.getKey()));
            } else {
                throw new RuntimeException("Restoring state for " + entry.getValue().getClass().getCanonicalName() + " not implemented");
            }
        }
    }

    public Scheduler getUiScheduler() {
        return uiScheduler;
    }

    private abstract class AutoSubscription<T> extends Subscriber<T> {
        @Override
        public void onCompleted() { }

        @Override
        public void onError(Throwable error) {
            errorHandler.handleError(error, getView());
        }
    }

    protected abstract class ViewSubscription<T> extends AutoSubscription<T> {
        { viewSubscriptions.add(this); }
    }

    protected abstract class FragmentSubscription<T> extends AutoSubscription<T>{
        { fragmentSubscriptions.add(this); }
    }
}
