package org.xelevra.architecture.base;

import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.provider.MediaStore;
import android.support.v7.app.AppCompatActivity;
import android.view.MenuItem;

import org.xelevra.architecture.util.Easy;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;

import icepick.Icepick;
import icepick.State;
import rx.Observable;
import rx.Scheduler;
import rx.Subscriber;
import rx.internal.util.SubscriptionList;
import rx.schedulers.Schedulers;
import rx.subjects.PublishSubject;
import rx.subjects.Subject;

public abstract class BaseActivity extends AppCompatActivity {
    public static final int CODE_GET_PHOTO_FROM_CAMERA =        Integer.parseInt("0000000100000000", 2);

    private FragmentStack fragmentStack;
    private final SubscriptionList subscriptions;
    private final UIExecutor uiExecutor;
    private Subject<Bundle, Bundle> photos;
    protected final Scheduler uiScheduler;

    @State
    protected int fragmentsLayoutId;
    @State
    Bundle imageCaptureData;

    {
        uiExecutor = new UIExecutor();
        uiScheduler = Schedulers.from(uiExecutor);
        subscriptions = new SubscriptionList();
        photos = PublishSubject.<Bundle>create().toSerialized();
    }

    public boolean takePhoto(Bundle data){
        Intent intent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);
        if(intent.resolveActivity(getPackageManager()) == null) {
            Easy.logE("Cannot find activity for camera");
            return false;
        }
        if(data == null) imageCaptureData = new Bundle();
        else imageCaptureData = data;
        try {
            File file = createImageFile();
            intent.putExtra(MediaStore.EXTRA_OUTPUT, Uri.fromFile(file));
            imageCaptureData.putString("file_name", file.getAbsolutePath());
        } catch (IOException e) {
            Easy.logE(e);
            return false;
        }
        startActivityForResult(intent, CODE_GET_PHOTO_FROM_CAMERA);
        return true;
    }

    public Observable<Bundle> observeCamera(){
        return photos.asObservable();
    }

    public FragmentStack getFragmentStack() {
        return fragmentStack;
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        Icepick.restoreInstanceState(this, savedInstanceState);
        if(fragmentsLayoutId == 0) throw new IllegalArgumentException("Define fragmentsLayoutId");
        fragmentStack = new FragmentStack(getSupportFragmentManager(), fragmentsLayoutId);
    }

    @Override
    protected void onPostCreate(Bundle savedInstanceState) {
        super.onPostCreate(savedInstanceState);
        uiExecutor.ready();
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        switch (item.getItemId()) {
            case android.R.id.home:
                onBackPressed();
                return true;
        }
        return super.onOptionsItemSelected(item);
    }

    protected void onSaveInstanceState(Bundle outState) {
        Icepick.saveInstanceState(this, outState);
        super.onSaveInstanceState(outState);
    }

        @Override
    public void onBackPressed() {
        if(getFragmentStack().peek() instanceof BackPressedListener) {
            if(((BackPressedListener) getFragmentStack().peek()).allowBackPress()) super.onBackPressed();
        } else {
            super.onBackPressed();
        }
    }

    @Override
    protected void onDestroy() {
        uiExecutor.pause();
        subscriptions.unsubscribe();
        subscriptions.clear();
        super.onDestroy();
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if(requestCode == CODE_GET_PHOTO_FROM_CAMERA){
            if(resultCode == RESULT_OK){
                photos.onNext(imageCaptureData);
            }
            imageCaptureData = null;
        }
    }

    private File createImageFile() throws IOException {
        // Create an image file name
        String timeStamp = new SimpleDateFormat("yyyyMMdd_HHmmss").format(new Date());
        String imageFileName = "JPEG_" + timeStamp + "_";
        File storageDir = Environment.getExternalStoragePublicDirectory(
                Environment.DIRECTORY_PICTURES);
        return File.createTempFile(
                imageFileName,
                ".jpg",
                storageDir
        );
    }

    protected abstract class ActivitySubscription<T> extends Subscriber<T> {
        {
            subscriptions.add(this);
        }

        @Override
        public void onCompleted() {
        }

        @Override
        public void onError(Throwable e) {
            Easy.logE(e);
        }
    }
}
