/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.http.impl;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.function.Supplier;
import javax.security.auth.DestroyFailedException;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.AuthorizeCallback;
import javax.security.sasl.RealmCallback;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.auth.callback.AuthenticationCompleteCallback;
import org.wildfly.security.auth.callback.AvailableRealmsCallback;
import org.wildfly.security.auth.callback.CredentialCallback;
import org.wildfly.security.credential.PasswordCredential;
import org.wildfly.security.http.HttpAuthenticationException;
import org.wildfly.security.http.HttpServerAuthenticationMechanism;
import org.wildfly.security.http.HttpServerMechanismsResponder;
import org.wildfly.security.http.HttpServerRequest;
import org.wildfly.security.http.HttpServerResponse;
import org.wildfly.security.http.impl.NonceManager;
import org.wildfly.security.mechanism.AuthenticationMechanismException;
import org.wildfly.security.mechanism.digest.DigestQuote;
import org.wildfly.security.mechanism.digest.DigestUtil;
import org.wildfly.security.password.TwoWayPassword;
import org.wildfly.security.password.interfaces.DigestPassword;
import org.wildfly.security.password.spec.DigestPasswordAlgorithmSpec;
import org.wildfly.security.util.ByteIterator;

final class DigestAuthenticationMechanism
implements HttpServerAuthenticationMechanism {
    private static final String CHALLENGE_PREFIX = "Digest ";
    private static final String OPAQUE_VALUE = "00000000000000000000000000000000";
    private static final byte COLON = 58;
    private final Supplier<Provider[]> providers;
    private final CallbackHandler callbackHandler;
    private final NonceManager nonceManager;
    private final String configuredRealm;
    private final String domain;

    DigestAuthenticationMechanism(CallbackHandler callbackHandler, NonceManager nonceManager, String configuredRealm, String domain, Supplier<Provider[]> providers) {
        this.callbackHandler = callbackHandler;
        this.nonceManager = nonceManager;
        this.configuredRealm = configuredRealm;
        this.domain = domain;
        this.providers = providers;
    }

    @Override
    public String getMechanismName() {
        return "DIGEST";
    }

    @Override
    public void evaluateRequest(HttpServerRequest request) throws HttpAuthenticationException {
        List<String> authorizationValues = request.getRequestHeaderValues("Authorization");
        if (authorizationValues != null) {
            for (String current : authorizationValues) {
                if (!current.startsWith(CHALLENGE_PREFIX)) continue;
                byte[] rawHeader = current.substring(CHALLENGE_PREFIX.length()).getBytes(StandardCharsets.UTF_8);
                try {
                    HashMap<String, byte[]> responseTokens = DigestUtil.parseResponse(rawHeader, StandardCharsets.UTF_8, false, ElytronMessages.httpDigest);
                    this.validateResponse(responseTokens, request);
                    return;
                }
                catch (AuthenticationMechanismException e) {
                    ElytronMessages.httpDigest.trace("Failed to parse or validate the response", e);
                    request.badRequest(e.toHttpAuthenticationException(), response -> this.prepareResponse(this.selectRealm(), response, false));
                    return;
                }
            }
        }
        request.noAuthenticationInProgress(response -> this.prepareResponse(this.selectRealm(), response, false));
    }

    private void validateResponse(HashMap<String, byte[]> responseTokens, HttpServerRequest request) throws AuthenticationMechanismException, HttpAuthenticationException {
        MessageDigest messageDigest;
        int nonceCount;
        String nonce = this.convertToken("nonce", responseTokens.get("nonce"));
        String messageRealm = this.convertToken("realm", responseTokens.get("realm"));
        if (!responseTokens.containsKey("nc")) {
            nonceCount = -1;
        } else {
            String nonceCountHex = this.convertToken("realm", responseTokens.get("nc"));
            nonceCount = Integer.parseInt(nonceCountHex, 16);
            if (nonceCount < 0) {
                throw ElytronMessages.httpDigest.invalidNonceCount(nonceCount);
            }
        }
        final byte[] salt = messageRealm.getBytes(StandardCharsets.UTF_8);
        boolean nonceValid = this.nonceManager.useNonce(nonce, salt, nonceCount);
        String username = this.convertToken("username", responseTokens.get("username"));
        if (!responseTokens.containsKey("uri")) {
            throw ElytronMessages.httpDigest.mechMissingDirective("uri");
        }
        byte[] digestUri = responseTokens.get("uri");
        if (!responseTokens.containsKey("response")) {
            throw ElytronMessages.httpDigest.mechMissingDirective("response");
        }
        byte[] response = ByteIterator.ofBytes(responseTokens.get("response")).hexDecode().drain();
        String algorithm = this.convertToken("algorithm", responseTokens.get("algorithm"));
        if (!"MD5".equals(algorithm)) {
            throw ElytronMessages.httpDigest.mechUnsupportedAlgorithm(algorithm);
        }
        try {
            messageDigest = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw ElytronMessages.httpDigest.mechMacAlgorithmNotSupported(e);
        }
        if (!this.checkRealm(messageRealm)) {
            throw ElytronMessages.httpDigest.mechDisallowedClientRealm(messageRealm);
        }
        String selectedRealm = this.selectRealm();
        if (username.length() == 0) {
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.authenticationFailed(), httpResponse -> this.prepareResponse(selectedRealm, httpResponse, false));
            return;
        }
        byte[] hA1 = this.getH_A1(messageDigest, username, messageRealm);
        if (hA1 == null) {
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.authenticationFailed(), httpResponse -> this.prepareResponse(selectedRealm, httpResponse, false));
            return;
        }
        byte[] calculatedResponse = this.calculateResponseDigest(messageDigest, hA1, nonce, request.getRequestMethod(), digestUri, responseTokens.get("qop"), responseTokens.get("cnonce"), responseTokens.get("nc"));
        if (!Arrays.equals(response, calculatedResponse)) {
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.mechResponseTokenMismatch(), httpResponse -> this.prepareResponse(selectedRealm, httpResponse, false));
            return;
        }
        if (!nonceValid) {
            request.authenticationInProgress(httpResponse -> this.prepareResponse(selectedRealm, httpResponse, true));
            return;
        }
        if (this.authorize(username)) {
            this.succeed();
            if (nonceCount < 0) {
                request.authenticationComplete(new HttpServerMechanismsResponder(){

                    @Override
                    public void sendResponse(HttpServerResponse response) throws HttpAuthenticationException {
                        DigestAuthenticationMechanism.this.sendAuthenticationInfoHeader(response, salt);
                    }
                });
            } else {
                request.authenticationComplete();
            }
        } else {
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.authorizationFailed(username), httpResponse -> httpResponse.setStatusCode(403));
        }
    }

    private void sendAuthenticationInfoHeader(HttpServerResponse response, byte[] salt) {
        String nextNonce = this.nonceManager.generateNonce(salt);
        response.addResponseHeader("Authentication-Info", "nextnonce=\"" + nextNonce + "\"");
    }

    private boolean checkRealm(String realm) throws AuthenticationMechanismException {
        String[] realms = this.getAvailableRealms();
        if (realms != null) {
            for (String current : realms) {
                if (!realm.equals(current)) continue;
                return true;
            }
        }
        return false;
    }

    private byte[] calculateResponseDigest(MessageDigest messageDigest, byte[] hA1, String nonce, String method, byte[] digestUri, byte[] qop, byte[] cnonce, byte[] nc) {
        messageDigest.update(method.getBytes(StandardCharsets.UTF_8));
        messageDigest.update((byte)58);
        byte[] hA2 = messageDigest.digest(digestUri);
        messageDigest.update(ByteIterator.ofBytes(hA1).hexEncode().drainToString().getBytes(StandardCharsets.UTF_8));
        messageDigest.update((byte)58);
        messageDigest.update(nonce.getBytes(StandardCharsets.UTF_8));
        if (qop != null) {
            messageDigest.update((byte)58);
            messageDigest.update(nc);
            messageDigest.update((byte)58);
            messageDigest.update(cnonce);
            messageDigest.update((byte)58);
            messageDigest.update(qop);
        }
        messageDigest.update((byte)58);
        return messageDigest.digest(ByteIterator.ofBytes(hA2).hexEncode().drainToString().getBytes(StandardCharsets.UTF_8));
    }

    private byte[] getH_A1(MessageDigest messageDigest, String username, String messageRealm) throws AuthenticationMechanismException {
        NameCallback nameCallback = new NameCallback("User name", username);
        RealmCallback realmCallback = new RealmCallback("User realm", messageRealm);
        byte[] response = null;
        response = this.getPredigestedSaltedPassword(realmCallback, nameCallback, "digest-md5");
        if (response != null) {
            return response;
        }
        response = this.getSaltedPasswordFromTwoWay(messageDigest, realmCallback, nameCallback);
        if (response != null) {
            return response;
        }
        response = this.getSaltedPasswordFromPasswordCallback(messageDigest, realmCallback, nameCallback);
        return response;
    }

    private String convertToken(String name, byte[] value) throws AuthenticationMechanismException {
        if (value == null) {
            throw ElytronMessages.httpDigest.mechMissingDirective(name);
        }
        return new String(value, StandardCharsets.UTF_8);
    }

    private String selectRealm() throws HttpAuthenticationException {
        try {
            if (this.configuredRealm != null) {
                if (!this.checkRealm(this.configuredRealm)) {
                    throw ElytronMessages.httpDigest.digestMechanismInvalidRealm(this.configuredRealm);
                }
                return this.configuredRealm;
            }
            String[] realms = this.getAvailableRealms();
            if (realms != null && realms.length > 0) {
                return realms[0];
            }
            throw ElytronMessages.httpDigest.digestMechanismRequireRealm();
        }
        catch (AuthenticationMechanismException e) {
            throw e.toHttpAuthenticationException();
        }
    }

    private String[] getAvailableRealms() throws AuthenticationMechanismException {
        AvailableRealmsCallback availableRealmsCallback = new AvailableRealmsCallback();
        try {
            this.callbackHandler.handle(new Callback[]{availableRealmsCallback});
            return availableRealmsCallback.getRealmNames();
        }
        catch (UnsupportedCallbackException ignored) {
            return new String[0];
        }
        catch (AuthenticationMechanismException e) {
            throw e;
        }
        catch (IOException e) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(e);
        }
    }

    private void prepareResponse(String realmName, HttpServerResponse response, boolean stale) throws HttpAuthenticationException {
        StringBuilder sb = new StringBuilder(CHALLENGE_PREFIX);
        sb.append("realm").append("=\"").append(DigestQuote.quote(realmName)).append("\"");
        if (this.domain != null) {
            sb.append(", ").append("domain").append("=\"").append(this.domain).append("\"");
        }
        sb.append(", ").append("nonce").append("=\"").append(this.nonceManager.generateNonce(realmName.getBytes(StandardCharsets.UTF_8))).append("\"");
        sb.append(", ").append("opaque").append("=\"").append(OPAQUE_VALUE).append("\"");
        if (stale) {
            sb.append(", ").append("stale").append("=true");
        }
        sb.append(", ").append("algorithm").append("=").append("MD5");
        sb.append(", ").append("qop").append("=").append("auth");
        response.addResponseHeader("WWW-Authenticate", sb.toString());
        response.setStatusCode(401);
    }

    private byte[] getPredigestedSaltedPassword(RealmCallback realmCallback, NameCallback nameCallback, String passwordAlgorithm) throws AuthenticationMechanismException {
        String realmName = realmCallback.getDefaultText();
        String userName = nameCallback.getDefaultName();
        DigestPasswordAlgorithmSpec parameterSpec = realmName != null && userName != null ? new DigestPasswordAlgorithmSpec(userName, realmName) : null;
        CredentialCallback credentialCallback = new CredentialCallback(PasswordCredential.class, passwordAlgorithm, parameterSpec);
        try {
            this.callbackHandler.handle(new Callback[]{realmCallback, nameCallback, credentialCallback});
            return credentialCallback.applyToCredential(PasswordCredential.class, c -> c.getPassword().castAndApply(DigestPassword.class, DigestPassword::getDigest));
        }
        catch (UnsupportedCallbackException e) {
            if (e.getCallback() == credentialCallback) {
                return null;
            }
            if (e.getCallback() == nameCallback) {
                throw ElytronMessages.httpDigest.mechCallbackHandlerDoesNotSupportUserName(e);
            }
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(e);
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
    }

    private byte[] getSaltedPasswordFromTwoWay(MessageDigest messageDigest, RealmCallback realmCallback, NameCallback nameCallback) throws AuthenticationMechanismException {
        CredentialCallback credentialCallback = new CredentialCallback(PasswordCredential.class, "clear");
        try {
            this.callbackHandler.handle(new Callback[]{realmCallback, nameCallback, credentialCallback});
        }
        catch (UnsupportedCallbackException e) {
            if (e.getCallback() == credentialCallback) {
                return null;
            }
            if (e.getCallback() == nameCallback) {
                throw ElytronMessages.httpDigest.mechCallbackHandlerDoesNotSupportUserName(e);
            }
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(e);
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
        TwoWayPassword password = credentialCallback.applyToCredential(PasswordCredential.class, c -> c.getPassword().castAs(TwoWayPassword.class));
        char[] passwordChars = DigestUtil.getTwoWayPasswordChars(password, this.providers, ElytronMessages.httpDigest);
        try {
            password.destroy();
        }
        catch (DestroyFailedException e) {
            ElytronMessages.httpDigest.credentialDestroyingFailed(e);
        }
        String realm = realmCallback.getDefaultText();
        String username = nameCallback.getDefaultName();
        byte[] digest_urp = DigestUtil.userRealmPasswordDigest(messageDigest, username, realm, passwordChars);
        Arrays.fill(passwordChars, '\u0000');
        return digest_urp;
    }

    private byte[] getSaltedPasswordFromPasswordCallback(MessageDigest messageDigest, RealmCallback realmCallback, NameCallback nameCallback) throws AuthenticationMechanismException {
        PasswordCallback passwordCallback = new PasswordCallback("User password", false);
        try {
            this.callbackHandler.handle(new Callback[]{realmCallback, nameCallback, passwordCallback});
        }
        catch (UnsupportedCallbackException e) {
            if (e.getCallback() == passwordCallback) {
                return null;
            }
            if (e.getCallback() == nameCallback) {
                throw ElytronMessages.httpDigest.mechCallbackHandlerDoesNotSupportUserName(e);
            }
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(e);
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
        char[] passwordChars = passwordCallback.getPassword();
        passwordCallback.clearPassword();
        if (passwordChars == null) {
            throw ElytronMessages.httpDigest.mechNoPasswordGiven();
        }
        String realm = realmCallback.getDefaultText();
        String username = nameCallback.getDefaultName();
        byte[] digest_urp = DigestUtil.userRealmPasswordDigest(messageDigest, username, realm, passwordChars);
        Arrays.fill(passwordChars, '\u0000');
        return digest_urp;
    }

    private boolean authorize(String username) throws AuthenticationMechanismException {
        AuthorizeCallback authorizeCallback = new AuthorizeCallback(username, username);
        try {
            this.callbackHandler.handle(new Callback[]{authorizeCallback});
            return authorizeCallback.isAuthorized();
        }
        catch (UnsupportedCallbackException e) {
            return false;
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
    }

    private void succeed() throws AuthenticationMechanismException {
        try {
            this.callbackHandler.handle(new Callback[]{AuthenticationCompleteCallback.SUCCEEDED});
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
    }

    private void fail() throws AuthenticationMechanismException {
        try {
            this.callbackHandler.handle(new Callback[]{AuthenticationCompleteCallback.FAILED});
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
    }
}

