/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.mechanism.scram;

import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.NoSuchElementException;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Supplier;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.mechanism.AuthenticationMechanismException;
import org.wildfly.security.mechanism.MechanismUtil;
import org.wildfly.security.mechanism.scram.ScramFinalClientMessage;
import org.wildfly.security.mechanism.scram.ScramFinalServerMessage;
import org.wildfly.security.mechanism.scram.ScramInitialClientMessage;
import org.wildfly.security.mechanism.scram.ScramInitialServerMessage;
import org.wildfly.security.mechanism.scram.ScramMechanism;
import org.wildfly.security.mechanism.scram.ScramServerErrorCode;
import org.wildfly.security.mechanism.scram.ScramUtil;
import org.wildfly.security.password.interfaces.ScramDigestPassword;
import org.wildfly.security.password.spec.IteratedSaltedPasswordAlgorithmSpec;
import org.wildfly.security.sasl.util.StringPrep;
import org.wildfly.security.util.ByteIterator;
import org.wildfly.security.util.ByteStringBuilder;
import org.wildfly.security.util.DecodeException;

public final class ScramClient {
    private final Supplier<Provider[]> providers;
    private final ScramMechanism mechanism;
    private final String authorizationId;
    private final CallbackHandler callbackHandler;
    private final SecureRandom secureRandom;
    private final byte[] bindingData;
    private final String bindingType;
    private final int minimumIterationCount;
    private final int maximumIterationCount;

    ScramClient(ScramMechanism mechanism, String authorizationId, CallbackHandler callbackHandler, SecureRandom secureRandom, byte[] bindingData, String bindingType, int minimumIterationCount, int maximumIterationCount, Supplier<Provider[]> providers) {
        this.mechanism = mechanism;
        this.authorizationId = authorizationId;
        this.callbackHandler = callbackHandler;
        this.secureRandom = secureRandom;
        this.bindingData = bindingData;
        this.bindingType = bindingType;
        this.minimumIterationCount = minimumIterationCount;
        this.maximumIterationCount = maximumIterationCount;
        this.providers = providers;
    }

    Random getRandom() {
        return this.secureRandom != null ? this.secureRandom : ThreadLocalRandom.current();
    }

    public ScramMechanism getMechanism() {
        return this.mechanism;
    }

    public String getAuthorizationId() {
        return this.authorizationId;
    }

    public String getBindingType() {
        return this.bindingType;
    }

    byte[] getRawBindingData() {
        return this.bindingData;
    }

    public byte[] getBindingData() {
        byte[] bindingData = this.bindingData;
        return bindingData == null ? null : (byte[])bindingData.clone();
    }

    public ScramInitialClientMessage getInitialResponse() throws AuthenticationMechanismException {
        boolean binding;
        NameCallback nameCallback = this.authorizationId == null || this.authorizationId.length() == 0 ? new NameCallback("User name") : new NameCallback("User name", this.authorizationId);
        try {
            MechanismUtil.handleCallbacks(this.mechanism.toString(), this.callbackHandler, nameCallback);
        }
        catch (UnsupportedCallbackException e) {
            throw ElytronMessages.log.mechCallbackHandlerDoesNotSupportUserName(this.mechanism.toString(), e);
        }
        String name = nameCallback.getName();
        if (name == null) {
            throw ElytronMessages.log.mechNoLoginNameGiven(this.mechanism.toString());
        }
        ByteStringBuilder encoded = new ByteStringBuilder();
        if (this.bindingData != null) {
            binding = true;
            if (this.mechanism.isPlus()) {
                encoded.append("p=");
                encoded.append(this.bindingType);
                encoded.append(',');
            } else {
                encoded.append("y,");
            }
        } else {
            binding = false;
            encoded.append("n,");
        }
        if (this.authorizationId != null) {
            encoded.append('a').append('=');
            StringPrep.encode(this.authorizationId, encoded, 1073758207L);
        }
        encoded.append(',');
        int initialPartIndex = encoded.length();
        encoded.append('n').append('=');
        StringPrep.encode(name, encoded, 1073758207L);
        encoded.append(',').append('r').append('=');
        byte[] nonce = ScramUtil.generateNonce(48, this.getRandom());
        encoded.append(nonce);
        return new ScramInitialClientMessage(this, name, binding, nonce, initialPartIndex, encoded.toArray());
    }

    public ScramInitialServerMessage parseInitialServerMessage(ScramInitialClientMessage initialResponse, byte[] bytes) throws AuthenticationMechanismException {
        int iterationCount;
        byte[] salt;
        byte[] serverNonce;
        byte[] challenge = (byte[])bytes.clone();
        ByteIterator bi = ByteIterator.ofBytes(challenge);
        try {
            if (bi.next() != 114 || bi.next() != 61) {
                throw ElytronMessages.log.mechInvalidMessageReceived(this.mechanism.toString());
            }
            byte[] clientNonce = initialResponse.getRawNonce();
            if (!bi.limitedTo(clientNonce.length).contentEquals(ByteIterator.ofBytes(clientNonce))) {
                throw ElytronMessages.log.mechNoncesDoNotMatch(this.mechanism.toString());
            }
            serverNonce = bi.delimitedBy(44).drain();
            bi.next();
            if (bi.next() != 115 || bi.next() != 61) {
                throw ElytronMessages.log.mechInvalidMessageReceived(this.mechanism.toString());
            }
            salt = bi.delimitedBy(44).base64Decode().drain();
            bi.next();
            if (bi.next() != 105 || bi.next() != 61) {
                throw ElytronMessages.log.mechInvalidMessageReceived(this.mechanism.toString());
            }
            iterationCount = ScramUtil.parsePosInt(bi);
            if (iterationCount < this.minimumIterationCount) {
                throw ElytronMessages.log.mechIterationCountIsTooLow(this.mechanism.toString(), iterationCount, this.minimumIterationCount);
            }
            if (iterationCount > this.maximumIterationCount) {
                throw ElytronMessages.log.mechIterationCountIsTooHigh(this.mechanism.toString(), iterationCount, this.maximumIterationCount);
            }
        }
        catch (NumberFormatException | NoSuchElementException | DecodeException ex) {
            throw ElytronMessages.log.mechInvalidMessageReceived(this.mechanism.toString());
        }
        return new ScramInitialServerMessage(initialResponse, serverNonce, salt, iterationCount, challenge);
    }

    public ScramFinalClientMessage handleInitialChallenge(ScramInitialClientMessage initialResponse, ScramInitialServerMessage initialChallenge) throws AuthenticationMechanismException {
        boolean trace = ElytronMessages.log.isTraceEnabled();
        if (initialResponse.getMechanism() != this.mechanism) {
            throw ElytronMessages.log.mechUnmatchedMechanism(this.mechanism.toString(), initialResponse.getMechanism().toString());
        }
        if (initialChallenge.getMechanism() != this.mechanism) {
            throw ElytronMessages.log.mechUnmatchedMechanism(this.mechanism.toString(), initialChallenge.getMechanism().toString());
        }
        boolean plus = this.mechanism.isPlus();
        ByteStringBuilder encoded = new ByteStringBuilder();
        encoded.append('c').append('=');
        ByteStringBuilder b2 = new ByteStringBuilder();
        if (this.bindingData != null) {
            if (trace) {
                ElytronMessages.log.tracef("[C] Binding data: %s%n", ByteIterator.ofBytes(this.bindingData).hexEncode().drainToString());
            }
            if (plus) {
                b2.append("p=");
                b2.append(this.bindingType);
            } else {
                b2.append('y');
            }
            b2.append(',');
            if (this.getAuthorizationId() != null) {
                b2.append("a=");
                StringPrep.encode(this.getAuthorizationId(), b2, 1073758207L);
            }
            b2.append(',');
            if (plus) {
                b2.append(this.bindingData);
            }
            encoded.appendLatin1(b2.iterate().base64Encode());
        } else {
            b2.append('n');
            b2.append(',');
            if (this.getAuthorizationId() != null) {
                b2.append("a=");
                StringPrep.encode(this.getAuthorizationId(), b2, 1073758207L);
            }
            b2.append(',');
            assert (!plus);
            encoded.appendLatin1(b2.iterate().base64Encode());
        }
        encoded.append(',').append('r').append('=').append(initialResponse.getRawNonce()).append(initialChallenge.getRawServerNonce());
        IteratedSaltedPasswordAlgorithmSpec parameters = new IteratedSaltedPasswordAlgorithmSpec(initialChallenge.getIterationCount(), initialChallenge.getRawSalt());
        ScramDigestPassword password = MechanismUtil.getPasswordCredential(initialResponse.getAuthenticationName(), this.callbackHandler, ScramDigestPassword.class, this.mechanism.getPasswordAlgorithm(), parameters, parameters, this.providers);
        byte[] saltedPassword = password.getDigest();
        if (trace) {
            ElytronMessages.log.tracef("[C] Client salted password: %s", ByteIterator.ofBytes(saltedPassword).hexEncode().drainToString());
        }
        try {
            Mac mac = Mac.getInstance(this.getMechanism().getHmacName());
            MessageDigest messageDigest = MessageDigest.getInstance(this.getMechanism().getMessageDigestName());
            mac.init(new SecretKeySpec(saltedPassword, mac.getAlgorithm()));
            byte[] clientKey = mac.doFinal(ScramUtil.CLIENT_KEY_BYTES);
            if (trace) {
                ElytronMessages.log.tracef("[C] Client key: %s", ByteIterator.ofBytes(clientKey).hexEncode().drainToString());
            }
            byte[] storedKey = messageDigest.digest(clientKey);
            if (trace) {
                ElytronMessages.log.tracef("[C] Stored key: %s%n", ByteIterator.ofBytes(storedKey).hexEncode().drainToString());
            }
            mac.init(new SecretKeySpec(storedKey, mac.getAlgorithm()));
            byte[] initialResponseBytes = initialResponse.getRawMessageBytes();
            mac.update(initialResponseBytes, initialResponse.getInitialPartIndex(), initialResponseBytes.length - initialResponse.getInitialPartIndex());
            if (trace) {
                ElytronMessages.log.tracef("[C] Using client first message: %s%n", ByteIterator.ofBytes(initialResponseBytes, initialResponse.getInitialPartIndex(), initialResponseBytes.length - initialResponse.getInitialPartIndex()).hexEncode().drainToString());
            }
            mac.update((byte)44);
            mac.update(initialChallenge.getRawMessageBytes());
            if (trace) {
                ElytronMessages.log.tracef("[C] Using server first message: %s%n", ByteIterator.ofBytes(initialChallenge.getRawMessageBytes()).hexEncode().drainToString());
            }
            mac.update((byte)44);
            encoded.updateMac(mac);
            if (trace) {
                ElytronMessages.log.tracef("[C] Using client final message without proof: %s%n", ByteIterator.ofBytes(encoded.toArray()).hexEncode().drainToString());
            }
            byte[] clientProof = mac.doFinal();
            if (trace) {
                ElytronMessages.log.tracef("[C] Client signature: %s%n", ByteIterator.ofBytes(clientProof).hexEncode().drainToString());
            }
            ScramUtil.xor(clientProof, clientKey);
            if (trace) {
                ElytronMessages.log.tracef("[C] Client proof: %s%n", ByteIterator.ofBytes(clientProof).hexEncode().drainToString());
            }
            int proofStart = encoded.length();
            encoded.append(',').append('p').append('=');
            encoded.appendLatin1(ByteIterator.ofBytes(clientProof).base64Encode());
            if (trace) {
                ElytronMessages.log.tracef("[C] Client final message: %s%n", ByteIterator.ofBytes(encoded.toArray()).hexEncode().drainToString());
            }
            return new ScramFinalClientMessage(initialResponse, initialChallenge, password, clientProof, encoded.toArray(), proofStart);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw ElytronMessages.log.mechMacAlgorithmNotSupported(this.mechanism.toString(), e);
        }
    }

    public ScramFinalServerMessage parseFinalServerMessage(byte[] messageBytes) throws AuthenticationMechanismException {
        byte[] sig;
        ByteIterator bi = ByteIterator.ofBytes(messageBytes);
        try {
            int c = bi.next();
            if (c == 101) {
                if (bi.next() == 61) {
                    throw ElytronMessages.log.scramServerRejectedAuthentication(this.mechanism.toString(), ScramServerErrorCode.fromErrorString(bi.delimitedBy(44).asUtf8String().drainToString()));
                }
                throw ElytronMessages.log.mechInvalidMessageReceived(this.mechanism.toString());
            }
            if (c != 118 || bi.next() != 61) {
                throw ElytronMessages.log.mechInvalidMessageReceived(this.mechanism.toString());
            }
            sig = bi.delimitedBy(44).base64Decode().drain();
            if (bi.hasNext()) {
                throw ElytronMessages.log.mechInvalidMessageReceived(this.mechanism.toString());
            }
        }
        catch (IllegalArgumentException e) {
            throw ElytronMessages.log.mechInvalidMessageReceived(this.mechanism.toString());
        }
        return new ScramFinalServerMessage(sig, messageBytes);
    }

    public void verifyFinalChallenge(ScramFinalClientMessage finalResponse, ScramFinalServerMessage finalChallenge) throws AuthenticationMechanismException {
        boolean trace = ElytronMessages.log.isTraceEnabled();
        try {
            Mac mac = Mac.getInstance(this.getMechanism().getHmacName());
            ScramDigestPassword password = finalResponse.getPassword();
            mac.init(new SecretKeySpec(password.getDigest(), mac.getAlgorithm()));
            byte[] serverKey = mac.doFinal(ScramUtil.SERVER_KEY_BYTES);
            if (trace) {
                ElytronMessages.log.tracef("[C] Server key: %s%n", ByteIterator.ofBytes(serverKey).hexEncode().drainToString());
            }
            mac.init(new SecretKeySpec(serverKey, mac.getAlgorithm()));
            byte[] clientFirstMessage = finalResponse.getInitialResponse().getRawMessageBytes();
            int bareStart = finalResponse.getInitialResponse().getInitialPartIndex();
            mac.update(clientFirstMessage, bareStart, clientFirstMessage.length - bareStart);
            mac.update((byte)44);
            byte[] serverFirstMessage = finalResponse.getInitialChallenge().getRawMessageBytes();
            mac.update(serverFirstMessage);
            mac.update((byte)44);
            byte[] clientFinalMessage = finalResponse.getRawMessageBytes();
            mac.update(clientFinalMessage, 0, finalResponse.getProofOffset());
            byte[] serverSignature = mac.doFinal();
            if (trace) {
                ElytronMessages.log.tracef("[C] Recovered server signature: %s%n", ByteIterator.ofBytes(serverSignature).hexEncode().drainToString());
            }
            if (!Arrays.equals(finalChallenge.getRawServerSignature(), serverSignature)) {
                throw ElytronMessages.log.mechServerAuthenticityCannotBeVerified(this.mechanism.toString());
            }
        }
        catch (IllegalArgumentException | InvalidKeyException | NoSuchAlgorithmException e) {
            throw ElytronMessages.log.mechMacAlgorithmNotSupported(this.mechanism.toString(), e);
        }
    }
}

