/*
 * All content copyright (c) Terracotta, Inc., except as may otherwise be noted in a separate copyright notice. All
 * rights reserved.
 */
package org.terracotta.collections;

import com.tc.object.bytecode.Clearable;
import com.tc.object.bytecode.ManagerUtil;
import com.tc.object.bytecode.NotClearable;
import com.tc.object.bytecode.NullManager;
import com.tc.util.Assert;

import java.util.AbstractCollection;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.TimeUnit;

class ConcurrentDistributedMapDsoArray<K, V> implements ConcurrentDistributedMapBackend<K, V>, NotClearable {

  // using ArrayList because it's faster then using an Array.
  private final List<ConcurrentDistributedMapDso<K, V>> storeList;
  private final int                                     segmentShift;
  private final int                                     segmentMask;

  private transient Random                              rndm;

  public ConcurrentDistributedMapDsoArray(final LockType lockType, final LockStrategy<? super K> lockStrategy,
                                          final int concurrency) {
    // ensure that DSO is active at construction time, ie. fail fast concurrency
    Assert.pre(!(ManagerUtil.getManager() instanceof NullManager));

    int sshift = 0;
    int ssize = 1;
    while (ssize < concurrency) {
      ++sshift;
      ssize <<= 1;
    }
    segmentShift = 32 - sshift;
    segmentMask = ssize - 1;

    this.storeList = new ArrayList<ConcurrentDistributedMapDso<K, V>>(ssize);

    if (storeList instanceof Clearable) {
      // ArrayList isn't currently clearable but it might be someday
      ((Clearable) storeList).setEvictionEnabled(false);
    }

    for (int i = 0; i < ssize; i++) {
      this.storeList.add(i, new ConcurrentDistributedMapDso<K, V>(lockType, lockStrategy));
    }

    onLoad();
  }

  private void onLoad() {
    rndm = new Random();
  }

  /**
   * Applies a supplemental hash function to a given hashCode, which defends against poor quality hash functions.
   */
  private static int hash(int h) {
    h += (h << 15) ^ 0xffffcd7d;
    h ^= (h >>> 10);
    h += (h << 3);
    h ^= (h >>> 6);
    h += (h << 2) + (h << 14);
    return h ^ (h >>> 16);
  }

  private int getIndexFromKey(final Object key) {
    int hash = hash(key.hashCode());
    return (hash >>> segmentShift) & segmentMask;
  }

  private ConcurrentDistributedMapDso<K, V> getMapFromKey(final Object key) {
    int index = getIndexFromKey(key);
    return storeList.get(index);
  }

  public Collection<java.util.Map.Entry<K, V>> getAllEntriesSnapshot() {
    return new AggregateEntriesSnapshotCollection();
  }

  public Collection<java.util.Map.Entry<K, V>> getAllLocalEntriesSnapshot() {
    return new AggregateLocalEntriesSnapshotCollection();
  }

  public void putNoReturn(final K key, final V value) {
    getMapFromKey(key).putNoReturn(key, value);
  }
  
  public void unlockedPutNoReturn(final K key, final V value) {
    getMapFromKey(key).unlockedPutNoReturn(key, value);
  }
  
  public V unlockedGet(final K key) {
    return getMapFromKey(key).unlockedGet(key);
  }

  public void removeNoReturn(final K key) {
    getMapFromKey(key).removeNoReturn(key);
  }
  
  public void unlockedRemoveNoReturn(final K key) {
    getMapFromKey(key).unlockedRemoveNoReturn(key);
  }

  public V putIfAbsent(final K key, final V value) {
    return getMapFromKey(key).putIfAbsent(key, value);
  }

  public boolean remove(final Object key, final Object value) {
    return getMapFromKey(key).remove(key, value);
  }

  public V replace(final K key, final V value) {
    return getMapFromKey(key).replace(key, value);
  }

  public boolean replace(final K key, final V oldValue, final V newValue) {
    return getMapFromKey(key).replace(key, oldValue, newValue);
  }

  public void clear() {
    for (ConcurrentDistributedMapDso<K, V> e : storeList) {
      e.clear();
    }
  }

  public boolean containsKey(final Object key) {
    return getMapFromKey(key).containsKey(key);
  }

  public boolean containsValue(final Object value) {
    for (ConcurrentDistributedMapDso<K, V> e : storeList) {
      if (e.containsValue(value)) { return true; }
    }
    return false;
  }

  public Set<java.util.Map.Entry<K, V>> entrySet() {
    return new AggregateEntrySet();
  }

  public V get(final Object key) {
    return getMapFromKey(key).get(key);
  }

  public V unsafeGet(final K key) {
    return getMapFromKey(key).unsafeGet(key);
  }

  public boolean isEmpty() {
    for (ConcurrentDistributedMapDso<K, V> e : storeList) {
      if (!e.isEmpty()) { return false; }
    }
    return true;
  }

  public Set<K> keySet() {
    return new AggregateKeySet();
  }

  public V put(final K key, final V value) {
    return getMapFromKey(key).put(key, value);
  }

  public void putAll(final Map<? extends K, ? extends V> m) {
    for (Object element : m.entrySet()) {
      Map.Entry<? extends K, ? extends V> entry = (java.util.Map.Entry<? extends K, ? extends V>) element;
      getMapFromKey(entry.getKey()).put(entry.getKey(), entry.getValue());
    }
  }

  public V remove(final Object key) {
    return getMapFromKey(key).remove(key);
  }

  public int size() {
    int size = 0;
    for (ConcurrentDistributedMapDso<K, V> e : storeList) {
      size += e.size();
    }
    return size;
  }

  public int localSize() {
    int localSize = 0;
    for (ConcurrentDistributedMapDso<K, V> e : storeList) {
      localSize += e.localSize();
    }
    return localSize;
  }

  public Collection<V> values() {
    return new AggregatedValuesCollection();
  }

  public FinegrainedLock createFinegrainedLock(final K key) {
    return getMapFromKey(key).createFinegrainedLock(key);
  }

  public void lockEntry(final K key) {
    getMapFromKey(key).lockEntry(key);
  }

  public void unlockEntry(final K key) {
    getMapFromKey(key).unlockEntry(key);
  }

  public String getLockIdForKey(final K key) {
    return getMapFromKey(key).getLockIdForKey(key);
  }

  public List<Map<K, ?>> getConstituentMaps() {
    return Collections.<Map<K, ?>> unmodifiableList(storeList);
  }

  public Map.Entry<K, V> getRandomEntry() {
    ConcurrentDistributedMapDso<K, V> random = storeList.get(rndm.nextInt(storeList.size()));
    if (!random.isEmpty()) {
      Map.Entry<K, V> entry = random.getRandomEntry();
      if (entry != null) { return entry; }
    }

    for (ConcurrentDistributedMapDso<K, V> e : storeList) {
      if (!e.isEmpty()) {
        Map.Entry<K, V> entry = e.getRandomEntry();
        if (entry != null) { return entry; }
      }
    }
    return null;
  }

  public Map.Entry<K, V> getRandomLocalEntry() {
    ConcurrentDistributedMapDso<K, V> random = storeList.get(rndm.nextInt(storeList.size()));
    if (random.localSize() != 0) {
      Map.Entry<K, V> entry = random.getRandomLocalEntry();
      if (entry != null) { return entry; }
    }

    for (ConcurrentDistributedMapDso<K, V> e : storeList) {
      if (e.localSize() != 0) {
        Map.Entry<K, V> entry = e.getRandomLocalEntry();
        if (entry != null) { return entry; }
      }
    }

    return null;
  }

  public boolean flush(final Object key, final Object value) {
    return getMapFromKey(key).flush(key, value);
  }

  public boolean tryRemove(final Object key, final long time, final TimeUnit unit) {
    return getMapFromKey(key).tryRemove(key, time, unit);
  }

  public MapSizeListener registerMapSizeListener(final MapSizeListener newListener) {
    MapSizeListener old = null;
    for (ConcurrentDistributedMapBackend e : storeList) {
      old = e.registerMapSizeListener(newListener);
    }
    return old;
  }

  private class AggregateKeySet extends BaseAggregateSet<K> {

    @Override
    public boolean contains(final Object o) {
      return ConcurrentDistributedMapDsoArray.this.containsKey(o);
    }

    @Override
    public boolean remove(final Object o) {
      return ConcurrentDistributedMapDsoArray.this.remove(o) != null;
    }

    @Override
    public Iterator<K> iterator() {
      return new AggregateSetIterator<K>() {

        @Override
        protected Iterator<K> getNextIterator() {
          return this.listIterator.next().keySet().iterator();
        }

      };
    }

  }

  private class AggregateEntrySet extends BaseAggregateSet<Map.Entry<K, V>> {

    @Override
    public Iterator<Map.Entry<K, V>> iterator() {
      return new AggregateSetIterator<Map.Entry<K, V>>() {

        @Override
        protected Iterator<java.util.Map.Entry<K, V>> getNextIterator() {
          return this.listIterator.next().entrySet().iterator();
        }

      };
    }

    @Override
    public boolean contains(final Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry<K, V> e = (Map.Entry<K, V>) o;
      V value = ConcurrentDistributedMapDsoArray.this.get(e.getKey());
      return value != null && value.equals(e.getValue());
    }

    @Override
    public boolean remove(final Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry<K, V> e = (Map.Entry<K, V>) o;
      V value = ConcurrentDistributedMapDsoArray.this.get(e.getKey());
      if (value != null && value.equals(e.getValue())) {
        return ConcurrentDistributedMapDsoArray.this.remove(e.getKey()) != null;
      } else {
        return false;
      }
    }

  }

  private class AggregatedValuesCollection extends AbstractCollection<V> {

    @Override
    public Iterator<V> iterator() {
      return new AggregateSetIterator<V>() {

        @Override
        protected Iterator<V> getNextIterator() {
          return this.listIterator.next().values().iterator();
        }

      };
    }

    @Override
    public int size() {
      return ConcurrentDistributedMapDsoArray.this.size();
    }

  }

  private abstract class BaseAggregateSet<T> extends AbstractSet<T> {

    @Override
    public int size() {
      return ConcurrentDistributedMapDsoArray.this.size();
    }

    @Override
    public void clear() {
      ConcurrentDistributedMapDsoArray.this.clear();
    }
  }

  private abstract class AggregateSetIterator<T> implements Iterator<T> {

    protected final Iterator<ConcurrentDistributedMapDso<K, V>> listIterator;
    protected Iterator<T>                                       currentIterator;

    protected abstract Iterator<T> getNextIterator();

    public AggregateSetIterator() {
      this.listIterator = storeList.iterator();
      while (listIterator.hasNext()) {
        currentIterator = getNextIterator();
        if (currentIterator.hasNext()) { return; }
      }
    }

    public boolean hasNext() {

      if (this.currentIterator == null) return false;
      boolean hasNext = false;

      if (this.currentIterator.hasNext()) {
        hasNext = true;
      } else {
        while (this.listIterator.hasNext()) {
          this.currentIterator = getNextIterator();
          if (this.currentIterator.hasNext()) { return true; }
        }
      }
      return hasNext;
    }

    public T next() {
      T item = null;

      if (currentIterator == null) return null;

      if (currentIterator.hasNext()) {
        item = currentIterator.next();

      } else {
        while (listIterator.hasNext()) {
          currentIterator = getNextIterator();

          if (currentIterator.hasNext()) {
            item = currentIterator.next();
            break;
          }
        }
      }

      return item;
    }

    public void remove() {
      currentIterator.remove();
    }

  }

  private class AggregateLocalEntriesSnapshotCollection extends AbstractCollection<Map.Entry<K, V>> {

    @Override
    public Iterator<java.util.Map.Entry<K, V>> iterator() {
      return new AggregateSetIterator<Entry<K, V>>() {
        @Override
        protected Iterator<java.util.Map.Entry<K, V>> getNextIterator() {
          return this.listIterator.next().getAllLocalEntriesSnapshot().iterator();
        }
      };
    }

    @Override
    public int size() {
      long size = 0;
      for (ConcurrentDistributedMapDso map : ConcurrentDistributedMapDsoArray.this.storeList) {
        size += map.getAllLocalEntriesSnapshot().size();
        if (size > Integer.MAX_VALUE) { return Integer.MAX_VALUE; }
      }

      return (int) size;
    }

    @Override
    public boolean contains(final Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry<K, V> e = (Map.Entry<K, V>) o;
      ConcurrentDistributedMapDso map = getMapFromKey(e.getKey());
      return map.getAllLocalEntriesSnapshot().contains(o);
    }

    @Override
    public boolean remove(final Object o) {
      throw new UnsupportedOperationException();
    }
  }

  private class AggregateEntriesSnapshotCollection extends AbstractCollection<Map.Entry<K, V>> {

    @Override
    public Iterator<java.util.Map.Entry<K, V>> iterator() {
      return new AggregateSetIterator<Entry<K, V>>() {
        @Override
        protected Iterator<java.util.Map.Entry<K, V>> getNextIterator() {
          return this.listIterator.next().getAllEntriesSnapshot().iterator();
        }
      };
    }

    @Override
    public int size() {
      return ConcurrentDistributedMapDsoArray.this.size();
    }

    @Override
    public boolean contains(final Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry<K, V> e = (Map.Entry<K, V>) o;
      ConcurrentDistributedMapDso map = getMapFromKey(e.getKey());
      return map.getAllEntriesSnapshot().contains(o);
    }

    @Override
    public boolean remove(final Object o) {
      throw new UnsupportedOperationException();
    }
  }
}
