/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.session.data.redis;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.core.NestedExceptionUtils;
import org.springframework.dao.NonTransientDataAccessException;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.connection.MessageListener;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.util.ByteUtils;
import org.springframework.session.DelegatingIndexResolver;
import org.springframework.session.FindByIndexNameSessionRepository;
import org.springframework.session.FlushMode;
import org.springframework.session.IndexResolver;
import org.springframework.session.MapSession;
import org.springframework.session.PrincipalNameIndexResolver;
import org.springframework.session.SaveMode;
import org.springframework.session.Session;
import org.springframework.session.data.redis.RedisSessionExpirationPolicy;
import org.springframework.session.events.SessionCreatedEvent;
import org.springframework.session.events.SessionDeletedEvent;
import org.springframework.session.events.SessionExpiredEvent;
import org.springframework.util.Assert;

public class RedisIndexedSessionRepository
implements FindByIndexNameSessionRepository<RedisSession>,
MessageListener {
    private static final Log logger = LogFactory.getLog(RedisIndexedSessionRepository.class);
    private static final String SPRING_SECURITY_CONTEXT = "SPRING_SECURITY_CONTEXT";
    public static final int DEFAULT_DATABASE = 0;
    public static final String DEFAULT_NAMESPACE = "spring:session";
    private int database = 0;
    private String namespace = "spring:session:";
    private String sessionCreatedChannelPrefix;
    private byte[] sessionCreatedChannelPrefixBytes;
    private String sessionDeletedChannel;
    private byte[] sessionDeletedChannelBytes;
    private String sessionExpiredChannel;
    private byte[] sessionExpiredChannelBytes;
    private String expiredKeyPrefix;
    private byte[] expiredKeyPrefixBytes;
    private final RedisOperations<Object, Object> sessionRedisOperations;
    private final RedisSessionExpirationPolicy expirationPolicy;
    private ApplicationEventPublisher eventPublisher = event -> {};
    private Integer defaultMaxInactiveInterval;
    private IndexResolver<Session> indexResolver = new DelegatingIndexResolver(new IndexResolver[]{new PrincipalNameIndexResolver()});
    private RedisSerializer<Object> defaultSerializer = new JdkSerializationRedisSerializer();
    private FlushMode flushMode = FlushMode.ON_SAVE;
    private SaveMode saveMode = SaveMode.ON_SET_ATTRIBUTE;

    public RedisIndexedSessionRepository(RedisOperations<Object, Object> sessionRedisOperations) {
        Assert.notNull(sessionRedisOperations, (String)"sessionRedisOperations cannot be null");
        this.sessionRedisOperations = sessionRedisOperations;
        this.expirationPolicy = new RedisSessionExpirationPolicy(sessionRedisOperations, this::getExpirationsKey, this::getSessionKey);
        this.configureSessionChannels();
    }

    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        Assert.notNull((Object)applicationEventPublisher, (String)"applicationEventPublisher cannot be null");
        this.eventPublisher = applicationEventPublisher;
    }

    public void setDefaultMaxInactiveInterval(int defaultMaxInactiveInterval) {
        this.defaultMaxInactiveInterval = defaultMaxInactiveInterval;
    }

    public void setIndexResolver(IndexResolver<Session> indexResolver) {
        Assert.notNull(indexResolver, (String)"indexResolver cannot be null");
        this.indexResolver = indexResolver;
    }

    public void setDefaultSerializer(RedisSerializer<Object> defaultSerializer) {
        Assert.notNull(defaultSerializer, (String)"defaultSerializer cannot be null");
        this.defaultSerializer = defaultSerializer;
    }

    public void setFlushMode(FlushMode flushMode) {
        Assert.notNull((Object)flushMode, (String)"flushMode cannot be null");
        this.flushMode = flushMode;
    }

    public void setSaveMode(SaveMode saveMode) {
        Assert.notNull((Object)saveMode, (String)"saveMode must not be null");
        this.saveMode = saveMode;
    }

    public void setDatabase(int database) {
        this.database = database;
        this.configureSessionChannels();
    }

    private void configureSessionChannels() {
        this.sessionCreatedChannelPrefix = this.namespace + "event:" + this.database + ":created:";
        this.sessionCreatedChannelPrefixBytes = this.sessionCreatedChannelPrefix.getBytes();
        this.sessionDeletedChannel = "__keyevent@" + this.database + "__:del";
        this.sessionDeletedChannelBytes = this.sessionDeletedChannel.getBytes();
        this.sessionExpiredChannel = "__keyevent@" + this.database + "__:expired";
        this.sessionExpiredChannelBytes = this.sessionExpiredChannel.getBytes();
        this.expiredKeyPrefix = this.namespace + "sessions:expires:";
        this.expiredKeyPrefixBytes = this.expiredKeyPrefix.getBytes();
    }

    public RedisOperations<Object, Object> getSessionRedisOperations() {
        return this.sessionRedisOperations;
    }

    public void save(RedisSession session) {
        session.save();
        if (session.isNew) {
            String sessionCreatedKey = this.getSessionCreatedChannel(session.getId());
            this.sessionRedisOperations.convertAndSend(sessionCreatedKey, (Object)session.delta);
            session.isNew = false;
        }
    }

    public void cleanupExpiredSessions() {
        this.expirationPolicy.cleanExpiredSessions();
    }

    public RedisSession findById(String id) {
        return this.getSession(id, false);
    }

    public Map<String, RedisSession> findByIndexNameAndIndexValue(String indexName, String indexValue) {
        if (!PRINCIPAL_NAME_INDEX_NAME.equals(indexName)) {
            return Collections.emptyMap();
        }
        String principalKey = this.getPrincipalKey(indexValue);
        Set sessionIds = this.sessionRedisOperations.boundSetOps((Object)principalKey).members();
        HashMap<String, RedisSession> sessions = new HashMap<String, RedisSession>(sessionIds.size());
        for (Object id : sessionIds) {
            RedisSession session = this.findById((String)id);
            if (session == null) continue;
            sessions.put(session.getId(), session);
        }
        return sessions;
    }

    private RedisSession getSession(String id, boolean allowExpired) {
        Map entries = this.getSessionBoundHashOperations(id).entries();
        if (entries.isEmpty()) {
            return null;
        }
        MapSession loaded = this.loadSession(id, entries);
        if (!allowExpired && loaded.isExpired()) {
            return null;
        }
        RedisSession result = new RedisSession(loaded, false);
        result.originalLastAccessTime = loaded.getLastAccessedTime();
        return result;
    }

    private MapSession loadSession(String id, Map<Object, Object> entries) {
        MapSession loaded = new MapSession(id);
        for (Map.Entry<Object, Object> entry : entries.entrySet()) {
            String key = (String)entry.getKey();
            if ("creationTime".equals(key)) {
                loaded.setCreationTime(Instant.ofEpochMilli((Long)entry.getValue()));
                continue;
            }
            if ("maxInactiveInterval".equals(key)) {
                loaded.setMaxInactiveInterval(Duration.ofSeconds(((Integer)entry.getValue()).intValue()));
                continue;
            }
            if ("lastAccessedTime".equals(key)) {
                loaded.setLastAccessedTime(Instant.ofEpochMilli((Long)entry.getValue()));
                continue;
            }
            if (!key.startsWith("sessionAttr:")) continue;
            loaded.setAttribute(key.substring("sessionAttr:".length()), entry.getValue());
        }
        return loaded;
    }

    public void deleteById(String sessionId) {
        RedisSession session = this.getSession(sessionId, true);
        if (session == null) {
            return;
        }
        this.cleanupPrincipalIndex(session);
        this.expirationPolicy.onDelete(session);
        String expireKey = this.getExpiredKey(session.getId());
        this.sessionRedisOperations.delete((Object)expireKey);
        session.setMaxInactiveInterval(Duration.ZERO);
        this.save(session);
    }

    public RedisSession createSession() {
        MapSession cached = new MapSession();
        if (this.defaultMaxInactiveInterval != null) {
            cached.setMaxInactiveInterval(Duration.ofSeconds(this.defaultMaxInactiveInterval.intValue()));
        }
        RedisSession session = new RedisSession(cached, true);
        session.flushImmediateIfNecessary();
        return session;
    }

    public void onMessage(Message message, byte[] pattern) {
        byte[] messageChannel = message.getChannel();
        if (ByteUtils.startsWith((byte[])messageChannel, (byte[])this.sessionCreatedChannelPrefixBytes)) {
            Map loaded = (Map)this.defaultSerializer.deserialize(message.getBody());
            this.handleCreated(loaded, new String(messageChannel));
            return;
        }
        byte[] messageBody = message.getBody();
        if (!ByteUtils.startsWith((byte[])messageBody, (byte[])this.expiredKeyPrefixBytes)) {
            return;
        }
        boolean isDeleted = Arrays.equals(messageChannel, this.sessionDeletedChannelBytes);
        if (isDeleted || Arrays.equals(messageChannel, this.sessionExpiredChannelBytes)) {
            int endIndex;
            String body = new String(messageBody);
            int beginIndex = body.lastIndexOf(":") + 1;
            String sessionId = body.substring(beginIndex, endIndex = body.length());
            RedisSession session = this.getSession(sessionId, true);
            if (session == null) {
                logger.warn((Object)("Unable to publish SessionDestroyedEvent for session " + sessionId));
                return;
            }
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Publishing SessionDestroyedEvent for session " + sessionId));
            }
            this.cleanupPrincipalIndex(session);
            if (isDeleted) {
                this.handleDeleted(session);
            } else {
                this.handleExpired(session);
            }
        }
    }

    private void cleanupPrincipalIndex(RedisSession session) {
        String sessionId = session.getId();
        Map indexes = this.indexResolver.resolveIndexesFor((Session)session);
        String principal = (String)indexes.get(PRINCIPAL_NAME_INDEX_NAME);
        if (principal != null) {
            this.sessionRedisOperations.boundSetOps((Object)this.getPrincipalKey(principal)).remove(new Object[]{sessionId});
        }
    }

    private void handleCreated(Map<Object, Object> loaded, String channel) {
        String id = channel.substring(channel.lastIndexOf(":") + 1);
        MapSession session = this.loadSession(id, loaded);
        this.publishEvent((ApplicationEvent)new SessionCreatedEvent((Object)this, (Session)session));
    }

    private void handleDeleted(RedisSession session) {
        this.publishEvent((ApplicationEvent)new SessionDeletedEvent((Object)this, (Session)session));
    }

    private void handleExpired(RedisSession session) {
        this.publishEvent((ApplicationEvent)new SessionExpiredEvent((Object)this, (Session)session));
    }

    private void publishEvent(ApplicationEvent event) {
        try {
            this.eventPublisher.publishEvent(event);
        }
        catch (Throwable ex) {
            logger.error((Object)("Error publishing " + event + "."), ex);
        }
    }

    public void setRedisKeyNamespace(String namespace) {
        Assert.hasText((String)namespace, (String)"namespace cannot be null or empty");
        this.namespace = namespace.trim() + ":";
        this.configureSessionChannels();
    }

    String getSessionKey(String sessionId) {
        return this.namespace + "sessions:" + sessionId;
    }

    String getPrincipalKey(String principalName) {
        return this.namespace + "index:" + FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME + ":" + principalName;
    }

    String getExpirationsKey(long expiration) {
        return this.namespace + "expirations:" + expiration;
    }

    private String getExpiredKey(String sessionId) {
        return this.getExpiredKeyPrefix() + sessionId;
    }

    private String getSessionCreatedChannel(String sessionId) {
        return this.getSessionCreatedChannelPrefix() + sessionId;
    }

    private String getExpiredKeyPrefix() {
        return this.expiredKeyPrefix;
    }

    public String getSessionCreatedChannelPrefix() {
        return this.sessionCreatedChannelPrefix;
    }

    public String getSessionDeletedChannel() {
        return this.sessionDeletedChannel;
    }

    public String getSessionExpiredChannel() {
        return this.sessionExpiredChannel;
    }

    private BoundHashOperations<Object, Object, Object> getSessionBoundHashOperations(String sessionId) {
        String key = this.getSessionKey(sessionId);
        return this.sessionRedisOperations.boundHashOps((Object)key);
    }

    static String getSessionAttrNameKey(String attributeName) {
        return "sessionAttr:" + attributeName;
    }

    final class RedisSession
    implements Session {
        private final MapSession cached;
        private Instant originalLastAccessTime;
        private Map<String, Object> delta = new HashMap<String, Object>();
        private boolean isNew;
        private String originalPrincipalName;
        private String originalSessionId;

        RedisSession(MapSession cached, boolean isNew) {
            this.cached = cached;
            this.isNew = isNew;
            this.originalSessionId = cached.getId();
            Map indexes = RedisIndexedSessionRepository.this.indexResolver.resolveIndexesFor((Session)this);
            this.originalPrincipalName = (String)indexes.get(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
            if (this.isNew) {
                this.delta.put("creationTime", cached.getCreationTime().toEpochMilli());
                this.delta.put("maxInactiveInterval", (int)cached.getMaxInactiveInterval().getSeconds());
                this.delta.put("lastAccessedTime", cached.getLastAccessedTime().toEpochMilli());
            }
            if (this.isNew || RedisIndexedSessionRepository.this.saveMode == SaveMode.ALWAYS) {
                this.getAttributeNames().forEach(attributeName -> this.delta.put(RedisIndexedSessionRepository.getSessionAttrNameKey(attributeName), cached.getAttribute(attributeName)));
            }
        }

        public void setLastAccessedTime(Instant lastAccessedTime) {
            this.cached.setLastAccessedTime(lastAccessedTime);
            this.delta.put("lastAccessedTime", this.getLastAccessedTime().toEpochMilli());
            this.flushImmediateIfNecessary();
        }

        public boolean isExpired() {
            return this.cached.isExpired();
        }

        public Instant getCreationTime() {
            return this.cached.getCreationTime();
        }

        public String getId() {
            return this.cached.getId();
        }

        public String changeSessionId() {
            return this.cached.changeSessionId();
        }

        public Instant getLastAccessedTime() {
            return this.cached.getLastAccessedTime();
        }

        public void setMaxInactiveInterval(Duration interval) {
            this.cached.setMaxInactiveInterval(interval);
            this.delta.put("maxInactiveInterval", (int)this.getMaxInactiveInterval().getSeconds());
            this.flushImmediateIfNecessary();
        }

        public Duration getMaxInactiveInterval() {
            return this.cached.getMaxInactiveInterval();
        }

        public <T> T getAttribute(String attributeName) {
            Object attributeValue = this.cached.getAttribute(attributeName);
            if (attributeValue != null && RedisIndexedSessionRepository.this.saveMode.equals((Object)SaveMode.ON_GET_ATTRIBUTE)) {
                this.delta.put(RedisIndexedSessionRepository.getSessionAttrNameKey(attributeName), attributeValue);
            }
            return (T)attributeValue;
        }

        public Set<String> getAttributeNames() {
            return this.cached.getAttributeNames();
        }

        public void setAttribute(String attributeName, Object attributeValue) {
            this.cached.setAttribute(attributeName, attributeValue);
            this.delta.put(RedisIndexedSessionRepository.getSessionAttrNameKey(attributeName), attributeValue);
            this.flushImmediateIfNecessary();
        }

        public void removeAttribute(String attributeName) {
            this.cached.removeAttribute(attributeName);
            this.delta.put(RedisIndexedSessionRepository.getSessionAttrNameKey(attributeName), null);
            this.flushImmediateIfNecessary();
        }

        private void flushImmediateIfNecessary() {
            if (RedisIndexedSessionRepository.this.flushMode == FlushMode.IMMEDIATE) {
                this.save();
            }
        }

        private void save() {
            this.saveChangeSessionId();
            this.saveDelta();
        }

        private void saveDelta() {
            if (this.delta.isEmpty()) {
                return;
            }
            String sessionId = this.getId();
            RedisIndexedSessionRepository.this.getSessionBoundHashOperations(sessionId).putAll(this.delta);
            String principalSessionKey = RedisIndexedSessionRepository.getSessionAttrNameKey(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
            String securityPrincipalSessionKey = RedisIndexedSessionRepository.getSessionAttrNameKey(RedisIndexedSessionRepository.SPRING_SECURITY_CONTEXT);
            if (this.delta.containsKey(principalSessionKey) || this.delta.containsKey(securityPrincipalSessionKey)) {
                String principal;
                if (this.originalPrincipalName != null) {
                    String originalPrincipalRedisKey = RedisIndexedSessionRepository.this.getPrincipalKey(this.originalPrincipalName);
                    RedisIndexedSessionRepository.this.sessionRedisOperations.boundSetOps((Object)originalPrincipalRedisKey).remove(new Object[]{sessionId});
                }
                Map indexes = RedisIndexedSessionRepository.this.indexResolver.resolveIndexesFor((Session)this);
                this.originalPrincipalName = principal = (String)indexes.get(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
                if (principal != null) {
                    String principalRedisKey = RedisIndexedSessionRepository.this.getPrincipalKey(principal);
                    RedisIndexedSessionRepository.this.sessionRedisOperations.boundSetOps((Object)principalRedisKey).add(new Object[]{sessionId});
                }
            }
            this.delta = new HashMap<String, Object>(this.delta.size());
            Long originalExpiration = this.originalLastAccessTime != null ? Long.valueOf(this.originalLastAccessTime.plus(this.getMaxInactiveInterval()).toEpochMilli()) : null;
            RedisIndexedSessionRepository.this.expirationPolicy.onExpirationUpdated(originalExpiration, this);
        }

        private void saveChangeSessionId() {
            String sessionId = this.getId();
            if (sessionId.equals(this.originalSessionId)) {
                return;
            }
            if (!this.isNew) {
                String originalSessionIdKey = RedisIndexedSessionRepository.this.getSessionKey(this.originalSessionId);
                String sessionIdKey = RedisIndexedSessionRepository.this.getSessionKey(sessionId);
                try {
                    RedisIndexedSessionRepository.this.sessionRedisOperations.rename((Object)originalSessionIdKey, (Object)sessionIdKey);
                }
                catch (NonTransientDataAccessException ex) {
                    this.handleErrNoSuchKeyError(ex);
                }
                String originalExpiredKey = RedisIndexedSessionRepository.this.getExpiredKey(this.originalSessionId);
                String expiredKey = RedisIndexedSessionRepository.this.getExpiredKey(sessionId);
                try {
                    RedisIndexedSessionRepository.this.sessionRedisOperations.rename((Object)originalExpiredKey, (Object)expiredKey);
                }
                catch (NonTransientDataAccessException ex) {
                    this.handleErrNoSuchKeyError(ex);
                }
            }
            this.originalSessionId = sessionId;
        }

        private void handleErrNoSuchKeyError(NonTransientDataAccessException ex) {
            if (!"ERR no such key".equals(NestedExceptionUtils.getMostSpecificCause((Throwable)ex).getMessage())) {
                throw ex;
            }
        }
    }
}

