/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.mongodb.core.query;

import com.mongodb.ReadConcern;
import com.mongodb.ReadPreference;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.bson.Document;
import org.springframework.data.domain.KeysetScrollPosition;
import org.springframework.data.domain.Limit;
import org.springframework.data.domain.OffsetScrollPosition;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.ScrollPosition;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.InvalidMongoDbApiUsageException;
import org.springframework.data.mongodb.core.ReadConcernAware;
import org.springframework.data.mongodb.core.ReadPreferenceAware;
import org.springframework.data.mongodb.core.query.Collation;
import org.springframework.data.mongodb.core.query.CriteriaDefinition;
import org.springframework.data.mongodb.core.query.Field;
import org.springframework.data.mongodb.core.query.Meta;
import org.springframework.data.mongodb.core.query.SerializationUtils;
import org.springframework.data.mongodb.util.BsonUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

public class Query
implements ReadConcernAware,
ReadPreferenceAware {
    private static final String RESTRICTED_TYPES_KEY = "_$RESTRICTED_TYPES";
    private Set<Class<?>> restrictedTypes = Collections.emptySet();
    private final Map<String, CriteriaDefinition> criteria = new LinkedHashMap<String, CriteriaDefinition>();
    @Nullable
    private Field fieldSpec = null;
    private Sort sort = Sort.unsorted();
    private long skip;
    private Limit limit = Limit.unlimited();
    private KeysetScrollPosition keysetScrollPosition;
    @Nullable
    private ReadConcern readConcern;
    @Nullable
    private ReadPreference readPreference;
    @Nullable
    private String hint;
    private Meta meta = new Meta();
    private Optional<Collation> collation = Optional.empty();

    public static Query query(CriteriaDefinition criteriaDefinition) {
        return new Query(criteriaDefinition);
    }

    public Query() {
    }

    public Query(CriteriaDefinition criteriaDefinition) {
        this.addCriteria(criteriaDefinition);
    }

    public Query addCriteria(CriteriaDefinition criteriaDefinition) {
        Assert.notNull((Object)criteriaDefinition, (String)"CriteriaDefinition must not be null");
        CriteriaDefinition existing = this.criteria.get(criteriaDefinition.getKey());
        String key = criteriaDefinition.getKey();
        if (existing != null) {
            throw new InvalidMongoDbApiUsageException(String.format("Due to limitations of the com.mongodb.BasicDocument, you can't add a second '%s' criteria; Query already contains '%s'", key, SerializationUtils.serializeToJsonSafely(existing.getCriteriaObject())));
        }
        this.criteria.put(key, criteriaDefinition);
        return this;
    }

    public Field fields() {
        if (this.fieldSpec == null) {
            this.fieldSpec = new Field();
        }
        return this.fieldSpec;
    }

    public Query skip(long skip) {
        this.skip = skip;
        return this;
    }

    public Query limit(int limit) {
        this.limit = limit > 0 ? Limit.of((int)limit) : Limit.unlimited();
        return this;
    }

    public Query limit(Limit limit) {
        Assert.notNull((Object)limit, (String)"Limit must not be null");
        if (limit.isUnlimited()) {
            this.limit = limit;
            return this;
        }
        return this.limit(limit.max());
    }

    public Query withHint(String hint) {
        Assert.hasText((String)hint, (String)"Hint must not be empty or null");
        this.hint = hint;
        return this;
    }

    public Query withReadConcern(ReadConcern readConcern) {
        Assert.notNull((Object)readConcern, (String)"ReadConcern must not be null");
        this.readConcern = readConcern;
        return this;
    }

    public Query withReadPreference(ReadPreference readPreference) {
        Assert.notNull((Object)readPreference, (String)"ReadPreference must not be null");
        this.readPreference = readPreference;
        return this;
    }

    @Override
    public boolean hasReadConcern() {
        return this.readConcern != null;
    }

    @Override
    public ReadConcern getReadConcern() {
        return this.readConcern;
    }

    @Override
    public boolean hasReadPreference() {
        return this.readPreference != null || this.getMeta().getFlags().contains((Object)Meta.CursorOption.SECONDARY_READS);
    }

    @Override
    public ReadPreference getReadPreference() {
        if (this.readPreference == null) {
            return this.getMeta().getFlags().contains((Object)Meta.CursorOption.SECONDARY_READS) ? ReadPreference.primaryPreferred() : null;
        }
        return this.readPreference;
    }

    public Query withHint(Document hint) {
        Assert.notNull((Object)hint, (String)"Hint must not be null");
        this.hint = hint.toJson();
        return this;
    }

    public Query with(Pageable pageable) {
        if (pageable.isPaged()) {
            this.limit = pageable.toLimit();
            this.skip = pageable.getOffset();
        }
        return this.with(pageable.getSort());
    }

    public Query with(ScrollPosition position) {
        Assert.notNull((Object)position, (String)"ScrollPosition must not be null");
        if (position instanceof OffsetScrollPosition) {
            OffsetScrollPosition offset = (OffsetScrollPosition)position;
            return this.with(offset);
        }
        if (position instanceof KeysetScrollPosition) {
            KeysetScrollPosition keyset = (KeysetScrollPosition)position;
            return this.with(keyset);
        }
        throw new IllegalArgumentException(String.format("ScrollPosition %s not supported", position));
    }

    public Query with(OffsetScrollPosition position) {
        Assert.notNull((Object)position, (String)"ScrollPosition must not be null");
        this.skip = position.getOffset();
        this.keysetScrollPosition = null;
        return this;
    }

    public Query with(KeysetScrollPosition position) {
        Assert.notNull((Object)position, (String)"ScrollPosition must not be null");
        this.skip = 0L;
        this.keysetScrollPosition = position;
        return this;
    }

    public boolean hasKeyset() {
        return this.keysetScrollPosition != null;
    }

    @Nullable
    public KeysetScrollPosition getKeyset() {
        return this.keysetScrollPosition;
    }

    public Query with(Sort sort) {
        Assert.notNull((Object)sort, (String)"Sort must not be null");
        if (sort.isUnsorted()) {
            return this;
        }
        sort.stream().filter(Sort.Order::isIgnoreCase).findFirst().ifPresent(it -> {
            throw new IllegalArgumentException(String.format("Given sort contained an Order for %s with ignore case; MongoDB does not support sorting ignoring case currently", it.getProperty()));
        });
        this.sort = this.sort.and(sort);
        return this;
    }

    public Set<Class<?>> getRestrictedTypes() {
        return this.restrictedTypes;
    }

    public Query restrict(Class<?> type, Class<?> ... additionalTypes) {
        Assert.notNull(type, (String)"Type must not be null");
        Assert.notNull(additionalTypes, (String)"AdditionalTypes must not be null");
        if (this.restrictedTypes == Collections.EMPTY_SET) {
            this.restrictedTypes = new HashSet(1 + additionalTypes.length);
        }
        this.restrictedTypes.add(type);
        if (additionalTypes.length > 0) {
            this.restrictedTypes.addAll(Arrays.asList(additionalTypes));
        }
        return this;
    }

    public Document getQueryObject() {
        Iterator<CriteriaDefinition> iterator;
        if (this.criteria.isEmpty() && this.restrictedTypes.isEmpty()) {
            return BsonUtils.EMPTY_DOCUMENT;
        }
        if (this.criteria.size() == 1 && this.restrictedTypes.isEmpty() && (iterator = this.criteria.values().iterator()).hasNext()) {
            CriteriaDefinition definition = iterator.next();
            return definition.getCriteriaObject();
        }
        Document document = new Document();
        for (CriteriaDefinition definition : this.criteria.values()) {
            document.putAll((Map)definition.getCriteriaObject());
        }
        if (!this.restrictedTypes.isEmpty()) {
            document.put(RESTRICTED_TYPES_KEY, this.getRestrictedTypes());
        }
        return document;
    }

    public Document getFieldsObject() {
        return this.fieldSpec == null ? BsonUtils.EMPTY_DOCUMENT : this.fieldSpec.getFieldsObject();
    }

    public Document getSortObject() {
        if (this.sort.isUnsorted()) {
            return BsonUtils.EMPTY_DOCUMENT;
        }
        Document document = new Document();
        this.sort.forEach(order -> document.put(order.getProperty(), (Object)(order.isAscending() ? 1 : -1)));
        return document;
    }

    public boolean isSorted() {
        return this.sort.isSorted();
    }

    public long getSkip() {
        return this.skip;
    }

    public boolean isLimited() {
        return this.limit.isLimited();
    }

    public int getLimit() {
        return this.limit.isUnlimited() ? 0 : this.limit.max();
    }

    @Nullable
    public String getHint() {
        return this.hint;
    }

    public Query maxTimeMsec(long maxTimeMsec) {
        this.meta.setMaxTimeMsec(maxTimeMsec);
        return this;
    }

    public Query maxTime(Duration timeout) {
        this.meta.setMaxTime(timeout);
        return this;
    }

    public Query comment(String comment) {
        this.meta.setComment(comment);
        return this;
    }

    public Query allowDiskUse(boolean allowDiskUse) {
        this.meta.setAllowDiskUse(allowDiskUse);
        return this;
    }

    public Query cursorBatchSize(int batchSize) {
        this.meta.setCursorBatchSize(batchSize);
        return this;
    }

    public Query noCursorTimeout() {
        this.meta.addFlag(Meta.CursorOption.NO_TIMEOUT);
        return this;
    }

    public Query exhaust() {
        this.meta.addFlag(Meta.CursorOption.EXHAUST);
        return this;
    }

    public Query allowSecondaryReads() {
        this.meta.addFlag(Meta.CursorOption.SECONDARY_READS);
        return this;
    }

    public Query partialResults() {
        this.meta.addFlag(Meta.CursorOption.PARTIAL);
        return this;
    }

    public Meta getMeta() {
        return this.meta;
    }

    public void setMeta(Meta meta) {
        Assert.notNull((Object)meta, (String)"Query meta might be empty but must not be null");
        this.meta = meta;
    }

    public Query collation(@Nullable Collation collation) {
        this.collation = Optional.ofNullable(collation);
        return this;
    }

    public Optional<Collation> getCollation() {
        return this.collation;
    }

    protected List<CriteriaDefinition> getCriteria() {
        return new ArrayList<CriteriaDefinition>(this.criteria.values());
    }

    public static Query of(final Query source) {
        Assert.notNull((Object)source, (String)"Source must not be null");
        final Document sourceFields = source.getFieldsObject();
        final Document sourceSort = source.getSortObject();
        final Document sourceQuery = source.getQueryObject();
        Query target = new Query(){

            @Override
            public Document getFieldsObject() {
                return BsonUtils.merge(sourceFields, super.getFieldsObject());
            }

            @Override
            public Document getSortObject() {
                return BsonUtils.merge(sourceSort, super.getSortObject());
            }

            @Override
            public Document getQueryObject() {
                return BsonUtils.merge(sourceQuery, super.getQueryObject());
            }

            @Override
            public boolean isSorted() {
                return source.isSorted() || super.isSorted();
            }
        };
        target.skip = source.getSkip();
        target.limit = source.isLimited() ? Limit.of((int)source.getLimit()) : Limit.unlimited();
        target.hint = source.getHint();
        target.collation = source.getCollation();
        target.restrictedTypes = new HashSet(source.getRestrictedTypes());
        if (source.getMeta().hasValues()) {
            target.setMeta(new Meta(source.getMeta()));
        }
        return target;
    }

    public String toString() {
        return String.format("Query: %s, Fields: %s, Sort: %s", SerializationUtils.serializeToJsonSafely(this.getQueryObject()), SerializationUtils.serializeToJsonSafely(this.getFieldsObject()), SerializationUtils.serializeToJsonSafely(this.getSortObject()));
    }

    public boolean equals(@Nullable Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || !this.getClass().equals(obj.getClass())) {
            return false;
        }
        return this.querySettingsEquals((Query)obj);
    }

    protected boolean querySettingsEquals(Query that) {
        boolean criteriaEqual = this.criteria.equals(that.criteria);
        boolean fieldsEqual = ObjectUtils.nullSafeEquals((Object)this.fieldSpec, (Object)that.fieldSpec);
        boolean sortEqual = this.sort.equals((Object)that.sort);
        boolean hintEqual = ObjectUtils.nullSafeEquals((Object)this.hint, (Object)that.hint);
        boolean skipEqual = this.skip == that.skip;
        boolean limitEqual = ObjectUtils.nullSafeEquals((Object)this.limit, (Object)that.limit);
        boolean metaEqual = ObjectUtils.nullSafeEquals((Object)this.meta, (Object)that.meta);
        boolean collationEqual = ObjectUtils.nullSafeEquals(this.collation.orElse(null), that.collation.orElse(null));
        return criteriaEqual && fieldsEqual && sortEqual && hintEqual && skipEqual && limitEqual && metaEqual && collationEqual;
    }

    public int hashCode() {
        int result = 17;
        result += 31 * this.criteria.hashCode();
        result += 31 * ObjectUtils.nullSafeHashCode((Object)this.fieldSpec);
        result += 31 * ObjectUtils.nullSafeHashCode((Object)this.sort);
        result += 31 * ObjectUtils.nullSafeHashCode((Object)this.hint);
        result = (int)((long)result + 31L * this.skip);
        result += 31 * this.limit.hashCode();
        result += 31 * ObjectUtils.nullSafeHashCode((Object)this.meta);
        return result += 31 * ObjectUtils.nullSafeHashCode(this.collation.orElse(null));
    }

    @Deprecated
    public static boolean isRestrictedTypeKey(String key) {
        return RESTRICTED_TYPES_KEY.equals(key);
    }
}

