/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.issue;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import java.time.Clock;
import java.time.OffsetDateTime;
import java.time.Period;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.CheckForNull;
import javax.annotation.Nullable;
import org.apache.commons.lang.BooleanUtils;
import org.sonar.api.rule.RuleKey;
import org.sonar.api.server.ServerSide;
import org.sonar.api.utils.DateUtils;
import org.sonar.core.util.stream.MoreCollectors;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.component.ComponentDto;
import org.sonar.db.component.SnapshotDto;
import org.sonar.db.organization.OrganizationDto;
import org.sonar.server.issue.IssueQuery;
import org.sonar.server.user.UserSession;
import org.sonar.server.ws.WsUtils;
import org.sonarqube.ws.client.issue.SearchWsRequest;

@ServerSide
public class IssueQueryFactory {
    public static final String LOGIN_MYSELF = "__me__";
    private static final String UNKNOWN = "<UNKNOWN>";
    private static final ComponentDto UNKNOWN_COMPONENT = new ComponentDto().setUuid("<UNKNOWN>").setProjectUuid("<UNKNOWN>");
    private final DbClient dbClient;
    private final Clock clock;
    private final UserSession userSession;

    public IssueQueryFactory(DbClient dbClient, Clock clock, UserSession userSession) {
        this.dbClient = dbClient;
        this.clock = clock;
        this.userSession = userSession;
    }

    public IssueQuery create(SearchWsRequest request) {
        try (DbSession dbSession = this.dbClient.openSession(false);){
            IssueQuery.Builder builder = IssueQuery.builder().issueKeys(request.getIssues()).severities(request.getSeverities()).statuses(request.getStatuses()).resolutions(request.getResolutions()).resolved(request.getResolved()).rules(IssueQueryFactory.stringsToRules(request.getRules())).assignees(this.buildAssignees(request.getAssignees())).languages(request.getLanguages()).tags(request.getTags()).types(request.getTypes()).assigned(request.getAssigned()).createdAt(DateUtils.parseDateOrDateTime((String)request.getCreatedAt())).createdBefore(DateUtils.parseEndingDateOrDateTime((String)request.getCreatedBefore())).facetMode(request.getFacetMode()).organizationUuid(this.convertOrganizationKeyToUuid(dbSession, request.getOrganization()));
            ArrayList<ComponentDto> allComponents = new ArrayList<ComponentDto>();
            boolean effectiveOnComponentOnly = this.mergeDeprecatedComponentParameters(dbSession, request, allComponents);
            this.addComponentParameters(builder, dbSession, effectiveOnComponentOnly, allComponents, request);
            this.setCreatedAfterFromRequest(dbSession, builder, request, allComponents);
            String sort = request.getSort();
            if (!Strings.isNullOrEmpty((String)sort)) {
                builder.sort(sort);
                builder.asc(request.getAsc());
            }
            IssueQuery issueQuery = builder.build();
            return issueQuery;
        }
    }

    private void setCreatedAfterFromDates(IssueQuery.Builder builder, @Nullable Date createdAfter, @Nullable String createdInLast, boolean createdAfterInclusive) {
        Preconditions.checkArgument((createdAfter == null || createdInLast == null ? 1 : 0) != 0, (Object)String.format("Parameters %s and %s cannot be set simultaneously", "createdAfter", "createdInLast"));
        Date actualCreatedAfter = createdAfter;
        if (createdInLast != null) {
            actualCreatedAfter = Date.from(OffsetDateTime.now(this.clock).minus(Period.parse("P" + createdInLast.toUpperCase(Locale.ENGLISH))).toInstant());
        }
        builder.createdAfter(actualCreatedAfter, createdAfterInclusive);
    }

    @CheckForNull
    private String convertOrganizationKeyToUuid(DbSession dbSession, @Nullable String organizationKey) {
        if (organizationKey == null) {
            return null;
        }
        Optional organization = this.dbClient.organizationDao().selectByKey(dbSession, organizationKey);
        return organization.map(OrganizationDto::getUuid).orElse(UNKNOWN);
    }

    private void setCreatedAfterFromRequest(DbSession dbSession, IssueQuery.Builder builder, SearchWsRequest request, List<ComponentDto> componentUuids) {
        Date createdAfter = DateUtils.parseStartingDateOrDateTime((String)request.getCreatedAfter());
        String createdInLast = request.getCreatedInLast();
        if (request.getSinceLeakPeriod() == null || !request.getSinceLeakPeriod().booleanValue()) {
            this.setCreatedAfterFromDates(builder, createdAfter, createdInLast, true);
        } else {
            WsUtils.checkRequest(createdAfter == null, "Parameters '%s' and '%s' cannot be set simultaneously", "createdAfter", "sinceLeakPeriod");
            Preconditions.checkArgument((componentUuids.size() == 1 ? 1 : 0) != 0, (Object)"One and only one component must be provided when searching since leak period");
            ComponentDto component = componentUuids.iterator().next();
            Date createdAfterFromSnapshot = this.findCreatedAfterFromComponentUuid(dbSession, component);
            this.setCreatedAfterFromDates(builder, createdAfterFromSnapshot, createdInLast, false);
        }
    }

    @CheckForNull
    private Date findCreatedAfterFromComponentUuid(DbSession dbSession, ComponentDto component) {
        Optional snapshot = this.dbClient.snapshotDao().selectLastAnalysisByComponentUuid(dbSession, component.uuid());
        return snapshot.map(s -> DateUtils.longToDate((Long)s.getPeriodDate())).orElse(null);
    }

    private List<String> buildAssignees(@Nullable List<String> assigneesFromParams) {
        ArrayList assignees = Lists.newArrayList();
        if (assigneesFromParams != null) {
            assignees.addAll(assigneesFromParams);
        }
        if (assignees.contains(LOGIN_MYSELF)) {
            String login = this.userSession.getLogin();
            if (login == null) {
                assignees.add(UNKNOWN);
            } else {
                assignees.add(login);
            }
        }
        return assignees;
    }

    private boolean mergeDeprecatedComponentParameters(DbSession session, SearchWsRequest request, List<ComponentDto> allComponents) {
        Boolean onComponentOnly = request.getOnComponentOnly();
        List components = request.getComponents();
        List componentUuids = request.getComponentUuids();
        List componentKeys = request.getComponentKeys();
        List componentRootUuids = request.getComponentRootUuids();
        List componentRoots = request.getComponentRoots();
        String branch = request.getBranch();
        boolean effectiveOnComponentOnly = false;
        Preconditions.checkArgument((boolean)IssueQueryFactory.atMostOneNonNullElement(components, componentUuids, componentKeys, componentRootUuids, componentRoots), (String)"At most one of the following parameters can be provided: %s, %s, %s, %s, %s", (Object[])new Object[]{"componentKeys", "componentUuids", "components", "componentRoots", "componentUuids"});
        if (componentRootUuids != null) {
            allComponents.addAll(this.getComponentsFromUuids(session, componentRootUuids));
        } else if (componentRoots != null) {
            allComponents.addAll(this.getComponentsFromKeys(session, componentRoots, branch));
        } else if (components != null) {
            allComponents.addAll(this.getComponentsFromKeys(session, components, branch));
            effectiveOnComponentOnly = true;
        } else if (componentUuids != null) {
            allComponents.addAll(this.getComponentsFromUuids(session, componentUuids));
            effectiveOnComponentOnly = BooleanUtils.isTrue((Boolean)onComponentOnly);
        } else if (componentKeys != null) {
            allComponents.addAll(this.getComponentsFromKeys(session, componentKeys, branch));
            effectiveOnComponentOnly = BooleanUtils.isTrue((Boolean)onComponentOnly);
        }
        return effectiveOnComponentOnly;
    }

    private static boolean atMostOneNonNullElement(Object ... objects) {
        return Arrays.stream(objects).filter(Objects::nonNull).count() <= 1L;
    }

    private void addComponentParameters(IssueQuery.Builder builder, DbSession session, boolean onComponentOnly, List<ComponentDto> components, SearchWsRequest request) {
        builder.onComponentOnly(onComponentOnly);
        if (onComponentOnly) {
            builder.componentUuids((Collection)components.stream().map(ComponentDto::uuid).collect(MoreCollectors.toList()));
            IssueQueryFactory.setBranch(builder, components.get(0), request.getBranch());
            return;
        }
        builder.authors(request.getAuthors());
        List projectUuids = request.getProjectUuids();
        List projectKeys = request.getProjectKeys();
        Preconditions.checkArgument((projectUuids == null || projectKeys == null ? 1 : 0) != 0, (Object)"Parameters projects and projectUuids cannot be set simultaneously");
        if (projectUuids != null) {
            builder.projectUuids(projectUuids);
        } else if (projectKeys != null) {
            List<ComponentDto> projects = this.getComponentsFromKeys(session, projectKeys, request.getBranch());
            builder.projectUuids((Collection)projects.stream().map(IssueQueryFactory::toProjectUuid).collect(MoreCollectors.toList()));
            IssueQueryFactory.setBranch(builder, projects.get(0), request.getBranch());
        }
        builder.moduleUuids(request.getModuleUuids());
        builder.directories(request.getDirectories());
        builder.fileUuids(request.getFileUuids());
        this.addComponentsBasedOnQualifier(builder, session, components, request);
    }

    private void addComponentsBasedOnQualifier(IssueQuery.Builder builder, DbSession dbSession, List<ComponentDto> components, SearchWsRequest request) {
        String qualifier;
        if (components.isEmpty()) {
            return;
        }
        if (components.stream().map(ComponentDto::uuid).anyMatch(uuid -> uuid.equals(UNKNOWN))) {
            builder.componentUuids(Collections.singleton(UNKNOWN));
            return;
        }
        Set qualifiers = (Set)components.stream().map(ComponentDto::qualifier).collect(MoreCollectors.toHashSet());
        Preconditions.checkArgument((qualifiers.size() == 1 ? 1 : 0) != 0, (String)"All components must have the same qualifier, found %s", (Object[])new Object[]{String.join((CharSequence)",", qualifiers)});
        IssueQueryFactory.setBranch(builder, components.get(0), request.getBranch());
        switch (qualifier = (String)qualifiers.iterator().next()) {
            case "VW": 
            case "SVW": {
                this.addViewsOrSubViews(builder, components);
                break;
            }
            case "APP": {
                this.addApplications(builder, dbSession, components, request);
                break;
            }
            case "TRK": {
                builder.projectUuids((Collection)components.stream().map(IssueQueryFactory::toProjectUuid).collect(MoreCollectors.toList()));
                break;
            }
            case "BRC": {
                builder.moduleRootUuids((Collection)components.stream().map(ComponentDto::uuid).collect(MoreCollectors.toList()));
                break;
            }
            case "DIR": {
                IssueQueryFactory.addDirectories(builder, components);
                break;
            }
            case "FIL": 
            case "UTS": {
                builder.fileUuids((Collection)components.stream().map(ComponentDto::uuid).collect(MoreCollectors.toList()));
                break;
            }
            default: {
                throw new IllegalArgumentException("Unable to set search root context for components " + Joiner.on((char)',').join(components));
            }
        }
    }

    private void addViewsOrSubViews(IssueQuery.Builder builder, Collection<ComponentDto> viewOrSubViewUuids) {
        List<String> filteredViewUuids = viewOrSubViewUuids.stream().filter(uuid -> this.userSession.hasComponentPermission("user", (ComponentDto)uuid)).map(ComponentDto::uuid).collect(Collectors.toList());
        if (filteredViewUuids.isEmpty()) {
            filteredViewUuids.add(UNKNOWN);
        }
        builder.viewUuids(filteredViewUuids);
    }

    private void addApplications(IssueQuery.Builder builder, DbSession dbSession, List<ComponentDto> applications, SearchWsRequest request) {
        Set<String> authorizedApplicationUuids = (Set<String>)applications.stream().filter(app -> this.userSession.hasComponentPermission("user", (ComponentDto)app)).map(ComponentDto::uuid).collect(MoreCollectors.toSet());
        builder.viewUuids(authorizedApplicationUuids.isEmpty() ? Collections.singleton(UNKNOWN) : authorizedApplicationUuids);
        this.addCreatedAfterByProjects(builder, dbSession, request, authorizedApplicationUuids);
    }

    private void addCreatedAfterByProjects(IssueQuery.Builder builder, DbSession dbSession, SearchWsRequest request, Set<String> applicationUuids) {
        if (request.getSinceLeakPeriod() == null || !request.getSinceLeakPeriod().booleanValue()) {
            return;
        }
        Set projectUuids = (Set)applicationUuids.stream().flatMap(app -> this.dbClient.componentDao().selectProjectsFromView(dbSession, app, app).stream()).collect(MoreCollectors.toSet());
        Map leakByProjects = (Map)this.dbClient.snapshotDao().selectLastAnalysesByRootComponentUuids(dbSession, (Collection)projectUuids).stream().filter(s -> s.getPeriodDate() != null).collect(MoreCollectors.uniqueIndex(SnapshotDto::getComponentUuid, s -> new IssueQuery.PeriodStart(DateUtils.longToDate((Long)s.getPeriodDate()), false)));
        builder.createdAfterByProjectUuids(leakByProjects);
    }

    private static void addDirectories(IssueQuery.Builder builder, List<ComponentDto> directories) {
        HashSet<String> directoryModuleUuids = new HashSet<String>();
        HashSet<String> directoryPaths = new HashSet<String>();
        for (ComponentDto directory : directories) {
            directoryModuleUuids.add(directory.moduleUuid());
            directoryPaths.add(directory.path());
        }
        builder.moduleUuids(directoryModuleUuids);
        builder.directories(directoryPaths);
    }

    private List<ComponentDto> getComponentsFromKeys(DbSession dbSession, Collection<String> componentKeys, @Nullable String branch) {
        List componentDtos;
        List list = componentDtos = branch == null ? this.dbClient.componentDao().selectByKeys(dbSession, componentKeys) : this.dbClient.componentDao().selectByKeysAndBranch(dbSession, componentKeys, branch);
        if (!componentKeys.isEmpty() && componentDtos.isEmpty()) {
            return Collections.singletonList(UNKNOWN_COMPONENT);
        }
        return componentDtos;
    }

    private List<ComponentDto> getComponentsFromUuids(DbSession dbSession, Collection<String> componentUuids) {
        List componentDtos = this.dbClient.componentDao().selectByUuids(dbSession, componentUuids);
        if (!componentUuids.isEmpty() && componentDtos.isEmpty()) {
            return Collections.singletonList(UNKNOWN_COMPONENT);
        }
        return componentDtos;
    }

    @VisibleForTesting
    static Collection<RuleKey> toRules(@Nullable Object o) {
        Collection<RuleKey> result = null;
        if (o != null) {
            if (o instanceof List) {
                result = IssueQueryFactory.stringsToRules((List)o);
            } else if (o instanceof String) {
                result = IssueQueryFactory.stringsToRules(Lists.newArrayList((Iterable)Splitter.on((char)',').omitEmptyStrings().split((CharSequence)((String)o))));
            }
        }
        return result;
    }

    @CheckForNull
    private static Collection<RuleKey> stringsToRules(@Nullable Collection<String> rules) {
        if (rules != null) {
            return Collections2.transform(rules, RuleKey::parse);
        }
        return null;
    }

    private static String toProjectUuid(ComponentDto componentDto) {
        String mainBranchProjectUuid = componentDto.getMainBranchProjectUuid();
        return mainBranchProjectUuid == null ? componentDto.projectUuid() : mainBranchProjectUuid;
    }

    private static void setBranch(IssueQuery.Builder builder, ComponentDto component, @Nullable String branch) {
        builder.branchUuid(branch == null ? null : component.projectUuid());
        builder.mainBranch(branch == null || component.equals((Object)UNKNOWN_COMPONENT) || !branch.equals(component.getBranch()));
    }
}

