/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.user;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.apache.commons.codec.digest.DigestUtils;
import org.sonar.api.config.Configuration;
import org.sonar.api.platform.NewUserHandler;
import org.sonar.api.server.ServerSide;
import org.sonar.core.util.stream.MoreCollectors;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.organization.OrganizationMemberDto;
import org.sonar.db.user.GroupDto;
import org.sonar.db.user.UserDto;
import org.sonar.db.user.UserGroupDto;
import org.sonar.server.organization.DefaultOrganizationProvider;
import org.sonar.server.organization.OrganizationCreation;
import org.sonar.server.organization.OrganizationFlags;
import org.sonar.server.user.ExternalIdentity;
import org.sonar.server.user.NewUser;
import org.sonar.server.user.NewUserNotifier;
import org.sonar.server.user.UpdateUser;
import org.sonar.server.user.index.UserIndexer;
import org.sonar.server.usergroups.DefaultGroupFinder;
import org.sonar.server.ws.WsUtils;

@ServerSide
public class UserUpdater {
    private static final String SQ_AUTHORITY = "sonarqube";
    private static final String LOGIN_PARAM = "Login";
    private static final String PASSWORD_PARAM = "Password";
    private static final String NAME_PARAM = "Name";
    private static final String EMAIL_PARAM = "Email";
    private static final int LOGIN_MIN_LENGTH = 2;
    private static final int LOGIN_MAX_LENGTH = 255;
    private static final int EMAIL_MAX_LENGTH = 100;
    private static final int NAME_MAX_LENGTH = 200;
    private final NewUserNotifier newUserNotifier;
    private final DbClient dbClient;
    private final UserIndexer userIndexer;
    private final OrganizationFlags organizationFlags;
    private final DefaultOrganizationProvider defaultOrganizationProvider;
    private final OrganizationCreation organizationCreation;
    private final DefaultGroupFinder defaultGroupFinder;
    private final Configuration config;

    public UserUpdater(NewUserNotifier newUserNotifier, DbClient dbClient, UserIndexer userIndexer, OrganizationFlags organizationFlags, DefaultOrganizationProvider defaultOrganizationProvider, OrganizationCreation organizationCreation, DefaultGroupFinder defaultGroupFinder, Configuration config) {
        this.newUserNotifier = newUserNotifier;
        this.dbClient = dbClient;
        this.userIndexer = userIndexer;
        this.organizationFlags = organizationFlags;
        this.defaultOrganizationProvider = defaultOrganizationProvider;
        this.organizationCreation = organizationCreation;
        this.defaultGroupFinder = defaultGroupFinder;
        this.config = config;
    }

    public UserDto createAndCommit(DbSession dbSession, NewUser newUser, Consumer<UserDto> beforeCommit) {
        String login = newUser.login();
        UserDto userDto = this.dbClient.userDao().selectByLogin(dbSession, newUser.login());
        if (userDto == null) {
            userDto = this.saveUser(dbSession, this.createDto(dbSession, newUser));
        } else {
            this.reactivateUser(dbSession, userDto, login, newUser);
        }
        beforeCommit.accept(userDto);
        this.userIndexer.commitAndIndex(dbSession, userDto);
        this.notifyNewUser(userDto.getLogin(), userDto.getName(), newUser.email());
        return userDto;
    }

    private void reactivateUser(DbSession dbSession, UserDto existingUser, String login, NewUser newUser) {
        Preconditions.checkArgument((!existingUser.isActive() ? 1 : 0) != 0, (String)"An active user with login '%s' already exists", (Object[])new Object[]{login});
        UpdateUser updateUser = UpdateUser.create(login).setName(newUser.name()).setEmail(newUser.email()).setScmAccounts(newUser.scmAccounts()).setExternalIdentity(newUser.externalIdentity());
        if (newUser.password() != null) {
            updateUser.setPassword(newUser.password());
        }
        this.setOnboarded(existingUser);
        this.updateDto(dbSession, updateUser, existingUser);
        this.updateUser(dbSession, existingUser);
        this.addUserToDefaultOrganizationAndDefaultGroup(dbSession, existingUser);
    }

    public void updateAndCommit(DbSession dbSession, UpdateUser updateUser, Consumer<UserDto> beforeCommit) {
        UserDto dto = this.dbClient.userDao().selectByLogin(dbSession, updateUser.login());
        WsUtils.checkFound(dto, "User with login '%s' has not been found", updateUser.login());
        boolean isUserUpdated = this.updateDto(dbSession, updateUser, dto);
        if (isUserUpdated) {
            this.updateUser(dbSession, dto);
            beforeCommit.accept(dto);
            this.userIndexer.commitAndIndex(dbSession, dto);
            this.notifyNewUser(dto.getLogin(), dto.getName(), dto.getEmail());
        } else {
            beforeCommit.accept(dto);
            dbSession.commit();
        }
    }

    private UserDto createDto(DbSession dbSession, NewUser newUser) {
        List<String> scmAccounts;
        String password;
        String email;
        String name;
        UserDto userDto = new UserDto();
        ArrayList<String> messages = new ArrayList<String>();
        String login = newUser.login();
        if (UserUpdater.validateLoginFormat(login, messages)) {
            userDto.setLogin(login);
        }
        if (UserUpdater.validateNameFormat(name = newUser.name(), messages)) {
            userDto.setName(name);
        }
        if ((email = newUser.email()) != null && UserUpdater.validateEmailFormat(email, messages)) {
            userDto.setEmail(email);
        }
        if ((password = newUser.password()) != null && UserUpdater.validatePasswords(password, messages)) {
            UserUpdater.setEncryptedPassword(password, userDto);
        }
        if ((scmAccounts = UserUpdater.sanitizeScmAccounts(newUser.scmAccounts())) != null && !scmAccounts.isEmpty() && this.validateScmAccounts(dbSession, scmAccounts, login, email, null, messages)) {
            userDto.setScmAccounts(scmAccounts);
        }
        UserUpdater.setExternalIdentity(userDto, newUser.externalIdentity());
        this.setOnboarded(userDto);
        WsUtils.checkRequest(messages.isEmpty(), messages);
        return userDto;
    }

    private boolean updateDto(DbSession dbSession, UpdateUser update, UserDto dto) {
        ArrayList messages = Lists.newArrayList();
        boolean changed = UserUpdater.updateName(update, dto, messages);
        changed |= UserUpdater.updateEmail(update, dto, messages);
        changed |= UserUpdater.updateExternalIdentity(update, dto);
        changed |= UserUpdater.updatePassword(update, dto, messages);
        WsUtils.checkRequest(messages.isEmpty(), messages);
        return changed |= this.updateScmAccounts(dbSession, update, dto, messages);
    }

    private static boolean updateName(UpdateUser updateUser, UserDto userDto, List<String> messages) {
        String name = updateUser.name();
        if (updateUser.isNameChanged() && UserUpdater.validateNameFormat(name, messages) && !Objects.equals(userDto.getName(), name)) {
            userDto.setName(name);
            return true;
        }
        return false;
    }

    private static boolean updateEmail(UpdateUser updateUser, UserDto userDto, List<String> messages) {
        String email = updateUser.email();
        if (updateUser.isEmailChanged() && UserUpdater.validateEmailFormat(email, messages) && !Objects.equals(userDto.getEmail(), email)) {
            userDto.setEmail(email);
            return true;
        }
        return false;
    }

    private static boolean updateExternalIdentity(UpdateUser updateUser, UserDto userDto) {
        ExternalIdentity externalIdentity = updateUser.externalIdentity();
        if (updateUser.isExternalIdentityChanged() && !UserUpdater.isSameExternalIdentity(userDto, externalIdentity)) {
            UserUpdater.setExternalIdentity(userDto, externalIdentity);
            return true;
        }
        return false;
    }

    private static boolean updatePassword(UpdateUser updateUser, UserDto userDto, List<String> messages) {
        String password = updateUser.password();
        if (updateUser.isPasswordChanged() && UserUpdater.validatePasswords(password, messages) && UserUpdater.checkPasswordChangeAllowed(userDto, messages)) {
            UserUpdater.setEncryptedPassword(password, userDto);
            return true;
        }
        return false;
    }

    private boolean updateScmAccounts(DbSession dbSession, UpdateUser updateUser, UserDto userDto, List<String> messages) {
        String email = updateUser.email();
        List<String> scmAccounts = UserUpdater.sanitizeScmAccounts(updateUser.scmAccounts());
        List existingScmAccounts = userDto.getScmAccountsAsList();
        if (!(!updateUser.isScmAccountsChanged() || existingScmAccounts.containsAll(scmAccounts) && scmAccounts.containsAll(existingScmAccounts))) {
            if (!scmAccounts.isEmpty()) {
                String newOrOldEmail;
                String string = newOrOldEmail = email != null ? email : userDto.getEmail();
                if (this.validateScmAccounts(dbSession, scmAccounts, userDto.getLogin(), newOrOldEmail, userDto, messages)) {
                    userDto.setScmAccounts(scmAccounts);
                }
            } else {
                userDto.setScmAccounts((String)null);
            }
            return true;
        }
        return false;
    }

    private static boolean isSameExternalIdentity(UserDto dto, @Nullable ExternalIdentity externalIdentity) {
        return externalIdentity != null && Objects.equals(dto.getExternalIdentity(), externalIdentity.getId()) && Objects.equals(dto.getExternalIdentityProvider(), externalIdentity.getProvider());
    }

    private static void setExternalIdentity(UserDto dto, @Nullable ExternalIdentity externalIdentity) {
        if (externalIdentity == null) {
            dto.setExternalIdentity(dto.getLogin());
            dto.setExternalIdentityProvider(SQ_AUTHORITY);
            dto.setLocal(true);
        } else {
            dto.setExternalIdentity(externalIdentity.getId());
            dto.setExternalIdentityProvider(externalIdentity.getProvider());
            dto.setLocal(false);
            dto.setSalt(null);
            dto.setCryptedPassword(null);
        }
    }

    private void setOnboarded(UserDto userDto) {
        boolean showOnboarding = this.config.getBoolean("sonar.onboardingTutorial.showToNewUsers").orElse(false);
        userDto.setOnboarded(!showOnboarding);
    }

    private static boolean checkNotEmptyParam(@Nullable String value, String param, List<String> messages) {
        if (Strings.isNullOrEmpty((String)value)) {
            messages.add(String.format("%s can't be empty", param));
            return false;
        }
        return true;
    }

    private static boolean validateLoginFormat(@Nullable String login, List<String> messages) {
        boolean isValid = UserUpdater.checkNotEmptyParam(login, LOGIN_PARAM, messages);
        if (!Strings.isNullOrEmpty((String)login)) {
            if (login.length() < 2) {
                messages.add(String.format("%s is too short (minimum is %s characters)", LOGIN_PARAM, 2));
                return false;
            }
            if (login.length() > 255) {
                messages.add(String.format("%s is too long (maximum is %s characters)", LOGIN_PARAM, 255));
                return false;
            }
            if (!login.matches("\\A\\w[\\w\\.\\-_@]+\\z")) {
                messages.add("Use only letters, numbers, and .-_@ please.");
                return false;
            }
        }
        return isValid;
    }

    private static boolean validateNameFormat(@Nullable String name, List<String> messages) {
        boolean isValid = UserUpdater.checkNotEmptyParam(name, NAME_PARAM, messages);
        if (name != null && name.length() > 200) {
            messages.add(String.format("%s is too long (maximum is %s characters)", NAME_PARAM, 200));
            return false;
        }
        return isValid;
    }

    private static boolean validateEmailFormat(@Nullable String email, List<String> messages) {
        if (email != null && email.length() > 100) {
            messages.add(String.format("%s is too long (maximum is %s characters)", EMAIL_PARAM, 100));
            return false;
        }
        return true;
    }

    private static boolean checkPasswordChangeAllowed(UserDto userDto, List<String> messages) {
        if (!userDto.isLocal()) {
            messages.add("Password cannot be changed when external authentication is used");
            return false;
        }
        return true;
    }

    private static boolean validatePasswords(@Nullable String password, List<String> messages) {
        if (password == null || password.length() == 0) {
            messages.add(String.format("%s can't be empty", PASSWORD_PARAM));
            return false;
        }
        return true;
    }

    private boolean validateScmAccounts(DbSession dbSession, List<String> scmAccounts, @Nullable String login, @Nullable String email, @Nullable UserDto existingUser, List<String> messages) {
        boolean isValid = true;
        for (String scmAccount : scmAccounts) {
            if (scmAccount.equals(login) || scmAccount.equals(email)) {
                messages.add("Login and email are automatically considered as SCM accounts");
                isValid = false;
                continue;
            }
            List matchingUsers = this.dbClient.userDao().selectByScmAccountOrLoginOrEmail(dbSession, scmAccount);
            ArrayList matchingUsersWithoutExistingUser = Lists.newArrayList();
            for (UserDto matchingUser : matchingUsers) {
                if (existingUser != null && matchingUser.getId().equals(existingUser.getId())) continue;
                matchingUsersWithoutExistingUser.add(matchingUser.getName() + " (" + matchingUser.getLogin() + ")");
            }
            if (matchingUsersWithoutExistingUser.isEmpty()) continue;
            messages.add(String.format("The scm account '%s' is already used by user(s) : '%s'", scmAccount, Joiner.on((String)", ").join((Iterable)matchingUsersWithoutExistingUser)));
            isValid = false;
        }
        return isValid;
    }

    private static List<String> sanitizeScmAccounts(@Nullable List<String> scmAccounts) {
        if (scmAccounts != null) {
            return (List)new HashSet<String>(scmAccounts).stream().map(Strings::emptyToNull).filter(Objects::nonNull).sorted(String::compareToIgnoreCase).collect(MoreCollectors.toList((int)scmAccounts.size()));
        }
        return Collections.emptyList();
    }

    private UserDto saveUser(DbSession dbSession, UserDto userDto) {
        userDto.setActive(true);
        UserDto res = this.dbClient.userDao().insert(dbSession, userDto);
        this.addUserToDefaultOrganizationAndDefaultGroup(dbSession, userDto);
        this.organizationCreation.createForUser(dbSession, userDto);
        return res;
    }

    private void updateUser(DbSession dbSession, UserDto dto) {
        dto.setActive(true);
        this.dbClient.userDao().update(dbSession, dto);
    }

    private static void setEncryptedPassword(String password, UserDto userDto) {
        SecureRandom random = new SecureRandom();
        byte[] salt = new byte[32];
        ((Random)random).nextBytes(salt);
        String saltHex = DigestUtils.sha1Hex((byte[])salt);
        userDto.setSalt(saltHex);
        userDto.setCryptedPassword(UserDto.encryptPassword((String)password, (String)saltHex));
    }

    private void notifyNewUser(String login, String name, @Nullable String email) {
        this.newUserNotifier.onNewUser(NewUserHandler.Context.builder().setLogin(login).setName(name).setEmail(email).build());
    }

    private static boolean isUserAlreadyMemberOfDefaultGroup(GroupDto defaultGroup, List<GroupDto> userGroups) {
        return userGroups.stream().anyMatch(group -> defaultGroup.getId().equals(group.getId()));
    }

    private void addUserToDefaultOrganizationAndDefaultGroup(DbSession dbSession, UserDto userDto) {
        if (this.organizationFlags.isEnabled(dbSession)) {
            return;
        }
        this.addUserToDefaultOrganization(dbSession, userDto);
        this.addDefaultGroup(dbSession, userDto);
    }

    private void addUserToDefaultOrganization(DbSession dbSession, UserDto userDto) {
        String defOrgUuid = this.defaultOrganizationProvider.get().getUuid();
        this.dbClient.organizationMemberDao().insert(dbSession, new OrganizationMemberDto().setOrganizationUuid(defOrgUuid).setUserId(userDto.getId()));
    }

    private void addDefaultGroup(DbSession dbSession, UserDto userDto) {
        String defOrgUuid = this.defaultOrganizationProvider.get().getUuid();
        List userGroups = this.dbClient.groupDao().selectByUserLogin(dbSession, userDto.getLogin());
        GroupDto defaultGroup = this.defaultGroupFinder.findDefaultGroup(dbSession, defOrgUuid);
        if (UserUpdater.isUserAlreadyMemberOfDefaultGroup(defaultGroup, userGroups)) {
            return;
        }
        this.dbClient.userGroupDao().insert(dbSession, new UserGroupDto().setUserId(userDto.getId().intValue()).setGroupId(defaultGroup.getId().intValue()));
    }
}

