/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.qualitygate;

import com.google.common.base.Strings;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.ObjectUtils;
import org.sonar.api.measures.Metric;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.metric.MetricDto;
import org.sonar.db.qualitygate.QualityGateConditionDto;
import org.sonar.db.qualitygate.QualityGateDto;
import org.sonar.server.computation.task.projectanalysis.qualitymodel.RatingGrid;
import org.sonar.server.exceptions.NotFoundException;
import org.sonar.server.qualitygate.ValidRatingMetrics;
import org.sonar.server.ws.WsUtils;

public class QualityGateConditionsUpdater {
    private static final List<String> RATING_VALID_INT_VALUES = Arrays.stream(RatingGrid.Rating.values()).map(r -> Integer.toString(r.getIndex())).collect(Collectors.toList());
    private final DbClient dbClient;

    public QualityGateConditionsUpdater(DbClient dbClient) {
        this.dbClient = dbClient;
    }

    public QualityGateConditionDto createCondition(DbSession dbSession, long qGateId, String metricKey, String operator, @Nullable String warningThreshold, @Nullable String errorThreshold, @Nullable Integer period) {
        this.getNonNullQgate(dbSession, qGateId);
        MetricDto metric = this.getNonNullMetric(dbSession, metricKey);
        QualityGateConditionsUpdater.validateCondition(metric, operator, warningThreshold, errorThreshold, period);
        QualityGateConditionsUpdater.checkConditionDoesNotAlreadyExistOnSameMetricAndPeriod(this.getConditions(dbSession, qGateId, null), metric, period);
        QualityGateConditionDto newCondition = new QualityGateConditionDto().setQualityGateId(qGateId).setMetricId((long)metric.getId().intValue()).setMetricKey(metric.getKey()).setOperator(operator).setWarningThreshold(warningThreshold).setErrorThreshold(errorThreshold).setPeriod(period);
        this.dbClient.gateConditionDao().insert(newCondition, dbSession);
        return newCondition;
    }

    public QualityGateConditionDto updateCondition(DbSession dbSession, long condId, String metricKey, String operator, @Nullable String warningThreshold, @Nullable String errorThreshold, @Nullable Integer period) {
        QualityGateConditionDto condition = this.getNonNullCondition(dbSession, condId);
        MetricDto metric = this.getNonNullMetric(dbSession, metricKey);
        QualityGateConditionsUpdater.validateCondition(metric, operator, warningThreshold, errorThreshold, period);
        QualityGateConditionsUpdater.checkConditionDoesNotAlreadyExistOnSameMetricAndPeriod(this.getConditions(dbSession, condition.getQualityGateId(), condition.getId()), metric, period);
        condition.setMetricId((long)metric.getId().intValue()).setMetricKey(metric.getKey()).setOperator(operator).setWarningThreshold(warningThreshold).setErrorThreshold(errorThreshold).setPeriod(period);
        this.dbClient.gateConditionDao().update(condition, dbSession);
        return condition;
    }

    private QualityGateDto getNonNullQgate(DbSession dbSession, long id) {
        QualityGateDto qGate = this.dbClient.qualityGateDao().selectById(dbSession, id);
        if (qGate == null) {
            throw new NotFoundException(String.format("There is no quality gate with id=%s", id));
        }
        return qGate;
    }

    private MetricDto getNonNullMetric(DbSession dbSession, String metricKey) {
        MetricDto metric = this.dbClient.metricDao().selectByKey(dbSession, metricKey);
        if (metric == null) {
            throw new NotFoundException(String.format("There is no metric with key=%s", metricKey));
        }
        return metric;
    }

    private QualityGateConditionDto getNonNullCondition(DbSession dbSession, long id) {
        QualityGateConditionDto condition = this.dbClient.gateConditionDao().selectById(id, dbSession);
        if (condition == null) {
            throw new NotFoundException("There is no condition with id=" + id);
        }
        return condition;
    }

    private Collection<QualityGateConditionDto> getConditions(DbSession dbSession, long qGateId, @Nullable Long conditionId) {
        Collection conditions = this.dbClient.gateConditionDao().selectForQualityGate(dbSession, qGateId);
        if (conditionId == null) {
            return conditions;
        }
        return this.dbClient.gateConditionDao().selectForQualityGate(dbSession, qGateId).stream().filter(condition -> condition.getId() != conditionId.longValue()).collect(Collectors.toList());
    }

    private static void validateCondition(MetricDto metric, String operator, @Nullable String warningThreshold, @Nullable String errorThreshold, @Nullable Integer period) {
        ArrayList<String> errors = new ArrayList<String>();
        QualityGateConditionsUpdater.validateMetric(metric, errors);
        QualityGateConditionsUpdater.checkOperator(metric, operator, errors);
        QualityGateConditionsUpdater.checkThresholds(warningThreshold, errorThreshold, errors);
        QualityGateConditionsUpdater.checkPeriod(metric, period, errors);
        QualityGateConditionsUpdater.checkRatingMetric(metric, warningThreshold, errorThreshold, period, errors);
        WsUtils.checkRequest(errors.isEmpty(), errors);
    }

    private static void validateMetric(MetricDto metric, List<String> errors) {
        QualityGateConditionsUpdater.check(QualityGateConditionsUpdater.isAlertable(metric), errors, "Metric '%s' cannot be used to define a condition.", metric.getKey());
    }

    private static boolean isAlertable(MetricDto metric) {
        return QualityGateConditionsUpdater.isAvailableForInit(metric) && BooleanUtils.isFalse((Boolean)metric.isHidden());
    }

    private static boolean isAvailableForInit(MetricDto metric) {
        return !metric.isDataType() && !"alert_status".equals(metric.getKey());
    }

    private static void checkOperator(MetricDto metric, String operator, List<String> errors) {
        Metric.ValueType valueType = Metric.ValueType.valueOf((String)metric.getValueType());
        QualityGateConditionsUpdater.check(QualityGateConditionDto.isOperatorAllowed((String)operator, (Metric.ValueType)valueType), errors, "Operator %s is not allowed for metric type %s.", operator, metric.getValueType());
    }

    private static void checkThresholds(@Nullable String warningThreshold, @Nullable String errorThreshold, List<String> errors) {
        QualityGateConditionsUpdater.check(warningThreshold != null || errorThreshold != null, errors, "At least one threshold (warning, error) must be set.", new String[0]);
    }

    private static void checkPeriod(MetricDto metric, @Nullable Integer period, List<String> errors) {
        if (period == null) {
            QualityGateConditionsUpdater.check(!metric.getKey().startsWith("new_"), errors, "A period must be selected for differential metrics.", new String[0]);
        } else {
            QualityGateConditionsUpdater.check(period == 1, errors, "The only valid quality gate period is 1, the leak period.", new String[0]);
        }
    }

    private static void checkConditionDoesNotAlreadyExistOnSameMetricAndPeriod(Collection<QualityGateConditionDto> conditions, MetricDto metric, @Nullable Integer period) {
        if (conditions.isEmpty()) {
            return;
        }
        boolean conditionExists = conditions.stream().anyMatch(c -> c.getMetricId() == (long)metric.getId().intValue() && ObjectUtils.equals((Object)c.getPeriod(), (Object)period));
        WsUtils.checkRequest(!conditionExists, period == null ? String.format("Condition on metric '%s' already exists.", metric.getShortName()) : String.format("Condition on metric '%s' over leak period already exists.", metric.getShortName()), new Object[0]);
    }

    private static void checkRatingMetric(MetricDto metric, @Nullable String warningThreshold, @Nullable String errorThreshold, @Nullable Integer period, List<String> errors) {
        if (!metric.getValueType().equals(Metric.ValueType.RATING.name())) {
            return;
        }
        if (!ValidRatingMetrics.isCoreRatingMetric(metric.getKey())) {
            errors.add(String.format("The metric '%s' cannot be used", metric.getShortName()));
        }
        if (period != null && !metric.getKey().startsWith("new_")) {
            errors.add(String.format("The metric '%s' cannot be used on the leak period", metric.getShortName()));
        }
        if (!QualityGateConditionsUpdater.isValidRating(warningThreshold)) {
            QualityGateConditionsUpdater.addInvalidRatingError(warningThreshold, errors);
            return;
        }
        if (!QualityGateConditionsUpdater.isValidRating(errorThreshold)) {
            QualityGateConditionsUpdater.addInvalidRatingError(errorThreshold, errors);
            return;
        }
        QualityGateConditionsUpdater.checkRatingGreaterThanOperator(warningThreshold, errors);
        QualityGateConditionsUpdater.checkRatingGreaterThanOperator(errorThreshold, errors);
    }

    private static void addInvalidRatingError(@Nullable String value, List<String> errors) {
        errors.add(String.format("'%s' is not a valid rating", value));
    }

    private static void checkRatingGreaterThanOperator(@Nullable String value, List<String> errors) {
        QualityGateConditionsUpdater.check(Strings.isNullOrEmpty((String)value) || !Objects.equals((Object)QualityGateConditionsUpdater.toRating(value), (Object)RatingGrid.Rating.E), errors, "There's no worse rating than E (%s)", value);
    }

    private static RatingGrid.Rating toRating(String value) {
        return RatingGrid.Rating.valueOf(Integer.parseInt(value));
    }

    private static boolean isValidRating(@Nullable String value) {
        return Strings.isNullOrEmpty((String)value) || RATING_VALID_INT_VALUES.contains(value);
    }

    private static boolean check(boolean expression, List<String> errors, String message, String ... args) {
        if (!expression) {
            errors.add(String.format(message, args));
        }
        return expression;
    }
}

