/**
 * Copyright 2005-2014 Restlet
 * 
 * The contents of this file are subject to the terms of one of the following
 * open source licenses: Apache 2.0 or LGPL 3.0 or LGPL 2.1 or CDDL 1.0 or EPL
 * 1.0 (the "Licenses"). You can select the license that you prefer but you may
 * not use this file except in compliance with one of these Licenses.
 * 
 * You can obtain a copy of the Apache 2.0 license at
 * http://www.opensource.org/licenses/apache-2.0
 * 
 * You can obtain a copy of the LGPL 3.0 license at
 * http://www.opensource.org/licenses/lgpl-3.0
 * 
 * You can obtain a copy of the LGPL 2.1 license at
 * http://www.opensource.org/licenses/lgpl-2.1
 * 
 * You can obtain a copy of the CDDL 1.0 license at
 * http://www.opensource.org/licenses/cddl1
 * 
 * You can obtain a copy of the EPL 1.0 license at
 * http://www.opensource.org/licenses/eclipse-1.0
 * 
 * See the Licenses for the specific language governing permissions and
 * limitations under the Licenses.
 * 
 * Alternatively, you can obtain a royalty free commercial license with less
 * limitations, transferable or non-transferable, directly at
 * http://restlet.com/products/restlet-framework
 * 
 * Restlet is a registered trademark of Restlet S.A.S.
 */

package org.restlet.ext.servlet.internal;

import java.util.Enumeration;

import org.restlet.Context;
import org.restlet.data.Form;
import org.restlet.engine.adapter.HttpRequest;
import org.restlet.engine.adapter.ServerAdapter;
import org.restlet.engine.adapter.ServerCall;

/**
 * Default server adapter from Servlet calls to Restlet calls. This class is
 * used by the {@code ServerServlet} to ensure that Servlet specific concepts
 * are properly transfered to Restlet.<br>
 * Especially, it makes sure that the "jsessionid" matrix parameter is removed
 * from the resource's reference, as this may interfer with the routing process,
 * and because this parameter is useless for Restlet-based applications.<br>
 * <br>
 * it also copies the Servlet's request attributes into the Restlet's request
 * attributes map.
 * 
 * @author Jeremy Gustie
 */
public class ServletServerAdapter extends ServerAdapter {

    /**
     * Constructor.
     * 
     * @param context
     *            The context to use.
     */
    public ServletServerAdapter(Context context) {
        super(context);
    }

    @Override
    public HttpRequest toRequest(ServerCall httpCall) {
        final HttpRequest result = super.toRequest(httpCall);

        // Remove the Servlet API "jsessionid" matrix parameter
        Form matrixForm = result.getResourceRef().getMatrixAsForm();
        if (matrixForm.removeAll("jsessionid", true)) {
            String lastSegment = result.getResourceRef().getLastSegment();
            final int matrixIndex = lastSegment.indexOf(';');
            if (matrixForm.isEmpty()) {
                // No matrix left
                lastSegment = lastSegment.substring(0, matrixIndex);
            } else {
                // Add the remaining matrix parameters back in
                lastSegment = lastSegment.substring(0, matrixIndex + 1)
                        + matrixForm.getMatrixString();
            }
            result.getResourceRef().setLastSegment(lastSegment);
        }

        if (httpCall instanceof ServletCall) {
            ServletCall servletCall = (ServletCall) httpCall;

            // Copy all Servlet's request attributes
            String attributeName;
            for (final Enumeration<String> namesEnum = servletCall.getRequest()
                    .getAttributeNames(); namesEnum.hasMoreElements();) {
                attributeName = namesEnum.nextElement();
                result.getAttributes().put(attributeName,
                        servletCall.getRequest().getAttribute(attributeName));
            }
        }

        return result;
    }

}
