/**
 * Copyright 2007-2012 Bull S.A.S.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.util.ant.archive.exploded;

import java.io.File;
import java.util.List;

import org.apache.tools.ant.Project;
import org.ow2.util.ant.archive.AbsArchive;
import org.ow2.util.ant.archive.Client;
import org.ow2.util.ant.archive.Ejb;
import org.ow2.util.ant.archive.War;
import org.ow2.util.ant.archive.api.IEar;
import org.ow2.util.ant.archive.info.EarInfo;

/**
 * Creates an EAR exploded archive.
 * @author Florent Benoit
 */
public class EarExploded extends AbsExplodedArchive implements IEar {

    /**
     * Path to the Standard deployment descriptor.
     */
    private static final String DEPLOYMENT_DESCRIPTOR = "META-INF/application.xml";

    /**
     * Path to the Specific deployment descriptor.
     */
    private static final String SPECIFIC_DEPLOYMENT_DESCRIPTOR = "META-INF/easybeans.xml";

    /**
     * Ear info object.
     */
    private EarInfo earInfo = null;


    /**
     * Creates an archive for the given project.
     * @param p the given project
     */
    public EarExploded(final Project p) {
        super(p);
    }

    /**
     * Sets the information about an EAR archive.
     * @param earInfo the object that holds data information.
     */
    public void setEarInfo(final EarInfo earInfo) {
        setArchiveInfo(earInfo);
        this.earInfo = earInfo;
    }

    /**
     * Gets the path to the standard deployment descriptor.
     * @return the path to the standard deployment descriptor.
     */
    @Override
    public String getDDStandardName() {
        return DEPLOYMENT_DESCRIPTOR;
    }

    /**
     * Execute the task.
     */
    @Override
    public void execute() {
        // create directory
        this.earInfo.getDest().mkdirs();

        // Include the DD if any and initial fileset
        super.execute();

        // Execute children by changing the destination directory
        List<Ejb> ejbs = this.earInfo.getEjbs();
        List<War> wars = this.earInfo.getWars();
        List<Client> clients = this.earInfo.getClients();

        if (ejbs != null) {
            for (Ejb ejb : ejbs) {
                updateArchive(ejb);
                ejb.execute();
            }
        }

        if (wars != null) {
            for (War war : wars) {
                updateArchive(war);
                war.execute();
            }
        }
        if (clients != null) {
            for (Client client : clients) {
                updateArchive(client);
                client.execute();
            }
        }

    }

    /**
     * Update the archive with some settings.
     * @param archive the archive to update
     */
    void updateArchive(final AbsArchive archive) {
        archive.setDest(new File(this.earInfo.getDest(), archive.getDest().getName()));
    }

    /**
     * Gets the path to the specific deployment descriptor.
     * @return the path to the specific deployment descriptor.
     */
    @Override
    public String getDDSpecificame() {
        return SPECIFIC_DEPLOYMENT_DESCRIPTOR;
    }
}
