/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch._types.query_dsl;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.opensearch._types.FieldValue;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;

// typedef: _types.query_dsl.TermQuery

@JsonpDeserializable
public class TermQuery extends QueryBase implements QueryVariant {
    // Single key dictionary
    private final String field;

    private final FieldValue value;

    @Nullable
    private final Boolean caseInsensitive;

    // ---------------------------------------------------------------------------------------------

    private TermQuery(Builder builder) {
        super(builder);
        this.field = ApiTypeHelper.requireNonNull(builder.field, this, "field");

        this.value = ApiTypeHelper.requireNonNull(builder.value, this, "value");
        this.caseInsensitive = builder.caseInsensitive;

    }

    public static TermQuery of(Function<Builder, ObjectBuilder<TermQuery>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Query variant kind.
     */
    @Override
    public Query.Kind _queryKind() {
        return Query.Kind.Term;
    }

    /**
     * Required - The target field
     */
    public final String field() {
        return this.field;
    }

    /**
     * Required - API name: {@code value}
     */
    public final FieldValue value() {
        return this.value;
    }

    /**
     * API name: {@code case_insensitive}
     */
    @Nullable
    public final Boolean caseInsensitive() {
        return this.caseInsensitive;
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject(this.field);

        super.serializeInternal(generator, mapper);
        generator.writeKey("value");
        this.value.serialize(generator, mapper);

        if (this.caseInsensitive != null) {
            generator.writeKey("case_insensitive");
            generator.write(this.caseInsensitive);

        }

        generator.writeEnd();

    }

    public Builder toBuilder() {
        return toBuilder(new Builder()).field(field).value(value).caseInsensitive(caseInsensitive);
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link TermQuery}.
     */

    public static class Builder extends QueryBase.AbstractBuilder<Builder> implements ObjectBuilder<TermQuery> {
        private String field;

        /**
         * Required - The target field
         */
        public final Builder field(String value) {
            this.field = value;
            return this;
        }

        private FieldValue value;

        @Nullable
        private Boolean caseInsensitive;

        /**
         * Required - API name: {@code value}
         */
        public final Builder value(FieldValue value) {
            this.value = value;
            return this;
        }

        /**
         * Required - API name: {@code value}
         */
        public final Builder value(Function<FieldValue.Builder, ObjectBuilder<FieldValue>> fn) {
            return this.value(fn.apply(new FieldValue.Builder()).build());
        }

        /**
         * API name: {@code case_insensitive}
         */
        public final Builder caseInsensitive(@Nullable Boolean value) {
            this.caseInsensitive = value;
            return this;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * Builds a {@link TermQuery}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public TermQuery build() {
            _checkSingleUse();

            return new TermQuery(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link TermQuery}
     */
    public static final JsonpDeserializer<TermQuery> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        TermQuery::setupTermQueryDeserializer
    );

    protected static void setupTermQueryDeserializer(ObjectDeserializer<TermQuery.Builder> op) {
        QueryBase.setupQueryBaseDeserializer(op);
        op.add(Builder::value, FieldValue._DESERIALIZER, "value");
        op.add(Builder::caseInsensitive, JsonpDeserializer.booleanDeserializer(), "case_insensitive");

        op.setKey(Builder::field, JsonpDeserializer.stringDeserializer());
        op.shortcutProperty("value");

    }

}
