/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.indices;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.json.PlainJsonSerializable;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: indices._types.DataStream

@JsonpDeserializable
public class DataStream implements PlainJsonSerializable {
    @Nullable
    private final Boolean hidden;

    @Nullable
    private final DataStreamTimestampField timestampField;

    // ---------------------------------------------------------------------------------------------

    private DataStream(Builder builder) {

        this.hidden = builder.hidden;
        this.timestampField = builder.timestampField;

    }

    public static DataStream of(Function<Builder, ObjectBuilder<DataStream>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * API name: {@code hidden}
     */
    @Nullable
    public final Boolean hidden() {
        return this.hidden;
    }

    /**
     * API name: {@code timestamp_field}
     */
    @Nullable
    public final DataStreamTimestampField timestampField() {
        return this.timestampField;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        if (this.hidden != null) {
            generator.writeKey("hidden");
            generator.write(this.hidden);
        }

        if (this.timestampField != null) {
            generator.writeKey("timestamp_field");
            this.timestampField.serialize(generator, mapper);
        }

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link DataStream}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<DataStream> {
        @Nullable
        private Boolean hidden;

        @Nullable
        private DataStreamTimestampField timestampField;

        /**
         * API name: {@code hidden}
         */
        public final Builder hidden(@Nullable Boolean value) {
            this.hidden = value;
            return this;
        }

        /**
         * API name: {@code timestamp_field}
         */
        public final Builder timestampField(@Nullable DataStreamTimestampField value) {
            this.timestampField = value;
            return this;
        }

        /**
         * API name: {@code timestamp_field}
         */
        public final Builder timestampField(Function<DataStreamTimestampField.Builder, ObjectBuilder<DataStreamTimestampField>> fn) {
            return this.timestampField(fn.apply(new DataStreamTimestampField.Builder()).build());
        }

        /**
         * Builds a {@link DataStream}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public DataStream build() {
            _checkSingleUse();

            return new DataStream(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link DataStream}
     */
    public static final JsonpDeserializer<DataStream> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        DataStream::setupDataStreamDeserializer
    );

    protected static void setupDataStreamDeserializer(ObjectDeserializer<DataStream.Builder> op) {

        op.add(Builder::hidden, JsonpDeserializer.booleanDeserializer(), "hidden");
        op.add(Builder::timestampField, DataStreamTimestampField._DESERIALIZER, "timestamp_field");

    }

}
