/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.ingest;

import jakarta.json.stream.JsonGenerator;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;

// typedef: ingest._types.ConvertProcessor

@JsonpDeserializable
public class ConvertProcessor extends ProcessorBase implements ProcessorVariant {
    private final String field;

    @Nullable
    private final Boolean ignoreMissing;

    private final String targetField;

    private final ConvertType type;

    // ---------------------------------------------------------------------------------------------

    private ConvertProcessor(Builder builder) {
        super(builder);

        this.field = ApiTypeHelper.requireNonNull(builder.field, this, "field");
        this.ignoreMissing = builder.ignoreMissing;
        this.targetField = ApiTypeHelper.requireNonNull(builder.targetField, this, "targetField");
        this.type = ApiTypeHelper.requireNonNull(builder.type, this, "type");

    }

    public static ConvertProcessor of(Function<Builder, ObjectBuilder<ConvertProcessor>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Processor variant kind.
     */
    @Override
    public Processor.Kind _processorKind() {
        return Processor.Kind.Convert;
    }

    /**
     * Required - API name: {@code field}
     */
    public final String field() {
        return this.field;
    }

    /**
     * API name: {@code ignore_missing}
     */
    @Nullable
    public final Boolean ignoreMissing() {
        return this.ignoreMissing;
    }

    /**
     * Required - API name: {@code target_field}
     */
    public final String targetField() {
        return this.targetField;
    }

    /**
     * Required - API name: {@code type}
     */
    public final ConvertType type() {
        return this.type;
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        super.serializeInternal(generator, mapper);
        generator.writeKey("field");
        generator.write(this.field);

        if (this.ignoreMissing != null) {
            generator.writeKey("ignore_missing");
            generator.write(this.ignoreMissing);

        }
        generator.writeKey("target_field");
        generator.write(this.targetField);

        generator.writeKey("type");
        this.type.serialize(generator, mapper);

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link ConvertProcessor}.
     */

    public static class Builder extends ProcessorBase.AbstractBuilder<Builder> implements ObjectBuilder<ConvertProcessor> {
        private String field;

        @Nullable
        private Boolean ignoreMissing;

        private String targetField;

        private ConvertType type;

        /**
         * Required - API name: {@code field}
         */
        public final Builder field(String value) {
            this.field = value;
            return this;
        }

        /**
         * API name: {@code ignore_missing}
         */
        public final Builder ignoreMissing(@Nullable Boolean value) {
            this.ignoreMissing = value;
            return this;
        }

        /**
         * Required - API name: {@code target_field}
         */
        public final Builder targetField(String value) {
            this.targetField = value;
            return this;
        }

        /**
         * Required - API name: {@code type}
         */
        public final Builder type(ConvertType value) {
            this.type = value;
            return this;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * Builds a {@link ConvertProcessor}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public ConvertProcessor build() {
            _checkSingleUse();

            return new ConvertProcessor(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link ConvertProcessor}
     */
    public static final JsonpDeserializer<ConvertProcessor> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        ConvertProcessor::setupConvertProcessorDeserializer
    );

    protected static void setupConvertProcessorDeserializer(ObjectDeserializer<ConvertProcessor.Builder> op) {
        ProcessorBase.setupProcessorBaseDeserializer(op);
        op.add(Builder::field, JsonpDeserializer.stringDeserializer(), "field");
        op.add(Builder::ignoreMissing, JsonpDeserializer.booleanDeserializer(), "ignore_missing");
        op.add(Builder::targetField, JsonpDeserializer.stringDeserializer(), "target_field");
        op.add(Builder::type, ConvertType._DESERIALIZER, "type");

    }

}
