/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.saml.saml2.metadata.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.AttributeMap;
import org.opensaml.core.xml.util.IndexedXMLObjectChildrenList;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.opensaml.saml.common.AbstractSignableSAMLObject;
import org.opensaml.saml.saml2.metadata.Extensions;
import org.opensaml.saml.saml2.metadata.AdditionalMetadataLocation;
import org.opensaml.saml.saml2.metadata.AffiliationDescriptor;
import org.opensaml.saml.saml2.metadata.AttributeAuthorityDescriptor;
import org.opensaml.saml.saml2.metadata.AuthnAuthorityDescriptor;
import org.opensaml.saml.saml2.metadata.ContactPerson;
import org.opensaml.saml.saml2.metadata.EntityDescriptor;
import org.opensaml.saml.saml2.metadata.IDPSSODescriptor;
import org.opensaml.saml.saml2.metadata.Organization;
import org.opensaml.saml.saml2.metadata.PDPDescriptor;
import org.opensaml.saml.saml2.metadata.RoleDescriptor;
import org.opensaml.saml.saml2.metadata.SPSSODescriptor;
import org.opensaml.xmlsec.signature.Signature;

import net.shibboleth.shared.annotation.constraint.Live;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;

/**
 * Concretate implementation of {@link EntityDescriptor}.
 */
public class EntityDescriptorImpl extends AbstractSignableSAMLObject implements EntityDescriptor {

    /** Entity ID of this Entity. */
    @Nullable private String entityID;

    /** ID attribute. */
    @Nullable private String id;

    /** validUntil attribute. */
    @Nullable private Instant validUntil;

    /** cacheDurection attribute. */
    @Nullable private Duration cacheDuration;

    /** Extensions child. */
    @Nullable private Extensions extensions;

    /** Role descriptors for this entity. */
    @Nonnull private final IndexedXMLObjectChildrenList<RoleDescriptor> roleDescriptors;

    /** Affiliatition descriptor for this entity. */
    @Nullable private AffiliationDescriptor affiliationDescriptor;

    /** Organization the administers this entity. */
    @Nullable private Organization organization;

    /** Contact persons for this entity. */
    @Nonnull private final XMLObjectChildrenList<ContactPerson> contactPersons;

    /** Additional metadata locations for this entity. */
    @Nonnull private final XMLObjectChildrenList<AdditionalMetadataLocation> additionalMetadata;

    /** "anyAttribute" attributes. */
    @Nonnull private final AttributeMap unknownAttributes;

    /**
     * Constructor.
     * 
     * @param namespaceURI the namespace the element is in
     * @param elementLocalName the local name of the XML element this Object represents
     * @param namespacePrefix the prefix for the given namespace
     */
    protected EntityDescriptorImpl(@Nullable final String namespaceURI, @Nonnull final String elementLocalName,
            @Nullable final String namespacePrefix) {
        super(namespaceURI, elementLocalName, namespacePrefix);
        roleDescriptors = new IndexedXMLObjectChildrenList<>(this);
        contactPersons = new XMLObjectChildrenList<>(this);
        additionalMetadata = new XMLObjectChildrenList<>(this);
        unknownAttributes = new AttributeMap(this);
    }

    /** {@inheritDoc} */
    @Nullable public String getEntityID() {
        return entityID;
    }

    /** {@inheritDoc} */
    public void setEntityID(@Nullable final String newId) {
        if (newId != null && newId.length() > 1024) {
            throw new IllegalArgumentException("Entity ID can not exceed 1024 characters in length");
        }
        entityID = prepareForAssignment(entityID, newId);
    }

    /** {@inheritDoc} */
    @Nullable public String getID() {
        return id;
    }

    /** {@inheritDoc} */
    public void setID(@Nullable final String newID) {
        final String oldID = id;
        id = prepareForAssignment(id, newID);
        registerOwnID(oldID, id);
    }

    /** {@inheritDoc} */
    public boolean isValid() {
        if (null == validUntil) {
            return true;
        }

        return Instant.now().isBefore(validUntil);
    }

    /** {@inheritDoc} */
    @Nullable public Instant getValidUntil() {
        return validUntil;
    }

    /** {@inheritDoc} */
    public void setValidUntil(@Nullable final Instant newValidUntil) {
        validUntil = prepareForAssignment(validUntil, newValidUntil);
    }

    /** {@inheritDoc} */
    @Nullable public Duration getCacheDuration() {
        return cacheDuration;
    }

    /** {@inheritDoc} */
    public void setCacheDuration(@Nullable final Duration duration) {
        cacheDuration = prepareForAssignment(cacheDuration, duration);
    }

    /** {@inheritDoc} */
    @Nullable public Extensions getExtensions() {
        return extensions;
    }

    /** {@inheritDoc} */
    public void setExtensions(@Nullable final Extensions newExtensions) {
        extensions = prepareForAssignment(extensions, newExtensions);
    }

    /** {@inheritDoc} */
    @Nonnull @Live public List<RoleDescriptor> getRoleDescriptors() {
        return roleDescriptors;
    }

    /** {@inheritDoc} */
    @SuppressWarnings("unchecked")
    @Nonnull @Live public List<RoleDescriptor> getRoleDescriptors(@Nonnull final QName typeOrName) {
        return (List<RoleDescriptor>) roleDescriptors.subList(typeOrName);
    }

    /** {@inheritDoc} */
    @Nonnull @NotLive @Unmodifiable public List<RoleDescriptor> getRoleDescriptors(@Nonnull final QName type,
            @Nonnull @NotEmpty final String supportedProtocol) {
        final ArrayList<RoleDescriptor> supportingRoleDescriptors = new ArrayList<>();
        for (final RoleDescriptor descriptor : roleDescriptors.subList(type)) {
            if (descriptor.isSupportedProtocol(supportedProtocol)) {
                supportingRoleDescriptors.add(descriptor);
            }
        }

        return supportingRoleDescriptors;
    }

    /** {@inheritDoc} */
    @Nullable public IDPSSODescriptor getIDPSSODescriptor(@Nonnull @NotEmpty final String supportedProtocol) {
        final List<RoleDescriptor> descriptors =
                getRoleDescriptors(IDPSSODescriptor.DEFAULT_ELEMENT_NAME, supportedProtocol);
        if (descriptors.size() > 0) {
            return (IDPSSODescriptor) descriptors.get(0);
        }

        return null;
    }

    /** {@inheritDoc} */
    @Nullable public SPSSODescriptor getSPSSODescriptor(@Nonnull @NotEmpty final String supportedProtocol) {
        final List<RoleDescriptor> descriptors =
                getRoleDescriptors(SPSSODescriptor.DEFAULT_ELEMENT_NAME, supportedProtocol);
        if (descriptors.size() > 0) {
            return (SPSSODescriptor) descriptors.get(0);
        }

        return null;
    }

    /** {@inheritDoc} */
    @Nullable public AuthnAuthorityDescriptor getAuthnAuthorityDescriptor(
            @Nonnull @NotEmpty final String supportedProtocol) {
        final List<RoleDescriptor> descriptors = getRoleDescriptors(AuthnAuthorityDescriptor.DEFAULT_ELEMENT_NAME,
                supportedProtocol);
        if (descriptors.size() > 0) {
            return (AuthnAuthorityDescriptor) descriptors.get(0);
        }

        return null;
    }

    /** {@inheritDoc} */
    @Nullable public AttributeAuthorityDescriptor getAttributeAuthorityDescriptor(
            @Nonnull @NotEmpty final String supportedProtocol) {
        final List<RoleDescriptor> descriptors = getRoleDescriptors(AttributeAuthorityDescriptor.DEFAULT_ELEMENT_NAME,
                supportedProtocol);
        if (descriptors.size() > 0) {
            return (AttributeAuthorityDescriptor) descriptors.get(0);
        }

        return null;
    }

    /** {@inheritDoc} */
    @Nullable public PDPDescriptor getPDPDescriptor(@Nonnull @NotEmpty final String supportedProtocol) {
        final List<RoleDescriptor> descriptors =
                getRoleDescriptors(PDPDescriptor.DEFAULT_ELEMENT_NAME, supportedProtocol);
        if (descriptors.size() > 0) {
            return (PDPDescriptor) descriptors.get(0);
        }

        return null;
    }

    /** {@inheritDoc} */
    @Nullable public AffiliationDescriptor getAffiliationDescriptor() {
        return affiliationDescriptor;
    }

    /** {@inheritDoc} */
    public void setAffiliationDescriptor(@Nullable final AffiliationDescriptor descriptor) {
        affiliationDescriptor = prepareForAssignment(affiliationDescriptor, descriptor);
    }

    /** {@inheritDoc} */
    @Nullable public Organization getOrganization() {
        return organization;
    }

    /** {@inheritDoc} */
    public void setOrganization(@Nullable final Organization newOrganization) {
        organization = prepareForAssignment(organization, newOrganization);
    }

    /** {@inheritDoc} */
    @Nonnull @Live public List<ContactPerson> getContactPersons() {
        return contactPersons;
    }

    /** {@inheritDoc} */
    @Nonnull @Live public List<AdditionalMetadataLocation> getAdditionalMetadataLocations() {
        return additionalMetadata;
    }

    /**
     * {@inheritDoc}
     */
    @Nonnull public AttributeMap getUnknownAttributes() {
        return unknownAttributes;
    }

    /** {@inheritDoc} */
    @Nullable public String getSignatureReferenceID() {
        return id;
    }

    /** {@inheritDoc} */
    @Nullable @NotLive @Unmodifiable public List<XMLObject> getOrderedChildren() {
        final ArrayList<XMLObject> children = new ArrayList<>();

        final Signature sig = getSignature();
        if (sig != null) {
            children.add(sig);
        }
        
        if (extensions != null) {
            children.add(extensions);
        }
        
        children.addAll(roleDescriptors);
        
        if (getAffiliationDescriptor() != null) {
            children.add(getAffiliationDescriptor());
        }
        
        if (getOrganization() != null) {
            children.add(getOrganization());
        }
        
        children.addAll(contactPersons);
        children.addAll(additionalMetadata);

        return CollectionSupport.copyToList(children);
    }
    
}