// For Library Version: 1.120.25

declare module "sap/ui/richtexteditor/library" {
  /**
   * Determines which editor component should be used for editing the text.
   */
  export enum EditorType {
    /**
     * Uses latest recommended TinyMCE version Current one is TinyMCE 6
     */
    TinyMCE = "TinyMCE",
    /**
     * Uses TinyMCE version 6 as editor
     */
    TinyMCE6 = "TinyMCE6",
  }
  /**
   * Interface for controls which are suitable as a Toolbar for RichTextEditor.
   *
   * @since 1.50
   */
  export interface IToolbar {
    __implements__sap_ui_richtexteditor_IToolbar: boolean;
  }
}

declare module "sap/ui/richtexteditor/RichTextEditor" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import { ID, CSSSize, TextDirection } from "sap/ui/core/library";

  import Event from "sap/ui/base/Event";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * The RichTextEditor-Control is used to enter formatted text. It uses the third-party component called
   * TinyMCE. In addition to the native toolbar, you can also use a toolbar built with SAPUI5 controls. Overview:
   *
   * With version 1.48 onward, aside from the native toolbar of the TinyMCE, the `RichTextEditor` can also
   * use a toolbar built with SAPUI5 controls. Which toolbar is used is taken into consideration only while
   * the control is being initialized and it will not be possible to change it during runtime, because of
   * lifecycle incompatibilities between the SAPUI5 and the third-party library. The custom toolbar acts like
   * a wrapper to the native toolbar and takes care of synchronizing the state of its internal controls with
   * the current state of the selection in the editor (bold, italics, font styles etc.).
   *
   * Restrictions:
   *
   * **Note: The `RichTextEditor` uses a third-party component and therefore some additional restrictions
   * apply for its proper usage and support. For more information see the Preamble section in {@link https://ui5.sap.com/#/topic/d4f3f1598373452bb73f2120930c133c sap.ui.richtexteditor}.
   * **
   *
   * Guidelines:
   * 	 -  The `RichTextEditor` should be used for desktop apps only. However, if it is essential for your
   *     use case, you can enable the mobile version of TinyMCE, whilst having in mind the restrictions. For more
   *     information see the {@link https://ui5.sap.com/#/topic/d4f3f1598373452bb73f2120930c133c sap.ui.richtexteditor documentation}.
   *
   * 	 -  In order to be usable, the control needs a minimum width 17.5 rem and height of 12.5 rem.
   * 	 -  Do not instantiate the `RichTextEditor` from a hidden container.
   * 	 -  Make sure you destroy the `RichTextEditor` instance instead of hiding it and create a new one when
   *     you show it again.
   *
   * Usage:
   *
   * When to use:
   * 	 - You want to enable users to enter text and other elements (tables, images) with different styles
   *     and colors.
   * 	 - You need to provide a tool for texts that require additional formatting.
   *
   *  When not to use:
   * 	 - You want to let users add simple text that doesn’t require formatting. Use {@link sap.m.TextArea text area }
   *     instead.
   * 	 - Use callbacks to the native third-party API with care, as there may be compatibility issues with
   *     later versions.
   *
   * Custom toolbar - adding and removing buttons: With version 1.102 it is possible to redefine the button
   * groups configuration in order to show only particular set of buttons in the custom toolbar of the `RichTextEditor`.
   * This is possible in few ways:
   * 	 - By providing the buttons group configurations as a whole by setting the `buttonGroups` property of
   *     the control.
   * 	 - By providing individual button group configuration via `addButtonGroup` method.
   *
   * Consider the following when choosing your approach:
   * 	 - Setting `buttonGroups` will override all current custom toolbar button group configurations. This
   *     method allows for total redefining of the custom toolbar groups.
   * 	 - Using `addButtonGroups` will try to add non-existing group into the configuration array, however,
   *     if such group configuration exists already, a warning will be logged and the new configuration will not
   *     be added. In order to replace the existing configuration you will need to remove it first via `removeButtonGroup`.
   *
   *
   * Below is a list of supported groups and buttons, which can be specified in the configuration objects
   * (format is <>: <>):
   * 	 - font-style: bold, italic, underline, strikethrough
   * 	 - font: fontfamily (fontselect in TinyMCE5), fontsize (fontsizeselect in TinyMCE5), forecolor, backcolor
   *
   * 	 - clipboard: cut, copy, paste
   * 	 - structure: bullist, numlist, outdent, indent
   * 	 - undo: undo, redo
   * 	 - insert: image, emoticons
   * 	 - link: link, unlink
   * 	 - text-align: alignleft, aligncenter, alignright, alignjustify  Additional supported groups, which
   *     can be added only after the editor is loaded (they can not be specified in the above mentioned configuration):
   *
   * 	 - table: table
   * 	 - styleselect: styleselect  **Note!** Adding configuration for "text-align" group with any buttons
   *     will still render a menu button with all available options inside. Removing/hiding the group can be achieved
   *     by invoking `removeButtonGroup` or `setShowGroupTextAlign(false)` depending on the desired result.
   *
   * **Note!** There is no synchronization between the `setShowGroup*` properties and the configuration object
   * that the application can specifying via `buttonGroups` in the constructor or on a later stage. This means
   * that in case new configuration is provided and for particular group the passed object contains property
   * `visible: true`, this group property will be respected no matter if the `RichTextEditor`'s property for
   * this particular group is set to `false`. Example: Providing the following object as group configuration:
   * { name: "font" visible: true } Will make the "font" group visible, no matter that calling `RichTextEditor.getShowGroupFont()`
   * returns `false`.
   */
  export default class RichTextEditor extends Control {
    /**
     * Constructor for a new RichTextEditor.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     * See:
     * 	{@link fiori:https://experience.sap.com/fiori-design-web/rich-text-editor/ Rich Text Editor}
     * 	{@link https://ui5.sap.com/#/topic/d4f3f1598373452bb73f2120930c133c}
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $RichTextEditorSettings
    );
    /**
     * Constructor for a new RichTextEditor.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     * See:
     * 	{@link fiori:https://experience.sap.com/fiori-design-web/rich-text-editor/ Rich Text Editor}
     * 	{@link https://ui5.sap.com/#/topic/d4f3f1598373452bb73f2120930c133c}
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $RichTextEditorSettings
    );

    /**
     * Creates a new subclass of class sap.ui.richtexteditor.RichTextEditor with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, RichTextEditor>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.ui.richtexteditor.RichTextEditor.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some ariaLabelledBy into the association {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     * @since 1.76.0
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addAriaLabelledBy(
      /**
       * The ariaLabelledBy to add; if empty, nothing is inserted
       */
      vAriaLabelledBy: ID | Control
    ): this;
    /**
     * Adds a button group to the editor. **Note**: Adding already existing group will not do anything. If a
     * button group needs to be changed, it first needs to be removed and then added by providing its name or
     * map object, containing its desired configuration.
     *
     *
     * @returns Control instance (for method chaining)
     */
    addButtonGroup(
      /**
       * Name/ID of a single button or object containing the group information
       */
      vGroup?:
        | {
            /**
             * Array of name/IDs of the buttons in the group
             */
            buttons?: string[];
            /**
             * Name/ID of the group.
             */
            name?: string;
            /**
             * (optional) The priority of the button group. Lower priorities are added first.
             */
            visible?: boolean;
            /**
             * (optional) Row number in which the button should be
             */
            row?: int;
            /**
             * (optional) The priority of the button group. Lower priorities are added first.
             */
            priority?: int;
            /**
             * (optional) The priority of the button group in the custom toolbar. Each default group in the custom toolbar
             * has a predefined `customToolbarPriority`. Lower priorities are added in first.
             */
            customToolbarPriority?: int;
          }
        | string
    ): RichTextEditor;
    /**
     * Adds some customButton to the aggregation {@link #getCustomButtons customButtons}.
     *
     * @since 1.48
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addCustomButton(
      /**
       * The customButton to add; if empty, nothing is inserted
       */
      oCustomButton: Control
    ): this;
    /**
     * Allows to add a plugin (that must already be installed on the server) to the RichTextEditor.
     *
     *
     * @returns Control instance (for method chaining)
     */
    addPlugin(
      /**
       * The plugin ID/name or an object with property "name", containing the ID/name of the plugin
       */
      mPlugin?: object | string
    ): RichTextEditor;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:beforeEditorInit beforeEditorInit} event of
     * this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.richtexteditor.RichTextEditor` itself.
     *
     * This event is fired right before the TinyMCE instance is created and can be used to change the settings
     * object that will be given to TinyMCE. The parameter "configuration" is the javascript object that will
     * be given to TinyMCE upon first instantiation. The configuration parameter contains a map that can be
     * changed in the case of TinyMCE. **Note:** In order to add plugins to the `RichTextEditor` control, you
     * have to use the `addPlugin` method. Adding plugins through the settings object may lead to synchronizing
     * issues between TinyMCE and `RichTextEditor` control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeEditorInit(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.richtexteditor.RichTextEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:beforeEditorInit beforeEditorInit} event of
     * this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.richtexteditor.RichTextEditor` itself.
     *
     * This event is fired right before the TinyMCE instance is created and can be used to change the settings
     * object that will be given to TinyMCE. The parameter "configuration" is the javascript object that will
     * be given to TinyMCE upon first instantiation. The configuration parameter contains a map that can be
     * changed in the case of TinyMCE. **Note:** In order to add plugins to the `RichTextEditor` control, you
     * have to use the `addPlugin` method. Adding plugins through the settings object may lead to synchronizing
     * issues between TinyMCE and `RichTextEditor` control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeEditorInit(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.richtexteditor.RichTextEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.richtexteditor.RichTextEditor` itself.
     *
     * Event is fired when the text in the field has changed AND the focus leaves the editor or when the Enter
     * key is pressed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: RichTextEditor$ChangeEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.richtexteditor.RichTextEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.richtexteditor.RichTextEditor` itself.
     *
     * Event is fired when the text in the field has changed AND the focus leaves the editor or when the Enter
     * key is pressed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: RichTextEditor$ChangeEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.richtexteditor.RichTextEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:ready ready} event of this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.richtexteditor.RichTextEditor` itself.
     *
     * Fired when the used editor is loaded and ready (its HTML is also created).
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachReady(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.richtexteditor.RichTextEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:ready ready} event of this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.richtexteditor.RichTextEditor` itself.
     *
     * Fired when the used editor is loaded and ready (its HTML is also created).
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachReady(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.richtexteditor.RichTextEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:readyRecurring readyRecurring} event of this
     * `sap.ui.richtexteditor.RichTextEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.richtexteditor.RichTextEditor` itself.
     *
     * Analogous to the ready event, the event is fired when the used editor is loaded and ready. But the event
     * is fired after every time the control is ready to use and not only once like the ready event.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachReadyRecurring(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.richtexteditor.RichTextEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:readyRecurring readyRecurring} event of this
     * `sap.ui.richtexteditor.RichTextEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.ui.richtexteditor.RichTextEditor` itself.
     *
     * Analogous to the ready event, the event is fired when the used editor is loaded and ready. But the event
     * is fired after every time the control is ready to use and not only once like the ready event.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachReadyRecurring(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.ui.richtexteditor.RichTextEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys all the customButtons in the aggregation {@link #getCustomButtons customButtons}.
     *
     * @since 1.48
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCustomButtons(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:beforeEditorInit beforeEditorInit} event of
     * this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachBeforeEditorInit(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:change change} event of this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: RichTextEditor$ChangeEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:ready ready} event of this `sap.ui.richtexteditor.RichTextEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachReady(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:readyRecurring readyRecurring} event of this
     * `sap.ui.richtexteditor.RichTextEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachReadyRecurring(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:beforeEditorInit beforeEditorInit} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireBeforeEditorInit(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:change change} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: RichTextEditor$ChangeEventParameters
    ): this;
    /**
     * Fires event {@link #event:ready ready} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireReady(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:readyRecurring readyRecurring} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireReadyRecurring(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Returns array of IDs of the elements which are the current targets of the association {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     * @since 1.76.0
     */
    getAriaLabelledBy(): ID[];
    /**
     * Gets current value of property {@link #getButtonGroups buttonGroups}.
     *
     * An array of button configurations. These configurations contain the names of buttons as array in the
     * property "buttons" and the name of the group in "name", they can also contain the "row" where the buttons
     * should be placed, a "priority" and whether the buttons are "visible". See method addButtonGroup() for
     * more details on the structure of the objects in this array. **Note:** `buttonGroups` is a feature from
     * the native editor. Its supported scope with `customToolbar` is limited up to the grouping- the user could
     * turn on/off a whole group, but modifying buttons within a group is not possible. In order to achieve
     * that functionality with `customToolbar`, the developer needs to turn the group off and instantiate their
     * own `sap.m.Button`(s) with the desired functionality.
     *
     * Default value is `[]`.
     *
     *
     * @returns Value of property `buttonGroups`
     */
    getButtonGroups(): object[];
    /**
     * Gets content of aggregation {@link #getCustomButtons customButtons}.
     *
     * Custom buttons are meant to extend the `RichTextEditor`'s custom toolbar. Though type is set to sap.ui.Control,
     * only sap.m.Button is allowed. **Note:** customButtons are available only when the customToolbar is enabled
     * and all the requirements are fulfilled.
     *
     * @since 1.48
     */
    getCustomButtons(): Control[];
    /**
     * Gets current value of property {@link #getCustomToolbar customToolbar}.
     *
     * Determines whether a Fiori Toolbar is used instead of the TinyMCE default toolbar one. It is applied
     * only when the EditorType is TinyMCE5 or TinyMCE6 and sap.m library is loaded. **Note:** The `customToolbar`
     * property will have effect only on initial loading. Changing it during runtime will not affect the initially
     * loaded toolbar.
     *
     * Default value is `false`.
     *
     * @since 1.48
     *
     * @returns Value of property `customToolbar`
     */
    getCustomToolbar(): boolean;
    /**
     * Gets current value of property {@link #getEditable editable}.
     *
     * Determines whether the editor content can be modified by the user. When set to "false" there might not
     * be any editor toolbar.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `editable`
     */
    getEditable(): boolean;
    /**
     * Gets current value of property {@link #getEditorLocation editorLocation}.
     *
     * Relative or absolute URL where the editor is available. Must be on the same server. **Note:** Any attempts
     * to set this property after the first rendering will not have any effect.
     *
     * Default value is `'js/tiny_mce6/tinymce.js'`.
     *
     * @deprecated (since 1.25.0) - The editorLocation is set implicitly when choosing the editorType.
     *
     * @returns Value of property `editorLocation`
     */
    getEditorLocation(): string;
    /**
     * Gets current value of property {@link #getEditorType editorType}.
     *
     * The editor implementation to use.
     *
     * Valid values are the ones found under sap.ui.richtexteditor.EditorType and any other editor identifier
     * that may be introduced by other groups (hence this is not an enumeration).
     *
     * **Notes:**
     * 	 - TinyMCE version 3 and 4 are no longer supported and cannot be used. If you set the property to TinyMCE,
     *     it will load TinyMCE version 6.
     * 	 - Any attempts to set this property after the first rendering will not have any effect.
     * 	 - The default value of the property will always resolve to the recommended version by UI5. Due to the
     *     different support cycles, we will be constantly getting newer TinyMCE versions and update the default
     *     value accordingly.
     * 	 - Usage of internal TinyMCE APIs is not recommended, since it might lead to issues upon TinyMCE version
     *     update.
     * 	 - Have in mind when choosing a specific TinyMCE version that there might be differences in the support
     *     rules compared to UI5, therefore we might be forced to remove any TinyMCE version.
     * 	 - TinyMCE 5 will be removed in future releases.
     *
     * Default value is `'TinyMCE'`.
     *
     *
     * @returns Value of property `editorType`
     */
    getEditorType(): string;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * Height of RichTextEditor control in CSS units. **Note:** If the height property results in a value smaller
     * than 200px, the minimum height of 200px will be applied. **Note:** If the "autoresize" TinyMCE plugin
     * is used, the height property is not taken into account.
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): CSSSize;
    /**
     * Returns the current editor's instance. CAUTION: using the native editor introduces a dependency to that
     * editor and breaks the wrapping character of the RichTextEditor control, so it should only be done in
     * justified cases.
     *
     *
     * @returns The native editor object (here: The TinyMCE editor instance)
     */
    getNativeApi(): object;
    /**
     * Gets current value of property {@link #getPlugins plugins}.
     *
     * An array of plugin configuration objects with the obligatory property "name". Each object has to contain
     * a property "name" which then contains the plugin name/ID.
     *
     * Default value is `[]`.
     *
     *
     * @returns Value of property `plugins`
     */
    getPlugins(): object[];
    /**
     * Gets current value of property {@link #getRequired required}.
     *
     * Determines whether a value is required.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `required`
     */
    getRequired(): boolean;
    /**
     * Gets current value of property {@link #getSanitizeValue sanitizeValue}.
     *
     * Determines whether to run the HTML sanitizer once the value (HTML markup) is applied or not. To configure
     * allowed URLs please use the validator API via {@link module:sap/base/security/URLListValidator#add URLListValidator API}.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `sanitizeValue`
     */
    getSanitizeValue(): boolean;
    /**
     * Gets current value of property {@link #getShowGroupClipboard showGroupClipboard}.
     *
     * Determines whether the toolbar button group containing commands like Cut, Copy and Paste is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showGroupClipboard`
     */
    getShowGroupClipboard(): boolean;
    /**
     * Gets current value of property {@link #getShowGroupFont showGroupFont}.
     *
     * Determines whether the toolbar button group containing commands like Font, Font Size and Colors is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `showGroupFont`
     */
    getShowGroupFont(): boolean;
    /**
     * Gets current value of property {@link #getShowGroupFontStyle showGroupFontStyle}.
     *
     * Determines whether the toolbar button group containing commands like Bold, Italic, Underline and Strikethrough
     * is available. Changing this after the initial rendering will result in some visible redrawing. Note:
     * This property will not be synchronized with group configuration provided via the buttonGroups property
     * or when groups are added on a later stage.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showGroupFontStyle`
     */
    getShowGroupFontStyle(): boolean;
    /**
     * Gets current value of property {@link #getShowGroupInsert showGroupInsert}.
     *
     * Determines whether the toolbar button group containing commands like Insert Image and Insert Smiley is
     * available. Changing this after the initial rendering will result in some visible redrawing. Note: This
     * property will not be synchronized with group configuration provided via the buttonGroups property or
     * when groups are added on a later stage.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `showGroupInsert`
     */
    getShowGroupInsert(): boolean;
    /**
     * Gets current value of property {@link #getShowGroupLink showGroupLink}.
     *
     * Determines whether the toolbar button group containing commands like Create Link and Remove Link is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `showGroupLink`
     */
    getShowGroupLink(): boolean;
    /**
     * Gets current value of property {@link #getShowGroupStructure showGroupStructure}.
     *
     * Determines whether the toolbar button group containing commands like Bullets and Indentation is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showGroupStructure`
     */
    getShowGroupStructure(): boolean;
    /**
     * Gets current value of property {@link #getShowGroupTextAlign showGroupTextAlign}.
     *
     * Determines whether the toolbar button group containing text alignment commands is available. Changing
     * this after the initial rendering will result in some visible redrawing. Note: This property will not
     * be synchronized with group configuration provided via the buttonGroups property or when groups are added
     * on a later stage.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showGroupTextAlign`
     */
    getShowGroupTextAlign(): boolean;
    /**
     * Gets current value of property {@link #getShowGroupUndo showGroupUndo}.
     *
     * Determines whether the toolbar button group containing commands like Undo and Redo is available. Changing
     * this after the initial rendering will result in some visible redrawing. Note: This property will not
     * be synchronized with group configuration provided via the buttonGroups property or when groups are added
     * on a later stage.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `showGroupUndo`
     */
    getShowGroupUndo(): boolean;
    /**
     * Gets current value of property {@link #getTextDirection textDirection}.
     *
     * The text direction
     *
     * Default value is `Inherit`.
     *
     *
     * @returns Value of property `textDirection`
     */
    getTextDirection(): TextDirection | keyof typeof TextDirection;
    /**
     * Gets current value of property {@link #getUseLegacyTheme useLegacyTheme}.
     *
     * Determines whether or not to use the legacy theme for the toolbar buttons. If this is set to false, the
     * default theme for the editor will be used (which might change slightly with every update). The legacy
     * theme has the disadvantage that not all functionality has its own icon, so using non default buttons
     * might lead to invisible buttons with the legacy theme - use the default editor theme in this case.
     *
     * Default value is `true`.
     *
     * @deprecated (since 1.97.0) - This property became obsolete after the deprecation of TinyMCE v3.
     *
     * @returns Value of property `useLegacyTheme`
     */
    getUseLegacyTheme(): boolean;
    /**
     * Gets current value of property {@link #getValue value}.
     *
     * An HTML string representing the editor content. Because this is HTML, the value cannot be generically
     * escaped to prevent cross-site scripting, so the application is responsible for doing so. Overwriting
     * this property would also reset editor's Undo manager and buttons "Undo"/"Redo" would be set to their
     * initial state.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `value`
     */
    getValue(): string;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Width of RichTextEditor control in CSS units.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Gets current value of property {@link #getWrapping wrapping}.
     *
     * Determines whether the text in the editor is wrapped. This does not affect the editor's value, only the
     * representation in the control.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `wrapping`
     */
    getWrapping(): boolean;
    /**
     * Checks for the provided `sap.ui.core.Control` in the aggregation {@link #getCustomButtons customButtons}.
     * and returns its index if found or -1 otherwise.
     *
     * @since 1.48
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfCustomButton(
      /**
       * The customButton whose index is looked for
       */
      oCustomButton: Control
    ): int;
    /**
     * Inserts a customButton into the aggregation {@link #getCustomButtons customButtons}.
     *
     * @since 1.48
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertCustomButton(
      /**
       * The customButton to insert; if empty, nothing is inserted
       */
      oCustomButton: Control,
      /**
       * The `0`-based index the customButton should be inserted at; for a negative value of `iIndex`, the customButton
       * is inserted at position 0; for a value greater than the current size of the aggregation, the customButton
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls in the association named {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     * @since 1.76.0
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllAriaLabelledBy(): ID[];
    /**
     * Removes all the controls from the aggregation {@link #getCustomButtons customButtons}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     * @since 1.48
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllCustomButtons(): Control[];
    /**
     * Removes an ariaLabelledBy from the association named {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     * @since 1.76.0
     *
     * @returns The removed ariaLabelledBy or `null`
     */
    removeAriaLabelledBy(
      /**
       * The ariaLabelledBy to be removed or its index or ID
       */
      vAriaLabelledBy: int | ID | Control
    ): ID | null;
    /**
     * Removes a button group from the editor.
     *
     *
     * @returns Control instance (for method chaining)
     */
    removeButtonGroup(
      /**
       * The name of the group to be removed.
       */
      sGroupName?: string
    ): RichTextEditor;
    /**
     * Removes a customButton from the aggregation {@link #getCustomButtons customButtons}.
     *
     * @since 1.48
     *
     * @returns The removed customButton or `null`
     */
    removeCustomButton(
      /**
       * The customButton to remove or its index or id
       */
      vCustomButton: int | string | Control
    ): Control | null;
    /**
     * Removes the plugin with the given name/ID from the list of plugins to load
     *
     *
     * @returns Control instance (for method chaining)
     */
    removePlugin(
      /**
       * The name/ID of the plugin to remove
       */
      sPluginName?: string
    ): RichTextEditor;
    /**
     * Sets the button groups to the editor.
     *
     *
     * @returns Control instance (for method chaining)
     */
    setButtonGroups(
      /**
       * Array of names or objects containing the group information
       */
      aGroups?: any[]
    ): RichTextEditor;
    /**
     * Sets a new value for property {@link #getCustomToolbar customToolbar}.
     *
     * Determines whether a Fiori Toolbar is used instead of the TinyMCE default toolbar one. It is applied
     * only when the EditorType is TinyMCE5 or TinyMCE6 and sap.m library is loaded. **Note:** The `customToolbar`
     * property will have effect only on initial loading. Changing it during runtime will not affect the initially
     * loaded toolbar.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     * @since 1.48
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCustomToolbar(
      /**
       * New value for property `customToolbar`
       */
      bCustomToolbar?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getEditable editable}.
     *
     * Determines whether the editor content can be modified by the user. When set to "false" there might not
     * be any editor toolbar.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEditable(
      /**
       * New value for property `editable`
       */
      bEditable?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getEditorLocation editorLocation}.
     *
     * Relative or absolute URL where the editor is available. Must be on the same server. **Note:** Any attempts
     * to set this property after the first rendering will not have any effect.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'js/tiny_mce6/tinymce.js'`.
     *
     * @deprecated (since 1.25.0) - The editorLocation is set implicitly when choosing the editorType.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEditorLocation(
      /**
       * New value for property `editorLocation`
       */
      sEditorLocation?: string
    ): this;
    /**
     * Switches the editor type and sets the default settings for the editor. All plugins and button groups
     * should be set after this has been invoked
     *
     *
     * @returns Control instance (for method chaining)
     */
    setEditorType(
      /**
       * Which editor type to be used (currently only 5 and 6)
       */
      sEditorType?: string
    ): RichTextEditor;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * Height of RichTextEditor control in CSS units. **Note:** If the height property results in a value smaller
     * than 200px, the minimum height of 200px will be applied. **Note:** If the "autoresize" TinyMCE plugin
     * is used, the height property is not taken into account.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      sHeight?: CSSSize
    ): this;
    /**
     * Sets the plugins to the editor.
     *
     *
     * @returns Control instance (for method chaining)
     */
    setPlugins(
      /**
       * Array of names or objects containing the plugin information
       */
      aPlugins?: any[]
    ): RichTextEditor;
    /**
     * Sets a new value for property {@link #getRequired required}.
     *
     * Determines whether a value is required.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setRequired(
      /**
       * New value for property `required`
       */
      bRequired?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getSanitizeValue sanitizeValue}.
     *
     * Determines whether to run the HTML sanitizer once the value (HTML markup) is applied or not. To configure
     * allowed URLs please use the validator API via {@link module:sap/base/security/URLListValidator#add URLListValidator API}.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSanitizeValue(
      /**
       * New value for property `sanitizeValue`
       */
      bSanitizeValue?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowGroupClipboard showGroupClipboard}.
     *
     * Determines whether the toolbar button group containing commands like Cut, Copy and Paste is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowGroupClipboard(
      /**
       * New value for property `showGroupClipboard`
       */
      bShowGroupClipboard?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowGroupFont showGroupFont}.
     *
     * Determines whether the toolbar button group containing commands like Font, Font Size and Colors is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowGroupFont(
      /**
       * New value for property `showGroupFont`
       */
      bShowGroupFont?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowGroupFontStyle showGroupFontStyle}.
     *
     * Determines whether the toolbar button group containing commands like Bold, Italic, Underline and Strikethrough
     * is available. Changing this after the initial rendering will result in some visible redrawing. Note:
     * This property will not be synchronized with group configuration provided via the buttonGroups property
     * or when groups are added on a later stage.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowGroupFontStyle(
      /**
       * New value for property `showGroupFontStyle`
       */
      bShowGroupFontStyle?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowGroupInsert showGroupInsert}.
     *
     * Determines whether the toolbar button group containing commands like Insert Image and Insert Smiley is
     * available. Changing this after the initial rendering will result in some visible redrawing. Note: This
     * property will not be synchronized with group configuration provided via the buttonGroups property or
     * when groups are added on a later stage.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowGroupInsert(
      /**
       * New value for property `showGroupInsert`
       */
      bShowGroupInsert?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowGroupLink showGroupLink}.
     *
     * Determines whether the toolbar button group containing commands like Create Link and Remove Link is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowGroupLink(
      /**
       * New value for property `showGroupLink`
       */
      bShowGroupLink?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowGroupStructure showGroupStructure}.
     *
     * Determines whether the toolbar button group containing commands like Bullets and Indentation is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowGroupStructure(
      /**
       * New value for property `showGroupStructure`
       */
      bShowGroupStructure?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowGroupTextAlign showGroupTextAlign}.
     *
     * Determines whether the toolbar button group containing text alignment commands is available. Changing
     * this after the initial rendering will result in some visible redrawing. Note: This property will not
     * be synchronized with group configuration provided via the buttonGroups property or when groups are added
     * on a later stage.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowGroupTextAlign(
      /**
       * New value for property `showGroupTextAlign`
       */
      bShowGroupTextAlign?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowGroupUndo showGroupUndo}.
     *
     * Determines whether the toolbar button group containing commands like Undo and Redo is available. Changing
     * this after the initial rendering will result in some visible redrawing. Note: This property will not
     * be synchronized with group configuration provided via the buttonGroups property or when groups are added
     * on a later stage.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowGroupUndo(
      /**
       * New value for property `showGroupUndo`
       */
      bShowGroupUndo?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getTextDirection textDirection}.
     *
     * The text direction
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Inherit`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTextDirection(
      /**
       * New value for property `textDirection`
       */
      sTextDirection?: TextDirection | keyof typeof TextDirection
    ): this;
    /**
     * Sets a new value for property {@link #getUseLegacyTheme useLegacyTheme}.
     *
     * Determines whether or not to use the legacy theme for the toolbar buttons. If this is set to false, the
     * default theme for the editor will be used (which might change slightly with every update). The legacy
     * theme has the disadvantage that not all functionality has its own icon, so using non default buttons
     * might lead to invisible buttons with the legacy theme - use the default editor theme in this case.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     * @deprecated (since 1.97.0) - This property became obsolete after the deprecation of TinyMCE v3.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUseLegacyTheme(
      /**
       * New value for property `useLegacyTheme`
       */
      bUseLegacyTheme?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getValue value}.
     *
     * An HTML string representing the editor content. Because this is HTML, the value cannot be generically
     * escaped to prevent cross-site scripting, so the application is responsible for doing so. Overwriting
     * this property would also reset editor's Undo manager and buttons "Undo"/"Redo" would be set to their
     * initial state.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValue(
      /**
       * New value for property `value`
       */
      sValue?: string
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Width of RichTextEditor control in CSS units.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
    /**
     * Sets a new value for property {@link #getWrapping wrapping}.
     *
     * Determines whether the text in the editor is wrapped. This does not affect the editor's value, only the
     * representation in the control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWrapping(
      /**
       * New value for property `wrapping`
       */
      bWrapping?: boolean
    ): this;
  }
  /**
   * Describes the settings that can be provided to the RichTextEditor constructor.
   */
  export interface $RichTextEditorSettings extends $ControlSettings {
    /**
     * An HTML string representing the editor content. Because this is HTML, the value cannot be generically
     * escaped to prevent cross-site scripting, so the application is responsible for doing so. Overwriting
     * this property would also reset editor's Undo manager and buttons "Undo"/"Redo" would be set to their
     * initial state.
     */
    value?: string | PropertyBindingInfo;

    /**
     * The text direction
     */
    textDirection?:
      | (TextDirection | keyof typeof TextDirection)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Width of RichTextEditor control in CSS units.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Height of RichTextEditor control in CSS units. **Note:** If the height property results in a value smaller
     * than 200px, the minimum height of 200px will be applied. **Note:** If the "autoresize" TinyMCE plugin
     * is used, the height property is not taken into account.
     */
    height?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * The editor implementation to use.
     *
     * Valid values are the ones found under sap.ui.richtexteditor.EditorType and any other editor identifier
     * that may be introduced by other groups (hence this is not an enumeration).
     *
     * **Notes:**
     * 	 - TinyMCE version 3 and 4 are no longer supported and cannot be used. If you set the property to TinyMCE,
     *     it will load TinyMCE version 6.
     * 	 - Any attempts to set this property after the first rendering will not have any effect.
     * 	 - The default value of the property will always resolve to the recommended version by UI5. Due to the
     *     different support cycles, we will be constantly getting newer TinyMCE versions and update the default
     *     value accordingly.
     * 	 - Usage of internal TinyMCE APIs is not recommended, since it might lead to issues upon TinyMCE version
     *     update.
     * 	 - Have in mind when choosing a specific TinyMCE version that there might be differences in the support
     *     rules compared to UI5, therefore we might be forced to remove any TinyMCE version.
     * 	 - TinyMCE 5 will be removed in future releases.
     */
    editorType?: string | PropertyBindingInfo;

    /**
     * Relative or absolute URL where the editor is available. Must be on the same server. **Note:** Any attempts
     * to set this property after the first rendering will not have any effect.
     *
     * @deprecated (since 1.25.0) - The editorLocation is set implicitly when choosing the editorType.
     */
    editorLocation?: string | PropertyBindingInfo;

    /**
     * Determines whether the editor content can be modified by the user. When set to "false" there might not
     * be any editor toolbar.
     */
    editable?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the toolbar button group containing commands like Bold, Italic, Underline and Strikethrough
     * is available. Changing this after the initial rendering will result in some visible redrawing. Note:
     * This property will not be synchronized with group configuration provided via the buttonGroups property
     * or when groups are added on a later stage.
     */
    showGroupFontStyle?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the toolbar button group containing text alignment commands is available. Changing
     * this after the initial rendering will result in some visible redrawing. Note: This property will not
     * be synchronized with group configuration provided via the buttonGroups property or when groups are added
     * on a later stage.
     */
    showGroupTextAlign?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the toolbar button group containing commands like Bullets and Indentation is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     */
    showGroupStructure?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the toolbar button group containing commands like Font, Font Size and Colors is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     */
    showGroupFont?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the toolbar button group containing commands like Cut, Copy and Paste is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     */
    showGroupClipboard?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the toolbar button group containing commands like Insert Image and Insert Smiley is
     * available. Changing this after the initial rendering will result in some visible redrawing. Note: This
     * property will not be synchronized with group configuration provided via the buttonGroups property or
     * when groups are added on a later stage.
     */
    showGroupInsert?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the toolbar button group containing commands like Create Link and Remove Link is available.
     * Changing this after the initial rendering will result in some visible redrawing. Note: This property
     * will not be synchronized with group configuration provided via the buttonGroups property or when groups
     * are added on a later stage.
     */
    showGroupLink?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the toolbar button group containing commands like Undo and Redo is available. Changing
     * this after the initial rendering will result in some visible redrawing. Note: This property will not
     * be synchronized with group configuration provided via the buttonGroups property or when groups are added
     * on a later stage.
     */
    showGroupUndo?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether the text in the editor is wrapped. This does not affect the editor's value, only the
     * representation in the control.
     */
    wrapping?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether a value is required.
     */
    required?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether to run the HTML sanitizer once the value (HTML markup) is applied or not. To configure
     * allowed URLs please use the validator API via {@link module:sap/base/security/URLListValidator#add URLListValidator API}.
     */
    sanitizeValue?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * An array of plugin configuration objects with the obligatory property "name". Each object has to contain
     * a property "name" which then contains the plugin name/ID.
     */
    plugins?: object[] | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether or not to use the legacy theme for the toolbar buttons. If this is set to false, the
     * default theme for the editor will be used (which might change slightly with every update). The legacy
     * theme has the disadvantage that not all functionality has its own icon, so using non default buttons
     * might lead to invisible buttons with the legacy theme - use the default editor theme in this case.
     *
     * @deprecated (since 1.97.0) - This property became obsolete after the deprecation of TinyMCE v3.
     */
    useLegacyTheme?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * An array of button configurations. These configurations contain the names of buttons as array in the
     * property "buttons" and the name of the group in "name", they can also contain the "row" where the buttons
     * should be placed, a "priority" and whether the buttons are "visible". See method addButtonGroup() for
     * more details on the structure of the objects in this array. **Note:** `buttonGroups` is a feature from
     * the native editor. Its supported scope with `customToolbar` is limited up to the grouping- the user could
     * turn on/off a whole group, but modifying buttons within a group is not possible. In order to achieve
     * that functionality with `customToolbar`, the developer needs to turn the group off and instantiate their
     * own `sap.m.Button`(s) with the desired functionality.
     */
    buttonGroups?: object[] | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines whether a Fiori Toolbar is used instead of the TinyMCE default toolbar one. It is applied
     * only when the EditorType is TinyMCE5 or TinyMCE6 and sap.m library is loaded. **Note:** The `customToolbar`
     * property will have effect only on initial loading. Changing it during runtime will not affect the initially
     * loaded toolbar.
     *
     * @since 1.48
     */
    customToolbar?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Custom buttons are meant to extend the `RichTextEditor`'s custom toolbar. Though type is set to sap.ui.Control,
     * only sap.m.Button is allowed. **Note:** customButtons are available only when the customToolbar is enabled
     * and all the requirements are fulfilled.
     *
     * @since 1.48
     */
    customButtons?:
      | Control[]
      | Control
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * Association to controls / IDs which label this control (see WAI-ARIA attribute `aria-labelledby`).
     *
     * @since 1.76.0
     */
    ariaLabelledBy?: Array<Control | string>;

    /**
     * Event is fired when the text in the field has changed AND the focus leaves the editor or when the Enter
     * key is pressed.
     */
    change?: (oEvent: RichTextEditor$ChangeEvent) => void;

    /**
     * Fired when the used editor is loaded and ready (its HTML is also created).
     */
    ready?: (oEvent: Event) => void;

    /**
     * Analogous to the ready event, the event is fired when the used editor is loaded and ready. But the event
     * is fired after every time the control is ready to use and not only once like the ready event.
     */
    readyRecurring?: (oEvent: Event) => void;

    /**
     * This event is fired right before the TinyMCE instance is created and can be used to change the settings
     * object that will be given to TinyMCE. The parameter "configuration" is the javascript object that will
     * be given to TinyMCE upon first instantiation. The configuration parameter contains a map that can be
     * changed in the case of TinyMCE. **Note:** In order to add plugins to the `RichTextEditor` control, you
     * have to use the `addPlugin` method. Adding plugins through the settings object may lead to synchronizing
     * issues between TinyMCE and `RichTextEditor` control.
     */
    beforeEditorInit?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the RichTextEditor#beforeEditorInit event.
   */
  export interface RichTextEditor$BeforeEditorInitEventParameters {}

  /**
   * Event object of the RichTextEditor#beforeEditorInit event.
   */
  export type RichTextEditor$BeforeEditorInitEvent = Event<
    RichTextEditor$BeforeEditorInitEventParameters,
    RichTextEditor
  >;

  /**
   * Parameters of the RichTextEditor#change event.
   */
  export interface RichTextEditor$ChangeEventParameters {
    /**
     * The new control value.
     */
    newValue?: string;
  }

  /**
   * Event object of the RichTextEditor#change event.
   */
  export type RichTextEditor$ChangeEvent = Event<
    RichTextEditor$ChangeEventParameters,
    RichTextEditor
  >;

  /**
   * Parameters of the RichTextEditor#ready event.
   */
  export interface RichTextEditor$ReadyEventParameters {}

  /**
   * Event object of the RichTextEditor#ready event.
   */
  export type RichTextEditor$ReadyEvent = Event<
    RichTextEditor$ReadyEventParameters,
    RichTextEditor
  >;

  /**
   * Parameters of the RichTextEditor#readyRecurring event.
   */
  export interface RichTextEditor$ReadyRecurringEventParameters {}

  /**
   * Event object of the RichTextEditor#readyRecurring event.
   */
  export type RichTextEditor$ReadyRecurringEvent = Event<
    RichTextEditor$ReadyRecurringEventParameters,
    RichTextEditor
  >;
}

declare namespace sap {
  interface IUI5DefineDependencyNames {
    "sap/ui/richtexteditor/library": undefined;

    "sap/ui/richtexteditor/RichTextEditor": undefined;
  }
}
