// For Library Version: 1.120.12

declare module "sap/suite/ui/commons/collaboration/BaseHelperService" {
  /**
   * Base class for the CollaborationHelpers
   *
   * @since 1.108
   */
  interface BaseHelperService {
    /**
     * Enables the Microsoft Teams collaboration functionality by providing a contact quick view with the options
     * to start a message, audio call or video call
     *
     *
     * @returns Instance of the contact quick view providing the data for the collaboration functionality
     */
    enableMinimalContactsCollaboration(
      /**
       * Provides the email of the contact to be used for the communication using Microsoft Teams
       */
      sEmail: string
    ): object;
    /**
     * Provides a list of all collaboration options
     *
     *
     * @returns Array of available options
     */
    getOptions(
      /**
       * Optional argument in case the consumer wants to influence the options, otherwise pass as undefined
       */
      oParams: {
        /**
         * Allow the 'Share as Chat' option to be available in case Microsoft Teams is the collaboration provider
         */
        isShareAsLinkEnabled: boolean;
        /**
         * Allow the 'Share as Tab' option to be available in case Microsoft Teams is the collaboration provider
         */
        isShareAsTabEnabled: boolean;
        /**
         * Allow the 'Share as Card' option to be available in case Microsoft Teams is the collaboration provider
         */
        isShareAsCardEnabled: boolean;
      }
    ): any[];
    /**
     * Provides the Microsoft Teams collaboration options for the contact quick view
     *
     *
     * @returns The Microsoft Teams collaboration options
     */
    getTeamsContactCollabOptions(): object;
    /**
     * Checks if collaboration with contacts is supported in Microsoft Teams
     *
     *
     * @returns A Boolean that determines whether or not the collaboration with contacts is supported in Microsoft
     * Teams
     */
    isContactsCollaborationSupported(): boolean;
    /**
     * Checks whether the feature flag is enabled and executes the code for the Adaptive Card Generation accordingly
     *
     *
     * @returns If set to true, Adaptive Card Generation is enabled
     */
    isFeatureFlagEnabled(): boolean;
    /**
     * Triggers the 'Share' operation
     */
    share(
      /**
       * JSON object of collaboration option that is clicked
       */
      oOption: Object,
      /**
       * Parameter object which contains the information to be shared
       */
      oParams: {
        /**
         * URL of the application which needs to be shared
         */
        url: string;
        /**
         * Title of the application which needs to be used during the integration
         */
        appTitle: string;
        /**
         * Title of the object page which needs to be used during the integration
         */
        subTitle: string;
        /**
         * Set the flag to 'true' to minimize the URL
         */
        minifyUrlForChat: boolean;
        /**
         * Adaptive card json for a given instance of the object page used for the ‘Share as Card’ option
         */
        cardManifest: Object;
        /**
         * ID of the card that needs to be stored and is constructed from SemanticObject_Action
         */
        cardId: string;
      }
    ): void;
  }
  const BaseHelperService: BaseHelperService;
  export default BaseHelperService;
}

declare module "sap/suite/ui/commons/collaboration/CollaborationHelper" {
  /**
   * CollaborationHelper for collaboration-related functionalities
   *
   * @since 1.108
   */
  interface CollaborationHelper {
    /**
     * Determines whether the app is running in the Microsoft Teams application. If the URL parameter "sap-collaboration-teams"
     * is set to true and if the sap-ushell-config is set to lean, the method ensures that the application runs
     * in the Microsoft Teams environment
     *
     *
     * @returns Return the resolved promise with the data if the conditions are met with 'true', else 'false'
     */
    isTeamsModeActive(): Promise<any>;
    /**
     * Method that checks whether the application is running in the Microsoft Teams environment. If yes, the
     * method disables the Avatar icon on the shell. This is done synchronously. The Method returns a promise.
     * The promise is resolved immediately if the URL is not compact. In case of a compact hash, the method
     * retrieves the original or complete hash and replaces it in the window. The method then resolves the promise.
     *
     *
     * @returns Return the resolved promise when the redirection of the URL is done.
     */
    processAndExpandHash(): Promise<any>;
  }
  const CollaborationHelper: CollaborationHelper;
  export default CollaborationHelper;
}

declare module "sap/suite/ui/commons/library" {
  /**
   * Enumeration of possible BulletChart display modes.
   *
   * @deprecated (since 1.34) - Deprecated. sap.suite.ui.microchart.BulletMicroChartModeType should be used.
   */
  export enum BulletChartMode {
    /**
     * Displays the Actual value.
     */
    Actual = "Actual",
    /**
     * Displays delta between the Actual and Threshold values.
     */
    Delta = "Delta",
  }
  /**
   * Comparison operators supported by the calculation builder.
   */
  export enum CalculationBuilderComparisonOperatorType {
    /**
     * Not equal to.
     */
    "!=" = "undefined",
    /**
     * Less than.
     */
    "<" = "undefined",
    /**
     * Less than or equal to.
     */
    "<=" = "undefined",
    /**
     * Equal to.
     */
    "=" = "undefined",
    /**
     * Greater than.
     */
    ">" = "undefined",
    /**
     * Greater than or equal to.
     */
    ">=" = "undefined",
  }
  /**
   * Functions supported by the calculation builder.
   *  To add a custom function, use {@link sap.suite.ui.commons.CalculationBuilderFunction}.
   */
  export enum CalculationBuilderFunctionType {
    /**
     * ABS.
     */
    ABS = "ABS",
    /**
     * Case.
     */
    Case = "CASE",
    /**
     * NDIV0.
     */
    NDIV0 = "NDIV0",
    /**
     * NODIM.
     */
    NODIM = "NODIM",
    /**
     * Round.
     */
    Round = "ROUND",
    /**
     * Round Down.
     */
    RoundDown = "RoundDown",
    /**
     * Round Up.
     */
    RoundUp = "RoundUp",
    /**
     * SQRT.
     */
    SQRT = "SQRT",
    /**
     * SUMCT.
     */
    SUMCT = "SUMCT",
    /**
     * SUMGT.
     */
    SUMGT = "SUMGT",
    /**
     * SUMRT.
     */
    SUMRT = "SUMRT",
  }
  /**
   * The types of items (operands) that can be used in a calculation builder expression.
   */
  export enum CalculationBuilderItemType {
    /**
     * Constant.
     *
     * @deprecated (since 1.77.0) - replaced by {@link sap.suite.ui.commons.CalculationBuilderItemType.Literal}.
     */
    Constant = "Constant",
    /**
     * Custom Function.
     *  Custom functions can be defined using {@link sap.suite.ui.commons.CalculationBuilderFunction}.
     */
    CustomFunction = "CustomFunction",
    /**
     * Custom arithmetic operator defined by the application.
     * Custom operators are not validated by the {@link sap.suite.ui.commons.CalculationBuilder} control.
     */
    CustomOperator = "CustomOperator",
    /**
     * Function.
     *  Default functions are defined by {@link sap.suite.ui.commons.CalculationBuilderFunctionType}.
     */
    Function = "Function",
    /**
     * Literal.
     *
     * @since 1.77.0
     */
    Literal = "Literal",
    /**
     * Operator.
     */
    Operator = "Operator",
    /**
     * Variable.
     */
    Variable = "Variable",
  }
  /**
   * Layout of the calculation builder.
   */
  export enum CalculationBuilderLayoutType {
    /**
     * Both the visual and text editors are displayed.
     */
    Default = "Default",
    /**
     * Only the text editor is displayed.
     */
    TextualOnly = "TextualOnly",
    /**
     * Only the visual editor is displayed.
     */
    VisualOnly = "VisualOnly",
    /**
     * Both the visual and text editors are displayed, but the text editor is read-only.
     */
    VisualTextualReadOnly = "VisualTextualReadOnly",
  }
  /**
   * Logical operators supported by the calculation builder.
   */
  export enum CalculationBuilderLogicalOperatorType {
    /**
     * Logical conjunction.
     */
    AND = "AND",
    /**
     * Negation.
     */
    NOT = "NOT",
    /**
     * Logical disjunction.
     */
    OR = "OR",
    /**
     * Exclusive disjunction.
     */
    XOR = "XOR",
  }
  /**
   * Arithmetic operators supported by the calculation builder.
   */
  export enum CalculationBuilderOperatorType {
    /**
     * Left bracket.
     */
    "(" = "undefined",
    /**
     * Right bracket.
     */
    ")" = "undefined",
    /**
     * Multiplication.
     */
    "*" = "undefined",
    /**
     * Addition.
     */
    "+" = "undefined",
    /**
     * Comma.
     */
    "," = "undefined",
    /**
     * Subtraction.
     */
    "-" = "undefined",
    /**
     * Division.
     */
    "/" = "undefined",
  }
  /**
   * Types of expression validation that define when the expression entered into the {@link sap.suite.ui.commons.CalculationBuilder }
   * is validated.
   */
  export enum CalculationBuilderValidationMode {}
  /**
   * Enumeration of possible theme specific background colors.
   *
   * @deprecated (since 1.34) - Deprecated. Moved to sapui5.runtime.
   */
  export enum CommonBackground {
    /**
     * Dark background color.
     */
    Dark = "Dark",
    /**
     * The darkest background color.
     */
    Darkest = "Darkest",
    /**
     * Extra dark background color.
     */
    ExtraDark = "ExtraDark",
    /**
     * Extra light background color.
     */
    ExtraLight = "ExtraLight",
    /**
     * Light background color.
     */
    Light = "Light",
    /**
     * The lightest background color.
     */
    Lightest = "Lightest",
    /**
     * Medium background color.
     */
    Medium = "Medium",
    /**
     * Medium light background color.
     */
    MediumLight = "MediumLight",
  }
  /**
   * The view of the ComparisonChart.
   *
   * @deprecated (since 1.34) - Deprecated. sap.suite.ui.microchart.ComparisonMicroChartViewType should be
   * used.
   */
  export enum ComparisonChartView {
    /**
     * Titles and values are displayed above the bars.
     */
    Normal = "Normal",
    /**
     * Titles and values are displayed in the same line with the bars.
     */
    Wide = "Wide",
  }
  /**
   * The marker for the deviation trend.
   *
   * @deprecated (since 1.34) - Deprecated. Moved to sapui5.runtime.
   */
  export enum DeviationIndicator {
    /**
     * The actual value is less than the target value.
     */
    Down = "Down",
    /**
     * No value.
     */
    None = "None",
    /**
     * The actual value is more than the target value.
     */
    Up = "Up",
  }
  /**
   * Enumeration of possible FacetOverview height settings.
   *
   * @deprecated (since 1.32) - Deprecated. Object page should be used instead.
   */
  export enum FacetOverviewHeight {
    /**
     * Content based height
     */
    Auto = "Auto",
    /**
     * Large height
     */
    L = "L",
    /**
     * Medium height
     */
    M = "M",
    /**
     * No value. The height of the control is defined by depricated height property.
     */
    None = "None",
    /**
     * Small height
     */
    S = "S",
    /**
     * Extra Large height
     */
    XL = "XL",
    /**
     * Extra small height
     */
    XS = "XS",
    /**
     * Extra extra large height
     */
    XXL = "XXL",
  }
  /**
   * Modes for the {@link sap.suite.ui.commons.CloudFilePicker}.
   */
  export enum FilePickerModes {
    /**
     * Allow selection of a File or Folder resource
     */
    All = "All",
    /**
     * Allow selection of File type of resource
     */
    FileOnly = "FileOnly",
    /**
     * Allow selection of Folder type of resource
     */
    FolderOnly = "FolderOnly",
  }
  /**
   * Runtime mode for the {@link sap.suite.ui.commons.CloudFilePicker}.
   */
  export enum FilePickerType {
    /**
     * Control runs in Export mode
     */
    Export = "Export",
    /**
     * Control runs in Upload mode
     */
    Upload = "Upload",
  }
  /**
   * Enumeration of possible frame types.
   *
   * @deprecated (since 1.34) - Deprecated. Moved to openUI5.
   */
  export enum FrameType {
    /**
     * The Auto frame type that adjusts the size of the control to the content.
     */
    Auto = "Auto",
    /**
     * The 1x1 frame type.
     */
    OneByOne = "OneByOne",
    /**
     * The 2x1 frame type.
     */
    TwoByOne = "TwoByOne",
    /**
     * The 2/3 frame type.
     */
    TwoThirds = "TwoThirds",
  }
  /**
   * The list of possible HeaderContainer views.
   *
   * @deprecated (since 1.48) - This control is deprecated since 1.48. Please use the equivalent sap.ui.core.Orientation.
   */
  export enum HeaderContainerView {
    /**
     * The horizontal orientation of the control.
     */
    Horizontal = "Horizontal",
    /**
     * The vertical orientation of the control.
     */
    Vertical = "Vertical",
  }
  /**
   * Action buttons for the {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer}.
   */
  export enum ImageEditorContainerButton {
    /**
     * Crop button.
     */
    Crop = "Crop",
    /**
     * Filter button.
     */
    Filter = "Filter",
    /**
     * Transform button.
     */
    Transform = "Transform",
  }
  /**
   * Mode types for {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer}.
   */
  export enum ImageEditorContainerMode {
    /**
     * Mode with rectangular and circular/oval crop options.
     */
    Crop = "Crop",
    /**
     * Mode with filter options, including grayscale, sepia, brightness, contrast, saturation, and some other.
     */
    Filter = "Filter",
    /**
     * Mode with resize, rotate, and flip options.
     */
    Transform = "Transform",
  }
  /**
   * Mode types for {@link sap.suite.ui.commons.imageeditor.ImageEditor}.
   */
  export enum ImageEditorMode {
    /**
     * Image editor mode with custom shape crop area.
     */
    CropCustomShape = "CropCustomShape",
    /**
     * Image editor mode with ellipse crop area.
     */
    CropEllipse = "CropEllipse",
    /**
     * Image editor mode with rectangle crop area.
     */
    CropRectangle = "CropRectangle",
    /**
     * Image editor mode that shows just the source image.
     * This mode is used by default.
     */
    Default = "Default",
    /**
     * Image editor mode with resize handlers.
     */
    Resize = "Resize",
  }
  /**
   * Image file format.
   */
  export enum ImageFormat {
    /**
     * JPEG file format.
     */
    Jpeg = "Jpeg",
    /**
     * PNG file format.
     */
    Png = "Png",
  }
  /**
   * Enumeration of possible PointTile size settings.
   *
   * @deprecated (since 1.34) - Deprecated. sap.m.InfoTileSize should be used.
   */
  export enum InfoTileSize {
    /**
     * The size of the tile depends on the device it is running on. It is large on desktop, medium on tablet
     * and small on phone.
     */
    Auto = "Auto",
    /**
     * Large size.
     */
    L = "L",
    /**
     * Medium size.
     */
    M = "M",
    /**
     * Small size.
     */
    S = "S",
    /**
     * Extra small size.
     */
    XS = "XS",
  }
  /**
   * Enumeration of possible InfoTile text color settings.
   *
   * @deprecated (since 1.34) - Deprecated. sap.m.InfoTileTextColor should be used.
   */
  export enum InfoTileTextColor {
    /**
     * Critical InfoTile text color.
     */
    Critical = "Critical",
    /**
     * Negative InfoTile text color.
     */
    Negative = "Negative",
    /**
     * Positive InfoTile text color.
     */
    Positive = "Positive",
  }
  /**
   * Enumeration of possible InfoTile value color settings.
   *
   * @deprecated (since 1.34) - Deprecated. sap.m.InfoTileValueColor should be used.
   */
  export enum InfoTileValueColor {
    /**
     * Critical InfoTile value color.
     */
    Critical = "Critical",
    /**
     * Error InfoTile value color.
     */
    Error = "Error",
    /**
     * Good InfoTile value color.
     */
    Good = "Good",
    /**
     * Neutral InfoTile value color.
     */
    Neutral = "Neutral",
  }
  /**
   * Supported Layout Types for {@link sap.suite.ui.commons.BaseContainer}.
   */
  export enum LayoutType {
    /**
     * Panels are rendered horizontally example pages and news
     */
    Horizontal = "Horizontal",
    /**
     * Panels are rendered side by side example todos and situaions, fav, recent and frequent apps
     */
    SideBySide = "SideBySide",
    /**
     * Panels are rendered vertically example insights cards and tiles
     */
    Vertical = "Vertical",
  }
  /**
   * Enumeration of possible load states for LoadableView.
   *
   * @deprecated (since 1.34) - Deprecated. sap.m.LoadState should be used.
   */
  export enum LoadState {
    /**
     * LoadableView disabled to load the control.
     */
    Disabled = "Disabled",
    /**
     * LoadableView failed to load the control.
     */
    Failed = "Failed",
    /**
     * LoadableView has loaded the control.
     */
    Loaded = "Loaded",
    /**
     * LoadableView is loading the control.
     */
    Loading = "Loading",
  }
  /**
   * The list of possible MicroAreaChart views.
   *
   * @deprecated (since 1.34) - Deprecated. sap.suite.ui.microchart.AreaMicroChartViewType should be used.
   */
  export enum MicroAreaChartView {
    /**
     * The view with labels on the top and bottom.
     */
    Normal = "Normal",
    /**
     * The view with labels on the left and right.
     */
    Wide = "Wide",
  }
  /**
   * Options that define how the micro process flow should be rendered inside its parent container.
   * These options can be useful when the width of the parent container does not allow for all nodes in the
   * micro process flow to be displayed on the same line.
   */
  export enum MicroProcessFlowRenderType {
    /**
     * The micro process flow nodes are not wrapped inside the parent container.
     * The nodes that do not fit into the width of the parent container are not displayed.
     */
    NoWrap = "NoWrap",
    /**
     * Two scrolling icons are added to the parent container, which allows navigation by scrolling through the
     * micro process flow.
     * Please note that the numbers displayed next to the scrolling icons are not recalculated dynamically when
     * you resize the browser window. If you want them to be recalculated, consider using the `ScrollingWithResizer`
     * render type instead.
     */
    Scrolling = "Scrolling",
    /**
     * Two scrolling icons are added to the parent container, with the number indicators updated automatically
     * when you resize the browser window.
     * This option allows scrolling through the micro process flow, just as the `Scrolling` option does, but
     * may slightly affect the performance. If using this render type affects your application's performance,
     * consider using the `Scrolling` render type instead.
     */
    ScrollingWithResizer = "ScrollingWithResizer",
    /**
     * The micro process flow nodes are wrapped inside the parent container.
     */
    Wrap = "Wrap",
  }
  /**
   * Describes the state of a connection label.
   */
  export enum ProcessFlowConnectionLabelState {
    /**
     * Critical connection label.
     */
    Critical = "Critical",
    /**
     * Negative connection label.
     */
    Negative = "Negative",
    /**
     * Neutral connection label.
     */
    Neutral = "Neutral",
    /**
     * Positive connection label.
     */
    Positive = "Positive",
  }
  /**
   * Describes the state of a connection.
   */
  export enum ProcessFlowConnectionState {
    /**
     * Dimmed connection.
     */
    Dimmed = "Dimmed",
    /**
     * Highlighted connection.
     */
    Highlighted = "Highlighted",
    /**
     * Regular connection.
     */
    Regular = "Regular",
    /**
     * Selected connection.
     */
    Selected = "Selected",
  }
  /**
   * Describes the type of a connection.
   */
  export enum ProcessFlowConnectionType {
    /**
     * Normal connection.
     */
    Normal = "Normal",
    /**
     * Planned connection.
     */
    Planned = "Planned",
  }
  /**
   * The ProcessFlow calculates the ProcessFlowDisplayState based on the 'focused' and 'highlighted' properties
   * of each node.
   */
  export enum ProcessFlowDisplayState {
    /**
     * The control is in the dimmed state
     */
    Dimmed = "Dimmed",
    /**
     * The control is in the combination of dimmed and focused display state
     */
    DimmedFocused = "DimmedFocused",
    /**
     * The control is in highlighted display state
     */
    Highlighted = "Highlighted",
    /**
     * The control is in the combination of highlighted and focused display state
     */
    HighlightedFocused = "HighlightedFocused",
    /**
     * The control is in the regular display state
     */
    Regular = "Regular",
    /**
     * The control is in the combination of regular and focused display state
     */
    RegularFocused = "RegularFocused",
    /**
     * The control is in the selected display state
     */
    Selected = "Selected",
    /**
     * The control is in the combination of selected and focused display state
     */
    SelectedFocused = "SelectedFocused",
    /**
     * The control is in the combination of selected and highlighted display state
     */
    SelectedHighlighted = "SelectedHighlighted",
    /**
     * The control is in the combination of selected, highlighted and focused display state
     */
    SelectedHighlightedFocused = "SelectedHighlightedFocused",
  }
  /**
   * This type is used in the 'state' property of the ProcessFlowLaneHeader. For example, app developers can
   * set the status of the lane header if lanes are displayed without documents. If the complete process flow
   * is displayed (that is, if the lane header is displayed with documents underneath), the given state values
   * of the lane header are ignored and will be calculated in the ProcessFlow according to the current state
   * of the documents.
   */
  export enum ProcessFlowLaneState {
    /**
     * The 'state' property is associated with the given value. Possible states are: positive, negative, neutral,
     * and planned.
     */
    state = "state",
    /**
     * In the 'state' array, the total amount of values needs to be 100%.
     */
    value = "value",
  }
  /**
   * Describes the state connected to the content it is representing in the Process Flow Node. The state is
   * also displayed in the Process Flow Lane Header as a color segment of the donut.
   */
  export enum ProcessFlowNodeState {
    /**
     * Critical status for a created document: with this document, a critical issue occurred, for example, the
     * business process can be interrupted.
     *
     * @since 1.42.0
     */
    Critical = "Critical",
    /**
     * Negative status for a created document: with this document, an issue occurred.
     */
    Negative = "Negative",
    /**
     * Neutral status for a created document: the document is in progress.
     */
    Neutral = "Neutral",
    /**
     * Planned status for a document: the document is planned to be started.
     */
    Planned = "Planned",
    /**
     * Planned, but negative status for a document: the planned document has an issue but has not yet been started.
     */
    PlannedNegative = "PlannedNegative",
    /**
     * Positive status for a created document: the document is done, finished, solved.
     */
    Positive = "Positive",
  }
  /**
   * Describes the type of a node. The type value could be single or aggregated. With this type, the application
   * can define if several nodes should be displayed as one aggregated node in a path per column to represent
   * a grouping of semantically equal nodes.
   */
  export enum ProcessFlowNodeType {
    /**
     * Aggregated node - several nodes are illustrated as a stack of nodes in the same path and in one column.
     */
    Aggregated = "Aggregated",
    /**
     * Single node - one node is illustrated in a column.
     */
    Single = "Single",
  }
  /**
   * The zoom level defines level of details for the node and how much space the process flow requires.
   */
  export enum ProcessFlowZoomLevel {
    /**
     * Zoom level for least details - only icon is displayed.
     */
    Four = "Four",
    /**
     * The full details with normal font size.
     */
    One = "One",
    /**
     * The details are the icon, title text and no additional texts.
     */
    Three = "Three",
    /**
     * The full detail view of the node but with smaller font size.
     */
    Two = "Two",
  }
  /**
   * File selection mode(Upload) for the {@link sap.suite.ui.commons.CloudFilePicker}.
   */
  export enum SelectionModes {
    /**
     * Allow selection of multiple files
     */
    MultiSelect = "MultiSelect",
    /**
     * Allow selection of single file
     */
    SingleSelect = "SingleSelectMaster",
  }
  /**
   * SelectionState
   *
   * @deprecated (since 1.48) - This Enumeration is deprecated as it is not used anywhere.
   */
  export enum SelectionState {
    /**
     * Not Selected
     */
    NotSelected = "NotSelected",
    /**
     * Selected
     */
    Selected = "Selected",
    /**
     * Semantic
     */
    Semantic = "Semantic",
  }
  /**
   * Defines the way how UnifiedThingGroup control is rendered.
   *
   * @deprecated (since 1.32) - Deprecated. Object page should be used instead.
   */
  export enum ThingGroupDesign {
    /**
     * In this design there is indentation between header and content of the group.
     */
    TopIndent = "TopIndent",
    /**
     * In this design there is no indentation between header and content of the group.
     */
    ZeroIndent = "ZeroIndent",
  }
  /**
   * The alignment of timeline posts relative to the timeline axis.
   */
  export enum TimelineAlignment {
    /**
     * Entries are displayed below the timeline axis.
     *  In a vertical timeline, entries are displayed to the right of the timeline axis. Synonym for Right.
     */
    Bottom = "Bottom",
    /**
     * Entries are displayed to the left of the timeline axis.
     *  In a horizontal timeline, entries are displayed above the timeline axis. Synonym for Top.
     */
    Left = "Left",
    /**
     * Entries are displayed to the right of the timeline axis.
     *  In a horizontal timeline, entries are displayed below the timeline axis. Synonym for Bottom.
     */
    Right = "Right",
    /**
     * Entries are displayed above the timeline axis.
     *  In a vertical timeline, entries are displayed to the left of the timeline axis. Synonym for Left.
     */
    Top = "Top",
  }
  /**
   * Defines the orientation of the timeline axis.
   */
  export enum TimelineAxisOrientation {
    /**
     * Horizontal timeline.
     */
    Horizontal = "Horizontal",
    /**
     * Vertical timeline.
     */
    Vertical = "Vertical",
  }
  /**
   * Filter type for the timeline.
   */
  export enum TimelineFilterType {
    /**
     * Items filter based on data, defined by the {@link sap.suite.ui.commons.TimelineItem#getFilterValue filterValue }
     * property or by a custom value.
     */
    Data = "Data",
    /**
     * Search results filter.
     */
    Search = "Search",
    /**
     * Time range filter, defined by the start date (`from`) and end date (`to`) of the time range.
     */
    Time = "Time",
  }
  /**
   * Type of grouping for timeline entries.
   */
  export enum TimelineGroupType {
    /**
     * Data is grouped by day.
     */
    Day = "Day",
    /**
     * Data is grouped by month.
     */
    Month = "Month",
    /**
     * No grouping is used.
     */
    None = "None",
    /**
     * Data is grouped by quarter.
     */
    Quarter = "Quarter",
    /**
     * Data is grouped by week.
     */
    Week = "Week",
    /**
     * Data is grouped by year.
     */
    Year = "Year",
  }
  /**
   * Type of the fadeout effect applied to the upper and lower margins of the visible timeline area.
   *
   * @deprecated (since 1.54.0) - Not Fiori.
   */
  export enum TimelineScrollingFadeout {
    /**
     * Timeline fades into the lower and upper margins of the visible area, but no scroll buttons are displayed.
     */
    Area = "Area",
    /**
     * Timeline fades into the lower and upper margins of the visible area, and scroll buttons are displayed.
     */
    AreaWithButtons = "AreaWithButtons",
    /**
     * Timeline does not fade out.
     */
    None = "None",
  }
  /**
   * Marker for the key value status.
   *
   * @deprecated (since 1.32) - Deprecated. Numeric content or any other standard Fiori control should be
   * used instead.
   */
  export enum ValueStatus {
    /**
     * Bad value.
     */
    Bad = "Bad",
    /**
     * Critical value.
     */
    Critical = "Critical",
    /**
     * Good value.
     */
    Good = "Good",
    /**
     * Positive value.
     */
    Neutral = "Neutral",
  }

  export namespace networkgraph {
    /**
     * Position of a custom action button.
     */
    enum ActionButtonPosition {
      /**
       * The action button is aligned to the left.
       */
      Left = "Left",
      /**
       * The action button is aligned to the right.
       */
      Right = "Right",
    }
    /**
     * Background color for the network graph.
     */
    enum BackgroundColor {
      /**
       * The default color of the parent container is used.
       */
      Default = "Default",
      /**
       * White.
       */
      White = "White",
    }
    /**
     * Semantic type of the node status.
     */
    enum ElementStatus {
      /**
       * A node type that communicates an error.
       */
      Error = "Error",
      /**
       * A node type that communicates information.
       */
      Information = "Information",
      /**
       * A standard node
       */
      Standard = "Standard",
      /**
       * A node type that communicates success.
       */
      Success = "Success",
      /**
       * A node type that communicates a warning.
       */
      Warning = "Warning",
    }
    /**
     * States of the Header checkbox.
     */
    enum HeaderCheckboxState {
      /**
       * Checkbox is visible and selected.
       */
      Checked = "Checked",
      /**
       * Checkbox is not visible.
       */
      Hidden = "Hidden",
      /**
       * Checkbox is visible and not selected.
       */
      Unchecked = "Unchecked",
    }
    /**
     * Types of layout algorithms that define the visual features and layout of the network graph.
     */
    enum LayoutRenderType {
      /**
       * Algorithm that arranges the nodes freely in a force-directed manner, based on the attractive and repulsive
       * forces within the graph.
       *  This layout algorithm ignores node groups, so all nodes appear as standalone nodes not grouped in any
       * way.
       */
      Forces = "Forces",
      /**
       * Algorithm that arranges nodes into layers and ensures support for free-form groups.
       */
      LayeredWithGroups = "LayeredWithGroups",
      /**
       * Algorithm that arranges nodes into swim lanes, which are single-layer groups.
       */
      SwimLanes = "SwimLanes",
      /**
       * Algorithm that supports nested groups, similar to `SwimLanes`, but arranges them into two columns only:
       * one on the left and one on the right.
       */
      TwoColumns = "TwoColumns",
    }
    /**
     * Direction of the arrow on the connector line.
     */
    enum LineArrowOrientation {
      /**
       * The arrows on the line point both ways.
       */
      Both = "Both",
      /**
       * The arrow points from child to parent.
       */
      ChildOf = "ChildOf",
      /**
       * The arrow is hidden.
       */
      None = "None",
      /**
       * The arrow points from parent to child.
       */
      ParentOf = "ParentOf",
    }
    /**
     * Position of the arrow on a connector line.
     */
    enum LineArrowPosition {
      /**
       * The arrow is placed at the end of the last line segment.
       */
      End = "End",
      /**
       * The arrow is placed in the middle of the last line segment. If the line has only one segment, the arrow
       * appears in the middle of the line.
       */
      Middle = "Middle",
      /**
       * The arrow is placed at the beginning of the first line segment.
       */
      Start = "Start",
    }
    /**
     * Type of connector line used in the network graph.
     */
    enum LineType {
      /**
       * Dashed connector line.
       */
      Dashed = "Dashed",
      /**
       * Dotted connector line.
       */
      Dotted = "Dotted",
      /**
       * Solid connector line.
       */
      Solid = "Solid",
    }
    /**
     * Type of node placement for Layered Algorithm. See {@link https://rtsys.informatik.uni-kiel.de/confluence/display/KIELER/KLay+Layered+Layout+Options#KLayLayeredLayoutOptions-nodePlacement}
     */
    enum NodePlacement {
      /**
       * Minimizes the number of edge bends at the expense of the graph size.
       */
      BrandesKoepf = "BrandesKoepf",
      /**
       * Calculates the most optimal layout balance.
       */
      LinearSegments = "LinearSegments",
      /**
       * Minimizes the area taken by the graph at the expense of everything else.
       */
      Simple = "Simple",
    }
    /**
     * Shape of a node in a network graph.
     */
    enum NodeShape {
      /**
       * Rectangular shape with an optional list of attributes.
       */
      Box = "Box",
      /**
       * Round shape with a text label below.
       */
      Circle = "Circle",
      /**
       * Shape for custom rendering.
       */
      Custom = "Custom",
    }
    /**
     * Orientation of layered layout.
     */
    enum Orientation {
      /**
       * The flow of the graph is bottom to top.
       */
      BottomTop = "BottomTop",
      /**
       * The flow of the graph is left to right.
       */
      LeftRight = "LeftRight",
      /**
       * The flow of the graph is right to left.
       */
      RightLeft = "RightLeft",
      /**
       * The flow of the graph is top to bottom.
       */
      TopBottom = "TopBottom",
    }
    /**
     * Determines how nodes are rendered. For optimal performance and usability, it is recommended that you
     * use HTML, which allows you to avoid dealing with SVG limitations.
     */
    enum RenderType {
      /**
       * Nodes are rendered as classic HTML.
       */
      Html = "Html",
      /**
       * Nodes are rendered as SVG.
       */
      Svg = "Svg",
    }
  }

  export namespace statusindicator {
    /**
     * The direction of animation.
     *
     *
     * The direction types `Up`, `Down`, `Left`, and `Right` are available when {@link sap.suite.ui.commons.statusindicator.FillingType }
     * is set to `Linear`.
     *  The direction types `Clockwise` and `Counterclockwise` are available when {@link sap.suite.ui.commons.statusindicator.FillingType }
     * is set to `Circular`.
     */
    enum FillingDirectionType {
      /**
       * Clockwise.
       */
      Clockwise = "Clockwise",
      /**
       * Counterclockwise.
       */
      CounterClockwise = "CounterClockwise",
      /**
       * From top to bottom.
       */
      Down = "Down",
      /**
       * From right to left.
       */
      Left = "Left",
      /**
       * From left to right.
       */
      Right = "Right",
      /**
       * From bottom upwards.
       */
      Up = "Up",
    }
    /**
     * The type of filling.
     */
    enum FillingType {
      /**
       * Clockwise or counterclockwise circular filling is applied.
       *
       *  For details, see {@link sap.suite.ui.commons.statusindicator.FillingDirectionType}.
       */
      Circular = "Circular",
      /**
       * The shape is filled with a linear gradient.
       */
      Linear = "Linear",
      /**
       * No filling is applied.
       */
      None = "None",
    }
    /**
     * The horizontal alignment of the status indicator within its parent container.
     */
    enum HorizontalAlignmentType {
      /**
       * Left.
       */
      Left = "Left",
      /**
       * Middle.
       */
      Middle = "Middle",
      /**
       * Right.
       */
      Right = "Right",
    }
    /**
     * Position of the label, relative to the status indicator.
     */
    enum LabelPositionType {
      /**
       * Bottom.
       */
      Bottom = "Bottom",
      /**
       * Left
       */
      Left = "Left",
      /**
       * Right.
       */
      Right = "Right",
      /**
       * Top.
       */
      Top = "Top",
    }
    /**
     * Predefined sizes of the status indicator.
     */
    enum SizeType {
      /**
       * Extra large status indicator.
       */
      ExtraLarge = "ExtraLarge",
      /**
       * Large status indicator.
       */
      Large = "Large",
      /**
       * Medium status indicator.
       */
      Medium = "Medium",
      /**
       * No size settings are applied.
       */
      None = "None",
      /**
       * Small status indicator.
       */
      Small = "Small",
    }
    /**
     * The vertical alignment of the status indicator within its parent container.
     */
    enum VerticalAlignmentType {
      /**
       * Bottom.
       */
      Bottom = "Bottom",
      /**
       * Middle.
       */
      Middle = "Middle",
      /**
       * Top.
       */
      Top = "Top",
    }
  }

  export namespace taccount {
    /**
     * The state of the {@link sap.suite.ui.commons.taccount.TAccountPanel} that defines how T accounts included
     * in the panel are displayed.
     */
    enum TAccountPanelState {
      /**
       * The T account panel displays T accounts as T shapes with debit and credit entries on either side of the
       * T.
       */
      Default = "Default",
      /**
       * The T account panel displays T accounts as an aggregated table, with each debit and credit entry listed
       * on a separate row.
       */
      Table = "Table",
    }
  }
}

declare module "sap/suite/ui/commons/AriaProperties" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import RenderManager from "sap/ui/core/RenderManager";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * `AriaProperties` provides ARIA-compliant properties for screen reader software that can be added to any
   * control renderer.
   *
   * @since 1.65.0
   */
  export default class AriaProperties extends UI5Element {
    /**
     * Constructor for a new AriaProperties.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.AriaProperties with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, AriaProperties>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.AriaProperties.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds the provided ARIA-compliant properties to a control.
     */
    static writeAriaProperties(
      /**
       * The RenderManager that can be used for writing to the render output buffer
       */
      oRm: RenderManager,
      /**
       * Default ARIA-compliant properties of the control
       */
      oDefaultProperties: object,
      /**
       * ARIA-compliant properties from data binding
       */
      oAriaProperties: AriaProperties
    ): void;
    /**
     * Gets current value of property {@link #getDescribedBy describedBy}.
     *
     * Identifies one or more elements that describe the object. See the related `labelledBy` property.
     *
     *
     * @returns Value of property `describedBy`
     */
    getDescribedBy(): string;
    /**
     * Gets current value of property {@link #getHasPopup hasPopup}.
     *
     * Indicates that the element has a popup context menu or a submenu.
     *
     *
     * @returns Value of property `hasPopup`
     */
    getHasPopup(): string;
    /**
     * Gets current value of property {@link #getLabel label}.
     *
     * Defines a string value that labels the current element. See the related `labelledBy` property.
     *
     *
     * @returns Value of property `label`
     */
    getLabel(): string;
    /**
     * Gets current value of property {@link #getLabelledBy labelledBy}.
     *
     * Identifies one or more elements that label the current element. See the related `label` and `describedBy`
     * properties.
     *
     *
     * @returns Value of property `labelledBy`
     */
    getLabelledBy(): string;
    /**
     * Gets current value of property {@link #getRole role}.
     *
     * Identifies the element role.
     *
     *
     * @returns Value of property `role`
     */
    getRole(): string;
    /**
     * Sets a new value for property {@link #getDescribedBy describedBy}.
     *
     * Identifies one or more elements that describe the object. See the related `labelledBy` property.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescribedBy(
      /**
       * New value for property `describedBy`
       */
      sDescribedBy?: string
    ): this;
    /**
     * Sets a new value for property {@link #getHasPopup hasPopup}.
     *
     * Indicates that the element has a popup context menu or a submenu.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHasPopup(
      /**
       * New value for property `hasPopup`
       */
      sHasPopup?: string
    ): this;
    /**
     * Sets a new value for property {@link #getLabel label}.
     *
     * Defines a string value that labels the current element. See the related `labelledBy` property.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLabel(
      /**
       * New value for property `label`
       */
      sLabel?: string
    ): this;
    /**
     * Sets a new value for property {@link #getLabelledBy labelledBy}.
     *
     * Identifies one or more elements that label the current element. See the related `label` and `describedBy`
     * properties.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLabelledBy(
      /**
       * New value for property `labelledBy`
       */
      sLabelledBy?: string
    ): this;
    /**
     * Sets a new value for property {@link #getRole role}.
     *
     * Identifies the element role.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setRole(
      /**
       * New value for property `role`
       */
      sRole?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the AriaProperties constructor.
   */
  export interface $AriaPropertiesSettings extends $ElementSettings {
    /**
     * Defines a string value that labels the current element. See the related `labelledBy` property.
     */
    label?: string | PropertyBindingInfo;

    /**
     * Identifies one or more elements that label the current element. See the related `label` and `describedBy`
     * properties.
     */
    labelledBy?: string | PropertyBindingInfo;

    /**
     * Identifies one or more elements that describe the object. See the related `labelledBy` property.
     */
    describedBy?: string | PropertyBindingInfo;

    /**
     * Identifies the element role.
     */
    role?: string | PropertyBindingInfo;

    /**
     * Indicates that the element has a popup context menu or a submenu.
     */
    hasPopup?: string | PropertyBindingInfo;
  }
}

declare module "sap/suite/ui/commons/BulletChart" {
  import {
    default as BulletMicroChart,
    $BulletMicroChartSettings,
  } from "sap/suite/ui/microchart/BulletMicroChart";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * Chart that displays an actual value as a horizontal bar in semantic color on the top of the background
   * bar, the numeric value, the scaling factor, along with the thresholds, and a target value as vertical
   * bars.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.BulletMicroChart.
   */
  export default class BulletChart extends BulletMicroChart {
    /**
     * Constructor for a new BulletChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.BulletMicroChart#constructor sap.suite.ui.microchart.BulletMicroChart }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $BulletChartSettings
    );
    /**
     * Constructor for a new BulletChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.BulletMicroChart#constructor sap.suite.ui.microchart.BulletMicroChart }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $BulletChartSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.BulletChart with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.BulletMicroChart.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, BulletChart>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.BulletChart.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the BulletChart constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.BulletMicroChart.
   */
  export interface $BulletChartSettings extends $BulletMicroChartSettings {}
}

declare module "sap/suite/ui/commons/BulletChartData" {
  import {
    default as BulletMicroChartData,
    $BulletMicroChartDataSettings,
  } from "sap/suite/ui/microchart/BulletMicroChartData";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * Thresholds data holder.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.BulletMicroChartData.
   */
  export default class BulletChartData extends BulletMicroChartData {
    /**
     * Constructor for a new BulletChartData.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.BulletMicroChartData#constructor sap.suite.ui.microchart.BulletMicroChartData }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $BulletChartDataSettings
    );
    /**
     * Constructor for a new BulletChartData.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.BulletMicroChartData#constructor sap.suite.ui.microchart.BulletMicroChartData }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $BulletChartDataSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.BulletChartData with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.BulletMicroChartData.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, BulletChartData>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.BulletChartData.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the BulletChartData constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.BulletMicroChartData.
   */
  export interface $BulletChartDataSettings
    extends $BulletMicroChartDataSettings {}
}

declare module "sap/suite/ui/commons/BusinessCard" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import { URI, CSSSize } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control allows you to display business card information including an image, first title (either
   * URL link or text), second title, and multiple text lines.
   *
   * @deprecated (since 1.32) - Deprecated. Standard Fiori technology should be used.
   */
  export default class BusinessCard extends Control {
    /**
     * Constructor for a new BusinessCard.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $BusinessCardSettings
    );
    /**
     * Constructor for a new BusinessCard.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $BusinessCardSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.BusinessCard with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, BusinessCard>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.BusinessCard.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Destroys the content in the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyContent(): this;
    /**
     * Destroys the firstTitle in the aggregation {@link #getFirstTitle firstTitle}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyFirstTitle(): this;
    /**
     * Gets content of aggregation {@link #getContent content}.
     *
     * The content of the BusinessCard body must be provided by the application developer.
     */
    getContent(): Control;
    /**
     * Gets content of aggregation {@link #getFirstTitle firstTitle}.
     *
     * Any control that can display a title. Content of this control appears at the first position in BusinessCard.
     * Recomended controls: sap.ui.commons.Label and sap.ui.commons.Link.
     */
    getFirstTitle(): Control;
    /**
     * Gets current value of property {@link #getIconPath iconPath}.
     *
     * A path to the BusinessCard icon.
     *
     *
     * @returns Value of property `iconPath`
     */
    getIconPath(): URI;
    /**
     * Gets current value of property {@link #getImageTooltip imageTooltip}.
     *
     * A tooltip that is set for an image.
     *
     *
     * @returns Value of property `imageTooltip`
     */
    getImageTooltip(): string;
    /**
     * Gets current value of property {@link #getSecondTitle secondTitle}.
     *
     * A short text line that describes this BusinessCard.
     *
     *
     * @returns Value of property `secondTitle`
     */
    getSecondTitle(): string;
    /**
     * Gets current value of property {@link #getType type}.
     *
     * Title of the BusinessCard that describes its type.
     *
     *
     * @returns Value of property `type`
     */
    getType(): string;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Width of the BusinessCard. Alternatively, CSS size can be set in %, px, or em.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Sets the aggregated {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setContent(
      /**
       * The content to set
       */
      oContent: Control
    ): this;
    /**
     * Sets the aggregated {@link #getFirstTitle firstTitle}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFirstTitle(
      /**
       * The firstTitle to set
       */
      oFirstTitle: Control
    ): this;
    /**
     * Sets a new value for property {@link #getIconPath iconPath}.
     *
     * A path to the BusinessCard icon.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIconPath(
      /**
       * New value for property `iconPath`
       */
      sIconPath?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getImageTooltip imageTooltip}.
     *
     * A tooltip that is set for an image.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setImageTooltip(
      /**
       * New value for property `imageTooltip`
       */
      sImageTooltip?: string
    ): this;
    /**
     * Sets a new value for property {@link #getSecondTitle secondTitle}.
     *
     * A short text line that describes this BusinessCard.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSecondTitle(
      /**
       * New value for property `secondTitle`
       */
      sSecondTitle?: string
    ): this;
    /**
     * Sets a new value for property {@link #getType type}.
     *
     * Title of the BusinessCard that describes its type.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setType(
      /**
       * New value for property `type`
       */
      sType?: string
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Width of the BusinessCard. Alternatively, CSS size can be set in %, px, or em.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
  }
  /**
   * Describes the settings that can be provided to the BusinessCard constructor.
   *
   * @deprecated (since 1.32) - Deprecated. Standard Fiori technology should be used.
   */
  export interface $BusinessCardSettings extends $ControlSettings {
    /**
     * Title of the BusinessCard that describes its type.
     */
    type?: string | PropertyBindingInfo;

    /**
     * A path to the BusinessCard icon.
     */
    iconPath?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * A short text line that describes this BusinessCard.
     */
    secondTitle?: string | PropertyBindingInfo;

    /**
     * Width of the BusinessCard. Alternatively, CSS size can be set in %, px, or em.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * A tooltip that is set for an image.
     */
    imageTooltip?: string | PropertyBindingInfo;

    /**
     * The content of the BusinessCard body must be provided by the application developer.
     */
    content?: Control;

    /**
     * Any control that can display a title. Content of this control appears at the first position in BusinessCard.
     * Recomended controls: sap.ui.commons.Label and sap.ui.commons.Link.
     */
    firstTitle?: Control;
  }
}

declare module "sap/suite/ui/commons/CalculationBuilder" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import CalculationBuilderFunction from "sap/suite/ui/commons/CalculationBuilderFunction";

  import CalculationBuilderGroup from "sap/suite/ui/commons/CalculationBuilderGroup";

  import CalculationBuilderItem from "sap/suite/ui/commons/CalculationBuilderItem";

  import Item from "sap/ui/core/Item";

  import CalculationBuilderVariable from "sap/suite/ui/commons/CalculationBuilderVariable";

  import {
    CalculationBuilderFunctionType,
    CalculationBuilderValidationMode,
  } from "sap/suite/ui/commons/library";

  import Event from "sap/ui/base/Event";

  import {
    AggregationBindingInfo,
    PropertyBindingInfo,
  } from "sap/ui/base/ManagedObject";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import CalculationBuilderValidationResult from "sap/suite/ui/commons/CalculationBuilderValidationResult";

  /**
   * Calculation Builder allows you to perform arithmetic calculations on literals and variables using standard
   * arithmetic operators as well as most common logical operators and functions.
   *  You can customize the sets of variables and functions that are visible in the calculation builder and
   * introduce your own custom functions, as needed.
   *  Arithmetic expressions can be entered using a touch-friendly visual editor or a textual editor that
   * provides autocomplete suggestions for variables and checks the expression syntax as you type.
   *
   * @since 1.56.0
   */
  export default class CalculationBuilder extends Control {
    /**
     * Constructor for a new calculation builder.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     * See:
     * 	{@link https://ui5.sap.com/#/topic/1db504725155424a8dc9fabd4147dd28 Calculation Builder}
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $CalculationBuilderSettings
    );
    /**
     * Constructor for a new calculation builder.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     * See:
     * 	{@link https://ui5.sap.com/#/topic/1db504725155424a8dc9fabd4147dd28 Calculation Builder}
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $CalculationBuilderSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.CalculationBuilder with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CalculationBuilder>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.CalculationBuilder.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some Function to the aggregation {@link #getFunctions functions}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addFunction(
      /**
       * The Function to add; if empty, nothing is inserted
       */
      oFunction: CalculationBuilderFunction
    ): this;
    /**
     * Adds some Group to the aggregation {@link #getGroups groups}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addGroup(
      /**
       * The Group to add; if empty, nothing is inserted
       */
      oGroup: CalculationBuilderGroup
    ): this;
    /**
     * Adds some item to the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addItem(
      /**
       * The item to add; if empty, nothing is inserted
       */
      oItem: CalculationBuilderItem
    ): this;
    /**
     * Adds some Operator to the aggregation {@link #getOperators operators}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addOperator(
      /**
       * The Operator to add; if empty, nothing is inserted
       */
      oOperator: Item
    ): this;
    /**
     * Adds some Variable to the aggregation {@link #getVariables variables}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addVariable(
      /**
       * The Variable to add; if empty, nothing is inserted
       */
      oVariable: CalculationBuilderVariable
    ): this;
    /**
     * Checks if the function is visible to the user.
     */
    allowFunction(
      /**
       * Name of the function
       */
      sFunction: CalculationBuilderFunctionType,
      /**
       * True if the function should be visible to the user
       */
      bAllow: boolean
    ): void;
    /**
     * Records a new error detected in the expression.
     */
    appendError(
      /**
       * Error object with contains following properties:
       */
      oError: {
        /**
         * Index of the item that contains errors
         */
        index?: object;
        /**
         * Title of the error
         */
        title?: number;
      }
    ): void;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:afterValidation afterValidation} event of this
     * `sap.suite.ui.commons.CalculationBuilder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CalculationBuilder` itself.
     *
     * This event is fired after the expression is validated.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAfterValidation(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CalculationBuilder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:afterValidation afterValidation} event of this
     * `sap.suite.ui.commons.CalculationBuilder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CalculationBuilder` itself.
     *
     * This event is fired after the expression is validated.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAfterValidation(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CalculationBuilder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.suite.ui.commons.CalculationBuilder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CalculationBuilder` itself.
     *
     * This event is fired when the order of items changes, or when some items are added or removed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CalculationBuilder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.suite.ui.commons.CalculationBuilder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CalculationBuilder` itself.
     *
     * This event is fired when the order of items changes, or when some items are added or removed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CalculationBuilder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validateFunction validateFunction} event of
     * this `sap.suite.ui.commons.CalculationBuilder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CalculationBuilder` itself.
     *
     * This event is fired for each custom function included in the expression.
     *  Custom functions can be defined using {@link sap.suite.ui.commons.CalculationBuilderFunction} and validated
     * using {@link sap.suite.ui.commons.CalculationBuilderValidationResult}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidateFunction(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: CalculationBuilder$ValidateFunctionEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CalculationBuilder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:validateFunction validateFunction} event of
     * this `sap.suite.ui.commons.CalculationBuilder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CalculationBuilder` itself.
     *
     * This event is fired for each custom function included in the expression.
     *  Custom functions can be defined using {@link sap.suite.ui.commons.CalculationBuilderFunction} and validated
     * using {@link sap.suite.ui.commons.CalculationBuilderValidationResult}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachValidateFunction(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: CalculationBuilder$ValidateFunctionEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CalculationBuilder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Binds aggregation {@link #getItems items} to model data.
     *
     * See {@link sap.ui.base.ManagedObject#bindAggregation ManagedObject.bindAggregation} for a detailed description
     * of the possible properties of `oBindingInfo`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    bindItems(
      /**
       * The binding information
       */
      oBindingInfo: AggregationBindingInfo
    ): this;
    /**
     * Destroys all the functions in the aggregation {@link #getFunctions functions}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyFunctions(): this;
    /**
     * Destroys all the groups in the aggregation {@link #getGroups groups}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyGroups(): this;
    /**
     * Destroys all the items in the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyItems(): this;
    /**
     * Destroys all the operators in the aggregation {@link #getOperators operators}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyOperators(): this;
    /**
     * Destroys all the variables in the aggregation {@link #getVariables variables}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyVariables(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:afterValidation afterValidation} event of
     * this `sap.suite.ui.commons.CalculationBuilder`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAfterValidation(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:change change} event of this `sap.suite.ui.commons.CalculationBuilder`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:validateFunction validateFunction} event of
     * this `sap.suite.ui.commons.CalculationBuilder`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachValidateFunction(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: CalculationBuilder$ValidateFunctionEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:afterValidation afterValidation} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAfterValidation(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:change change} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:validateFunction validateFunction} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireValidateFunction(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: CalculationBuilder$ValidateFunctionEventParameters
    ): this;
    /**
     * Gets current value of property {@link #getAllowComparisonOperators allowComparisonOperators}.
     *
     * Defines whether comparison operators (<, >, <=, >=, =, !=) are allowed.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `allowComparisonOperators`
     */
    getAllowComparisonOperators(): boolean;
    /**
     * Gets current value of property {@link #getAllowLogicalOperators allowLogicalOperators}.
     *
     * Defines whether logical operators (AND, OR, XOR, NOT) are allowed.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `allowLogicalOperators`
     */
    getAllowLogicalOperators(): boolean;
    /**
     * Gets the current value of property `allowStringConstants`.
     *
     * @deprecated (since 1.77.0) - replaced by {@link sap.suite.ui.commons.CalculationBuilder#getAllowStringLiterals}.
     *
     * @returns Value of property `allowStringConstants`
     */
    getAllowStringConstants(): boolean;
    /**
     * Gets current value of property {@link #getAllowStringLiterals allowStringLiterals}.
     *
     * Defines whether string literals are allowed.
     *  If set to `true`, the calculation builder doesn't validate strings as errors.
     *  Number literals are also allowed regardless of this setting.
     *
     * Default value is `false`.
     *
     * @since 1.77.0
     *
     * @returns Value of property `allowStringLiterals`
     */
    getAllowStringLiterals(): boolean;
    /**
     * Gets current value of property {@link #getAllowSuggestions allowSuggestions}.
     *
     * Defines whether autocomplete suggestions are shown when you type. Available only for desktop devices.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `allowSuggestions`
     */
    getAllowSuggestions(): boolean;
    /**
     * Gets current value of property {@link #getDisabledDefaultTokens disabledDefaultTokens}.
     *
     * Default operators or functions that are disabled. Delimiter for multiple items is ';'.
     *  Example of usage: +;-;sqrt
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `disabledDefaultTokens`
     */
    getDisabledDefaultTokens(): string;
    /**
     * Returns syntax errors.
     */
    getErrors(): void;
    /**
     * Gets current value of property {@link #getExpression expression}.
     *
     * Holds the arithmetic expression.
     *  Use either this property or aggregation `Items`. Not both.
     *
     *
     * @returns Value of property `expression`
     */
    getExpression(): string;
    /**
     * Gets content of aggregation {@link #getFunctions functions}.
     *
     * Holds the custom functions that can be used in the calculation builder.
     */
    getFunctions(): CalculationBuilderFunction[];
    /**
     * Gets content of aggregation {@link #getGroups groups}.
     *
     * Defines groups of variables.
     *  Variables can be organized into multiple groups for better orientation among different types of variables.
     */
    getGroups(): CalculationBuilderGroup[];
    /**
     * Returns the input toolbar of the calculation builder.
     *
     *
     * @returns Input toolbar
     */
    getInputToolbar(): Object;
    /**
     * Gets content of aggregation {@link #getItems items}.
     *
     * Holds the items (operators and operands) to be displayed in the calculation builder.
     */
    getItems(): CalculationBuilderItem[];
    /**
     * Gets current value of property {@link #getLayoutType layoutType}.
     *
     * Defines the layout type used for the calculation builder.
     *  The layout may include a visual editor, a text editor, or both. In addition, you can set the text editor
     * to be read-only.
     *
     * Default value is `"Default"`.
     *
     *
     * @returns Value of property `layoutType`
     */
    getLayoutType(): string;
    /**
     * Gets content of aggregation {@link #getOperators operators}.
     *
     * Holds additional operators defined by the application.
     * These operators are not validated.
     */
    getOperators(): Item[];
    /**
     * Gets current value of property {@link #getReadOnly readOnly}.
     *
     * Defines whether the control is read-only.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `readOnly`
     */
    getReadOnly(): boolean;
    /**
     * Gets current value of property {@link #getShowInputToolbar showInputToolbar}.
     *
     * Defines whether the input toolbar is visible.
     *  The input toolbar contains operators and functions that can be used in the expression.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `showInputToolbar`
     */
    getShowInputToolbar(): boolean;
    /**
     * Gets current value of property {@link #getShowToolbar showToolbar}.
     *
     * Defines whether the toolbar is visible.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showToolbar`
     */
    getShowToolbar(): boolean;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * The title of the calculation builder element.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Returns the toolbar of the calculation builder.
     *
     *
     * @returns Toolbar
     */
    getToolbar(): Object;
    /**
     * Gets current value of property {@link #getValidationMode validationMode}.
     *
     * Defines when the expression is validated.
     *
     * 	 - If set to `LiveChange`, the input is validated as you type.
     * 	 - If set to `FocusOut`, the input is validated, once you press Enter or Tab.
     *
     * Default value is `LiveChange`.
     *
     *
     * @returns Value of property `validationMode`
     */
    getValidationMode():
      | CalculationBuilderValidationMode
      | keyof typeof CalculationBuilderValidationMode;
    /**
     * Gets content of aggregation {@link #getVariables variables}.
     *
     * Holds the variables that can be used in the calculation builder.
     */
    getVariables(): CalculationBuilderVariable[];
    /**
     * Gets current value of property {@link #getWrapItemsInExpression wrapItemsInExpression}.
     *
     * Defines whether the expression should be wrapped inside the calculation builder field.
     *  If set to `false`, the expression is rearranged into a single scrollable row.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `wrapItemsInExpression`
     */
    getWrapItemsInExpression(): boolean;
    /**
     * Checks for the provided `sap.suite.ui.commons.CalculationBuilderFunction` in the aggregation {@link #getFunctions functions}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfFunction(
      /**
       * The Function whose index is looked for
       */
      oFunction: CalculationBuilderFunction
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.CalculationBuilderGroup` in the aggregation {@link #getGroups groups}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfGroup(
      /**
       * The Group whose index is looked for
       */
      oGroup: CalculationBuilderGroup
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.CalculationBuilderItem` in the aggregation {@link #getItems items}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfItem(
      /**
       * The item whose index is looked for
       */
      oItem: CalculationBuilderItem
    ): int;
    /**
     * Checks for the provided `sap.ui.core.Item` in the aggregation {@link #getOperators operators}. and returns
     * its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfOperator(
      /**
       * The Operator whose index is looked for
       */
      oOperator: Item
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.CalculationBuilderVariable` in the aggregation {@link #getVariables variables}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfVariable(
      /**
       * The Variable whose index is looked for
       */
      oVariable: CalculationBuilderVariable
    ): int;
    /**
     * Inserts a Function into the aggregation {@link #getFunctions functions}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertFunction(
      /**
       * The Function to insert; if empty, nothing is inserted
       */
      oFunction: CalculationBuilderFunction,
      /**
       * The `0`-based index the Function should be inserted at; for a negative value of `iIndex`, the Function
       * is inserted at position 0; for a value greater than the current size of the aggregation, the Function
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a Group into the aggregation {@link #getGroups groups}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertGroup(
      /**
       * The Group to insert; if empty, nothing is inserted
       */
      oGroup: CalculationBuilderGroup,
      /**
       * The `0`-based index the Group should be inserted at; for a negative value of `iIndex`, the Group is inserted
       * at position 0; for a value greater than the current size of the aggregation, the Group is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a item into the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertItem(
      /**
       * The item to insert; if empty, nothing is inserted
       */
      oItem: CalculationBuilderItem,
      /**
       * The `0`-based index the item should be inserted at; for a negative value of `iIndex`, the item is inserted
       * at position 0; for a value greater than the current size of the aggregation, the item is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a Operator into the aggregation {@link #getOperators operators}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertOperator(
      /**
       * The Operator to insert; if empty, nothing is inserted
       */
      oOperator: Item,
      /**
       * The `0`-based index the Operator should be inserted at; for a negative value of `iIndex`, the Operator
       * is inserted at position 0; for a value greater than the current size of the aggregation, the Operator
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a Variable into the aggregation {@link #getVariables variables}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertVariable(
      /**
       * The Variable to insert; if empty, nothing is inserted
       */
      oVariable: CalculationBuilderVariable,
      /**
       * The `0`-based index the Variable should be inserted at; for a negative value of `iIndex`, the Variable
       * is inserted at position 0; for a value greater than the current size of the aggregation, the Variable
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getFunctions functions}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllFunctions(): CalculationBuilderFunction[];
    /**
     * Removes all the controls from the aggregation {@link #getGroups groups}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllGroups(): CalculationBuilderGroup[];
    /**
     * Removes all the controls from the aggregation {@link #getItems items}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllItems(): CalculationBuilderItem[];
    /**
     * Removes all the controls from the aggregation {@link #getOperators operators}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllOperators(): Item[];
    /**
     * Removes all the controls from the aggregation {@link #getVariables variables}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllVariables(): CalculationBuilderVariable[];
    /**
     * Removes a Function from the aggregation {@link #getFunctions functions}.
     *
     *
     * @returns The removed Function or `null`
     */
    removeFunction(
      /**
       * The Function to remove or its index or id
       */
      vFunction: int | string | CalculationBuilderFunction
    ): CalculationBuilderFunction | null;
    /**
     * Removes a Group from the aggregation {@link #getGroups groups}.
     *
     *
     * @returns The removed Group or `null`
     */
    removeGroup(
      /**
       * The Group to remove or its index or id
       */
      vGroup: int | string | CalculationBuilderGroup
    ): CalculationBuilderGroup | null;
    /**
     * Removes a item from the aggregation {@link #getItems items}.
     *
     *
     * @returns The removed item or `null`
     */
    removeItem(
      /**
       * The item to remove or its index or id
       */
      vItem: int | string | CalculationBuilderItem
    ): CalculationBuilderItem | null;
    /**
     * Removes a Operator from the aggregation {@link #getOperators operators}.
     *
     *
     * @returns The removed Operator or `null`
     */
    removeOperator(
      /**
       * The Operator to remove or its index or id
       */
      vOperator: int | string | Item
    ): Item | null;
    /**
     * Removes a Variable from the aggregation {@link #getVariables variables}.
     *
     *
     * @returns The removed Variable or `null`
     */
    removeVariable(
      /**
       * The Variable to remove or its index or id
       */
      vVariable: int | string | CalculationBuilderVariable
    ): CalculationBuilderVariable | null;
    /**
     * Sets a new value for property {@link #getAllowComparisonOperators allowComparisonOperators}.
     *
     * Defines whether comparison operators (<, >, <=, >=, =, !=) are allowed.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAllowComparisonOperators(
      /**
       * New value for property `allowComparisonOperators`
       */
      bAllowComparisonOperators?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getAllowLogicalOperators allowLogicalOperators}.
     *
     * Defines whether logical operators (AND, OR, XOR, NOT) are allowed.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAllowLogicalOperators(
      /**
       * New value for property `allowLogicalOperators`
       */
      bAllowLogicalOperators?: boolean
    ): this;
    /**
     * Sets a new value for property `allowStringConstants`.
     *
     * @deprecated (since 1.77.0) - replaced by {@link sap.suite.ui.commons.CalculationBuilder#setAllowStringLiterals}.
     *
     * @returns Reference to `this` to allow method chaining
     */
    setAllowStringConstants(
      /**
       * New value for property `allowStringConstants`.
       */
      bAllowStringConstants: boolean
    ): CalculationBuilder;
    /**
     * Sets a new value for property {@link #getAllowStringLiterals allowStringLiterals}.
     *
     * Defines whether string literals are allowed.
     *  If set to `true`, the calculation builder doesn't validate strings as errors.
     *  Number literals are also allowed regardless of this setting.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     * @since 1.77.0
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAllowStringLiterals(
      /**
       * New value for property `allowStringLiterals`
       */
      bAllowStringLiterals?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getAllowSuggestions allowSuggestions}.
     *
     * Defines whether autocomplete suggestions are shown when you type. Available only for desktop devices.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAllowSuggestions(
      /**
       * New value for property `allowSuggestions`
       */
      bAllowSuggestions?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getDisabledDefaultTokens disabledDefaultTokens}.
     *
     * Default operators or functions that are disabled. Delimiter for multiple items is ';'.
     *  Example of usage: +;-;sqrt
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDisabledDefaultTokens(
      /**
       * New value for property `disabledDefaultTokens`
       */
      sDisabledDefaultTokens?: string
    ): this;
    /**
     * Sets a new value for property {@link #getExpression expression}.
     *
     * Holds the arithmetic expression.
     *  Use either this property or aggregation `Items`. Not both.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setExpression(
      /**
       * New value for property `expression`
       */
      sExpression?: string
    ): this;
    /**
     * Sets a new value for property {@link #getLayoutType layoutType}.
     *
     * Defines the layout type used for the calculation builder.
     *  The layout may include a visual editor, a text editor, or both. In addition, you can set the text editor
     * to be read-only.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Default"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLayoutType(
      /**
       * New value for property `layoutType`
       */
      sLayoutType?: string
    ): this;
    /**
     * Sets a new value for property {@link #getReadOnly readOnly}.
     *
     * Defines whether the control is read-only.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setReadOnly(
      /**
       * New value for property `readOnly`
       */
      bReadOnly?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowInputToolbar showInputToolbar}.
     *
     * Defines whether the input toolbar is visible.
     *  The input toolbar contains operators and functions that can be used in the expression.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowInputToolbar(
      /**
       * New value for property `showInputToolbar`
       */
      bShowInputToolbar?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowToolbar showToolbar}.
     *
     * Defines whether the toolbar is visible.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowToolbar(
      /**
       * New value for property `showToolbar`
       */
      bShowToolbar?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * The title of the calculation builder element.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValidationMode validationMode}.
     *
     * Defines when the expression is validated.
     *
     * 	 - If set to `LiveChange`, the input is validated as you type.
     * 	 - If set to `FocusOut`, the input is validated, once you press Enter or Tab.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `LiveChange`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValidationMode(
      /**
       * New value for property `validationMode`
       */
      sValidationMode?:
        | CalculationBuilderValidationMode
        | keyof typeof CalculationBuilderValidationMode
    ): this;
    /**
     * Sets a new value for property {@link #getWrapItemsInExpression wrapItemsInExpression}.
     *
     * Defines whether the expression should be wrapped inside the calculation builder field.
     *  If set to `false`, the expression is rearranged into a single scrollable row.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWrapItemsInExpression(
      /**
       * New value for property `wrapItemsInExpression`
       */
      bWrapItemsInExpression?: boolean
    ): this;
    /**
     * Unbinds aggregation {@link #getItems items} from model data.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    unbindItems(): this;
    /**
     * Updates the list of displayed errors
     *  You can call this method when you change errors outside default calculation builder events.
     */
    updateErrorsDisplay(): void;
    /**
     * Creates or updates the currently selected item.
     *  Call this function only when using the `customView` aggregation in the {@link sap.suite.ui.commons.CalculationBuilderGroup}.
     */
    updateOrCreateItem(
      /**
       * New key for current item (new or old)
       */
      sKey: string
    ): void;
    /**
     * Validates the current expression and displays errors.
     */
    validate(): void;
    /**
     * Checks if the expression syntax is valid.
     *
     *
     * @returns aErrors Array of errors found.
     */
    validateParts(): any[];
  }
  /**
   * Describes the settings that can be provided to the CalculationBuilder constructor.
   */
  export interface $CalculationBuilderSettings extends $ControlSettings {
    /**
     * Holds the arithmetic expression.
     *  Use either this property or aggregation `Items`. Not both.
     */
    expression?: string | PropertyBindingInfo;

    /**
     * The title of the calculation builder element.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Defines whether the toolbar is visible.
     */
    showToolbar?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether the expression should be wrapped inside the calculation builder field.
     *  If set to `false`, the expression is rearranged into a single scrollable row.
     */
    wrapItemsInExpression?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the layout type used for the calculation builder.
     *  The layout may include a visual editor, a text editor, or both. In addition, you can set the text editor
     * to be read-only.
     */
    layoutType?: string | PropertyBindingInfo;

    /**
     * Defines whether the input toolbar is visible.
     *  The input toolbar contains operators and functions that can be used in the expression.
     */
    showInputToolbar?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether the control is read-only.
     */
    readOnly?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether comparison operators (<, >, <=, >=, =, !=) are allowed.
     */
    allowComparisonOperators?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether logical operators (AND, OR, XOR, NOT) are allowed.
     */
    allowLogicalOperators?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether autocomplete suggestions are shown when you type. Available only for desktop devices.
     */
    allowSuggestions?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether string constants are allowed.
     *  If set to `true`, the calculation builder doesn't validate custom strings as errors.
     *
     * @deprecated (since 1.77.0) - replaced by the `allowStringLiterals` property.
     */
    allowStringConstants?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether string literals are allowed.
     *  If set to `true`, the calculation builder doesn't validate strings as errors.
     *  Number literals are also allowed regardless of this setting.
     *
     * @since 1.77.0
     */
    allowStringLiterals?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines when the expression is validated.
     *
     * 	 - If set to `LiveChange`, the input is validated as you type.
     * 	 - If set to `FocusOut`, the input is validated, once you press Enter or Tab.
     */
    validationMode?:
      | (
          | CalculationBuilderValidationMode
          | keyof typeof CalculationBuilderValidationMode
        )
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Default operators or functions that are disabled. Delimiter for multiple items is ';'.
     *  Example of usage: +;-;sqrt
     */
    disabledDefaultTokens?: string | PropertyBindingInfo;

    /**
     * Holds the items (operators and operands) to be displayed in the calculation builder.
     */
    items?:
      | CalculationBuilderItem[]
      | CalculationBuilderItem
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * Holds the variables that can be used in the calculation builder.
     */
    variables?:
      | CalculationBuilderVariable[]
      | CalculationBuilderVariable
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * Holds the custom functions that can be used in the calculation builder.
     */
    functions?:
      | CalculationBuilderFunction[]
      | CalculationBuilderFunction
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * Holds additional operators defined by the application.
     * These operators are not validated.
     */
    operators?: Item[] | Item | AggregationBindingInfo | `{${string}}`;

    /**
     * Defines groups of variables.
     *  Variables can be organized into multiple groups for better orientation among different types of variables.
     */
    groups?:
      | CalculationBuilderGroup[]
      | CalculationBuilderGroup
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * This event is fired for each custom function included in the expression.
     *  Custom functions can be defined using {@link sap.suite.ui.commons.CalculationBuilderFunction} and validated
     * using {@link sap.suite.ui.commons.CalculationBuilderValidationResult}.
     */
    validateFunction?: (
      oEvent: CalculationBuilder$ValidateFunctionEvent
    ) => void;

    /**
     * This event is fired when the order of items changes, or when some items are added or removed.
     */
    change?: (oEvent: Event) => void;

    /**
     * This event is fired after the expression is validated.
     */
    afterValidation?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the CalculationBuilder#afterValidation event.
   */
  export interface CalculationBuilder$AfterValidationEventParameters {}

  /**
   * Event object of the CalculationBuilder#afterValidation event.
   */
  export type CalculationBuilder$AfterValidationEvent = Event<
    CalculationBuilder$AfterValidationEventParameters,
    CalculationBuilder
  >;

  /**
   * Parameters of the CalculationBuilder#change event.
   */
  export interface CalculationBuilder$ChangeEventParameters {}

  /**
   * Event object of the CalculationBuilder#change event.
   */
  export type CalculationBuilder$ChangeEvent = Event<
    CalculationBuilder$ChangeEventParameters,
    CalculationBuilder
  >;

  /**
   * Parameters of the CalculationBuilder#validateFunction event.
   */
  export interface CalculationBuilder$ValidateFunctionEventParameters {
    definition?: object;

    customFunction?: object;

    result?: CalculationBuilderValidationResult;
  }

  /**
   * Event object of the CalculationBuilder#validateFunction event.
   */
  export type CalculationBuilder$ValidateFunctionEvent = Event<
    CalculationBuilder$ValidateFunctionEventParameters,
    CalculationBuilder
  >;
}

declare module "sap/suite/ui/commons/CalculationBuilderFunction" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import CalculationBuilderItem from "sap/suite/ui/commons/CalculationBuilderItem";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * A custom function to be used in the calculation builder.
   *  The default functions available for the calculation builder are defined in {@link sap.suite.ui.commons.CalculationBuilderFunctionType}.
   *
   * @since 1.56.0
   */
  export default class CalculationBuilderFunction extends Control {
    /**
     * Constructor for a new custom function.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $CalculationBuilderFunctionSettings
    );
    /**
     * Constructor for a new custom function.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $CalculationBuilderFunctionSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.CalculationBuilderFunction with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CalculationBuilderFunction>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.CalculationBuilderFunction.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some item to the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addItem(
      /**
       * The item to add; if empty, nothing is inserted
       */
      oItem: CalculationBuilderItem
    ): this;
    /**
     * Destroys all the items in the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyItems(): this;
    /**
     * Gets current value of property {@link #getDescription description}.
     *
     * Description of the function.
     *  The description is displayed in the functions menu on the calculation builder's toolbar. If no description
     * is specified, it is generated automatically based on the `key` property and the parameters defined in
     * the `items` aggregation.
     *
     *
     * @returns Value of property `description`
     */
    getDescription(): string;
    /**
     * Gets content of aggregation {@link #getItems items}.
     *
     * Holds the parameters that can be passed to the function.
     *  For example, if the function has two parameters, the `items` aggregation may consist of:
     * 	 - A {@link sap.suite.ui.commons.CalculationBuilderItem} with an empty key for the first argument entered
     *     by the user.
     * 	 - A {@link sap.suite.ui.commons.CalculationBuilderItem} with a "`,`" (comma) key for the comma that
     *     separates the arguments.
     * 	 - A {@link sap.suite.ui.commons.CalculationBuilderItem} with an empty key for the second argument entered
     *     by the user.
     */
    getItems(): CalculationBuilderItem[];
    /**
     * Gets current value of property {@link #getKey key}.
     *
     * A key associated with the function. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Gets current value of property {@link #getLabel label}.
     *
     * Label for the function.
     *  The label is displayed in the visual editor of the calculation builder and in the functions menu on
     * the calculation builder's toolbar.
     *
     *
     * @returns Value of property `label`
     */
    getLabel(): string;
    /**
     * Gets current value of property {@link #getUseDefaultValidation useDefaultValidation}.
     *
     * Defines whether default validation should be used.
     *  Set this property to `false` to provide your own validation algorithm using {@link sap.suite.ui.commons.CalculationBuilderValidationResult}.
     *  If you do not provide your own validation algorithm for this custom function, set this property to `true`,
     * and the function will be validated by the calculation builder.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `useDefaultValidation`
     */
    getUseDefaultValidation(): boolean;
    /**
     * Checks for the provided `sap.suite.ui.commons.CalculationBuilderItem` in the aggregation {@link #getItems items}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfItem(
      /**
       * The item whose index is looked for
       */
      oItem: CalculationBuilderItem
    ): int;
    /**
     * Inserts a item into the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertItem(
      /**
       * The item to insert; if empty, nothing is inserted
       */
      oItem: CalculationBuilderItem,
      /**
       * The `0`-based index the item should be inserted at; for a negative value of `iIndex`, the item is inserted
       * at position 0; for a value greater than the current size of the aggregation, the item is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getItems items}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllItems(): CalculationBuilderItem[];
    /**
     * Removes a item from the aggregation {@link #getItems items}.
     *
     *
     * @returns The removed item or `null`
     */
    removeItem(
      /**
       * The item to remove or its index or id
       */
      vItem: int | string | CalculationBuilderItem
    ): CalculationBuilderItem | null;
    /**
     * Sets a new value for property {@link #getDescription description}.
     *
     * Description of the function.
     *  The description is displayed in the functions menu on the calculation builder's toolbar. If no description
     * is specified, it is generated automatically based on the `key` property and the parameters defined in
     * the `items` aggregation.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescription(
      /**
       * New value for property `description`
       */
      sDescription?: string
    ): this;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * A key associated with the function. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
    /**
     * Sets a new value for property {@link #getLabel label}.
     *
     * Label for the function.
     *  The label is displayed in the visual editor of the calculation builder and in the functions menu on
     * the calculation builder's toolbar.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLabel(
      /**
       * New value for property `label`
       */
      sLabel?: string
    ): this;
    /**
     * Sets a new value for property {@link #getUseDefaultValidation useDefaultValidation}.
     *
     * Defines whether default validation should be used.
     *  Set this property to `false` to provide your own validation algorithm using {@link sap.suite.ui.commons.CalculationBuilderValidationResult}.
     *  If you do not provide your own validation algorithm for this custom function, set this property to `true`,
     * and the function will be validated by the calculation builder.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUseDefaultValidation(
      /**
       * New value for property `useDefaultValidation`
       */
      bUseDefaultValidation?: boolean
    ): this;
  }
  /**
   * Describes the settings that can be provided to the CalculationBuilderFunction constructor.
   */
  export interface $CalculationBuilderFunctionSettings
    extends $ControlSettings {
    /**
     * A key associated with the function. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     */
    key?: string | PropertyBindingInfo;

    /**
     * Label for the function.
     *  The label is displayed in the visual editor of the calculation builder and in the functions menu on
     * the calculation builder's toolbar.
     */
    label?: string | PropertyBindingInfo;

    /**
     * Description of the function.
     *  The description is displayed in the functions menu on the calculation builder's toolbar. If no description
     * is specified, it is generated automatically based on the `key` property and the parameters defined in
     * the `items` aggregation.
     */
    description?: string | PropertyBindingInfo;

    /**
     * Defines whether default validation should be used.
     *  Set this property to `false` to provide your own validation algorithm using {@link sap.suite.ui.commons.CalculationBuilderValidationResult}.
     *  If you do not provide your own validation algorithm for this custom function, set this property to `true`,
     * and the function will be validated by the calculation builder.
     */
    useDefaultValidation?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Holds the parameters that can be passed to the function.
     *  For example, if the function has two parameters, the `items` aggregation may consist of:
     * 	 - A {@link sap.suite.ui.commons.CalculationBuilderItem} with an empty key for the first argument entered
     *     by the user.
     * 	 - A {@link sap.suite.ui.commons.CalculationBuilderItem} with a "`,`" (comma) key for the comma that
     *     separates the arguments.
     * 	 - A {@link sap.suite.ui.commons.CalculationBuilderItem} with an empty key for the second argument entered
     *     by the user.
     */
    items?:
      | CalculationBuilderItem[]
      | CalculationBuilderItem
      | AggregationBindingInfo
      | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/CalculationBuilderGroup" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * A variable group is used to organize the variables that the user can insert into the {@link sap.suite.ui.commons.CalculationBuilder }
   * expression.
   *
   * @since 1.60.0
   */
  export default class CalculationBuilderGroup extends Control {
    /**
     * Constructor for a new calculation builder variable group.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $CalculationBuilderGroupSettings
    );
    /**
     * Constructor for a new calculation builder variable group.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $CalculationBuilderGroupSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.CalculationBuilderGroup with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CalculationBuilderGroup>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.CalculationBuilderGroup.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:setSelection setSelection} event of this `sap.suite.ui.commons.CalculationBuilderGroup`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CalculationBuilderGroup` itself.
     *
     * This event is fired when the user selects the variable group.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSetSelection(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: CalculationBuilderGroup$SetSelectionEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CalculationBuilderGroup`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:setSelection setSelection} event of this `sap.suite.ui.commons.CalculationBuilderGroup`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CalculationBuilderGroup` itself.
     *
     * This event is fired when the user selects the variable group.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSetSelection(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: CalculationBuilderGroup$SetSelectionEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CalculationBuilderGroup`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys the customView in the aggregation {@link #getCustomView customView}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCustomView(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:setSelection setSelection} event of this `sap.suite.ui.commons.CalculationBuilderGroup`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachSetSelection(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: CalculationBuilderGroup$SetSelectionEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:setSelection setSelection} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireSetSelection(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: CalculationBuilderGroup$SetSelectionEventParameters
    ): this;
    /**
     * Gets content of aggregation {@link #getCustomView customView}.
     *
     * Holds the variables included in the variable group.
     */
    getCustomView(): Control;
    /**
     * Gets current value of property {@link #getDescription description}.
     *
     * Additional description for the variable group.
     *
     *
     * @returns Value of property `description`
     */
    getDescription(): string;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * Icon for the variable group. This icon is displayed in the Variables menu.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): string;
    /**
     * Gets current value of property {@link #getKey key}.
     *
     * A key associated with the group. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * Label for the group.
     *  The label is displayed in the visual editor of the calculation builder and in the variables menu on
     * the calculation builder's toolbar.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets the aggregated {@link #getCustomView customView}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCustomView(
      /**
       * The customView to set
       */
      oCustomView: Control
    ): this;
    /**
     * Sets a new value for property {@link #getDescription description}.
     *
     * Additional description for the variable group.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescription(
      /**
       * New value for property `description`
       */
      sDescription?: string
    ): this;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * Icon for the variable group. This icon is displayed in the Variables menu.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: string
    ): this;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * A key associated with the group. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * Label for the group.
     *  The label is displayed in the visual editor of the calculation builder and in the variables menu on
     * the calculation builder's toolbar.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the CalculationBuilderGroup constructor.
   */
  export interface $CalculationBuilderGroupSettings extends $ControlSettings {
    /**
     * A key associated with the group. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     */
    key?: string | PropertyBindingInfo;

    /**
     * Label for the group.
     *  The label is displayed in the visual editor of the calculation builder and in the variables menu on
     * the calculation builder's toolbar.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Additional description for the variable group.
     */
    description?: string | PropertyBindingInfo;

    /**
     * Icon for the variable group. This icon is displayed in the Variables menu.
     */
    icon?: string | PropertyBindingInfo;

    /**
     * Holds the variables included in the variable group.
     */
    customView?: Control;

    /**
     * This event is fired when the user selects the variable group.
     */
    setSelection?: (oEvent: CalculationBuilderGroup$SetSelectionEvent) => void;
  }

  /**
   * Parameters of the CalculationBuilderGroup#setSelection event.
   */
  export interface CalculationBuilderGroup$SetSelectionEventParameters {
    key?: string;
  }

  /**
   * Event object of the CalculationBuilderGroup#setSelection event.
   */
  export type CalculationBuilderGroup$SetSelectionEvent = Event<
    CalculationBuilderGroup$SetSelectionEventParameters,
    CalculationBuilderGroup
  >;
}

declare module "sap/suite/ui/commons/CalculationBuilderItem" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import CalculationBuilderVariable from "sap/suite/ui/commons/CalculationBuilderVariable";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Each of the items used as building blocks to create an arithmetic expression in the calculation builder.
   *
   * @since 1.56.0
   */
  export default class CalculationBuilderItem extends Control {
    /**
     * Constructor for a new item used in the expression.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $CalculationBuilderItemSettings
    );
    /**
     * Constructor for a new item used in the expression.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $CalculationBuilderItemSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.CalculationBuilderItem with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CalculationBuilderItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.CalculationBuilderItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getKey key}.
     *
     * A key associated with the item. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Returns the type of the item.
     *  Available item types are defined in {@link sap.suite.ui.commons.CalculationBuilderItemType}.
     *
     *
     * @returns Type of the item
     */
    getType(): string;
    /**
     * Checks if there is a variable object related to this item.
     *
     *
     * @returns Variable object paired with this item, if there is any.
     */
    getVariable(): CalculationBuilderVariable;
    /**
     * Checks if the item is expandable.
     *
     *
     * @returns True if the item is expandable.
     */
    isExpandable(): boolean;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * A key associated with the item. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the CalculationBuilderItem constructor.
   */
  export interface $CalculationBuilderItemSettings extends $ControlSettings {
    /**
     * A key associated with the item. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     */
    key?: string | PropertyBindingInfo;
  }
}

declare module "sap/suite/ui/commons/CalculationBuilderValidationResult" {
  import {
    default as ManagedObject,
    $ManagedObjectSettings,
  } from "sap/ui/base/ManagedObject";

  import ManagedObjectMetadata from "sap/ui/base/ManagedObjectMetadata";

  /**
   * This can be used for creating your own validation algorithm for custom functions.
   *  Custom functions can be defined using {@link sap.suite.ui.commons.CalculationBuilderFunction}.
   *
   * @since 1.56.0
   */
  export default class CalculationBuilderValidationResult extends ManagedObject {
    /**
     * Constructor for a new validation result.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.CalculationBuilderValidationResult with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.ManagedObject.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CalculationBuilderValidationResult>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.CalculationBuilderValidationResult.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ManagedObjectMetadata;
  }
  /**
   * Describes the settings that can be provided to the CalculationBuilderValidationResult constructor.
   */
  export interface $CalculationBuilderValidationResultSettings
    extends $ManagedObjectSettings {}
}

declare module "sap/suite/ui/commons/CalculationBuilderVariable" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import CalculationBuilderItem from "sap/suite/ui/commons/CalculationBuilderItem";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * A variable used in the expression entered into the calculation builder.
   *
   * @since 1.56.0
   */
  export default class CalculationBuilderVariable extends Control {
    /**
     * Constructor for a new variable.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $CalculationBuilderVariableSettings
    );
    /**
     * Constructor for a new variable.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $CalculationBuilderVariableSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.CalculationBuilderVariable with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CalculationBuilderVariable>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.CalculationBuilderVariable.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some Item to the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addItem(
      /**
       * The Item to add; if empty, nothing is inserted
       */
      oItem: CalculationBuilderItem
    ): this;
    /**
     * Destroys all the items in the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyItems(): this;
    /**
     * Gets current value of property {@link #getGroup group}.
     *
     * The group where this variable belongs.
     *  To define variable groups, use {@link sap.suite.ui.commons.CalculationBuilderGroup}.
     *
     *
     * @returns Value of property `group`
     */
    getGroup(): string;
    /**
     * Gets content of aggregation {@link #getItems items}.
     *
     * Holds the items included in the variable.
     */
    getItems(): CalculationBuilderItem[];
    /**
     * Gets current value of property {@link #getKey key}.
     *
     * A key associated with the variable. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Gets current value of property {@link #getLabel label}.
     *
     * Label for the variable.
     *  The label is displayed in the visual editor of the calculation builder and in the variables menu on
     * the calculation builder's toolbar.
     *
     *
     * @returns Value of property `label`
     */
    getLabel(): string;
    /**
     * Checks for the provided `sap.suite.ui.commons.CalculationBuilderItem` in the aggregation {@link #getItems items}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfItem(
      /**
       * The Item whose index is looked for
       */
      oItem: CalculationBuilderItem
    ): int;
    /**
     * Inserts a Item into the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertItem(
      /**
       * The Item to insert; if empty, nothing is inserted
       */
      oItem: CalculationBuilderItem,
      /**
       * The `0`-based index the Item should be inserted at; for a negative value of `iIndex`, the Item is inserted
       * at position 0; for a value greater than the current size of the aggregation, the Item is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getItems items}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllItems(): CalculationBuilderItem[];
    /**
     * Removes a Item from the aggregation {@link #getItems items}.
     *
     *
     * @returns The removed Item or `null`
     */
    removeItem(
      /**
       * The Item to remove or its index or id
       */
      vItem: int | string | CalculationBuilderItem
    ): CalculationBuilderItem | null;
    /**
     * Sets a new value for property {@link #getGroup group}.
     *
     * The group where this variable belongs.
     *  To define variable groups, use {@link sap.suite.ui.commons.CalculationBuilderGroup}.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setGroup(
      /**
       * New value for property `group`
       */
      sGroup: string
    ): this;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * A key associated with the variable. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
    /**
     * Sets a new value for property {@link #getLabel label}.
     *
     * Label for the variable.
     *  The label is displayed in the visual editor of the calculation builder and in the variables menu on
     * the calculation builder's toolbar.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLabel(
      /**
       * New value for property `label`
       */
      sLabel?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the CalculationBuilderVariable constructor.
   */
  export interface $CalculationBuilderVariableSettings
    extends $ControlSettings {
    /**
     * A key associated with the variable. This property is mandatory.
     *  The key is displayed in the text editor area of the calculation builder.
     */
    key?: string | PropertyBindingInfo;

    /**
     * Label for the variable.
     *  The label is displayed in the visual editor of the calculation builder and in the variables menu on
     * the calculation builder's toolbar.
     */
    label?: string | PropertyBindingInfo;

    /**
     * The group where this variable belongs.
     *  To define variable groups, use {@link sap.suite.ui.commons.CalculationBuilderGroup}.
     */
    group?: string | PropertyBindingInfo;

    /**
     * Holds the items included in the variable.
     */
    items?:
      | CalculationBuilderItem[]
      | CalculationBuilderItem
      | AggregationBindingInfo
      | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/ChartContainer" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ChartContainerContent from "sap/suite/ui/commons/ChartContainerContent";

  import Icon from "sap/ui/core/Icon";

  import Event from "sap/ui/base/Event";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import ScrollEnablement from "sap/ui/core/delegate/ScrollEnablement";

  import OverflowToolbar from "sap/m/OverflowToolbar";

  import { CSSSize } from "sap/ui/core/library";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * Provides a toolbar with generic functions for tables and charts based on the VizFrame control like zoom,
   * display in fullscreen mode, toggle the legend, switch between chart types, and changes of the chart dimension.
   * The controls of the content aggregation are positioned below the toolbar. Additional functions can be
   * added to the toolbar with the customIcons aggregation.
   */
  export default class ChartContainer extends Control {
    /**
     * Constructor for a new ChartContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ChartContainerSettings
    );
    /**
     * Constructor for a new ChartContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ChartContainerSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ChartContainer with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ChartContainer>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ChartContainer.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some content to the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addContent(
      /**
       * The content to add; if empty, nothing is inserted
       */
      oContent: ChartContainerContent
    ): this;
    /**
     * Adds some customIcon to the aggregation {@link #getCustomIcons customIcons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addCustomIcon(
      /**
       * The customIcon to add; if empty, nothing is inserted
       */
      oCustomIcon: Icon
    ): this;
    /**
     * Adds some dimensionSelector to the aggregation {@link #getDimensionSelectors dimensionSelectors}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addDimensionSelector(
      /**
       * The dimensionSelector to add; if empty, nothing is inserted
       */
      oDimensionSelector: Control
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:contentChange contentChange} event of this `sap.suite.ui.commons.ChartContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ChartContainer` itself.
     *
     * Event fired when a user changes the displayed content.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachContentChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ChartContainer$ContentChangeEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ChartContainer`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:contentChange contentChange} event of this `sap.suite.ui.commons.ChartContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ChartContainer` itself.
     *
     * Event fired when a user changes the displayed content.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachContentChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ChartContainer$ContentChangeEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ChartContainer`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:customZoomInPress customZoomInPress} event of
     * this `sap.suite.ui.commons.ChartContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ChartContainer` itself.
     *
     * Custom event for zoom in.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCustomZoomInPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ChartContainer`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:customZoomInPress customZoomInPress} event of
     * this `sap.suite.ui.commons.ChartContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ChartContainer` itself.
     *
     * Custom event for zoom in.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCustomZoomInPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ChartContainer`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:customZoomOutPress customZoomOutPress} event
     * of this `sap.suite.ui.commons.ChartContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ChartContainer` itself.
     *
     * Custom event for zoom out.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCustomZoomOutPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ChartContainer`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:customZoomOutPress customZoomOutPress} event
     * of this `sap.suite.ui.commons.ChartContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ChartContainer` itself.
     *
     * Custom event for zoom out.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCustomZoomOutPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ChartContainer`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:personalizationPress personalizationPress} event
     * of this `sap.suite.ui.commons.ChartContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ChartContainer` itself.
     *
     * Event fired when a user clicks on the personalization icon.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPersonalizationPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ChartContainer`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:personalizationPress personalizationPress} event
     * of this `sap.suite.ui.commons.ChartContainer`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ChartContainer` itself.
     *
     * Event fired when a user clicks on the personalization icon.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPersonalizationPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ChartContainer`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys all the content in the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyContent(): this;
    /**
     * Destroys all the customIcons in the aggregation {@link #getCustomIcons customIcons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCustomIcons(): this;
    /**
     * Destroys all the dimensionSelectors in the aggregation {@link #getDimensionSelectors dimensionSelectors}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyDimensionSelectors(): this;
    /**
     * Destroys the toolbar in the aggregation {@link #getToolbar toolbar}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyToolbar(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:contentChange contentChange} event of this
     * `sap.suite.ui.commons.ChartContainer`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachContentChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ChartContainer$ContentChangeEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:customZoomInPress customZoomInPress} event
     * of this `sap.suite.ui.commons.ChartContainer`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachCustomZoomInPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:customZoomOutPress customZoomOutPress} event
     * of this `sap.suite.ui.commons.ChartContainer`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachCustomZoomOutPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:personalizationPress personalizationPress }
     * event of this `sap.suite.ui.commons.ChartContainer`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPersonalizationPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:contentChange contentChange} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireContentChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ChartContainer$ContentChangeEventParameters
    ): this;
    /**
     * Fires event {@link #event:customZoomInPress customZoomInPress} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireCustomZoomInPress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:customZoomOutPress customZoomOutPress} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireCustomZoomOutPress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:personalizationPress personalizationPress} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePersonalizationPress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getAutoAdjustHeight autoAdjustHeight}.
     *
     * Determine whether to stretch the chart height to the maximum possible height of ChartContainer's parent
     * container. As a prerequisite, the parent container needs to have a fixed value height or be able to determine
     * height from its parent.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `autoAdjustHeight`
     */
    getAutoAdjustHeight(): boolean;
    /**
     * Gets content of aggregation {@link #getContent content}.
     *
     * ChartToolBar Content aggregation. Only sap.viz.ui5.controls.VizFrame, sap.m.Table and sap.ui.table.Table
     * can be embedded. If not specified explicitly, the rendering order of the charts is determined by the
     * sequence of contents provided by the application via this aggregation. This means, per default the first
     * chart of the aggregation will be rendered within the container.
     */
    getContent(): ChartContainerContent[];
    /**
     * Gets content of aggregation {@link #getCustomIcons customIcons}.
     *
     * This aggregation contains the custom icons that should be displayed additionally on the toolbar. It is
     * not guaranteed that the same instance of the sap.ui.core.Icon control will be used within the toolbar,
     * but the toolbar will contain a sap.m.OverflowToolbarButton with an icon property equal to the src property
     * of the sap.ui.core.Icon provided in the aggregation. If a press event is triggered by the icon displayed
     * on the toolbar, then the press handler of the original sap.ui.core.Icon control is used. The instance
     * of the control, that has triggered the press event, can be accessed using the "controlReference" parameter
     * of the event object.
     */
    getCustomIcons(): Icon[];
    /**
     * Gets content of aggregation {@link #getDimensionSelectors dimensionSelectors}.
     *
     * Dimension Selects.
     */
    getDimensionSelectors(): Control[];
    /**
     * Gets current value of property {@link #getEnableScroll enableScroll}.
     *
     * If set to `true`, the Container control has its own scroll bar, with the scrolling taking place within
     * the Container control itself.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `enableScroll`
     */
    getEnableScroll(): boolean;
    /**
     * Gets current value of property {@link #getFullScreen fullScreen}.
     *
     * Display the chart and the toolbar in full screen or normal mode.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `fullScreen`
     */
    getFullScreen(): boolean;
    /**
     * Returns the current instance of the delegate to other controls.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The current instance of the delegate
     */
    getScrollDelegate(): ScrollEnablement;
    /**
     * Returns the currently selected content control.
     *
     *
     * @returns The currently selected content
     */
    getSelectedContent(): Control;
    /**
     * Gets current value of property {@link #getSelectorGroupLabel selectorGroupLabel}.
     *
     * Custom Label for Selectors Group.
     *
     * @deprecated (since 1.32.0) - Obsolete property as sap.m.Toolbar is replaced by sap.m.OverflowToolbar.
     *
     * @returns Value of property `selectorGroupLabel`
     */
    getSelectorGroupLabel(): string;
    /**
     * Gets current value of property {@link #getShowFullScreen showFullScreen}.
     *
     * Set to true to display the full screen icon. Set to false to hide it.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `showFullScreen`
     */
    getShowFullScreen(): boolean;
    /**
     * Gets current value of property {@link #getShowLegend showLegend}.
     *
     * Set to true to display the charts' legends. Set to false to hide them. See also showLegendButton.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showLegend`
     */
    getShowLegend(): boolean;
    /**
     * Gets current value of property {@link #getShowLegendButton showLegendButton}.
     *
     * Set to true or false to display or hide a button for controlling the visbility of the chart's legend.
     * Please be aware that setting this property to true indirectly is setting showLegend to false. If you
     * need to hide the button but to show the legend, you need to set showLegend at a later point in time (onBeforeRendering).
     * The execution order of the combined properties is not guaranteed by the control.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showLegendButton`
     */
    getShowLegendButton(): boolean;
    /**
     * Gets current value of property {@link #getShowPersonalization showPersonalization}.
     *
     * Set to true to display the personalization icon. Set to false to hide it.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `showPersonalization`
     */
    getShowPersonalization(): boolean;
    /**
     * Gets current value of property {@link #getShowSelectionDetails showSelectionDetails}.
     *
     * Set to true to display the 'Details' button that opens a popup dialog with details about the selected
     * data from the VizFrame based chart.
     *
     * Default value is `false`.
     *
     * @since 1.48.0
     *
     * @returns Value of property `showSelectionDetails`
     */
    getShowSelectionDetails(): boolean;
    /**
     * Gets current value of property {@link #getShowZoom showZoom}.
     *
     * Set to true to display zoom icons. Set to false to hide them.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showZoom`
     */
    getShowZoom(): boolean;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * String shown if there are no dimensions to display.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Gets content of aggregation {@link #getToolbar toolbar}.
     *
     * Overflow ToolBar. If an external toolbar is used, it will be integrated with the embedded toolbar via
     * a placeholder. This placeholder is mandatory, and it needs to be of type 'sap.suite.ui.commons.ChartContainerToolbarPlaceholder'.
     */
    getToolbar(): OverflowToolbar;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Defines the width of the Container.
     *
     * Default value is `"100%"`.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Gets current value of property {@link #getWrapLabels wrapLabels}.
     *
     * Set to true to wrap text labels in the dialog that opens when the user clicks or taps the 'Details' button.
     *
     * Default value is `false`.
     *
     * @since 1.58.0
     *
     * @returns Value of property `wrapLabels`
     */
    getWrapLabels(): boolean;
    /**
     * Checks for the provided `sap.suite.ui.commons.ChartContainerContent` in the aggregation {@link #getContent content}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfContent(
      /**
       * The content whose index is looked for
       */
      oContent: ChartContainerContent
    ): int;
    /**
     * Checks for the provided `sap.ui.core.Icon` in the aggregation {@link #getCustomIcons customIcons}. and
     * returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfCustomIcon(
      /**
       * The customIcon whose index is looked for
       */
      oCustomIcon: Icon
    ): int;
    /**
     * Checks for the provided `sap.ui.core.Control` in the aggregation {@link #getDimensionSelectors dimensionSelectors}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfDimensionSelector(
      /**
       * The dimensionSelector whose index is looked for
       */
      oDimensionSelector: Control
    ): int;
    /**
     * Inserts a content into the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertContent(
      /**
       * The content to insert; if empty, nothing is inserted
       */
      oContent: ChartContainerContent,
      /**
       * The `0`-based index the content should be inserted at; for a negative value of `iIndex`, the content
       * is inserted at position 0; for a value greater than the current size of the aggregation, the content
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a customIcon into the aggregation {@link #getCustomIcons customIcons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertCustomIcon(
      /**
       * The customIcon to insert; if empty, nothing is inserted
       */
      oCustomIcon: Icon,
      /**
       * The `0`-based index the customIcon should be inserted at; for a negative value of `iIndex`, the customIcon
       * is inserted at position 0; for a value greater than the current size of the aggregation, the customIcon
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a dimensionSelector into the aggregation {@link #getDimensionSelectors dimensionSelectors}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertDimensionSelector(
      /**
       * The dimensionSelector to insert; if empty, nothing is inserted
       */
      oDimensionSelector: Control,
      /**
       * The `0`-based index the dimensionSelector should be inserted at; for a negative value of `iIndex`, the
       * dimensionSelector is inserted at position 0; for a value greater than the current size of the aggregation,
       * the dimensionSelector is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getContent content}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllContent(): ChartContainerContent[];
    /**
     * Removes all the controls from the aggregation {@link #getCustomIcons customIcons}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllCustomIcons(): Icon[];
    /**
     * Removes all the controls from the aggregation {@link #getDimensionSelectors dimensionSelectors}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllDimensionSelectors(): Control[];
    /**
     * Removes a content from the aggregation {@link #getContent content}.
     *
     *
     * @returns The removed content or `null`
     */
    removeContent(
      /**
       * The content to remove or its index or id
       */
      vContent: int | string | ChartContainerContent
    ): ChartContainerContent | null;
    /**
     * Removes a customIcon from the aggregation {@link #getCustomIcons customIcons}.
     *
     *
     * @returns The removed customIcon or `null`
     */
    removeCustomIcon(
      /**
       * The customIcon to remove or its index or id
       */
      vCustomIcon: int | string | Icon
    ): Icon | null;
    /**
     * Removes a dimensionSelector from the aggregation {@link #getDimensionSelectors dimensionSelectors}.
     *
     *
     * @returns The removed dimensionSelector or `null`
     */
    removeDimensionSelector(
      /**
       * The dimensionSelector to remove or its index or id
       */
      vDimensionSelector: int | string | Control
    ): Control | null;
    /**
     * Sets a new value for property {@link #getAutoAdjustHeight autoAdjustHeight}.
     *
     * Determine whether to stretch the chart height to the maximum possible height of ChartContainer's parent
     * container. As a prerequisite, the parent container needs to have a fixed value height or be able to determine
     * height from its parent.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAutoAdjustHeight(
      /**
       * New value for property `autoAdjustHeight`
       */
      bAutoAdjustHeight?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getEnableScroll enableScroll}.
     *
     * If set to `true`, the Container control has its own scroll bar, with the scrolling taking place within
     * the Container control itself.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEnableScroll(
      /**
       * New value for property `enableScroll`
       */
      bEnableScroll?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getFullScreen fullScreen}.
     *
     * Display the chart and the toolbar in full screen or normal mode.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFullScreen(
      /**
       * New value for property `fullScreen`
       */
      bFullScreen?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowFullScreen showFullScreen}.
     *
     * Set to true to display the full screen icon. Set to false to hide it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowFullScreen(
      /**
       * New value for property `showFullScreen`
       */
      bShowFullScreen?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowLegend showLegend}.
     *
     * Set to true to display the charts' legends. Set to false to hide them. See also showLegendButton.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowLegend(
      /**
       * New value for property `showLegend`
       */
      bShowLegend?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowLegendButton showLegendButton}.
     *
     * Set to true or false to display or hide a button for controlling the visbility of the chart's legend.
     * Please be aware that setting this property to true indirectly is setting showLegend to false. If you
     * need to hide the button but to show the legend, you need to set showLegend at a later point in time (onBeforeRendering).
     * The execution order of the combined properties is not guaranteed by the control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowLegendButton(
      /**
       * New value for property `showLegendButton`
       */
      bShowLegendButton?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowPersonalization showPersonalization}.
     *
     * Set to true to display the personalization icon. Set to false to hide it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowPersonalization(
      /**
       * New value for property `showPersonalization`
       */
      bShowPersonalization?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowSelectionDetails showSelectionDetails}.
     *
     * Set to true to display the 'Details' button that opens a popup dialog with details about the selected
     * data from the VizFrame based chart.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     * @since 1.48.0
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowSelectionDetails(
      /**
       * New value for property `showSelectionDetails`
       */
      bShowSelectionDetails?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowZoom showZoom}.
     *
     * Set to true to display zoom icons. Set to false to hide them.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowZoom(
      /**
       * New value for property `showZoom`
       */
      bShowZoom?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * String shown if there are no dimensions to display.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
    /**
     * Sets the aggregated {@link #getToolbar toolbar}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setToolbar(
      /**
       * The toolbar to set
       */
      oToolbar: OverflowToolbar
    ): this;
    /**
     * Sets the Chart Container width.
     *
     *
     * @returns this ChartContainer reference for chaining.
     */
    setWidth(
      /**
       * Width in string format.
       */
      sValue: string
    ): ChartContainer;
    /**
     * Sets a new value for property {@link #getWrapLabels wrapLabels}.
     *
     * Set to true to wrap text labels in the dialog that opens when the user clicks or taps the 'Details' button.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     * @since 1.58.0
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWrapLabels(
      /**
       * New value for property `wrapLabels`
       */
      bWrapLabels?: boolean
    ): this;
    /**
     * Switches the currently viewed content (triggers re-rendering).
     */
    switchChart(
      /**
       * The new content (Chart or Table) to be displayed
       */
      chart: Control
    ): void;
    /**
     * Updates ChartContainer and re-renders all its contents.
     *
     *
     * @returns Reference to this in order to allow method chaining
     */
    updateChartContainer(): ChartContainer;
  }
  /**
   * Describes the settings that can be provided to the ChartContainer constructor.
   */
  export interface $ChartContainerSettings extends $ControlSettings {
    /**
     * Set to true to display the personalization icon. Set to false to hide it.
     */
    showPersonalization?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Set to true to display the full screen icon. Set to false to hide it.
     */
    showFullScreen?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Display the chart and the toolbar in full screen or normal mode.
     */
    fullScreen?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Set to true to display the charts' legends. Set to false to hide them. See also showLegendButton.
     */
    showLegend?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * String shown if there are no dimensions to display.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Custom Label for Selectors Group.
     *
     * @deprecated (since 1.32.0) - Obsolete property as sap.m.Toolbar is replaced by sap.m.OverflowToolbar.
     */
    selectorGroupLabel?: string | PropertyBindingInfo;

    /**
     * Determine whether to stretch the chart height to the maximum possible height of ChartContainer's parent
     * container. As a prerequisite, the parent container needs to have a fixed value height or be able to determine
     * height from its parent.
     */
    autoAdjustHeight?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Set to true to display zoom icons. Set to false to hide them.
     */
    showZoom?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Set to true or false to display or hide a button for controlling the visbility of the chart's legend.
     * Please be aware that setting this property to true indirectly is setting showLegend to false. If you
     * need to hide the button but to show the legend, you need to set showLegend at a later point in time (onBeforeRendering).
     * The execution order of the combined properties is not guaranteed by the control.
     */
    showLegendButton?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Set to true to display the 'Details' button that opens a popup dialog with details about the selected
     * data from the VizFrame based chart.
     *
     * @since 1.48.0
     */
    showSelectionDetails?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Set to true to wrap text labels in the dialog that opens when the user clicks or taps the 'Details' button.
     *
     * @since 1.58.0
     */
    wrapLabels?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * If set to `true`, the Container control has its own scroll bar, with the scrolling taking place within
     * the Container control itself.
     */
    enableScroll?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the width of the Container.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Dimension Selects.
     */
    dimensionSelectors?:
      | Control[]
      | Control
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * ChartToolBar Content aggregation. Only sap.viz.ui5.controls.VizFrame, sap.m.Table and sap.ui.table.Table
     * can be embedded. If not specified explicitly, the rendering order of the charts is determined by the
     * sequence of contents provided by the application via this aggregation. This means, per default the first
     * chart of the aggregation will be rendered within the container.
     */
    content?:
      | ChartContainerContent[]
      | ChartContainerContent
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * Overflow ToolBar. If an external toolbar is used, it will be integrated with the embedded toolbar via
     * a placeholder. This placeholder is mandatory, and it needs to be of type 'sap.suite.ui.commons.ChartContainerToolbarPlaceholder'.
     */
    toolbar?: OverflowToolbar;

    /**
     * This aggregation contains the custom icons that should be displayed additionally on the toolbar. It is
     * not guaranteed that the same instance of the sap.ui.core.Icon control will be used within the toolbar,
     * but the toolbar will contain a sap.m.OverflowToolbarButton with an icon property equal to the src property
     * of the sap.ui.core.Icon provided in the aggregation. If a press event is triggered by the icon displayed
     * on the toolbar, then the press handler of the original sap.ui.core.Icon control is used. The instance
     * of the control, that has triggered the press event, can be accessed using the "controlReference" parameter
     * of the event object.
     */
    customIcons?: Icon[] | Icon | AggregationBindingInfo | `{${string}}`;

    /**
     * Event fired when a user clicks on the personalization icon.
     */
    personalizationPress?: (oEvent: Event) => void;

    /**
     * Event fired when a user changes the displayed content.
     */
    contentChange?: (oEvent: ChartContainer$ContentChangeEvent) => void;

    /**
     * Custom event for zoom in.
     */
    customZoomInPress?: (oEvent: Event) => void;

    /**
     * Custom event for zoom out.
     */
    customZoomOutPress?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the ChartContainer#contentChange event.
   */
  export interface ChartContainer$ContentChangeEventParameters {
    /**
     * Id of the selected item.
     */
    selectedItemId?: string;
  }

  /**
   * Event object of the ChartContainer#contentChange event.
   */
  export type ChartContainer$ContentChangeEvent = Event<
    ChartContainer$ContentChangeEventParameters,
    ChartContainer
  >;

  /**
   * Parameters of the ChartContainer#customZoomInPress event.
   */
  export interface ChartContainer$CustomZoomInPressEventParameters {}

  /**
   * Event object of the ChartContainer#customZoomInPress event.
   */
  export type ChartContainer$CustomZoomInPressEvent = Event<
    ChartContainer$CustomZoomInPressEventParameters,
    ChartContainer
  >;

  /**
   * Parameters of the ChartContainer#customZoomOutPress event.
   */
  export interface ChartContainer$CustomZoomOutPressEventParameters {}

  /**
   * Event object of the ChartContainer#customZoomOutPress event.
   */
  export type ChartContainer$CustomZoomOutPressEvent = Event<
    ChartContainer$CustomZoomOutPressEventParameters,
    ChartContainer
  >;

  /**
   * Parameters of the ChartContainer#personalizationPress event.
   */
  export interface ChartContainer$PersonalizationPressEventParameters {}

  /**
   * Event object of the ChartContainer#personalizationPress event.
   */
  export type ChartContainer$PersonalizationPressEvent = Event<
    ChartContainer$PersonalizationPressEventParameters,
    ChartContainer
  >;
}

declare module "sap/suite/ui/commons/ChartContainerContent" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import SelectionDetailsFacade from "sap/m/SelectionDetailsFacade";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Content aggregation for ChartContainer.
   */
  export default class ChartContainerContent extends Control {
    /**
     * Constructor for a new ChartContainerContent.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ChartContainerContentSettings
    );
    /**
     * Constructor for a new ChartContainerContent.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ChartContainerContentSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ChartContainerContent with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ChartContainerContent>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ChartContainerContent.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Destroys the content in the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyContent(): this;
    /**
     * Gets content of aggregation {@link #getContent content}.
     *
     * Chart or Table as content of the ChartToolBar. Supported Types: sap.viz.ui5.controls.VizFrame, sap.m.Table,
     * sap.ui.table.Table
     */
    getContent(): Control;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * Icon of the Chart.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): string;
    /**
     * Getter for the Selection Details facade.
     *
     * @since 1.48
     *
     * @returns The reduced Selection Details facade for outer framework usages.
     */
    getSelectionDetails(): SelectionDetailsFacade;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * Title of the Chart/Table
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets the aggregated {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setContent(
      /**
       * The content to set
       */
      oContent: Control
    ): this;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * Icon of the Chart.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: string
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * Title of the Chart/Table
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ChartContainerContent constructor.
   */
  export interface $ChartContainerContentSettings extends $ControlSettings {
    /**
     * Icon of the Chart.
     */
    icon?: string | PropertyBindingInfo;

    /**
     * Title of the Chart/Table
     */
    title?: string | PropertyBindingInfo;

    /**
     * Chart or Table as content of the ChartToolBar. Supported Types: sap.viz.ui5.controls.VizFrame, sap.m.Table,
     * sap.ui.table.Table
     */
    content?: Control;
  }
}

declare module "sap/suite/ui/commons/ChartContainerToolbarPlaceholder" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * Provides a placeholder for the embedded chart container toolbar.
   */
  export default class ChartContainerToolbarPlaceholder extends Control {
    /**
     * Constructor for a new ChartContainerToolbarPlaceholder.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.ui.core.Control#constructor sap.ui.core.Control }
     * can be used.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $ChartContainerToolbarPlaceholderSettings
    );
    /**
     * Constructor for a new ChartContainerToolbarPlaceholder.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.ui.core.Control#constructor sap.ui.core.Control }
     * can be used.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $ChartContainerToolbarPlaceholderSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ChartContainerToolbarPlaceholder with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ChartContainerToolbarPlaceholder>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ChartContainerToolbarPlaceholder.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the ChartContainerToolbarPlaceholder constructor.
   */
  export interface $ChartContainerToolbarPlaceholderSettings
    extends $ControlSettings {}
}

declare module "sap/suite/ui/commons/ChartTile" {
  import {
    default as InfoTile,
    $InfoTileSettings,
  } from "sap/suite/ui/commons/InfoTile";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control is the implementation of the InfoTile to show a comparison or bullet chart.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export default class ChartTile extends InfoTile {
    /**
     * Constructor for a new ChartTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $ChartTileSettings
    );
    /**
     * Constructor for a new ChartTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $ChartTileSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ChartTile with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.InfoTile.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ChartTile>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ChartTile.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getUnit unit}.
     *
     * The percent sign, the currency symbol, or the unit of measure.
     *
     *
     * @returns Value of property `unit`
     */
    getUnit(): string;
    /**
     * Sets a new value for property {@link #getUnit unit}.
     *
     * The percent sign, the currency symbol, or the unit of measure.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUnit(
      /**
       * New value for property `unit`
       */
      sUnit?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ChartTile constructor.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export interface $ChartTileSettings extends $InfoTileSettings {
    /**
     * The percent sign, the currency symbol, or the unit of measure.
     */
    unit?: string | PropertyBindingInfo;
  }
}

declare module "sap/suite/ui/commons/CloudFilePicker" {
  import { default as Dialog, $DialogSettings } from "sap/m/Dialog";

  import Event from "sap/ui/base/Event";

  import {
    FilePickerModes,
    FilePickerType,
    SelectionModes,
  } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { URI } from "sap/ui/core/library";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * @since 1.101
   * @experimental
   */
  export default class CloudFilePicker extends Dialog {
    /**
     * Constructor of the CloudFilePicker
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.CloudFilePicker with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.m.Dialog.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CloudFilePicker>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.CloudFilePicker.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:cancel cancel} event of this `sap.suite.ui.commons.CloudFilePicker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CloudFilePicker` itself.
     *
     * Event is fired when the cancel button is pressed
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCancel(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CloudFilePicker`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:cancel cancel} event of this `sap.suite.ui.commons.CloudFilePicker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CloudFilePicker` itself.
     *
     * Event is fired when the cancel button is pressed
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCancel(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CloudFilePicker`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:select select} event of this `sap.suite.ui.commons.CloudFilePicker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CloudFilePicker` itself.
     *
     * Event is fired when the selection is made
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSelect(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: CloudFilePicker$SelectEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CloudFilePicker`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:select select} event of this `sap.suite.ui.commons.CloudFilePicker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.CloudFilePicker` itself.
     *
     * Event is fired when the selection is made
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSelect(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: CloudFilePicker$SelectEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.CloudFilePicker`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:cancel cancel} event of this `sap.suite.ui.commons.CloudFilePicker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachCancel(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:select select} event of this `sap.suite.ui.commons.CloudFilePicker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachSelect(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: CloudFilePicker$SelectEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:cancel cancel} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireCancel(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:select select} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireSelect(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: CloudFilePicker$SelectEventParameters
    ): this;
    /**
     * Gets current value of property {@link #getConfirmButtonText confirmButtonText}.
     *
     * Overwrites the default text for the confirmation button.
     *
     * Default value is `...see text or source`.
     *
     *
     * @returns Value of property `confirmButtonText`
     */
    getConfirmButtonText(): string;
    /**
     * Gets current value of property {@link #getDuplicateMessage duplicateMessage}.
     *
     * Overwrites the default text for the duplicate message popup. It is relevant only if "enableDuplicateCheck"
     * is set to true.
     *
     *
     * @returns Value of property `duplicateMessage`
     */
    getDuplicateMessage(): string;
    /**
     * Gets current value of property {@link #getEnableDuplicateCheck enableDuplicateCheck}.
     *
     * Specifies whether duplicate file check logic is needed.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `enableDuplicateCheck`
     */
    getEnableDuplicateCheck(): boolean;
    /**
     * Gets current value of property {@link #getFileNameMandatory fileNameMandatory}.
     *
     * Specifies whether file name is mandatory to perform confirmation action
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `fileNameMandatory`
     */
    getFileNameMandatory(): boolean;
    /**
     * Gets current value of property {@link #getFilePickerMode filePickerMode}.
     *
     * Allow the type of resources that can be selected.
     *
     * Default value is `"All"`.
     *
     *
     * @returns Value of property `filePickerMode`
     */
    getFilePickerMode(): FilePickerModes | keyof typeof FilePickerModes;
    /**
     * Gets current value of property {@link #getFilePickerType filePickerType}.
     *
     * Specifies the mode in which the control is running
     *
     * Default value is `"Export"`.
     *
     *
     * @returns Value of property `filePickerType`
     */
    getFilePickerType(): FilePickerType | keyof typeof FilePickerType;
    /**
     * Gets current value of property {@link #getMIMEType MIMEType}.
     *
     * Specifies the MIME type of the file during Export
     *
     *
     * @returns Value of property `MIMEType`
     */
    getMIMEType(): string;
    /**
     * Gets current value of property {@link #getSelectionMode selectionMode}.
     *
     * Specifies the Selection mode of the table
     *
     * Default value is `"SingleSelectMaster"`.
     *
     *
     * @returns Value of property `selectionMode`
     */
    getSelectionMode(): SelectionModes;
    /**
     * Gets current value of property {@link #getServiceUrl serviceUrl}.
     *
     * Url of the FileShare OData V4 service.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `serviceUrl`
     */
    getServiceUrl(): URI;
    /**
     * Gets current value of property {@link #getSharedModel sharedModel}.
     *
     * Model shared from export as dialog
     *
     *
     * @returns Value of property `sharedModel`
     */
    getSharedModel(): object;
    /**
     * Gets current value of property {@link #getSuggestedFileName suggestedFileName}.
     *
     * File name could be provided in case File picker control is used for Export/Save As scenario for selecting
     * the location. Value will be displayed in the File Name control on the dialog.
     *
     *
     * @returns Value of property `suggestedFileName`
     */
    getSuggestedFileName(): string;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * Specifies the text for selectButton.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets a new value for property {@link #getConfirmButtonText confirmButtonText}.
     *
     * Overwrites the default text for the confirmation button.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `...see text or source`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setConfirmButtonText(
      /**
       * New value for property `confirmButtonText`
       */
      sConfirmButtonText?: string
    ): this;
    /**
     * Sets a new value for property {@link #getDuplicateMessage duplicateMessage}.
     *
     * Overwrites the default text for the duplicate message popup. It is relevant only if "enableDuplicateCheck"
     * is set to true.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDuplicateMessage(
      /**
       * New value for property `duplicateMessage`
       */
      sDuplicateMessage: string
    ): this;
    /**
     * Sets a new value for property {@link #getEnableDuplicateCheck enableDuplicateCheck}.
     *
     * Specifies whether duplicate file check logic is needed.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEnableDuplicateCheck(
      /**
       * New value for property `enableDuplicateCheck`
       */
      bEnableDuplicateCheck?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getFileNameMandatory fileNameMandatory}.
     *
     * Specifies whether file name is mandatory to perform confirmation action
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFileNameMandatory(
      /**
       * New value for property `fileNameMandatory`
       */
      bFileNameMandatory?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getFilePickerMode filePickerMode}.
     *
     * Allow the type of resources that can be selected.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"All"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFilePickerMode(
      /**
       * New value for property `filePickerMode`
       */
      sFilePickerMode?: FilePickerModes | keyof typeof FilePickerModes
    ): this;
    /**
     * Sets a new value for property {@link #getFilePickerType filePickerType}.
     *
     * Specifies the mode in which the control is running
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Export"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFilePickerType(
      /**
       * New value for property `filePickerType`
       */
      sFilePickerType?: FilePickerType | keyof typeof FilePickerType
    ): this;
    /**
     * Sets a new value for property {@link #getMIMEType MIMEType}.
     *
     * Specifies the MIME type of the file during Export
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMIMEType(
      /**
       * New value for property `MIMEType`
       */
      sMIMEType: string
    ): this;
    /**
     * Sets a new value for property {@link #getSelectionMode selectionMode}.
     *
     * Specifies the Selection mode of the table
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"SingleSelectMaster"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSelectionMode(
      /**
       * New value for property `selectionMode`
       */
      sSelectionMode?: SelectionModes
    ): this;
    /**
     * Sets a new value for property {@link #getServiceUrl serviceUrl}.
     *
     * Url of the FileShare OData V4 service.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setServiceUrl(
      /**
       * New value for property `serviceUrl`
       */
      sServiceUrl?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getSharedModel sharedModel}.
     *
     * Model shared from export as dialog
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSharedModel(
      /**
       * New value for property `sharedModel`
       */
      oSharedModel?: object
    ): this;
    /**
     * Sets a new value for property {@link #getSuggestedFileName suggestedFileName}.
     *
     * File name could be provided in case File picker control is used for Export/Save As scenario for selecting
     * the location. Value will be displayed in the File Name control on the dialog.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSuggestedFileName(
      /**
       * New value for property `suggestedFileName`
       */
      sSuggestedFileName: string
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * Specifies the text for selectButton.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the CloudFilePicker constructor.
   *
   * @experimental
   */
  export interface $CloudFilePickerSettings extends $DialogSettings {
    /**
     * Url of the FileShare OData V4 service.
     */
    serviceUrl?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * Model shared from export as dialog
     */
    sharedModel?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Overwrites the default text for the confirmation button.
     */
    confirmButtonText?: string | PropertyBindingInfo;

    /**
     * Allow the type of resources that can be selected.
     */
    filePickerMode?:
      | (FilePickerModes | keyof typeof FilePickerModes)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Specifies the text for selectButton.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Specifies whether duplicate file check logic is needed.
     */
    enableDuplicateCheck?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Overwrites the default text for the duplicate message popup. It is relevant only if "enableDuplicateCheck"
     * is set to true.
     */
    duplicateMessage?: string | PropertyBindingInfo;

    /**
     * File name could be provided in case File picker control is used for Export/Save As scenario for selecting
     * the location. Value will be displayed in the File Name control on the dialog.
     */
    suggestedFileName?: string | PropertyBindingInfo;

    /**
     * Specifies whether file name is mandatory to perform confirmation action
     */
    fileNameMandatory?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Specifies the mode in which the control is running
     */
    filePickerType?:
      | (FilePickerType | keyof typeof FilePickerType)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Specifies the MIME type of the file during Export
     */
    MIMEType?: string | PropertyBindingInfo;

    /**
     * Specifies the Selection mode of the table
     */
    selectionMode?: SelectionModes | PropertyBindingInfo | `{${string}}`;

    /**
     * Event is fired when the selection is made
     */
    select?: (oEvent: CloudFilePicker$SelectEvent) => void;

    /**
     * Event is fired when the cancel button is pressed
     */
    cancel?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the CloudFilePicker#cancel event.
   */
  export interface CloudFilePicker$CancelEventParameters {}

  /**
   * Event object of the CloudFilePicker#cancel event.
   */
  export type CloudFilePicker$CancelEvent = Event<
    CloudFilePicker$CancelEventParameters,
    CloudFilePicker
  >;

  /**
   * Parameters of the CloudFilePicker#select event.
   */
  export interface CloudFilePicker$SelectEventParameters {
    /**
     * Specifies whether an existing file is being overwritten in a file share.
     */
    replaceExistingFile?: boolean;

    /**
     * Specifies the name of the selected file.
     */
    selectedFileName?: string;

    /**
     * Specifies the details of the seleced file.
     */
    selectedFiles?: /* was: sap.suite.ui.commons.CloudFileInfo */ any[];

    /**
     * Specifies the details of the folder of a selected file.
     */
    selectedFolder?: /* was: sap.suite.ui.commons.CloudFileInfo */ any;
  }

  /**
   * Event object of the CloudFilePicker#select event.
   */
  export type CloudFilePicker$SelectEvent = Event<
    CloudFilePicker$SelectEventParameters,
    CloudFilePicker
  >;
}

declare module "sap/suite/ui/commons/collaboration/ServiceContainer" {
  /**
   * Service container to get the collaboration type.
   *
   * @since 1.108
   */
  interface ServiceContainer {
    /**
     * Method that returns the collaboration service object as 'active' on the system. Microsoft Teams is supported
     * as a collaboration option and must be enabled using the communication service SAP_COM_0860. The type
     * definition and class are only available internally and are not intended for external consumers.
     *
     *
     * @returns Returns the promise that is resolved to the instance of the collaboration service.
     */
    getServiceAsync(): Promise</* was: sap.suite.ui.commons.collaboration.BaseHelperService */ any>;
  }
  const ServiceContainer: ServiceContainer;
  export default ServiceContainer;
}

declare module "sap/suite/ui/commons/ColumnData" {
  import {
    default as ColumnMicroChartData,
    $ColumnMicroChartDataSettings,
  } from "sap/suite/ui/microchart/ColumnMicroChartData";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * Column data holder.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ColumnMicroChartData.
   */
  export default class ColumnData extends ColumnMicroChartData {
    /**
     * Constructor for a new ColumnData.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ColumnMicroChartData#constructor sap.suite.ui.microchart.ColumnMicroChartData }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ColumnDataSettings
    );
    /**
     * Constructor for a new ColumnData.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ColumnMicroChartData#constructor sap.suite.ui.microchart.ColumnMicroChartData }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ColumnDataSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ColumnData with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.ColumnMicroChartData.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ColumnData>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ColumnData.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the ColumnData constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ColumnMicroChartData.
   */
  export interface $ColumnDataSettings extends $ColumnMicroChartDataSettings {}
}

declare module "sap/suite/ui/commons/ColumnMicroChart" {
  import {
    default as ColumnMicroChart1,
    $ColumnMicroChartSettings as $ColumnMicroChartSettings1,
  } from "sap/suite/ui/microchart/ColumnMicroChart";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This control shows a column chart.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ColumnMicroChart.
   */
  export default class ColumnMicroChart extends ColumnMicroChart1 {
    /**
     * Constructor for a new ColumnMicroChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ColumnMicroChart#constructor sap.suite.ui.microchart.ColumnMicroChart }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ColumnMicroChartSettings
    );
    /**
     * Constructor for a new ColumnMicroChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ColumnMicroChart#constructor sap.suite.ui.microchart.ColumnMicroChart }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ColumnMicroChartSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ColumnMicroChart with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.ColumnMicroChart.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ColumnMicroChart>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ColumnMicroChart.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the ColumnMicroChart constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ColumnMicroChart.
   */
  export interface $ColumnMicroChartSettings
    extends $ColumnMicroChartSettings1 {}
}

declare module "sap/suite/ui/commons/ColumnMicroChartLabel" {
  import {
    default as ColumnMicroChartLabel1,
    $ColumnMicroChartLabelSettings as $ColumnMicroChartLabelSettings1,
  } from "sap/suite/ui/microchart/ColumnMicroChartLabel";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This element contains data for a label in ColumnMicroChart control.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ColumnMicroChartLabel.
   */
  export default class ColumnMicroChartLabel extends ColumnMicroChartLabel1 {
    /**
     * Constructor for a new ColumnMicroChartLabel.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ColumnMicroChartLabel#constructor sap.suite.ui.microchart.ColumnMicroChartLabel }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ColumnMicroChartLabelSettings
    );
    /**
     * Constructor for a new ColumnMicroChartLabel.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ColumnMicroChartLabel#constructor sap.suite.ui.microchart.ColumnMicroChartLabel }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ColumnMicroChartLabelSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ColumnMicroChartLabel with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.ColumnMicroChartLabel.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ColumnMicroChartLabel>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ColumnMicroChartLabel.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the ColumnMicroChartLabel constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ColumnMicroChartLabel.
   */
  export interface $ColumnMicroChartLabelSettings
    extends $ColumnMicroChartLabelSettings1 {}
}

declare module "sap/suite/ui/commons/ComparisonChart" {
  import {
    default as ComparisonMicroChart,
    $ComparisonMicroChartSettings,
  } from "sap/suite/ui/microchart/ComparisonMicroChart";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This control shows a comparison chart.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ComparisonMicroChart.
   */
  export default class ComparisonChart extends ComparisonMicroChart {
    /**
     * Constructor for a new ComparisonChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ComparisonMicroChart#constructor sap.suite.ui.microchart.ComparisonMicroChart }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ComparisonChartSettings
    );
    /**
     * Constructor for a new ComparisonChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ComparisonMicroChart#constructor sap.suite.ui.microchart.ComparisonMicroChart }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ComparisonChartSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ComparisonChart with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.ComparisonMicroChart.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ComparisonChart>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ComparisonChart.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the ComparisonChart constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ComparisonMicroChart.
   */
  export interface $ComparisonChartSettings
    extends $ComparisonMicroChartSettings {}
}

declare module "sap/suite/ui/commons/ComparisonData" {
  import {
    default as ComparisonMicroChartData,
    $ComparisonMicroChartDataSettings,
  } from "sap/suite/ui/microchart/ComparisonMicroChartData";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * Comparison tile value holder.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ComparisonMicroChartData.
   */
  export default class ComparisonData extends ComparisonMicroChartData {
    /**
     * Constructor for a new ComparisonData.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ComparisonMicroChartData#constructor sap.suite.ui.microchart.ComparisonMicroChartData }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ComparisonDataSettings
    );
    /**
     * Constructor for a new ComparisonData.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.ComparisonMicroChartData#constructor sap.suite.ui.microchart.ComparisonMicroChartData }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ComparisonDataSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ComparisonData with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.ComparisonMicroChartData.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ComparisonData>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ComparisonData.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the ComparisonData constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.ComparisonMicroChartData.
   */
  export interface $ComparisonDataSettings
    extends $ComparisonMicroChartDataSettings {}
}

declare module "sap/suite/ui/commons/DateRangeScroller" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import { ID } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import DateFormat from "sap/ui/core/format/DateFormat";

  import Event from "sap/ui/base/Event";

  /**
   * The DateRangeScroller provides a method to scroll through a series of time periods, each of which is
   * represented by a starting date and an ending date, known as the date range. The user may scroll to the
   * previous or next date range. Several predefined ranges are supported such as day, week, work week, month,
   * and year.
   *
   * @deprecated (since 1.34) - Deprecated. Not Fiori.
   */
  export default class DateRangeScroller extends Control {
    /**
     * Constructor for a new DateRangeScroller.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $DateRangeScrollerSettings
    );
    /**
     * Constructor for a new DateRangeScroller.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $DateRangeScrollerSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.DateRangeScroller with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DateRangeScroller>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.DateRangeScroller.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some ariaDescribedBy into the association {@link #getAriaDescribedBy ariaDescribedBy}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addAriaDescribedBy(
      /**
       * The ariaDescribedBy to add; if empty, nothing is inserted
       */
      vAriaDescribedBy: ID | Control
    ): this;
    /**
     * Adds some ariaLabelledBy into the association {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addAriaLabelledBy(
      /**
       * The ariaLabelledBy to add; if empty, nothing is inserted
       */
      vAriaLabelledBy: ID | Control
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeScroller`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeScroller` itself.
     *
     * This event is fired whenever the date range is changed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DateRangeScroller$ChangeEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeScroller`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeScroller`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeScroller` itself.
     *
     * This event is fired whenever the date range is changed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: DateRangeScroller$ChangeEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeScroller`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Decrement the date range by a time period increment according to the the date range type and fire the
     * dateChange event.
     *
     *
     * @returns `this` to allow method chaining
     */
    decrementDateRange(): DateRangeScroller;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeScroller`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: DateRangeScroller$ChangeEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:change change} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: DateRangeScroller$ChangeEventParameters
    ): this;
    /**
     * Returns array of IDs of the elements which are the current targets of the association {@link #getAriaDescribedBy ariaDescribedBy}.
     */
    getAriaDescribedBy(): ID[];
    /**
     * Returns array of IDs of the elements which are the current targets of the association {@link #getAriaLabelledBy ariaLabelledBy}.
     */
    getAriaLabelledBy(): ID[];
    /**
     * Returns an object containing startDate and endDate set to the current start and end Date objects.
     *
     *
     * @returns The date range containing start and end Date.
     */
    getDateRange(): Object;
    /**
     * Increment the date range by a time period increment according to the the date range type and fire the
     * dateChange event.
     *
     *
     * @returns `this` to allow method chaining.
     */
    incrementDateRange(): DateRangeScroller;
    /**
     * Removes all the controls in the association named {@link #getAriaDescribedBy ariaDescribedBy}.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllAriaDescribedBy(): ID[];
    /**
     * Removes all the controls in the association named {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllAriaLabelledBy(): ID[];
    /**
     * Removes an ariaDescribedBy from the association named {@link #getAriaDescribedBy ariaDescribedBy}.
     *
     *
     * @returns The removed ariaDescribedBy or `null`
     */
    removeAriaDescribedBy(
      /**
       * The ariaDescribedBy to be removed or its index or ID
       */
      vAriaDescribedBy: int | ID | Control
    ): ID | null;
    /**
     * Removes an ariaLabelledBy from the association named {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     *
     * @returns The removed ariaLabelledBy or `null`
     */
    removeAriaLabelledBy(
      /**
       * The ariaLabelledBy to be removed or its index or ID
       */
      vAriaLabelledBy: int | ID | Control
    ): ID | null;
    /**
     * Setter for dateFormat which is used for formating the dates If passed object is null or is of incorrect
     * type, control's default formatting will be used.
     */
    setDateFormat(
      /**
       * Date format which should be used
       */
      oDateFormat: DateFormat
    ): void;
    /**
     * Set a custom date range beginning with the specified start date that increments/decrements `iDuration`
     * days at a time.
     *
     *
     * @returns `this` to allow method chaining.
     */
    setDateRangeCustom(
      /**
       * The initial date for the custom range.
       */
      dInitialDate: Date,
      /**
       * The number of days in the custom range, including the start date.
       */
      iDuration?: int
    ): DateRangeScroller;
    /**
     * Set a date range equal to a single day, starting with the given initial date.
     *
     *
     * @returns `this` to allow method chaining.
     */
    setDateRangeDay(
      /**
       * The initial date.
       */
      dInitialDate: Date
    ): DateRangeScroller;
    /**
     * Set a date range that increments/decrements one calendar month at a time. The month date range begins
     * on the first day of the month (beginning of the day) and ends on the last day of the month (end of the
     * day).
     *
     *
     * @returns `this` to allow method chaining.
     */
    setDateRangeMonth(
      /**
       * Any date that falls within the desired month.
       */
      dInitialDate: Date
    ): DateRangeScroller;
    /**
     * Set a date range that steps one week at a time. The starting date is set according to the first day of
     * the week (default is Monday) and the initial date. For example, if the first day of the week is set to
     * Tuesday and the initial date is Wednesday, January 9, 2013, then the starting date will be Tuesday, January
     * 8, 2013.
     *
     * The last day of the range is calculated by adding (duration - 1) days to the starting date.
     *
     *
     * @returns `this` to allow method chaining.
     */
    setDateRangeWeek(
      /**
       * Any date that falls on the desired week. Start/end dates will be adjusted according to the first day
       * of the week.
       */
      dInitialDate: Date,
      /**
       * Settings for the object
       */
      oSettings?: {
        /**
         * The number of days in the range with a minimum of 1 and maximum of 7.
         */
        duration?: int;
        /**
         * The starting day for the range. Valid values are 0-6, with 0=Sunday and 6=Saturday.
         */
        iFirstDayOfWeek?: int;
      }
    ): DateRangeScroller;
    /**
     * Set a date range that increments/decrements one calendar year at a time. The year date range begins on
     * the first day of the year (beginning of the day) and ends on the last day of the year (end of the day).
     *
     *
     * @returns `this` to allow method chaining.
     */
    setDateRangeYear(
      /**
       * Any date that falls within the desired year.
       */
      dInitialDate: Date
    ): DateRangeScroller;
  }
  /**
   * Describes the settings that can be provided to the DateRangeScroller constructor.
   *
   * @deprecated (since 1.34) - Deprecated. Not Fiori.
   */
  export interface $DateRangeScrollerSettings extends $ControlSettings {
    /**
     * Association to controls / ids which describe this control (see WAI-ARIA attribute aria-describedby).
     */
    ariaDescribedBy?: Array<Control | string>;

    /**
     * Association to controls / ids which label this control (see WAI-ARIA attribute aria-labelledby).
     */
    ariaLabelledBy?: Array<Control | string>;

    /**
     * This event is fired whenever the date range is changed.
     */
    change?: (oEvent: DateRangeScroller$ChangeEvent) => void;
  }

  /**
   * Parameters of the DateRangeScroller#change event.
   */
  export interface DateRangeScroller$ChangeEventParameters {
    /**
     * Object containing startDate and endDate Date properties signifying the start and ending dates of the
     * currently selected range.
     */
    dateRange?: any;
  }

  /**
   * Event object of the DateRangeScroller#change event.
   */
  export type DateRangeScroller$ChangeEvent = Event<
    DateRangeScroller$ChangeEventParameters,
    DateRangeScroller
  >;
}

declare module "sap/suite/ui/commons/DateRangeSlider" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { CSSSize } from "sap/ui/core/library";

  import DateFormat from "sap/ui/core/format/DateFormat";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * The Date Range Slider provides the user with a Range Slider control that is optimized for use with Dates.
   *
   * @deprecated (since 1.34) - Deprecated. Not Fiori.
   */
  export default class DateRangeSlider extends Control {
    /**
     * Constructor for a new DateRangeSlider.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $DateRangeSliderSettings
    );
    /**
     * Constructor for a new DateRangeSlider.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $DateRangeSliderSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.DateRangeSlider with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DateRangeSlider>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.DateRangeSlider.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeSlider`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeSlider` itself.
     *
     * This event is fired when user changes completes the selection of a new date using a grip. It contains
     * the Date object value of each grip such that the left grip value is held by the value property and the
     * right grip value is held by the value2 property.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeSlider`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeSlider`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeSlider` itself.
     *
     * This event is fired when user changes completes the selection of a new date using a grip. It contains
     * the Date object value of each grip such that the left grip value is held by the value property and the
     * right grip value is held by the value2 property.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeSlider`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:liveChange liveChange} event of this `sap.suite.ui.commons.DateRangeSlider`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeSlider` itself.
     *
     * This live event is fired as the user slides a grip with the mouse. It contains the Date object value
     * of each grip such that the left grip value is held by the value property and the right grip value is
     * held by the value2 property.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachLiveChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeSlider`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:liveChange liveChange} event of this `sap.suite.ui.commons.DateRangeSlider`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeSlider` itself.
     *
     * This live event is fired as the user slides a grip with the mouse. It contains the Date object value
     * of each grip such that the left grip value is held by the value property and the right grip value is
     * held by the value2 property.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachLiveChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeSlider`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeSlider`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:liveChange liveChange} event of this `sap.suite.ui.commons.DateRangeSlider`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachLiveChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:change change} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:liveChange liveChange} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireLiveChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getEditable editable}.
     *
     * This property switches the enabled state of the control. Using the date range slider interactively requires
     * this property to be true.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `editable`
     */
    getEditable(): boolean;
    /**
     * Gets current value of property {@link #getEnabled enabled}.
     *
     * This property switches the enabled state of the control. Disabled fields have different colors, and can
     * not be focused.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `enabled`
     */
    getEnabled(): boolean;
    /**
     * Gets current value of property {@link #getLabels labels}.
     *
     * Labels to be displayed instead of dates.
     *
     *
     * @returns Value of property `labels`
     */
    getLabels(): string[];
    /**
     * Gets current value of property {@link #getMax max}.
     *
     * Maximum date for the slider.
     *
     *
     * @returns Value of property `max`
     */
    getMax(): object;
    /**
     * Gets current value of property {@link #getMin min}.
     *
     * Minimum date for the slider.
     *
     *
     * @returns Value of property `min`
     */
    getMin(): object;
    /**
     * Gets current value of property {@link #getPinGrip pinGrip}.
     *
     * Pin the left grip so that user cannot move it with the keyboard or mouse.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `pinGrip`
     */
    getPinGrip(): boolean;
    /**
     * Gets current value of property {@link #getPinGrip2 pinGrip2}.
     *
     * Pin the right grip so that user cannot move it with the keyboard or mouse.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `pinGrip2`
     */
    getPinGrip2(): boolean;
    /**
     * Gets current value of property {@link #getShowBubbles showBubbles}.
     *
     * Property to show or hide bubbles. Default is true.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showBubbles`
     */
    getShowBubbles(): boolean;
    /**
     * Gets current value of property {@link #getSmallStepWidth smallStepWidth}.
     *
     * The grips of the control can only be moved in steps of this width.
     *
     *
     * @returns Value of property `smallStepWidth`
     */
    getSmallStepWidth(): float;
    /**
     * Gets current value of property {@link #getStepLabels stepLabels}.
     *
     * Display a date label above each tick.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `stepLabels`
     */
    getStepLabels(): boolean;
    /**
     * Gets current value of property {@link #getTotalUnits totalUnits}.
     *
     * Number of units between ticks.
     *
     *
     * @returns Value of property `totalUnits`
     */
    getTotalUnits(): int;
    /**
     * Gets current value of property {@link #getValue value}.
     *
     * Date value of the left grip.
     *
     *
     * @returns Value of property `value`
     */
    getValue(): object;
    /**
     * Gets current value of property {@link #getValue2 value2}.
     *
     * Date value of the right grip.
     *
     *
     * @returns Value of property `value2`
     */
    getValue2(): object;
    /**
     * Gets current value of property {@link #getVisible visible}.
     *
     * This property switches the visible state of the control. Invisible date range slider are not rendered.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `visible`
     */
    getVisible(): boolean;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Width of the horizontal date range slider.
     *
     * Default value is `'100%'`.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Setter for dateFormat which is used to format the dates for Labels, bubble texts, and tool tips. If passed
     * object is null or is of incorrect type, _oDateRangeSliderInternal's default formatting will be used.
     */
    setDateFormat(
      /**
       * Date format which is used to format the dates.
       */
      oDateFormat: DateFormat
    ): void;
    /**
     * Set Date Range Slider Granularity to Day
     *
     *
     * @returns `this` to allow method chaining.
     */
    setDayGranularity(): DateRangeSlider;
    /**
     * Sets a new value for property {@link #getEditable editable}.
     *
     * This property switches the enabled state of the control. Using the date range slider interactively requires
     * this property to be true.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEditable(
      /**
       * New value for property `editable`
       */
      bEditable?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getEnabled enabled}.
     *
     * This property switches the enabled state of the control. Disabled fields have different colors, and can
     * not be focused.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEnabled(
      /**
       * New value for property `enabled`
       */
      bEnabled?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getLabels labels}.
     *
     * Labels to be displayed instead of dates.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLabels(
      /**
       * New value for property `labels`
       */
      sLabels?: string[]
    ): this;
    /**
     * Set max property for the DateRangeSlider. This sets the maximum date for the slider.
     */
    setMax(
      /**
       * The max date.
       */
      dMax: Date
    ): void;
    /**
     * Set min property for the DateRangeSlider. This sets the minimum date for the slider.
     */
    setMin(
      /**
       * The min date.
       */
      dMin: Date
    ): void;
    /**
     * Set Date Range Slider Granularity to Month
     *
     *
     * @returns `this` to allow method chaining.
     */
    setMonthGranularity(): DateRangeSlider;
    /**
     * Sets a new value for property {@link #getPinGrip pinGrip}.
     *
     * Pin the left grip so that user cannot move it with the keyboard or mouse.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPinGrip(
      /**
       * New value for property `pinGrip`
       */
      bPinGrip?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getPinGrip2 pinGrip2}.
     *
     * Pin the right grip so that user cannot move it with the keyboard or mouse.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPinGrip2(
      /**
       * New value for property `pinGrip2`
       */
      bPinGrip2?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowBubbles showBubbles}.
     *
     * Property to show or hide bubbles. Default is true.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowBubbles(
      /**
       * New value for property `showBubbles`
       */
      bShowBubbles?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getSmallStepWidth smallStepWidth}.
     *
     * The grips of the control can only be moved in steps of this width.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSmallStepWidth(
      /**
       * New value for property `smallStepWidth`
       */
      fSmallStepWidth?: float
    ): this;
    /**
     * Sets a new value for property {@link #getStepLabels stepLabels}.
     *
     * Display a date label above each tick.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setStepLabels(
      /**
       * New value for property `stepLabels`
       */
      bStepLabels?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getTotalUnits totalUnits}.
     *
     * Number of units between ticks.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTotalUnits(
      /**
       * New value for property `totalUnits`
       */
      iTotalUnits?: int
    ): this;
    /**
     * Set value property for the DateRangeSlider. This sets the value date for the slider.
     */
    setValue(
      /**
       * The value date.
       */
      dValue: Date
    ): void;
    /**
     * Sets a new value for property {@link #getValue2 value2}.
     *
     * Date value of the right grip.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValue2(
      /**
       * New value for property `value2`
       */
      oValue2?: object
    ): this;
    /**
     * Sets a new value for property {@link #getVisible visible}.
     *
     * This property switches the visible state of the control. Invisible date range slider are not rendered.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setVisible(
      /**
       * New value for property `visible`
       */
      bVisible?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Width of the horizontal date range slider.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'100%'`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
  }
  /**
   * Describes the settings that can be provided to the DateRangeSlider constructor.
   *
   * @deprecated (since 1.34) - Deprecated. Not Fiori.
   */
  export interface $DateRangeSliderSettings extends $ControlSettings {
    /**
     * Width of the horizontal date range slider.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * This property switches the enabled state of the control. Disabled fields have different colors, and can
     * not be focused.
     */
    enabled?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * This property switches the enabled state of the control. Using the date range slider interactively requires
     * this property to be true.
     */
    editable?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * This property switches the visible state of the control. Invisible date range slider are not rendered.
     */
    visible?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Property to show or hide bubbles. Default is true.
     */
    showBubbles?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The grips of the control can only be moved in steps of this width.
     */
    smallStepWidth?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Number of units between ticks.
     */
    totalUnits?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Display a date label above each tick.
     */
    stepLabels?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Labels to be displayed instead of dates.
     */
    labels?: string[] | PropertyBindingInfo | `{${string}}`;

    /**
     * Minimum date for the slider.
     */
    min?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Maximum date for the slider.
     */
    max?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Date value of the left grip.
     */
    value?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Date value of the right grip.
     */
    value2?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Pin the left grip so that user cannot move it with the keyboard or mouse.
     */
    pinGrip?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Pin the right grip so that user cannot move it with the keyboard or mouse.
     */
    pinGrip2?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * This event is fired when user changes completes the selection of a new date using a grip. It contains
     * the Date object value of each grip such that the left grip value is held by the value property and the
     * right grip value is held by the value2 property.
     */
    change?: (oEvent: Event) => void;

    /**
     * This live event is fired as the user slides a grip with the mouse. It contains the Date object value
     * of each grip such that the left grip value is held by the value property and the right grip value is
     * held by the value2 property.
     */
    liveChange?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the DateRangeSlider#change event.
   */
  export interface DateRangeSlider$ChangeEventParameters {}

  /**
   * Event object of the DateRangeSlider#change event.
   */
  export type DateRangeSlider$ChangeEvent = Event<
    DateRangeSlider$ChangeEventParameters,
    DateRangeSlider
  >;

  /**
   * Parameters of the DateRangeSlider#liveChange event.
   */
  export interface DateRangeSlider$LiveChangeEventParameters {}

  /**
   * Event object of the DateRangeSlider#liveChange event.
   */
  export type DateRangeSlider$LiveChangeEvent = Event<
    DateRangeSlider$LiveChangeEventParameters,
    DateRangeSlider
  >;
}

declare module "sap/suite/ui/commons/DateRangeSliderInternal" {
  import {
    default as RangeSlider,
    $RangeSliderSettings,
  } from "sap/ui/commons/RangeSlider";

  import Event from "sap/ui/base/Event";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import {
    Slider$ChangeEventParameters,
    Slider$LiveChangeEventParameters,
  } from "sap/ui/commons/Slider";

  /**
   * The Date Range Slider provides the user with a Range Slider control that is optimized for use with Dates.
   *
   * @deprecated (since 1.34) - Deprecated. Not Fiori.
   */
  export default class DateRangeSliderInternal extends RangeSlider {
    /**
     * Constructor for a new DateRangeSliderInternal.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $DateRangeSliderInternalSettings
    );
    /**
     * Constructor for a new DateRangeSliderInternal.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $DateRangeSliderInternalSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.DateRangeSliderInternal with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.commons.RangeSlider.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DateRangeSliderInternal>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.DateRangeSliderInternal.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeSliderInternal`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeSliderInternal` itself.
     *
     * This event is fired when user changes the positions of the grips of the control. It contains value as
     * Date object and value2 as Date object which are based on the positions of two grips.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeSliderInternal`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeSliderInternal`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeSliderInternal` itself.
     *
     * This event is fired when user changes the positions of the grips of the control. It contains value as
     * Date object and value2 as Date object which are based on the positions of two grips.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeSliderInternal`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:liveChange liveChange} event of this `sap.suite.ui.commons.DateRangeSliderInternal`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeSliderInternal` itself.
     *
     * This live event is fired when user changes the positions of the grips of the control. It contains value
     * as Date object and value2 as Date object which are based on the positions of two grips.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachLiveChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeSliderInternal`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:liveChange liveChange} event of this `sap.suite.ui.commons.DateRangeSliderInternal`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.DateRangeSliderInternal` itself.
     *
     * This live event is fired when user changes the positions of the grips of the control. It contains value
     * as Date object and value2 as Date object which are based on the positions of two grips.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachLiveChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.DateRangeSliderInternal`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:change change} event of this `sap.suite.ui.commons.DateRangeSliderInternal`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:liveChange liveChange} event of this `sap.suite.ui.commons.DateRangeSliderInternal`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachLiveChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:change change} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:liveChange liveChange} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireLiveChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getPinGrip pinGrip}.
     *
     * Property to pin Grip (left grip) of the slider so that user cannot move it with key or mouse clicks.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `pinGrip`
     */
    getPinGrip(): boolean;
    /**
     * Gets current value of property {@link #getPinGrip2 pinGrip2}.
     *
     * Property to pin Grip2 (right grip) of the slider so that user cannot move it with key or mouse clicks.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `pinGrip2`
     */
    getPinGrip2(): boolean;
    /**
     * Gets current value of property {@link #getShowBubbles showBubbles}.
     *
     * Property to show or hide bubbles. Default is true.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showBubbles`
     */
    getShowBubbles(): boolean;
    /**
     * Sets a new value for property {@link #getPinGrip pinGrip}.
     *
     * Property to pin Grip (left grip) of the slider so that user cannot move it with key or mouse clicks.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPinGrip(
      /**
       * New value for property `pinGrip`
       */
      bPinGrip?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getPinGrip2 pinGrip2}.
     *
     * Property to pin Grip2 (right grip) of the slider so that user cannot move it with key or mouse clicks.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPinGrip2(
      /**
       * New value for property `pinGrip2`
       */
      bPinGrip2?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowBubbles showBubbles}.
     *
     * Property to show or hide bubbles. Default is true.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowBubbles(
      /**
       * New value for property `showBubbles`
       */
      bShowBubbles?: boolean
    ): this;
  }
  /**
   * Describes the settings that can be provided to the DateRangeSliderInternal constructor.
   *
   * @deprecated (since 1.34) - Deprecated. Not Fiori.
   */
  export interface $DateRangeSliderInternalSettings
    extends $RangeSliderSettings {
    /**
     * Property to show or hide bubbles. Default is true.
     */
    showBubbles?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Property to pin Grip (left grip) of the slider so that user cannot move it with key or mouse clicks.
     */
    pinGrip?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Property to pin Grip2 (right grip) of the slider so that user cannot move it with key or mouse clicks.
     */
    pinGrip2?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * This event is fired when user changes the positions of the grips of the control. It contains value as
     * Date object and value2 as Date object which are based on the positions of two grips.
     */
    change?: (oEvent: Event) => void;

    /**
     * This live event is fired when user changes the positions of the grips of the control. It contains value
     * as Date object and value2 as Date object which are based on the positions of two grips.
     */
    liveChange?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the DateRangeSliderInternal#change event.
   */
  export interface DateRangeSliderInternal$ChangeEventParameters
    extends Slider$ChangeEventParameters {}

  /**
   * Event object of the DateRangeSliderInternal#change event.
   */
  export type DateRangeSliderInternal$ChangeEvent = Event<
    DateRangeSliderInternal$ChangeEventParameters,
    DateRangeSliderInternal
  >;

  /**
   * Parameters of the DateRangeSliderInternal#liveChange event.
   */
  export interface DateRangeSliderInternal$LiveChangeEventParameters
    extends Slider$LiveChangeEventParameters {}

  /**
   * Event object of the DateRangeSliderInternal#liveChange event.
   */
  export type DateRangeSliderInternal$LiveChangeEvent = Event<
    DateRangeSliderInternal$LiveChangeEventParameters,
    DateRangeSliderInternal
  >;
}

declare module "sap/suite/ui/commons/DeltaMicroChart" {
  import {
    default as DeltaMicroChart1,
    $DeltaMicroChartSettings as $DeltaMicroChartSettings1,
  } from "sap/suite/ui/microchart/DeltaMicroChart";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This control displays a delta of two values as a chart.
   *
   * @deprecated (since 1.34) - Deprecated. sap.suite.ui.microchart.DeltaMicroChart should be used.
   */
  export default class DeltaMicroChart extends DeltaMicroChart1 {
    /**
     * Constructor for a new DeltaMicroChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.DeltaMicroChart#constructor sap.suite.ui.microchart.DeltaMicroChart }
     * can be used.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $DeltaMicroChartSettings
    );
    /**
     * Constructor for a new DeltaMicroChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.DeltaMicroChart#constructor sap.suite.ui.microchart.DeltaMicroChart }
     * can be used.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $DeltaMicroChartSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.DeltaMicroChart with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.DeltaMicroChart.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DeltaMicroChart>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.DeltaMicroChart.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the DeltaMicroChart constructor.
   *
   * @deprecated (since 1.34) - Deprecated. sap.suite.ui.microchart.DeltaMicroChart should be used.
   */
  export interface $DeltaMicroChartSettings extends $DeltaMicroChartSettings1 {}
}

declare module "sap/suite/ui/commons/DynamicContainer" {
  import { default as SlideTile, $SlideTileSettings } from "sap/m/SlideTile";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * The control that displays multiple GenericTile controls as changing slides.
   *
   * @deprecated (since 1.34) - Deprecated. Moved to openui5.
   */
  export default class DynamicContainer extends SlideTile {
    /**
     * Constructor for a new DynamicContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.m.SlideTile#constructor sap.m.SlideTile }
     * can be used.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $DynamicContainerSettings
    );
    /**
     * Constructor for a new DynamicContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.m.SlideTile#constructor sap.m.SlideTile }
     * can be used.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $DynamicContainerSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.DynamicContainer with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.m.SlideTile.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, DynamicContainer>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.DynamicContainer.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the DynamicContainer constructor.
   *
   * @deprecated (since 1.34) - Deprecated. Moved to openui5.
   */
  export interface $DynamicContainerSettings extends $SlideTileSettings {}
}

declare module "sap/suite/ui/commons/FacetOverview" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import { CSSSize } from "sap/ui/core/library";

  import { FacetOverviewHeight } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control is used in UnifiedThingInspector to display the preview of the facet content.
   *
   * @deprecated (since 1.32) - Deprecated. Object page should be used instead.
   */
  export default class FacetOverview extends Control {
    /**
     * Constructor for a new FacetOverview.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $FacetOverviewSettings
    );
    /**
     * Constructor for a new FacetOverview.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $FacetOverviewSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.FacetOverview with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, FacetOverview>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.FacetOverview.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:heightChange heightChange} event of this `sap.suite.ui.commons.FacetOverview`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.FacetOverview` itself.
     *
     * This event is fired when a new value to the heightType or height property is set.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHeightChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.FacetOverview`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:heightChange heightChange} event of this `sap.suite.ui.commons.FacetOverview`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.FacetOverview` itself.
     *
     * This event is fired when a new value to the heightType or height property is set.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHeightChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.FacetOverview`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.FacetOverview`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.FacetOverview` itself.
     *
     * The event is fired when the user chooses the control. Provides an event with parameter id, the ID of
     * the chosen control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: FacetOverview$PressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.FacetOverview`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.FacetOverview`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.FacetOverview` itself.
     *
     * The event is fired when the user chooses the control. Provides an event with parameter id, the ID of
     * the chosen control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: FacetOverview$PressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.FacetOverview`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys the content in the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyContent(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:heightChange heightChange} event of this `sap.suite.ui.commons.FacetOverview`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachHeightChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.FacetOverview`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: FacetOverview$PressEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:heightChange heightChange} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireHeightChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: FacetOverview$PressEventParameters
    ): this;
    /**
     * Gets content of aggregation {@link #getContent content}.
     *
     * The content that appears in the left part of the control.
     */
    getContent(): Control;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * Defines the height of the control only if the heightType prperty is set to None. Must be set in rems
     * for the correct work in the UnifiedThingInspector.
     *
     * Default value is `'10rem'`.
     *
     * @deprecated (since 1.17.1) - This property was replaced by heightType property.
     *
     * @returns Value of property `height`
     */
    getHeight(): CSSSize;
    /**
     * Gets current value of property {@link #getHeightType heightType}.
     *
     * Indicates the height of the control in the predifened values. If set to None, then the height is defined
     * by the depricated height property.
     *
     * Default value is `"None"`.
     *
     *
     * @returns Value of property `heightType`
     */
    getHeightType(): FacetOverviewHeight | keyof typeof FacetOverviewHeight;
    /**
     * Gets current value of property {@link #getQuantity quantity}.
     *
     * Displays a label with the number of items in the right part of the control.
     *
     * Default value is `-1`.
     *
     *
     * @returns Value of property `quantity`
     */
    getQuantity(): int;
    /**
     * Gets current value of property {@link #getRowSpan rowSpan}.
     *
     * Defines the number of rows that the control represents in an external layout.
     *
     * Default value is `1`.
     *
     * @deprecated (since 1.17.1) - It is not used any more for the laoyut calculations in UnifiedThingInspector.
     *
     * @returns Value of property `rowSpan`
     */
    getRowSpan(): int;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * This property is shown in the upper left part of control.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Defines the width of the control. By default, the value is empty and the control inhertis the size from
     * its content.
     *
     * Default value is `'auto'`.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Sets the aggregated {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setContent(
      /**
       * The content to set
       */
      oContent: Control
    ): this;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * Defines the height of the control only if the heightType prperty is set to None. Must be set in rems
     * for the correct work in the UnifiedThingInspector.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'10rem'`.
     *
     * @deprecated (since 1.17.1) - This property was replaced by heightType property.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      sHeight?: CSSSize
    ): this;
    /**
     * Sets a new value for property {@link #getHeightType heightType}.
     *
     * Indicates the height of the control in the predifened values. If set to None, then the height is defined
     * by the depricated height property.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"None"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeightType(
      /**
       * New value for property `heightType`
       */
      sHeightType?: FacetOverviewHeight | keyof typeof FacetOverviewHeight
    ): this;
    /**
     * Sets a new value for property {@link #getQuantity quantity}.
     *
     * Displays a label with the number of items in the right part of the control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `-1`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setQuantity(
      /**
       * New value for property `quantity`
       */
      iQuantity?: int
    ): this;
    /**
     * Sets a new value for property {@link #getRowSpan rowSpan}.
     *
     * Defines the number of rows that the control represents in an external layout.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `1`.
     *
     * @deprecated (since 1.17.1) - It is not used any more for the laoyut calculations in UnifiedThingInspector.
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setRowSpan(
      /**
       * New value for property `rowSpan`
       */
      iRowSpan?: int
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * This property is shown in the upper left part of control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Defines the width of the control. By default, the value is empty and the control inhertis the size from
     * its content.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'auto'`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
  }
  /**
   * Describes the settings that can be provided to the FacetOverview constructor.
   *
   * @deprecated (since 1.32) - Deprecated. Object page should be used instead.
   */
  export interface $FacetOverviewSettings extends $ControlSettings {
    /**
     * This property is shown in the upper left part of control.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Displays a label with the number of items in the right part of the control.
     */
    quantity?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the width of the control. By default, the value is empty and the control inhertis the size from
     * its content.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the height of the control only if the heightType prperty is set to None. Must be set in rems
     * for the correct work in the UnifiedThingInspector.
     *
     * @deprecated (since 1.17.1) - This property was replaced by heightType property.
     */
    height?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the number of rows that the control represents in an external layout.
     *
     * @deprecated (since 1.17.1) - It is not used any more for the laoyut calculations in UnifiedThingInspector.
     */
    rowSpan?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Indicates the height of the control in the predifened values. If set to None, then the height is defined
     * by the depricated height property.
     */
    heightType?:
      | (FacetOverviewHeight | keyof typeof FacetOverviewHeight)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * The content that appears in the left part of the control.
     */
    content?: Control;

    /**
     * The event is fired when the user chooses the control. Provides an event with parameter id, the ID of
     * the chosen control.
     */
    press?: (oEvent: FacetOverview$PressEvent) => void;

    /**
     * This event is fired when a new value to the heightType or height property is set.
     */
    heightChange?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the FacetOverview#heightChange event.
   */
  export interface FacetOverview$HeightChangeEventParameters {}

  /**
   * Event object of the FacetOverview#heightChange event.
   */
  export type FacetOverview$HeightChangeEvent = Event<
    FacetOverview$HeightChangeEventParameters,
    FacetOverview
  >;

  /**
   * Parameters of the FacetOverview#press event.
   */
  export interface FacetOverview$PressEventParameters {
    /**
     * The control ID.
     */
    id?: string;
  }

  /**
   * Event object of the FacetOverview#press event.
   */
  export type FacetOverview$PressEvent = Event<
    FacetOverview$PressEventParameters,
    FacetOverview
  >;
}

declare module "sap/suite/ui/commons/FeedItem" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import { URI } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This element represents a news feed item.
   *
   * @deprecated (since 1.32) - Deprecated. Generic Tile should be used instead.
   */
  export default class FeedItem extends UI5Element {
    /**
     * Constructor for a new FeedItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $FeedItemSettings
    );
    /**
     * Constructor for a new FeedItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $FeedItemSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.FeedItem with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, FeedItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.FeedItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getImage image}.
     *
     * The background image for the feed item.
     *
     *
     * @returns Value of property `image`
     */
    getImage(): URI;
    /**
     * Gets current value of property {@link #getLink link}.
     *
     * The target location of the feed item.
     *
     *
     * @returns Value of property `link`
     */
    getLink(): URI;
    /**
     * Gets current value of property {@link #getPublicationDate publicationDate}.
     *
     * The date the feed was published.
     *
     *
     * @returns Value of property `publicationDate`
     */
    getPublicationDate(): object;
    /**
     * Gets current value of property {@link #getSource source}.
     *
     * The source of the feed item.
     *
     *
     * @returns Value of property `source`
     */
    getSource(): string;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * The title of the feed item.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets a new value for property {@link #getImage image}.
     *
     * The background image for the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setImage(
      /**
       * New value for property `image`
       */
      sImage?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getLink link}.
     *
     * The target location of the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLink(
      /**
       * New value for property `link`
       */
      sLink?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getPublicationDate publicationDate}.
     *
     * The date the feed was published.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPublicationDate(
      /**
       * New value for property `publicationDate`
       */
      oPublicationDate?: object
    ): this;
    /**
     * Sets a new value for property {@link #getSource source}.
     *
     * The source of the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSource(
      /**
       * New value for property `source`
       */
      sSource?: string
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * The title of the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the FeedItem constructor.
   *
   * @deprecated (since 1.32) - Deprecated. Generic Tile should be used instead.
   */
  export interface $FeedItemSettings extends $ElementSettings {
    /**
     * The title of the feed item.
     */
    title?: string | PropertyBindingInfo;

    /**
     * The background image for the feed item.
     */
    image?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * The target location of the feed item.
     */
    link?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * The source of the feed item.
     */
    source?: string | PropertyBindingInfo;

    /**
     * The date the feed was published.
     */
    publicationDate?: object | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/FeedItemHeader" {
  import {
    default as ListItemBase,
    $ListItemBaseSettings,
  } from "sap/m/ListItemBase";

  import { URI } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control displays feed item header information.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export default class FeedItemHeader extends ListItemBase {
    /**
     * Constructor for a new FeedItemHeader.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $FeedItemHeaderSettings
    );
    /**
     * Constructor for a new FeedItemHeader.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $FeedItemHeaderSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.FeedItemHeader with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.m.ListItemBase.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, FeedItemHeader>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.FeedItemHeader.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getDescription description}.
     *
     * The description of the feed item.
     *
     *
     * @returns Value of property `description`
     */
    getDescription(): string;
    /**
     * Gets current value of property {@link #getImage image}.
     *
     * The image associated with the feed item.
     *
     *
     * @returns Value of property `image`
     */
    getImage(): URI;
    /**
     * Gets current value of property {@link #getLink link}.
     *
     * The target location of the feed item.
     *
     *
     * @returns Value of property `link`
     */
    getLink(): URI;
    /**
     * Gets current value of property {@link #getPublicationDate publicationDate}.
     *
     * The date the feed item was published.
     *
     *
     * @returns Value of property `publicationDate`
     */
    getPublicationDate(): object;
    /**
     * Gets current value of property {@link #getSource source}.
     *
     * The source of the feed item.
     *
     *
     * @returns Value of property `source`
     */
    getSource(): string;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * The title of the feed item.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets a new value for property {@link #getDescription description}.
     *
     * The description of the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescription(
      /**
       * New value for property `description`
       */
      sDescription?: string
    ): this;
    /**
     * Sets a new value for property {@link #getImage image}.
     *
     * The image associated with the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setImage(
      /**
       * New value for property `image`
       */
      sImage?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getLink link}.
     *
     * The target location of the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLink(
      /**
       * New value for property `link`
       */
      sLink?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getPublicationDate publicationDate}.
     *
     * The date the feed item was published.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPublicationDate(
      /**
       * New value for property `publicationDate`
       */
      oPublicationDate?: object
    ): this;
    /**
     * Sets a new value for property {@link #getSource source}.
     *
     * The source of the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSource(
      /**
       * New value for property `source`
       */
      sSource?: string
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * The title of the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the FeedItemHeader constructor.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export interface $FeedItemHeaderSettings extends $ListItemBaseSettings {
    /**
     * The title of the feed item.
     */
    title?: string | PropertyBindingInfo;

    /**
     * The image associated with the feed item.
     */
    image?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * The target location of the feed item.
     */
    link?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * The source of the feed item.
     */
    source?: string | PropertyBindingInfo;

    /**
     * The date the feed item was published.
     */
    publicationDate?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * The description of the feed item.
     */
    description?: string | PropertyBindingInfo;
  }
}

declare module "sap/suite/ui/commons/FeedTile" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import FeedItem from "sap/suite/ui/commons/FeedItem";

  import Event from "sap/ui/base/Event";

  import { URI } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import Model from "sap/ui/model/Model";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * This control displays news feeds.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export default class FeedTile extends Control {
    /**
     * Constructor for a new FeedTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $FeedTileSettings
    );
    /**
     * Constructor for a new FeedTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $FeedTileSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.FeedTile with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, FeedTile>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.FeedTile.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some item to the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addItem(
      /**
       * The item to add; if empty, nothing is inserted
       */
      oItem: FeedItem
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.FeedTile`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.FeedTile` itself.
     *
     * The event fired when the user clicks on the control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.FeedTile` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.FeedTile`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.FeedTile` itself.
     *
     * The event fired when the user clicks on the control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.FeedTile` itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys all the items in the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyItems(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.FeedTile`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getDefaultImages defaultImages}.
     *
     * A list of default images that is cycled by the control when no image is available for a feed item or
     * when no feed items exist. After a random image is displayed first time, control cycles through this list
     * of images.
     *
     *
     * @returns Value of property `defaultImages`
     */
    getDefaultImages(): URI[];
    /**
     * Gets current value of property {@link #getDisplayArticleImage displayArticleImage}.
     *
     * To display article Image or not. If it is true, the article Image will be displayed based on precedence.
     * If it is false, the default image will be displayed.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `displayArticleImage`
     */
    getDisplayArticleImage(): boolean;
    /**
     * Gets current value of property {@link #getDisplayDuration displayDuration}.
     *
     * The length of time in seconds the control displays each feed item. Default value is 5 seconds.
     *
     * Default value is `5`.
     *
     *
     * @returns Value of property `displayDuration`
     */
    getDisplayDuration(): int;
    /**
     * Gets content of aggregation {@link #getItems items}.
     *
     * The feed items the control cycles through.
     */
    getItems(): FeedItem[];
    /**
     * Gets current value of property {@link #getSource source}.
     *
     * The source of the feed item.
     *
     *
     * @returns Value of property `source`
     */
    getSource(): string;
    /**
     * Checks for the provided `sap.suite.ui.commons.FeedItem` in the aggregation {@link #getItems items}. and
     * returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfItem(
      /**
       * The item whose index is looked for
       */
      oItem: FeedItem
    ): int;
    /**
     * Inserts a item into the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertItem(
      /**
       * The item to insert; if empty, nothing is inserted
       */
      oItem: FeedItem,
      /**
       * The `0`-based index the item should be inserted at; for a negative value of `iIndex`, the item is inserted
       * at position 0; for a value greater than the current size of the aggregation, the item is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getItems items}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllItems(): FeedItem[];
    /**
     * Removes a item from the aggregation {@link #getItems items}.
     *
     *
     * @returns The removed item or `null`
     */
    removeItem(
      /**
       * The item to remove or its index or id
       */
      vItem: int | string | FeedItem
    ): FeedItem | null;
    /**
     * Validate the array of URI first and then set the defaultImages property
     */
    setDefaultImages(
      /**
       * Default image URIs
       */
      oDefaultImages: URI[]
    ): void;
    /**
     * Sets a new value for property {@link #getDisplayArticleImage displayArticleImage}.
     *
     * To display article Image or not. If it is true, the article Image will be displayed based on precedence.
     * If it is false, the default image will be displayed.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDisplayArticleImage(
      /**
       * New value for property `displayArticleImage`
       */
      bDisplayArticleImage?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getDisplayDuration displayDuration}.
     *
     * The length of time in seconds the control displays each feed item. Default value is 5 seconds.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `5`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDisplayDuration(
      /**
       * New value for property `displayDuration`
       */
      iDisplayDuration?: int
    ): this;
    /**
     * Sets a new value for property {@link #getSource source}.
     *
     * The source of the feed item.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSource(
      /**
       * New value for property `source`
       */
      sSource?: string
    ): this;
    /**
     * Set a new model of feed items, such as when a feed aggregator has collected the latest feed items. This
     * model is staged (not immediately set on the control) to avoid re-rendering before the currently displayed
     * article is faded out. Therefore a smooth transition between the display of feed items is always maintained.
     */
    stageModel(
      /**
       * Model of new feed items.
       */
      oModel: Model
    ): void;
  }
  /**
   * Describes the settings that can be provided to the FeedTile constructor.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export interface $FeedTileSettings extends $ControlSettings {
    /**
     * The length of time in seconds the control displays each feed item. Default value is 5 seconds.
     */
    displayDuration?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * To display article Image or not. If it is true, the article Image will be displayed based on precedence.
     * If it is false, the default image will be displayed.
     */
    displayArticleImage?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The source of the feed item.
     */
    source?: string | PropertyBindingInfo;

    /**
     * A list of default images that is cycled by the control when no image is available for a feed item or
     * when no feed items exist. After a random image is displayed first time, control cycles through this list
     * of images.
     */
    defaultImages?: URI[] | PropertyBindingInfo | `{${string}}`;

    /**
     * The feed items the control cycles through.
     */
    items?: FeedItem[] | FeedItem | AggregationBindingInfo | `{${string}}`;

    /**
     * The event fired when the user clicks on the control.
     */
    press?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the FeedTile#press event.
   */
  export interface FeedTile$PressEventParameters {}

  /**
   * Event object of the FeedTile#press event.
   */
  export type FeedTile$PressEvent = Event<
    FeedTile$PressEventParameters,
    FeedTile
  >;
}

declare module "sap/suite/ui/commons/GenericTile" {
  import {
    default as GenericTile1,
    $GenericTileSettings as $GenericTileSettings1,
  } from "sap/m/GenericTile";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * The tile control that displays the title, description, and customizable main area.
   *
   * @deprecated (since 1.34) - Deprecated. Moved to openui5.
   */
  export default class GenericTile extends GenericTile1 {
    /**
     * Constructor for a new GenericTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.m.GenericTile#constructor sap.m.GenericTile }
     * can be used.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $GenericTileSettings
    );
    /**
     * Constructor for a new GenericTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.m.GenericTile#constructor sap.m.GenericTile }
     * can be used.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $GenericTileSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.GenericTile with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.m.GenericTile.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, GenericTile>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.GenericTile.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the GenericTile constructor.
   *
   * @deprecated (since 1.34) - Deprecated. Moved to openui5.
   */
  export interface $GenericTileSettings extends $GenericTileSettings1 {}
}

declare module "sap/suite/ui/commons/GenericTile2X2" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import TileContent2X2 from "sap/suite/ui/commons/TileContent2X2";

  import Event from "sap/ui/base/Event";

  import { URI } from "sap/ui/core/library";

  import {
    FrameType,
    InfoTileSize,
    LoadState,
  } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * The tile control that displays the title, description, and customizable main area.
   *
   * @deprecated (since 1.34) - Deprecated.
   */
  export default class GenericTile2X2 extends Control {
    /**
     * Constructor for a new GenericTile2X2.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $GenericTile2X2Settings
    );
    /**
     * Constructor for a new GenericTile2X2.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $GenericTile2X2Settings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.GenericTile2X2 with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, GenericTile2X2>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.GenericTile2X2.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some tileContent to the aggregation {@link #getTileContent tileContent}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addTileContent(
      /**
       * The tileContent to add; if empty, nothing is inserted
       */
      oTileContent: TileContent2X2
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.GenericTile2X2`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.GenericTile2X2` itself.
     *
     * The event is fired when the user chooses the tile.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.GenericTile2X2`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.GenericTile2X2`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.GenericTile2X2` itself.
     *
     * The event is fired when the user chooses the tile.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.GenericTile2X2`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys the icon in the aggregation {@link #getIcon icon}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyIcon(): this;
    /**
     * Destroys all the tileContent in the aggregation {@link #getTileContent tileContent}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyTileContent(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.GenericTile2X2`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getBackgroundImage backgroundImage}.
     *
     * The URI of the background image.
     *
     *
     * @returns Value of property `backgroundImage`
     */
    getBackgroundImage(): URI;
    /**
     * Gets current value of property {@link #getFailedText failedText}.
     *
     * The message that appears when the control is in the Failed state.
     *
     *
     * @returns Value of property `failedText`
     */
    getFailedText(): string;
    /**
     * Gets current value of property {@link #getFrameType frameType}.
     *
     * The frame type: 1x1 or 2x1.
     *
     * Default value is `"OneByOne"`.
     *
     *
     * @returns Value of property `frameType`
     */
    getFrameType(): FrameType | keyof typeof FrameType;
    /**
     * Gets current value of property {@link #getHeader header}.
     *
     * The header of the tile.
     *
     *
     * @returns Value of property `header`
     */
    getHeader(): string;
    /**
     * Gets current value of property {@link #getHeaderImage headerImage}.
     *
     * The image to be displayed as a graphical element within the header. This can be an image or an icon from
     * the icon font.
     *
     *
     * @returns Value of property `headerImage`
     */
    getHeaderImage(): URI;
    /**
     * Gets content of aggregation {@link #getIcon icon}.
     *
     * An icon or image to be displayed in the control.
     */
    getIcon(): Control;
    /**
     * Gets current value of property {@link #getImageDescription imageDescription}.
     *
     * Description of a header image that is used in the tooltip.
     *
     *
     * @returns Value of property `imageDescription`
     */
    getImageDescription(): string;
    /**
     * Gets current value of property {@link #getSize size}.
     *
     * The size of the tile. If not set, then the default size is applied based on the device tile.
     *
     * Default value is `"Auto"`.
     *
     *
     * @returns Value of property `size`
     */
    getSize(): InfoTileSize | keyof typeof InfoTileSize;
    /**
     * Gets current value of property {@link #getState state}.
     *
     * The load status.
     *
     * Default value is `"Loaded"`.
     *
     *
     * @returns Value of property `state`
     */
    getState(): LoadState | keyof typeof LoadState;
    /**
     * Gets current value of property {@link #getSubheader subheader}.
     *
     * The subheader of the tile.
     *
     *
     * @returns Value of property `subheader`
     */
    getSubheader(): string;
    /**
     * Gets content of aggregation {@link #getTileContent tileContent}.
     *
     * The switchable view that depends on the tile type.
     */
    getTileContent(): TileContent2X2[];
    /**
     * Checks for the provided `sap.suite.ui.commons.TileContent2X2` in the aggregation {@link #getTileContent tileContent}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfTileContent(
      /**
       * The tileContent whose index is looked for
       */
      oTileContent: TileContent2X2
    ): int;
    /**
     * Inserts a tileContent into the aggregation {@link #getTileContent tileContent}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertTileContent(
      /**
       * The tileContent to insert; if empty, nothing is inserted
       */
      oTileContent: TileContent2X2,
      /**
       * The `0`-based index the tileContent should be inserted at; for a negative value of `iIndex`, the tileContent
       * is inserted at position 0; for a value greater than the current size of the aggregation, the tileContent
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getTileContent tileContent}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllTileContent(): TileContent2X2[];
    /**
     * Removes a tileContent from the aggregation {@link #getTileContent tileContent}.
     *
     *
     * @returns The removed tileContent or `null`
     */
    removeTileContent(
      /**
       * The tileContent to remove or its index or id
       */
      vTileContent: int | string | TileContent2X2
    ): TileContent2X2 | null;
    /**
     * Sets a new value for property {@link #getBackgroundImage backgroundImage}.
     *
     * The URI of the background image.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBackgroundImage(
      /**
       * New value for property `backgroundImage`
       */
      sBackgroundImage?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getFailedText failedText}.
     *
     * The message that appears when the control is in the Failed state.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFailedText(
      /**
       * New value for property `failedText`
       */
      sFailedText?: string
    ): this;
    /**
     * Sets a new value for property {@link #getFrameType frameType}.
     *
     * The frame type: 1x1 or 2x1.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"OneByOne"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFrameType(
      /**
       * New value for property `frameType`
       */
      sFrameType?: FrameType | keyof typeof FrameType
    ): this;
    /**
     * Sets a new value for property {@link #getHeader header}.
     *
     * The header of the tile.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeader(
      /**
       * New value for property `header`
       */
      sHeader?: string
    ): this;
    /**
     * Sets a new value for property {@link #getHeaderImage headerImage}.
     *
     * The image to be displayed as a graphical element within the header. This can be an image or an icon from
     * the icon font.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeaderImage(
      /**
       * New value for property `headerImage`
       */
      sHeaderImage?: URI
    ): this;
    /**
     * Sets the aggregated {@link #getIcon icon}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * The icon to set
       */
      oIcon: Control
    ): this;
    /**
     * Sets a new value for property {@link #getImageDescription imageDescription}.
     *
     * Description of a header image that is used in the tooltip.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setImageDescription(
      /**
       * New value for property `imageDescription`
       */
      sImageDescription?: string
    ): this;
    /**
     * Sets a new value for property {@link #getSize size}.
     *
     * The size of the tile. If not set, then the default size is applied based on the device tile.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Auto"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSize(
      /**
       * New value for property `size`
       */
      sSize?: InfoTileSize | keyof typeof InfoTileSize
    ): this;
    /**
     * Sets a new value for property {@link #getState state}.
     *
     * The load status.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Loaded"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setState(
      /**
       * New value for property `state`
       */
      sState?: LoadState | keyof typeof LoadState
    ): this;
    /**
     * Sets a new value for property {@link #getSubheader subheader}.
     *
     * The subheader of the tile.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSubheader(
      /**
       * New value for property `subheader`
       */
      sSubheader?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the GenericTile2X2 constructor.
   *
   * @deprecated (since 1.34) - Deprecated.
   */
  export interface $GenericTile2X2Settings extends $ControlSettings {
    /**
     * The header of the tile.
     */
    header?: string | PropertyBindingInfo;

    /**
     * The subheader of the tile.
     */
    subheader?: string | PropertyBindingInfo;

    /**
     * The message that appears when the control is in the Failed state.
     */
    failedText?: string | PropertyBindingInfo;

    /**
     * The size of the tile. If not set, then the default size is applied based on the device tile.
     */
    size?:
      | (InfoTileSize | keyof typeof InfoTileSize)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * The URI of the background image.
     */
    backgroundImage?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * The image to be displayed as a graphical element within the header. This can be an image or an icon from
     * the icon font.
     */
    headerImage?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * The frame type: 1x1 or 2x1.
     */
    frameType?:
      | (FrameType | keyof typeof FrameType)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * The load status.
     */
    state?:
      | (LoadState | keyof typeof LoadState)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Description of a header image that is used in the tooltip.
     */
    imageDescription?: string | PropertyBindingInfo;

    /**
     * The switchable view that depends on the tile type.
     */
    tileContent?:
      | TileContent2X2[]
      | TileContent2X2
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * An icon or image to be displayed in the control.
     */
    icon?: Control;

    /**
     * The event is fired when the user chooses the tile.
     */
    press?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the GenericTile2X2#press event.
   */
  export interface GenericTile2X2$PressEventParameters {}

  /**
   * Event object of the GenericTile2X2#press event.
   */
  export type GenericTile2X2$PressEvent = Event<
    GenericTile2X2$PressEventParameters,
    GenericTile2X2
  >;
}

declare module "sap/suite/ui/commons/HarveyBallMicroChart" {
  import {
    default as HarveyBallMicroChart1,
    $HarveyBallMicroChartSettings as $HarveyBallMicroChartSettings1,
  } from "sap/suite/ui/microchart/HarveyBallMicroChart";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This chart shows the part comparative to total.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.HarveyBallMicroChart.
   */
  export default class HarveyBallMicroChart extends HarveyBallMicroChart1 {
    /**
     * Constructor for a new HarveyBallMicroChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.HarveyBallMicroChart#constructor sap.suite.ui.microchart.HarveyBallMicroChart }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $HarveyBallMicroChartSettings
    );
    /**
     * Constructor for a new HarveyBallMicroChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.HarveyBallMicroChart#constructor sap.suite.ui.microchart.HarveyBallMicroChart }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $HarveyBallMicroChartSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.HarveyBallMicroChart with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.HarveyBallMicroChart.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, HarveyBallMicroChart>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.HarveyBallMicroChart.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the HarveyBallMicroChart constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.HarveyBallMicroChart.
   */
  export interface $HarveyBallMicroChartSettings
    extends $HarveyBallMicroChartSettings1 {}
}

declare module "sap/suite/ui/commons/HarveyBallMicroChartItem" {
  import {
    default as HarveyBallMicroChartItem1,
    $HarveyBallMicroChartItemSettings as $HarveyBallMicroChartItemSettings1,
  } from "sap/suite/ui/microchart/HarveyBallMicroChartItem";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * The configuration of the slice on the pie chart.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.HarveyBallMicroChartItem.
   */
  export default class HarveyBallMicroChartItem extends HarveyBallMicroChartItem1 {
    /**
     * Constructor for a new HarveyBallMicroChartItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.HarveyBallMicroChartItem#constructor sap.suite.ui.microchart.HarveyBallMicroChartItem }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $HarveyBallMicroChartItemSettings
    );
    /**
     * Constructor for a new HarveyBallMicroChartItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.HarveyBallMicroChartItem#constructor sap.suite.ui.microchart.HarveyBallMicroChartItem }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $HarveyBallMicroChartItemSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.HarveyBallMicroChartItem with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.HarveyBallMicroChartItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, HarveyBallMicroChartItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.HarveyBallMicroChartItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the HarveyBallMicroChartItem constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.HarveyBallMicroChartItem.
   */
  export interface $HarveyBallMicroChartItemSettings
    extends $HarveyBallMicroChartItemSettings1 {}
}

declare module "sap/suite/ui/commons/HeaderCell" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import HeaderCellItem from "sap/suite/ui/commons/HeaderCellItem";

  import { CSSSize } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control contains 4 cells (West, North, East, South). It can display one or more controls in different
   * layouts. Each aggregation must contain only one instance of HeaderCellItem.
   *
   * @deprecated (since 1.44.0) - HeaderCell control is no longer used. Please use other container controls
   * instead (like sap.m.VBox or sap.m.HBox).
   */
  export default class HeaderCell extends Control {
    /**
     * Constructor for a new HeaderCell.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $HeaderCellSettings
    );
    /**
     * Constructor for a new HeaderCell.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $HeaderCellSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.HeaderCell with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, HeaderCell>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.HeaderCell.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Destroys the east in the aggregation {@link #getEast east}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyEast(): this;
    /**
     * Destroys the north in the aggregation {@link #getNorth north}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyNorth(): this;
    /**
     * Destroys the south in the aggregation {@link #getSouth south}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroySouth(): this;
    /**
     * Destroys the west in the aggregation {@link #getWest west}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyWest(): this;
    /**
     * Gets content of aggregation {@link #getEast east}.
     *
     * Object that contains control to render in east area of the HeaderCell.
     */
    getEast(): HeaderCellItem;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * Height of the HeaderCell control.
     *
     * Default value is `'106px'`.
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): CSSSize;
    /**
     * Gets current value of property {@link #getHeigth heigth}.
     *
     * Height of the HeaderCell control.
     *
     * Default value is `'100px'`.
     *
     * @deprecated (since 1.20.2) - Wrong property name
     *
     * @returns Value of property `heigth`
     */
    getHeigth(): CSSSize;
    /**
     * Gets content of aggregation {@link #getNorth north}.
     *
     * Object that contains control to render in north area of the HeaderCell.
     */
    getNorth(): HeaderCellItem;
    /**
     * Gets content of aggregation {@link #getSouth south}.
     *
     * Object that contains control to render in south area of the HeaderCell.
     */
    getSouth(): HeaderCellItem;
    /**
     * Gets content of aggregation {@link #getWest west}.
     *
     * Object that contains control to render in west area of the HeaderCell.
     */
    getWest(): HeaderCellItem;
    /**
     * Sets the aggregated {@link #getEast east}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEast(
      /**
       * The east to set
       */
      oEast: HeaderCellItem
    ): this;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * Height of the HeaderCell control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'106px'`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      sHeight?: CSSSize
    ): this;
    /**
     * Sets a new value for property {@link #getHeigth heigth}.
     *
     * Height of the HeaderCell control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'100px'`.
     *
     * @deprecated (since 1.20.2) - Wrong property name
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeigth(
      /**
       * New value for property `heigth`
       */
      sHeigth?: CSSSize
    ): this;
    /**
     * Sets the aggregated {@link #getNorth north}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setNorth(
      /**
       * The north to set
       */
      oNorth: HeaderCellItem
    ): this;
    /**
     * Sets the aggregated {@link #getSouth south}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSouth(
      /**
       * The south to set
       */
      oSouth: HeaderCellItem
    ): this;
    /**
     * Sets the aggregated {@link #getWest west}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWest(
      /**
       * The west to set
       */
      oWest: HeaderCellItem
    ): this;
  }
  /**
   * Describes the settings that can be provided to the HeaderCell constructor.
   *
   * @deprecated (since 1.44.0) - HeaderCell control is no longer used. Please use other container controls
   * instead (like sap.m.VBox or sap.m.HBox).
   */
  export interface $HeaderCellSettings extends $ControlSettings {
    /**
     * Height of the HeaderCell control.
     *
     * @deprecated (since 1.20.2) - Wrong property name
     */
    heigth?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Height of the HeaderCell control.
     */
    height?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Object that contains control to render in west area of the HeaderCell.
     */
    west?: HeaderCellItem;

    /**
     * Object that contains control to render in north area of the HeaderCell.
     */
    north?: HeaderCellItem;

    /**
     * Object that contains control to render in east area of the HeaderCell.
     */
    east?: HeaderCellItem;

    /**
     * Object that contains control to render in south area of the HeaderCell.
     */
    south?: HeaderCellItem;
  }
}

declare module "sap/suite/ui/commons/HeaderCellItem" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import Control from "sap/ui/core/Control";

  import { CSSSize } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Object that contains instance of control and information about height. It should be used inside sap.suite.ui.commons.HeaderCell
   *
   * @deprecated (since 1.44.0) - The HeaderCellItem was deprecated because of the deprecation of HeaderCell.
   * Please see HeaderCell for replacement advice.
   */
  export default class HeaderCellItem extends UI5Element {
    /**
     * Constructor for a new HeaderCellItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $HeaderCellItemSettings
    );
    /**
     * Constructor for a new HeaderCellItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $HeaderCellItemSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.HeaderCellItem with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, HeaderCellItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.HeaderCellItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Destroys the content in the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyContent(): this;
    /**
     * Gets content of aggregation {@link #getContent content}.
     *
     * Instance of UI5 Control that is used as content.
     */
    getContent(): Control;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * Height of area to occupy by control.
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): CSSSize;
    /**
     * Sets the aggregated {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setContent(
      /**
       * The content to set
       */
      oContent: Control
    ): this;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * Height of area to occupy by control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      sHeight?: CSSSize
    ): this;
  }
  /**
   * Describes the settings that can be provided to the HeaderCellItem constructor.
   *
   * @deprecated (since 1.44.0) - The HeaderCellItem was deprecated because of the deprecation of HeaderCell.
   * Please see HeaderCell for replacement advice.
   */
  export interface $HeaderCellItemSettings extends $ElementSettings {
    /**
     * Height of area to occupy by control.
     */
    height?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Instance of UI5 Control that is used as content.
     */
    content?: Control;
  }
}

declare module "sap/suite/ui/commons/HeaderContainer" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import { ObjectHeaderContainer, BackgroundDesign } from "sap/m/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { HeaderContainerView } from "sap/suite/ui/commons/library";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * The container that provides a horizontal layout. It provides a horizontal scroll on the tablet and phone.
   * On the desktop, it provides scroll left and scroll right buttons. This control supports keyboard navigation.
   * You can use left and right arrow keys to navigate through the inner content. The Home key puts focus
   * on the first control and the End key puts focus on the last control. Use Enter or Space to choose the
   * control.
   *
   * @deprecated (since 1.44) - This control is deprecated in this library since 1.44 and moved to sap.m library
   * that is also part of openUI5.
   */
  export default class HeaderContainer
    extends Control
    implements ObjectHeaderContainer {
    __implements__sap_m_ObjectHeaderContainer: boolean;
    /**
     * Constructor for a new HeaderContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $HeaderContainerSettings
    );
    /**
     * Constructor for a new HeaderContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $HeaderContainerSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.HeaderContainer with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, HeaderContainer>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.HeaderContainer.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some item to the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addItem(
      /**
       * The item to add; if empty, nothing is inserted
       */
      oItem: Control
    ): this;
    /**
     * Destroys all the items in the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyItems(): this;
    /**
     * Gets current value of property {@link #getBackgroundDesign backgroundDesign}.
     *
     * Specifies the background color of the content. The visualization of the different options depends on
     * the used theme.
     *
     * Default value is `"Transparent"`.
     *
     * @since 1.38
     *
     * @returns Value of property `backgroundDesign`
     */
    getBackgroundDesign(): BackgroundDesign | keyof typeof BackgroundDesign;
    /**
     * Gets content of aggregation {@link #getItems items}.
     *
     * Items to add to HeaderContainer.
     */
    getItems(): Control[];
    /**
     * Gets current value of property {@link #getScrollStep scrollStep}.
     *
     * Number of pixels to scroll when the user chooses Next or Previous buttons. Relevant only for desktop.
     *
     * Default value is `300`.
     *
     *
     * @returns Value of property `scrollStep`
     */
    getScrollStep(): int;
    /**
     * Gets current value of property {@link #getScrollTime scrollTime}.
     *
     * Scroll animation time in milliseconds.
     *
     * Default value is `500`.
     *
     *
     * @returns Value of property `scrollTime`
     */
    getScrollTime(): int;
    /**
     * Gets current value of property {@link #getShowDividers showDividers}.
     *
     * If set to true, shows dividers between scrollable items.
     *
     * Default value is `true`.
     *
     * @since 1.25
     *
     * @returns Value of property `showDividers`
     */
    getShowDividers(): boolean;
    /**
     * Gets current value of property {@link #getView view}.
     *
     * The view of the HeaderContainer. There are two view modes: horizontal and vertical. In horizontal mode
     * the content controls are displayed next to each other, in vertical mode the content controls are displayed
     * on top of each other.
     *
     * Default value is `"Horizontal"`.
     *
     * @since 1.25
     *
     * @returns Value of property `view`
     */
    getView(): HeaderContainerView | keyof typeof HeaderContainerView;
    /**
     * Checks for the provided `sap.ui.core.Control` in the aggregation {@link #getItems items}. and returns
     * its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfItem(
      /**
       * The item whose index is looked for
       */
      oItem: Control
    ): int;
    /**
     * Inserts a item into the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertItem(
      /**
       * The item to insert; if empty, nothing is inserted
       */
      oItem: Control,
      /**
       * The `0`-based index the item should be inserted at; for a negative value of `iIndex`, the item is inserted
       * at position 0; for a value greater than the current size of the aggregation, the item is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getItems items}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllItems(): Control[];
    /**
     * Removes a item from the aggregation {@link #getItems items}.
     *
     *
     * @returns The removed item or `null`
     */
    removeItem(
      /**
       * The item to remove or its index or id
       */
      vItem: int | string | Control
    ): Control | null;
    /**
     * Sets a new value for property {@link #getBackgroundDesign backgroundDesign}.
     *
     * Specifies the background color of the content. The visualization of the different options depends on
     * the used theme.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Transparent"`.
     *
     * @since 1.38
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBackgroundDesign(
      /**
       * New value for property `backgroundDesign`
       */
      sBackgroundDesign?: BackgroundDesign | keyof typeof BackgroundDesign
    ): this;
    /**
     * Sets a new value for property {@link #getScrollStep scrollStep}.
     *
     * Number of pixels to scroll when the user chooses Next or Previous buttons. Relevant only for desktop.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `300`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setScrollStep(
      /**
       * New value for property `scrollStep`
       */
      iScrollStep?: int
    ): this;
    /**
     * Sets a new value for property {@link #getScrollTime scrollTime}.
     *
     * Scroll animation time in milliseconds.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `500`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setScrollTime(
      /**
       * New value for property `scrollTime`
       */
      iScrollTime?: int
    ): this;
    /**
     * Sets a new value for property {@link #getShowDividers showDividers}.
     *
     * If set to true, shows dividers between scrollable items.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     * @since 1.25
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowDividers(
      /**
       * New value for property `showDividers`
       */
      bShowDividers?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getView view}.
     *
     * The view of the HeaderContainer. There are two view modes: horizontal and vertical. In horizontal mode
     * the content controls are displayed next to each other, in vertical mode the content controls are displayed
     * on top of each other.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Horizontal"`.
     *
     * @since 1.25
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setView(
      /**
       * New value for property `view`
       */
      sView?: HeaderContainerView | keyof typeof HeaderContainerView
    ): this;
  }
  /**
   * Describes the settings that can be provided to the HeaderContainer constructor.
   *
   * @deprecated (since 1.44) - This control is deprecated in this library since 1.44 and moved to sap.m library
   * that is also part of openUI5.
   */
  export interface $HeaderContainerSettings extends $ControlSettings {
    /**
     * Number of pixels to scroll when the user chooses Next or Previous buttons. Relevant only for desktop.
     */
    scrollStep?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Scroll animation time in milliseconds.
     */
    scrollTime?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * If set to true, shows dividers between scrollable items.
     *
     * @since 1.25
     */
    showDividers?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The view of the HeaderContainer. There are two view modes: horizontal and vertical. In horizontal mode
     * the content controls are displayed next to each other, in vertical mode the content controls are displayed
     * on top of each other.
     *
     * @since 1.25
     */
    view?:
      | (HeaderContainerView | keyof typeof HeaderContainerView)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Specifies the background color of the content. The visualization of the different options depends on
     * the used theme.
     *
     * @since 1.38
     */
    backgroundDesign?:
      | (BackgroundDesign | keyof typeof BackgroundDesign)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Items to add to HeaderContainer.
     */
    items?: Control[] | Control | AggregationBindingInfo | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/imageeditor/CropCustomShapeHistoryItem" {
  import HistoryItem from "sap/suite/ui/commons/imageeditor/HistoryItem";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Holds information about an {@link sap.suite.ui.commons.imageeditor.ImageEditor} action history item that
   * was created by the {@link sap.suite.ui.commons.imageeditor.ImageEditor#rectangleCrop} method.
   *
   * @since 1.67.0
   */
  export default class CropCustomShapeHistoryItem extends HistoryItem {
    /**
     * Constructor for a new CropCustomShapeHistoryItem.
     */
    constructor(
      /**
       * Property bag
       */
      mProperties: {
        /**
         * X value
         */
        x: int;
        /**
         * Y value
         */
        y: int;
        /**
         * Width of the rectangle
         */
        width: int;
        /**
         * Height of the rectangle
         */
        height: int;
        /**
         * Previous width of the rectangle
         */
        oldWidth: int;
        /**
         * Previous height of the rectangle
         */
        oldHeight: int;
      }
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.CropCustomShapeHistoryItem with name
     * `sClassName` and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.imageeditor.HistoryItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CropCustomShapeHistoryItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.CropCustomShapeHistoryItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
  }
}

declare module "sap/suite/ui/commons/imageeditor/CropEllipseHistoryItem" {
  import HistoryItem from "sap/suite/ui/commons/imageeditor/HistoryItem";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Holds information about an {@link sap.suite.ui.commons.imageeditor.ImageEditor} action history item that
   * was created by the {@link sap.suite.ui.commons.imageeditor.ImageEditor#ellipseCrop} method.
   *
   * @since 1.66.0
   */
  export default class CropEllipseHistoryItem extends HistoryItem {
    /**
     * Constructor for a new CropEllipseHistoryItem.
     */
    constructor(
      /**
       * Property bag
       */
      mProperties: {
        /**
         * X value
         */
        x: int;
        /**
         * Y value
         */
        y: int;
        /**
         * X radius value
         */
        rx: int;
        /**
         * Y radius value
         */
        ry: int;
        /**
         * Width of the ellipse
         */
        width: int;
        /**
         * Height of the ellipse
         */
        height: int;
        /**
         * Previous width of the ellipse
         */
        oldWidth: int;
        /**
         * Previous height of the ellipse
         */
        oldHeight: int;
      }
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.CropEllipseHistoryItem with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.imageeditor.HistoryItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CropEllipseHistoryItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.CropEllipseHistoryItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
  }
}

declare module "sap/suite/ui/commons/imageeditor/CropRectangleHistoryItem" {
  import HistoryItem from "sap/suite/ui/commons/imageeditor/HistoryItem";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Holds information about an {@link sap.suite.ui.commons.imageeditor.ImageEditor} action history item that
   * was created by the {@link sap.suite.ui.commons.imageeditor.ImageEditor#rectangleCrop} method.
   *
   * @since 1.66.0
   */
  export default class CropRectangleHistoryItem extends HistoryItem {
    /**
     * Constructor for a new CropRectangleHistoryItem.
     */
    constructor(
      /**
       * Property bag
       */
      mProperties: {
        /**
         * X value
         */
        x: int;
        /**
         * Y value
         */
        y: int;
        /**
         * Width of the rectangle
         */
        width: int;
        /**
         * Height of the rectangle
         */
        height: int;
        /**
         * Previous width of the rectangle
         */
        oldWidth: int;
        /**
         * Previous height of the rectangle
         */
        oldHeight: int;
      }
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.CropRectangleHistoryItem with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.imageeditor.HistoryItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CropRectangleHistoryItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.CropRectangleHistoryItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
  }
}

declare module "sap/suite/ui/commons/imageeditor/CustomSizeItem" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import { URI } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * A `CustomSizeItem` can be used in the {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer} aggregations
   * to define custom items on the Resize, Crop Rectangle, and Crop Ellipse panels.
   *
   * @since 1.66.0
   */
  export default class CustomSizeItem extends UI5Element {
    /**
     * Constructor for a new CustomSizeItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $CustomSizeItemSettings
    );
    /**
     * Constructor for a new CustomSizeItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $CustomSizeItemSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.CustomSizeItem with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, CustomSizeItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.CustomSizeItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * * Defines the height of the `CustomSizeItem`.
     * This property is mandatory and must be greater than 0.
     *
     * Default value is `0`.
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): float;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * Defines the icon to be displayed in the `CustomSizeItem`.
     * If no icon is specified, the default image is used, which is derived from the set width and height.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): URI;
    /**
     * Gets current value of property {@link #getLabel label}.
     *
     * Defines the label of the `CustomSizeItem`.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `label`
     */
    getLabel(): string;
    /**
     * Gets current value of property {@link #getRelative relative}.
     *
     * Specifies whether the `width` and `height` properties use relative values (`true`) or absolute values
     * (`false`).
     * When used in the `customResizeItems` aggregation of an {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer},
     * width and height are multiplied by the `width` and `height` properties of the `CustomSizeItem`.
     * When used in the `customRectangleCropItems` or `customEllipseCropItems` aggregations of an {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer},
     * aspect ratio is set to the `width` and `height` properties of the `CustomSizeItem`.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `relative`
     */
    getRelative(): boolean;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Defines the width of the `CustomSizeItem`.
     * This property is mandatory and must be greater than 0.
     *
     * Default value is `0`.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): float;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * * Defines the height of the `CustomSizeItem`.
     * This property is mandatory and must be greater than 0.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      fHeight?: float
    ): this;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * Defines the icon to be displayed in the `CustomSizeItem`.
     * If no icon is specified, the default image is used, which is derived from the set width and height.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getLabel label}.
     *
     * Defines the label of the `CustomSizeItem`.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLabel(
      /**
       * New value for property `label`
       */
      sLabel?: string
    ): this;
    /**
     * Sets a new value for property {@link #getRelative relative}.
     *
     * Specifies whether the `width` and `height` properties use relative values (`true`) or absolute values
     * (`false`).
     * When used in the `customResizeItems` aggregation of an {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer},
     * width and height are multiplied by the `width` and `height` properties of the `CustomSizeItem`.
     * When used in the `customRectangleCropItems` or `customEllipseCropItems` aggregations of an {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer},
     * aspect ratio is set to the `width` and `height` properties of the `CustomSizeItem`.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setRelative(
      /**
       * New value for property `relative`
       */
      bRelative?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Defines the width of the `CustomSizeItem`.
     * This property is mandatory and must be greater than 0.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      fWidth?: float
    ): this;
  }
  /**
   * Describes the settings that can be provided to the CustomSizeItem constructor.
   */
  export interface $CustomSizeItemSettings extends $ElementSettings {
    /**
     * Defines the width of the `CustomSizeItem`.
     * This property is mandatory and must be greater than 0.
     */
    width?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * * Defines the height of the `CustomSizeItem`.
     * This property is mandatory and must be greater than 0.
     */
    height?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the label of the `CustomSizeItem`.
     */
    label?: string | PropertyBindingInfo;

    /**
     * Defines the icon to be displayed in the `CustomSizeItem`.
     * If no icon is specified, the default image is used, which is derived from the set width and height.
     */
    icon?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * Specifies whether the `width` and `height` properties use relative values (`true`) or absolute values
     * (`false`).
     * When used in the `customResizeItems` aggregation of an {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer},
     * width and height are multiplied by the `width` and `height` properties of the `CustomSizeItem`.
     * When used in the `customRectangleCropItems` or `customEllipseCropItems` aggregations of an {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer},
     * aspect ratio is set to the `width` and `height` properties of the `CustomSizeItem`.
     */
    relative?: boolean | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/imageeditor/FilterHistoryItem" {
  import HistoryItem from "sap/suite/ui/commons/imageeditor/HistoryItem";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Holds information about an {@link sap.suite.ui.commons.imageeditor.ImageEditor} action history item that
   * was created by one of the methods that apply filters to the image, such as {@link sap.suite.ui.commons.imageeditor.ImageEditor#sepia},
   * {@link sap.suite.ui.commons.imageeditor.ImageEditor#grayscale}, {@link sap.suite.ui.commons.imageeditor.ImageEditor#saturate},
   * {@link sap.suite.ui.commons.imageeditor.ImageEditor#invert}, {@link sap.suite.ui.commons.imageeditor.ImageEditor#brightness},
   * and {@link sap.suite.ui.commons.imageeditor.ImageEditor#contrast}.
   *
   * @since 1.66.0
   */
  export default class FilterHistoryItem extends HistoryItem {
    /**
     * Constructor for a new FilterHistoryItem.
     */
    constructor(
      /**
       * Property bag
       */
      mProperties: {
        /**
         * Filter type
         */
        type: string;
        /**
         * Filter value
         */
        value: int;
        /**
         * Filter unit
         */
        unit: string;
      }
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.FilterHistoryItem with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.imageeditor.HistoryItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, FilterHistoryItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.FilterHistoryItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
    /**
     * Gets the type of the filter applied.
     *
     *
     * @returns Type of the filter
     */
    getType(): string;
    /**
     * Gets the units of the filter value.
     *
     *
     * @returns Unit of the filter
     */
    getUnit(): string;
    /**
     * Gets the filter value (its intensity).
     *
     *
     * @returns Value of the filter
     */
    getValue(): int;
  }
}

declare module "sap/suite/ui/commons/imageeditor/FlipHistoryItem" {
  import HistoryItem from "sap/suite/ui/commons/imageeditor/HistoryItem";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Holds information about an {@link sap.suite.ui.commons.imageeditor.ImageEditor} action history item that
   * was created by one of the flip methods, such as {@link sap.suite.ui.commons.imageeditor.ImageEditor#flipVertical},
   * {@link sap.suite.ui.commons.imageeditor.ImageEditor#flipHorizontal}, and {@link sap.suite.ui.commons.imageeditor.ImageEditor#flip}.
   *
   * @since 1.66.0
   */
  export default class FlipHistoryItem extends HistoryItem {
    /**
     * Constructor for a new FlipHistoryItem.
     */
    constructor(
      /**
       * Property bag
       */
      mProperties: {
        /**
         * Whether vertical flip has been performed
         */
        vertical: boolean;
        /**
         * Whether horizontal flip has been performed
         */
        horizontal: boolean;
      }
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.FlipHistoryItem with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.imageeditor.HistoryItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, FlipHistoryItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.FlipHistoryItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
  }
}

declare module "sap/suite/ui/commons/imageeditor/HistoryItem" {
  import BaseObject from "sap/ui/base/Object";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Abstract base class for all {@link sap.suite.ui.commons.imageeditor.ImageEditor} action history items.
   *
   * @since 1.66.0
   */
  export default abstract class HistoryItem extends BaseObject {
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.HistoryItem with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, HistoryItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.HistoryItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
  }
}

declare module "sap/suite/ui/commons/imageeditor/ImageEditor" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import HistoryItem from "sap/suite/ui/commons/imageeditor/HistoryItem";

  import { ImageFormat, ImageEditorMode } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Image editor displays an image and provides API and visual controls to edit it.
   *
   * @since 1.66.0
   */
  export default class ImageEditor extends Control {
    /**
     * Constructor for a new ImageEditor.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ImageEditorSettings
    );
    /**
     * Constructor for a new ImageEditor.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ImageEditorSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.ImageEditor with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ImageEditor>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.ImageEditor.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Saves current preview to action history, if there is any.
     */
    applyPreview(): void;
    /**
     * Crops the image based on the currently shown crop area.
     * This method works only when the {@link sap.suite.ui.commons.ImageEditorMode} is set to `CropRectangle`
     * or `CropEllipse`.
     */
    applyVisibleCrop(
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:cropAreaChanged cropAreaChanged} event of this
     * `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired when the size of the crop area is changed by the user while in the `CropRectangle`
     * or `CropEllipse` mode ({@link sap.suite.ui.commons.ImageEditorMode}).
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCropAreaChanged(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ImageEditor$CropAreaChangedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:cropAreaChanged cropAreaChanged} event of this
     * `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired when the size of the crop area is changed by the user while in the `CropRectangle`
     * or `CropEllipse` mode ({@link sap.suite.ui.commons.ImageEditorMode}).
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCropAreaChanged(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ImageEditor$CropAreaChangedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:customShapeError customShapeError} event of
     * this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired if there is an error loading the image specified in the `customShapeSrc` property,
     * for example, when the URL is unreachable.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCustomShapeError(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:customShapeError customShapeError} event of
     * this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired if there is an error loading the image specified in the `customShapeSrc` property,
     * for example, when the URL is unreachable.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCustomShapeError(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:customShapeLoaded customShapeLoaded} event of
     * this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired after the image specified in the `customShapeSrc` property is successfully loaded.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCustomShapeLoaded(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:customShapeLoaded customShapeLoaded} event of
     * this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired after the image specified in the `customShapeSrc` property is successfully loaded.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCustomShapeLoaded(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:error error} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired if there is an error loading the image specified in the `src` property, for example,
     * when the URL is unreachable.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachError(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:error error} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired if there is an error loading the image specified in the `src` property, for example,
     * when the URL is unreachable.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachError(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:historyChanged historyChanged} event of this
     * `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired each time there is a change in the history of recent actions or in the recent actions
     * history index.
     * This may happen either when an action is performed on the image or when the {@link #undo}, {@link #redo},
     * or {@link #jumpToHistory} methods are called.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHistoryChanged(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:historyChanged historyChanged} event of this
     * `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired each time there is a change in the history of recent actions or in the recent actions
     * history index.
     * This may happen either when an action is performed on the image or when the {@link #undo}, {@link #redo},
     * or {@link #jumpToHistory} methods are called.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHistoryChanged(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:loaded loaded} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired after the image specified in the `src` property is successfully loaded.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachLoaded(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:loaded loaded} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired after the image specified in the `src` property is successfully loaded.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachLoaded(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:sizeChanged sizeChanged} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired when the size of the image is changed by the user while in the `Resize` mode ({@link sap.suite.ui.commons.ImageEditorMode}).
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSizeChanged(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ImageEditor$SizeChangedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:sizeChanged sizeChanged} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired when the size of the image is changed by the user while in the `Resize` mode ({@link sap.suite.ui.commons.ImageEditorMode}).
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSizeChanged(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ImageEditor$SizeChangedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:zoomChanged zoomChanged} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired when the zoom level is changed by the user interaction.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachZoomChanged(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ImageEditor$ZoomChangedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:zoomChanged zoomChanged} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.imageeditor.ImageEditor` itself.
     *
     * This event is fired when the zoom level is changed by the user interaction.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachZoomChanged(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ImageEditor$ZoomChangedEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.imageeditor.ImageEditor`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Applies the brightness filter to the image.
     */
    brightness(
      /**
       * Filter strength. Applicable range of values is from 0 to 500
       */
      iValue: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Cancels the current preview and restores the image to its last recorded state.
     */
    cancelPreview(): void;
    /**
     * Applies the contrast filter to the image.
     */
    contrast(
      /**
       * Amount of the filter strength. Applicable range of values is from 0 to 500
       */
      iValue: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Crops the image to a new shape based on the loaded custom shape and the specified pixel parameters.
     */
    customShapeCrop(
      /**
       * X coordinate of the top left corner of the crop area
       */
      iX: int,
      /**
       * Y coordinate of the top left corner of the crop area
       */
      iY: int,
      /**
       * Width of the crop area
       */
      iWidth: int,
      /**
       * Height of the crop area
       */
      iHeight: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:cropAreaChanged cropAreaChanged} event of
     * this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachCropAreaChanged(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ImageEditor$CropAreaChangedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:customShapeError customShapeError} event of
     * this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachCustomShapeError(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:customShapeLoaded customShapeLoaded} event
     * of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachCustomShapeLoaded(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:error error} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachError(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:historyChanged historyChanged} event of this
     * `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachHistoryChanged(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:loaded loaded} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachLoaded(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:sizeChanged sizeChanged} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachSizeChanged(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ImageEditor$SizeChangedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:zoomChanged zoomChanged} event of this `sap.suite.ui.commons.imageeditor.ImageEditor`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachZoomChanged(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ImageEditor$ZoomChangedEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Crops the image to a new circular or oval shape based on the specified pixel parameters.
     */
    ellipseCrop(
      /**
       * X coordinate of the center of the crop area
       */
      iX: int,
      /**
       * Y coordinate of the center of the crop area
       */
      iY: int,
      /**
       * X radius of the crop area
       */
      iXRadius: int,
      /**
       * Y radius of the crop area
       */
      iYRadius: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Fires event {@link #event:cropAreaChanged cropAreaChanged} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireCropAreaChanged(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ImageEditor$CropAreaChangedEventParameters
    ): this;
    /**
     * Fires event {@link #event:customShapeError customShapeError} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireCustomShapeError(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:customShapeLoaded customShapeLoaded} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireCustomShapeLoaded(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:error error} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireError(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:historyChanged historyChanged} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireHistoryChanged(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:loaded loaded} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireLoaded(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:sizeChanged sizeChanged} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireSizeChanged(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ImageEditor$SizeChangedEventParameters
    ): this;
    /**
     * Fires event {@link #event:zoomChanged zoomChanged} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireZoomChanged(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ImageEditor$ZoomChangedEventParameters
    ): this;
    /**
     * Flips the image based on the specified parameters.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    flip(
      /**
       * Whether to flip the image vertically
       */
      bVertical: boolean,
      /**
       * Whether to flip the image horizontally
       */
      bHorizontal: boolean,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): ImageEditor;
    /**
     * Flips the whole image horizontally.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    flipHorizontal(
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): ImageEditor;
    /**
     * Flips the whole image vertically.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    flipVertical(
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): ImageEditor;
    /**
     * Gets the current crop area position and size.
     *
     *
     * @returns Crop area with properties x, y, width, and height
     */
    getCropArea(): {
      x: int;

      y: int;

      width: int;

      height: int;
    };
    /**
     * Returns `true` when the `src` property is set and the specified image has been successfuly loaded.
     *
     *
     * @returns Whether the image is loaded or not
     */
    getCustomShapeLoaded(): boolean;
    /**
     * Gets current value of property {@link #getCustomShapeSrc customShapeSrc}.
     *
     * Image source for the custom shape used in custom shape cropping.
     * Can be a file or Blob object, a URL address of the image, a Base64 data URL string, or any other source
     * supported by the <img> HTML tag.
     * **Note:** In Internet Explorer, `customShape` doesn't support SVG files and will throw and exception
     * if SVG files are used.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `customShapeSrc`
     */
    getCustomShapeSrc(): any;
    /**
     * Returns the name of the image file if a file was specified in the`src` property and its name is available.
     * Otherwise returns an empty string.
     *
     *
     * @returns File name
     */
    getFileName(): string;
    /**
     * Returns the height of the image.
     *
     *
     * @returns height of the last non-preview image
     */
    getHeight(): int;
    /**
     * Returns an array of all action history items.
     *
     *
     * @returns Array of action history items
     */
    getHistory(): HistoryItem[];
    /**
     * Returns the blob object containing the image in a specified format. If the original format is not supported
     * by the browser's `canvas.toDataURL` method, either the original format of the image is used or a PNG
     * is used.
     *
     *
     * @returns Promise that resolves into the blob object
     */
    getImageAsBlob(
      /**
       * File format of the returned image blob. If the selected format is not supported by the browser's `canvas.toDataURL`
       * method, PNG is returned instead.
       */
      sFormat?: ImageFormat | keyof typeof ImageFormat,
      /**
       * Only applied when JPEG format is used. Quality of the returned image, ranging from 0 to 1. 0 implies
       * highly compressed image and 1 implies uncompressed image. When no value is provided, canvas default compression
       * level is used.
       */
      fQuality?: float
    ): Promise<Blob>;
    /**
     * Returns Base64 Data URL of the image in the specified format. If the format is not specified, either
     * the original format of the image is used or PNG, if the original format is not supported by the browser's
     * `canvas.toDataURL` method.
     *
     *
     * @returns Base64 Data URL
     */
    getImageDataURL(
      /**
       * File format of the returned Data URL. If the selected format is not suppoirted by the browser's `canvas.toDataURL`
       * method, PNG is returned instead.
       */
      sFormat?: ImageFormat | keyof typeof ImageFormat,
      /**
       * Only applied when JPEG format is used. Quality of the returned image, ranging from 0 to 1. 0 means highly
       * compressed image, 1 means uncompressed image. When no value is provided, canvas default compression level
       * is used.
       */
      fQuality?: float
    ): string;
    /**
     * Returns Base64 Data URL of the image in JPEG format.
     *
     *
     * @returns Base64 Data URL
     */
    getImageJpegDataURL(
      /**
       * Quality of the returned image, ranging from 0 to 1. 0 means highly compressed image, 1 means uncompressed
       * image. When no value is provided, canvas default compression level is used.
       */
      fQuality?: float
    ): string;
    /**
     * Returns Base64 Data URL of the image in PNG format.
     *
     *
     * @returns Base64 Data URL
     */
    getImagePngDataURL(): string;
    /**
     * Gets current value of property {@link #getKeepCropAspectRatio keepCropAspectRatio}.
     *
     * Defines whether aspect ratio should be kept when the crop area is expanded or shrunk in the `CropRectangle`
     * or `CropEllipse` modes ({@link sap.suite.ui.commons.ImageEditorMode}).
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `keepCropAspectRatio`
     */
    getKeepCropAspectRatio(): boolean;
    /**
     * Gets current value of property {@link #getKeepResizeAspectRatio keepResizeAspectRatio}.
     *
     * Defines whether aspect ratio should be kept when the image is resized in the `Resize` mode ({@link sap.suite.ui.commons.ImageEditorMode})
     * or using the {@link #setWidth} or {@link #setHeight} methods.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `keepResizeAspectRatio`
     */
    getKeepResizeAspectRatio(): boolean;
    /**
     * Returns `true` when the `customShapeSrc` property is set and the specified image has been successfuly
     * loaded.
     *
     *
     * @returns Whether the image is loaded or not
     */
    getLoaded(): boolean;
    /**
     * Gets current value of property {@link #getMode mode}.
     *
     * Image editor mode.
     * Four modes are available: resize mode, crop to rectangle mode, crop to circle mode, and display image
     * mode (default).
     * The default mode displays the image to be edited.
     *
     * Default value is `Default`.
     *
     *
     * @returns Value of property `mode`
     */
    getMode(): ImageEditorMode | keyof typeof ImageEditorMode;
    /**
     * Returns the height of the currently shown image, even if it is in preview state.
     *
     *
     * @returns height of the current image, even in preview state
     */
    getPreviewHeight(): int;
    /**
     * Gets the width of the currently shown image, even if it is in preview state.
     *
     *
     * @returns width of the current image, even when in preview state
     */
    getPreviewWidth(): int;
    /**
     * Gets current value of property {@link #getScaleCropArea scaleCropArea}.
     *
     * Defines the behavior of the crop area when zooming in or out.
     * If set to `true`, the crop area is scaled up or down when the zoom level is changed.
     * If set to `false`, the crop area remains unchanged and keeps its size.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `scaleCropArea`
     */
    getScaleCropArea(): boolean;
    /**
     * Gets current value of property {@link #getSrc src}.
     *
     * Image source.
     * Can be a file or Blob object, a URL address of the image, a Base64 data URL string, or any other source
     * supported by the <img> HTML tag.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `src`
     */
    getSrc(): any;
    /**
     * Gets current value of property {@link #getUseExifOrientation useExifOrientation}.
     *
     * Defines the behaviour where if the property is set to true, the images are displayed as per the exif
     * orientation
     *
     * Default value is `false`.
     *
     * @since 1.118.0
     *
     * @returns Value of property `useExifOrientation`
     */
    getUseExifOrientation(): boolean;
    /**
     * Gets the width of the image.
     *
     *
     * @returns width of the last non-preview image
     */
    getWidth(): int;
    /**
     * Returns the current zoom level.
     *
     *
     * @returns Current zoom level
     */
    getZoomLevel(): int;
    /**
     * Applies the grayscale filter to the image.
     */
    grayscale(
      /**
       * Filter strength. Applicable range of values is from 0 to 100
       */
      iValue: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Inverts the colors of the image.
     */
    invert(
      /**
       * Filter strength. Applicable range of values is from 0 to 100
       */
      iValue: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Changes the state to an action history item at the given index.
     */
    jumpToHistory(
      /**
       * Index of the history item. Newest state has the index of 0, the most recent change has the index of 1.
       * The index increases with each preceding action.
       */
      iIndex: int
    ): void;
    /**
     * Opens Save As dialog in browser.
     */
    openSaveDialog(
      /**
       * File name prefilled in the the save as dialog. Default name is "image".
       */
      sFileName?: string,
      /**
       * File format of the returned image Blob. If the format is not specified, either the original format of
       * the image is used or PNG, if the original format is not supported by the browser's `canvas.toDataURL`
       * method.
       */
      sFormat?: ImageFormat | keyof typeof ImageFormat,
      /**
       * Only applied when JPEG format is used. Quality of the returned image, ranging from 0 to 1. 0 means highly
       * compressed image, 1 means uncompressed image. When no value is provided, canvas default compression level
       * is used.
       */
      fQuality?: float
    ): void;
    /**
     * Crops the image to a new rectangle-shaped image based on the specified pixel parameters.
     */
    rectangleCrop(
      /**
       * X coordinate of the top left corner of the crop area
       */
      iX: int,
      /**
       * Y coordinate of the top left corner of the crop area
       */
      iY: int,
      /**
       * Width of the crop area
       */
      iWidth: int,
      /**
       * Height of the crop area
       */
      iHeight: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Goes one action forward in the action history, if possible.
     */
    redo(): void;
    /**
     * Rotates the image to the specified angle, ranging from -360 to 360 degrees.
     * Use a positive value for clockwise rotation and a negative value for counterclockwise rotation.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    rotate(
      /**
       * Number of degrees to rotate the image. Applicable range of values is from -360 to 360
       */
      iDegrees: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): ImageEditor;
    /**
     * Applies the saturation filter to the image.
     */
    saturate(
      /**
       * Amount of the filter strength. Applicable range of values is from 0 to 500
       */
      iValue: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Applies the sepia filter to the image.
     */
    sepia(
      /**
       * Filter strength. Applicable range of values is from 0 to 100
       */
      iValue: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): void;
    /**
     * Sets the crop area to the position and size.
     * The size is expected to be in pixel values corresponding to the actual size of the image.
     * Crop area is automatically constrained, so that it doesn't overflow the edges of the image.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    setCropArea(
      /**
       * X coordinate of the top left corner of the crop area
       */
      iX: int,
      /**
       * Y coordinate of the top left corner of the crop area
       */
      iY: int,
      /**
       * Width of the crop area
       */
      iWidth: int,
      /**
       * Height of the crop area
       */
      iHeight: int
    ): ImageEditor;
    /**
     * Sets the size of the crop area to comply with the specified aspect ratio.
     * Centers the crop area, so that there is always some space from the edges of the image.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    setCropAreaByRatio(
      /**
       * Ratio width
       */
      iWidth: int,
      /**
       * Ratio height
       */
      iHeight: int
    ): ImageEditor;
    /**
     * Resizes the crop area to the specified height and width.
     * The size is expected to be in pixel values corresponding to the actual size of the image.
     * Crop area is automatically constrained, so that it doesn't overflow the edges of the image.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    setCropAreaBySize(
      /**
       * Width of the cropping area
       */
      iWidth: int,
      /**
       * Height of the cropping area
       */
      iHeight: int
    ): ImageEditor;
    /**
     * Sets the size of the crop area to comply with the aspect ratio of the loaded custom shape.
     * Centers the crop area, so that there is always some space from the edges of the image.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    setCustomShapeCropAreaRatio(): ImageEditor;
    /**
     * Sets a new value for property {@link #getCustomShapeSrc customShapeSrc}.
     *
     * Image source for the custom shape used in custom shape cropping.
     * Can be a file or Blob object, a URL address of the image, a Base64 data URL string, or any other source
     * supported by the <img> HTML tag.
     * **Note:** In Internet Explorer, `customShape` doesn't support SVG files and will throw and exception
     * if SVG files are used.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCustomShapeSrc(
      /**
       * New value for property `customShapeSrc`
       */
      oCustomShapeSrc?: any
    ): this;
    /**
     * Sets the height of the image in pixels.
     * If the property `keepResizeAspectRatio` is set to `true`, changes the current width as well (even in
     * the preview mode).
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    setHeight(
      /**
       * New height of the image in pixels
       */
      iHeight: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): ImageEditor;
    /**
     * Sets a new value for property {@link #getKeepCropAspectRatio keepCropAspectRatio}.
     *
     * Defines whether aspect ratio should be kept when the crop area is expanded or shrunk in the `CropRectangle`
     * or `CropEllipse` modes ({@link sap.suite.ui.commons.ImageEditorMode}).
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKeepCropAspectRatio(
      /**
       * New value for property `keepCropAspectRatio`
       */
      bKeepCropAspectRatio?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getKeepResizeAspectRatio keepResizeAspectRatio}.
     *
     * Defines whether aspect ratio should be kept when the image is resized in the `Resize` mode ({@link sap.suite.ui.commons.ImageEditorMode})
     * or using the {@link #setWidth} or {@link #setHeight} methods.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKeepResizeAspectRatio(
      /**
       * New value for property `keepResizeAspectRatio`
       */
      bKeepResizeAspectRatio?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getMode mode}.
     *
     * Image editor mode.
     * Four modes are available: resize mode, crop to rectangle mode, crop to circle mode, and display image
     * mode (default).
     * The default mode displays the image to be edited.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Default`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMode(
      /**
       * New value for property `mode`
       */
      sMode?: ImageEditorMode | keyof typeof ImageEditorMode
    ): this;
    /**
     * Sets a new value for property {@link #getScaleCropArea scaleCropArea}.
     *
     * Defines the behavior of the crop area when zooming in or out.
     * If set to `true`, the crop area is scaled up or down when the zoom level is changed.
     * If set to `false`, the crop area remains unchanged and keeps its size.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setScaleCropArea(
      /**
       * New value for property `scaleCropArea`
       */
      bScaleCropArea?: boolean
    ): this;
    /**
     * Sets the size of the image in pixels.
     *
     *
     * @returns Instance of the ImageEditor for method chaining
     */
    setSize(
      /**
       * New width of the image
       */
      iWidth: int,
      /**
       * New height of the image
       */
      iHeight: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): ImageEditor;
    /**
     * Sets a new value for property {@link #getSrc src}.
     *
     * Image source.
     * Can be a file or Blob object, a URL address of the image, a Base64 data URL string, or any other source
     * supported by the <img> HTML tag.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSrc(
      /**
       * New value for property `src`
       */
      oSrc?: any
    ): this;
    /**
     * Sets a new value for property {@link #getUseExifOrientation useExifOrientation}.
     *
     * Defines the behaviour where if the property is set to true, the images are displayed as per the exif
     * orientation
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     * @since 1.118.0
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUseExifOrientation(
      /**
       * New value for property `useExifOrientation`
       */
      bUseExifOrientation?: boolean
    ): this;
    /**
     * Sets the width of the image in pixels.
     * If the property `keepResizeAspectRatio` is set to `true`, changes the current image height as well (even
     * in preview state).
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    setWidth(
      /**
       * New width of the image in pixels
       */
      iWidth: int,
      /**
       * Indicates whether to show this action in preview only (`true`) or record it in action history as well
       * (`false`)
       */
      bPreview: boolean
    ): ImageEditor;
    /**
     * Goes one action back in the action history, if possible.
     */
    undo(): void;
    /**
     * Zooms in or out to the specified zoom level.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    zoom(
      /**
       * Level of zoom
       */
      fZoom: float,
      /**
       * Map which contains following parameters properties:
       */
      mParameters?: {
        /**
         * x position in the image from/to which the zoom should happen
         */
        x?: number;
        /**
         * y position in the image from/to which the zoom should happen
         */
        y?: number;
      }
    ): ImageEditor;
    /**
     * Zooms in one step closer.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    zoomIn(
      /**
       * Map that contains following parameters properties:
       */
      mParameters?: {
        /**
         * x position from/to which the zoom should happen
         */
        x?: number;
        /**
         * y position from/to which the zoom should happen
         */
        y?: number;
      }
    ): ImageEditor;
    /**
     * Zooms out one step further.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    zoomOut(
      /**
       * Map that contains following parameters properties:
       */
      mParameters?: {
        /**
         * x position from/to which the zoom should happen
         */
        x?: number;
        /**
         * y position from/to which the zoom should happen
         */
        y?: number;
      }
    ): ImageEditor;
    /**
     * Fits the image into the currently visible area. This method can only be called after the image is loaded
     * and rendered.
     *
     *
     * @returns instance of the ImageEditor for method chaining
     */
    zoomToFit(): ImageEditor;
  }
  /**
   * Describes the settings that can be provided to the ImageEditor constructor.
   */
  export interface $ImageEditorSettings extends $ControlSettings {
    /**
     * Image source.
     * Can be a file or Blob object, a URL address of the image, a Base64 data URL string, or any other source
     * supported by the <img> HTML tag.
     */
    src?: any | PropertyBindingInfo | `{${string}}`;

    /**
     * Image source for the custom shape used in custom shape cropping.
     * Can be a file or Blob object, a URL address of the image, a Base64 data URL string, or any other source
     * supported by the <img> HTML tag.
     * **Note:** In Internet Explorer, `customShape` doesn't support SVG files and will throw and exception
     * if SVG files are used.
     */
    customShapeSrc?: any | PropertyBindingInfo | `{${string}}`;

    /**
     * Image editor mode.
     * Four modes are available: resize mode, crop to rectangle mode, crop to circle mode, and display image
     * mode (default).
     * The default mode displays the image to be edited.
     */
    mode?:
      | (ImageEditorMode | keyof typeof ImageEditorMode)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Defines whether aspect ratio should be kept when the crop area is expanded or shrunk in the `CropRectangle`
     * or `CropEllipse` modes ({@link sap.suite.ui.commons.ImageEditorMode}).
     */
    keepCropAspectRatio?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether aspect ratio should be kept when the image is resized in the `Resize` mode ({@link sap.suite.ui.commons.ImageEditorMode})
     * or using the {@link #setWidth} or {@link #setHeight} methods.
     */
    keepResizeAspectRatio?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the behavior of the crop area when zooming in or out.
     * If set to `true`, the crop area is scaled up or down when the zoom level is changed.
     * If set to `false`, the crop area remains unchanged and keeps its size.
     */
    scaleCropArea?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines the behaviour where if the property is set to true, the images are displayed as per the exif
     * orientation
     *
     * @since 1.118.0
     */
    useExifOrientation?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * This event is fired after the image specified in the `src` property is successfully loaded.
     */
    loaded?: (oEvent: Event) => void;

    /**
     * This event is fired if there is an error loading the image specified in the `src` property, for example,
     * when the URL is unreachable.
     */
    error?: (oEvent: Event) => void;

    /**
     * This event is fired after the image specified in the `customShapeSrc` property is successfully loaded.
     */
    customShapeLoaded?: (oEvent: Event) => void;

    /**
     * This event is fired if there is an error loading the image specified in the `customShapeSrc` property,
     * for example, when the URL is unreachable.
     */
    customShapeError?: (oEvent: Event) => void;

    /**
     * This event is fired when the size of the image is changed by the user while in the `Resize` mode ({@link sap.suite.ui.commons.ImageEditorMode}).
     */
    sizeChanged?: (oEvent: ImageEditor$SizeChangedEvent) => void;

    /**
     * This event is fired when the size of the crop area is changed by the user while in the `CropRectangle`
     * or `CropEllipse` mode ({@link sap.suite.ui.commons.ImageEditorMode}).
     */
    cropAreaChanged?: (oEvent: ImageEditor$CropAreaChangedEvent) => void;

    /**
     * This event is fired when the zoom level is changed by the user interaction.
     */
    zoomChanged?: (oEvent: ImageEditor$ZoomChangedEvent) => void;

    /**
     * This event is fired each time there is a change in the history of recent actions or in the recent actions
     * history index.
     * This may happen either when an action is performed on the image or when the {@link #undo}, {@link #redo},
     * or {@link #jumpToHistory} methods are called.
     */
    historyChanged?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the ImageEditor#cropAreaChanged event.
   */
  export interface ImageEditor$CropAreaChangedEventParameters {
    cropArea?: object;

    originalCropArea?: object;
  }

  /**
   * Event object of the ImageEditor#cropAreaChanged event.
   */
  export type ImageEditor$CropAreaChangedEvent = Event<
    ImageEditor$CropAreaChangedEventParameters,
    ImageEditor
  >;

  /**
   * Parameters of the ImageEditor#customShapeError event.
   */
  export interface ImageEditor$CustomShapeErrorEventParameters {}

  /**
   * Event object of the ImageEditor#customShapeError event.
   */
  export type ImageEditor$CustomShapeErrorEvent = Event<
    ImageEditor$CustomShapeErrorEventParameters,
    ImageEditor
  >;

  /**
   * Parameters of the ImageEditor#customShapeLoaded event.
   */
  export interface ImageEditor$CustomShapeLoadedEventParameters {}

  /**
   * Event object of the ImageEditor#customShapeLoaded event.
   */
  export type ImageEditor$CustomShapeLoadedEvent = Event<
    ImageEditor$CustomShapeLoadedEventParameters,
    ImageEditor
  >;

  /**
   * Parameters of the ImageEditor#error event.
   */
  export interface ImageEditor$ErrorEventParameters {}

  /**
   * Event object of the ImageEditor#error event.
   */
  export type ImageEditor$ErrorEvent = Event<
    ImageEditor$ErrorEventParameters,
    ImageEditor
  >;

  /**
   * Parameters of the ImageEditor#historyChanged event.
   */
  export interface ImageEditor$HistoryChangedEventParameters {}

  /**
   * Event object of the ImageEditor#historyChanged event.
   */
  export type ImageEditor$HistoryChangedEvent = Event<
    ImageEditor$HistoryChangedEventParameters,
    ImageEditor
  >;

  /**
   * Parameters of the ImageEditor#loaded event.
   */
  export interface ImageEditor$LoadedEventParameters {}

  /**
   * Event object of the ImageEditor#loaded event.
   */
  export type ImageEditor$LoadedEvent = Event<
    ImageEditor$LoadedEventParameters,
    ImageEditor
  >;

  /**
   * Parameters of the ImageEditor#sizeChanged event.
   */
  export interface ImageEditor$SizeChangedEventParameters {
    width?: int;

    height?: int;

    originalWidth?: int;

    originalHeight?: int;
  }

  /**
   * Event object of the ImageEditor#sizeChanged event.
   */
  export type ImageEditor$SizeChangedEvent = Event<
    ImageEditor$SizeChangedEventParameters,
    ImageEditor
  >;

  /**
   * Parameters of the ImageEditor#zoomChanged event.
   */
  export interface ImageEditor$ZoomChangedEventParameters {
    zoom?: int;
  }

  /**
   * Event object of the ImageEditor#zoomChanged event.
   */
  export type ImageEditor$ZoomChangedEvent = Event<
    ImageEditor$ZoomChangedEventParameters,
    ImageEditor
  >;
}

declare module "sap/suite/ui/commons/imageeditor/ImageEditorContainer" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import CustomSizeItem from "sap/suite/ui/commons/imageeditor/CustomSizeItem";

  import {
    ImageEditorContainerButton,
    ImageEditorContainerMode,
  } from "sap/suite/ui/commons/library";

  import ImageEditor from "sap/suite/ui/commons/imageeditor/ImageEditor";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import OverflowToolbar from "sap/m/OverflowToolbar";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * This control acts as a wrapper around the {@link sap.suite.ui.commons.imageeditor.ImageEditor} control.
   * It provides additional image editing capabilities for the convenience of your users.
   *
   * Toolbar Customization: There are two ways to customize the toolbar, but they cannot be used together:
   *
   * 	 - Use `enabledButtons property to select the order and visibility of the buttons provided by
   *     the ImageEditorContainer` control. You can use the `customToolbarControls` aggregation to add custom
   *     controls that will be rendered in the toolbar.
   * 	 - Use the {@link #getToolbar} method together with the {@link #getToolbarIds} method to customize the
   *     toolbar as you need.
   * The changes made to the toolbar will be kept unless some of the methods from previous approach are used.
   * The previous approach involves rerendering of the toolbar, which causes all changes to be lost. **Note:**
   * If you use this approach, make sure that you make your changes to the mobile toolbars as well, so that
   * your application can be used on mobile devices.
   * There are three mobile toolbars available:
   * 	The header toolbar that can be customized using the method {@link #getMobileHeaderToolbar} and is intended
   * for custom buttons.
   * 	 - The zoom toolbar contains zoom functionality that can be customized using the methods {@link #getMobileZoomToolbar }
   *     and {@link #getMobileZoomToolbarIds}.
   * 	 - The footer toolbar that usually includes the `ImageEditorContainer` buttons and can be customized
   *     using the methods {@link #getMobileFooterToolbar} and {@link #getMobileFooterToolbarIds}.
   *
   *
   * @since 1.66.0
   */
  export default class ImageEditorContainer extends Control {
    /**
     * Constructor for a new ImageEditorContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ImageEditorContainerSettings
    );
    /**
     * Constructor for a new ImageEditorContainer.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ImageEditorContainerSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.ImageEditorContainer with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ImageEditorContainer>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.ImageEditorContainer.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some customEllipseCropItem to the aggregation {@link #getCustomEllipseCropItems customEllipseCropItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addCustomEllipseCropItem(
      /**
       * The customEllipseCropItem to add; if empty, nothing is inserted
       */
      oCustomEllipseCropItem: CustomSizeItem
    ): this;
    /**
     * Adds some customRectangleCropItem to the aggregation {@link #getCustomRectangleCropItems customRectangleCropItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addCustomRectangleCropItem(
      /**
       * The customRectangleCropItem to add; if empty, nothing is inserted
       */
      oCustomRectangleCropItem: CustomSizeItem
    ): this;
    /**
     * Adds some customResizeItem to the aggregation {@link #getCustomResizeItems customResizeItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addCustomResizeItem(
      /**
       * The customResizeItem to add; if empty, nothing is inserted
       */
      oCustomResizeItem: CustomSizeItem
    ): this;
    /**
     * Adds some customToolbarControl to the aggregation {@link #getCustomToolbarControls customToolbarControls}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addCustomToolbarControl(
      /**
       * The customToolbarControl to add; if empty, nothing is inserted
       */
      oCustomToolbarControl: Control
    ): this;
    /**
     * Destroys all the customEllipseCropItems in the aggregation {@link #getCustomEllipseCropItems customEllipseCropItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCustomEllipseCropItems(): this;
    /**
     * Destroys all the customRectangleCropItems in the aggregation {@link #getCustomRectangleCropItems customRectangleCropItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCustomRectangleCropItems(): this;
    /**
     * Destroys all the customResizeItems in the aggregation {@link #getCustomResizeItems customResizeItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCustomResizeItems(): this;
    /**
     * Destroys all the customToolbarControls in the aggregation {@link #getCustomToolbarControls customToolbarControls}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCustomToolbarControls(): this;
    /**
     * Destroys the imageEditor in the aggregation {@link #getImageEditor imageEditor}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyImageEditor(): this;
    /**
     * Gets content of aggregation {@link #getCustomEllipseCropItems customEllipseCropItems}.
     *
     * Custom items for Ellipse Crop option panel.
     */
    getCustomEllipseCropItems(): CustomSizeItem[];
    /**
     * Gets content of aggregation {@link #getCustomRectangleCropItems customRectangleCropItems}.
     *
     * Custom items for Rectangle Crop option panel.
     */
    getCustomRectangleCropItems(): CustomSizeItem[];
    /**
     * Gets content of aggregation {@link #getCustomResizeItems customResizeItems}.
     *
     * Custom items for Resize option panel.
     */
    getCustomResizeItems(): CustomSizeItem[];
    /**
     * Gets content of aggregation {@link #getCustomToolbarControls customToolbarControls}.
     *
     * Custom controls that are added into the toolbar of `ImageEditorContainer`. Only use controls that are
     * supported by {@link sap.m.OverflowToolbar}.
     */
    getCustomToolbarControls(): Control[];
    /**
     * Gets current value of property {@link #getEnabledButtons enabledButtons}.
     *
     * Controls order and availability of the buttons in the header panel of ImageEditorContainer.
     *
     * Default value is `[Filter, Transform, Crop]`.
     *
     *
     * @returns Value of property `enabledButtons`
     */
    getEnabledButtons(): Array<
      ImageEditorContainerButton | keyof typeof ImageEditorContainerButton
    >;
    /**
     * Gets content of aggregation {@link #getImageEditor imageEditor}.
     *
     * {@link sap.suite.ui.commons.imageeditor.ImageEditor} containing the image for editing.
     * If no {@link sap.suite.ui.commons.imageeditor.ImageEditor} is specified, the `ImageEditorContainer` has
     * no effect.
     */
    getImageEditor(): ImageEditor;
    /**
     * Returns the mobile footer toolbar of the `ImageEditorContainer` for customization of its content.
     *
     *
     * @returns Mobile footer toolbar
     */
    getMobileFooterToolbar(): OverflowToolbar;
    /**
     * Returns IDs of the default items in the mobile footer toolbar. Can be used for toolbar customization.
     *
     *
     * @returns IDs of the default items in the header toolbar
     */
    getMobileFooterToolbarIds(): {
      filter: string;

      crop: string;

      transform: string;

      undo: string;

      history: string;

      apply: string;
    };
    /**
     * Returns the mobile header toolbar of the `ImageEditorContainer` for customization of its content.
     *
     *
     * @returns Mobile header toolbar
     */
    getMobileHeaderToolbar(): OverflowToolbar;
    /**
     * Returns the mobile zoom toolbar of the `ImageEditorContainer` for customization of its content.
     *
     * @since 1.68
     *
     * @returns Mobile zoom toolbar
     */
    getMobileZoomToolbar(): OverflowToolbar;
    /**
     * Returns IDs of the default items in the mobile zoom toolbar. Can be used for toolbar customization.
     *
     * @since 1.68
     *
     * @returns IDs of the default items in the header toolbar
     */
    getMobileZoomToolbarIds(): {
      zoomIn: string;

      zoomOut: string;

      zoomToFit: string;

      zoomLabel: string;

      scaleSwitch: string;

      scaleLabel: string;
    };
    /**
     * Gets current value of property {@link #getMode mode}.
     *
     * Image editor container mode.
     * Three modes are available: transform mode, crop mode, and filter mode (default).
     * The default mode provides filters for image editing: brightness, contrast, saturation, sepia grayscale,
     * and others.
     *
     * Default value is `Filter`.
     *
     *
     * @returns Value of property `mode`
     */
    getMode(): ImageEditorContainerMode | keyof typeof ImageEditorContainerMode;
    /**
     * Returns the header toolbar of the `ImageEditorContainer` for customization of its content.
     *
     *
     * @returns Header toolbar
     */
    getToolbar(): OverflowToolbar;
    /**
     * Returns IDs of the default items in the header toolbar. Can be used for toolbar customization.
     *
     *
     * @returns Ids of the default items in the header toolbar
     */
    getToolbarIds(): {
      filter: string;

      crop: string;

      transform: string;

      undo: string;

      history: string;

      zoomIn: string;

      zoomOut: string;

      zoomToFit: string;

      zoomLabel: string;

      separator: string;
    };
    /**
     * Checks for the provided `sap.suite.ui.commons.imageeditor.CustomSizeItem` in the aggregation {@link #getCustomEllipseCropItems customEllipseCropItems}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfCustomEllipseCropItem(
      /**
       * The customEllipseCropItem whose index is looked for
       */
      oCustomEllipseCropItem: CustomSizeItem
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.imageeditor.CustomSizeItem` in the aggregation {@link #getCustomRectangleCropItems customRectangleCropItems}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfCustomRectangleCropItem(
      /**
       * The customRectangleCropItem whose index is looked for
       */
      oCustomRectangleCropItem: CustomSizeItem
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.imageeditor.CustomSizeItem` in the aggregation {@link #getCustomResizeItems customResizeItems}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfCustomResizeItem(
      /**
       * The customResizeItem whose index is looked for
       */
      oCustomResizeItem: CustomSizeItem
    ): int;
    /**
     * Checks for the provided `sap.ui.core.Control` in the aggregation {@link #getCustomToolbarControls customToolbarControls}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfCustomToolbarControl(
      /**
       * The customToolbarControl whose index is looked for
       */
      oCustomToolbarControl: Control
    ): int;
    /**
     * Inserts a customEllipseCropItem into the aggregation {@link #getCustomEllipseCropItems customEllipseCropItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertCustomEllipseCropItem(
      /**
       * The customEllipseCropItem to insert; if empty, nothing is inserted
       */
      oCustomEllipseCropItem: CustomSizeItem,
      /**
       * The `0`-based index the customEllipseCropItem should be inserted at; for a negative value of `iIndex`,
       * the customEllipseCropItem is inserted at position 0; for a value greater than the current size of the
       * aggregation, the customEllipseCropItem is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a customRectangleCropItem into the aggregation {@link #getCustomRectangleCropItems customRectangleCropItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertCustomRectangleCropItem(
      /**
       * The customRectangleCropItem to insert; if empty, nothing is inserted
       */
      oCustomRectangleCropItem: CustomSizeItem,
      /**
       * The `0`-based index the customRectangleCropItem should be inserted at; for a negative value of `iIndex`,
       * the customRectangleCropItem is inserted at position 0; for a value greater than the current size of the
       * aggregation, the customRectangleCropItem is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a customResizeItem into the aggregation {@link #getCustomResizeItems customResizeItems}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertCustomResizeItem(
      /**
       * The customResizeItem to insert; if empty, nothing is inserted
       */
      oCustomResizeItem: CustomSizeItem,
      /**
       * The `0`-based index the customResizeItem should be inserted at; for a negative value of `iIndex`, the
       * customResizeItem is inserted at position 0; for a value greater than the current size of the aggregation,
       * the customResizeItem is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a customToolbarControl into the aggregation {@link #getCustomToolbarControls customToolbarControls}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertCustomToolbarControl(
      /**
       * The customToolbarControl to insert; if empty, nothing is inserted
       */
      oCustomToolbarControl: Control,
      /**
       * The `0`-based index the customToolbarControl should be inserted at; for a negative value of `iIndex`,
       * the customToolbarControl is inserted at position 0; for a value greater than the current size of the
       * aggregation, the customToolbarControl is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getCustomEllipseCropItems customEllipseCropItems}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllCustomEllipseCropItems(): CustomSizeItem[];
    /**
     * Removes all the controls from the aggregation {@link #getCustomRectangleCropItems customRectangleCropItems}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllCustomRectangleCropItems(): CustomSizeItem[];
    /**
     * Removes all the controls from the aggregation {@link #getCustomResizeItems customResizeItems}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllCustomResizeItems(): CustomSizeItem[];
    /**
     * Removes all the controls from the aggregation {@link #getCustomToolbarControls customToolbarControls}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllCustomToolbarControls(): Control[];
    /**
     * Removes a customEllipseCropItem from the aggregation {@link #getCustomEllipseCropItems customEllipseCropItems}.
     *
     *
     * @returns The removed customEllipseCropItem or `null`
     */
    removeCustomEllipseCropItem(
      /**
       * The customEllipseCropItem to remove or its index or id
       */
      vCustomEllipseCropItem: int | string | CustomSizeItem
    ): CustomSizeItem | null;
    /**
     * Removes a customRectangleCropItem from the aggregation {@link #getCustomRectangleCropItems customRectangleCropItems}.
     *
     *
     * @returns The removed customRectangleCropItem or `null`
     */
    removeCustomRectangleCropItem(
      /**
       * The customRectangleCropItem to remove or its index or id
       */
      vCustomRectangleCropItem: int | string | CustomSizeItem
    ): CustomSizeItem | null;
    /**
     * Removes a customResizeItem from the aggregation {@link #getCustomResizeItems customResizeItems}.
     *
     *
     * @returns The removed customResizeItem or `null`
     */
    removeCustomResizeItem(
      /**
       * The customResizeItem to remove or its index or id
       */
      vCustomResizeItem: int | string | CustomSizeItem
    ): CustomSizeItem | null;
    /**
     * Removes a customToolbarControl from the aggregation {@link #getCustomToolbarControls customToolbarControls}.
     *
     *
     * @returns The removed customToolbarControl or `null`
     */
    removeCustomToolbarControl(
      /**
       * The customToolbarControl to remove or its index or id
       */
      vCustomToolbarControl: int | string | Control
    ): Control | null;
    /**
     * Sets a new value for property {@link #getEnabledButtons enabledButtons}.
     *
     * Controls order and availability of the buttons in the header panel of ImageEditorContainer.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `[Filter, Transform, Crop]`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEnabledButtons(
      /**
       * New value for property `enabledButtons`
       */
      sEnabledButtons?: Array<
        ImageEditorContainerButton | keyof typeof ImageEditorContainerButton
      >
    ): this;
    /**
     * Sets the aggregated {@link #getImageEditor imageEditor}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setImageEditor(
      /**
       * The imageEditor to set
       */
      oImageEditor: ImageEditor
    ): this;
    /**
     * Sets a new value for property {@link #getMode mode}.
     *
     * Image editor container mode.
     * Three modes are available: transform mode, crop mode, and filter mode (default).
     * The default mode provides filters for image editing: brightness, contrast, saturation, sepia grayscale,
     * and others.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Filter`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMode(
      /**
       * New value for property `mode`
       */
      sMode?: ImageEditorContainerMode | keyof typeof ImageEditorContainerMode
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ImageEditorContainer constructor.
   */
  export interface $ImageEditorContainerSettings extends $ControlSettings {
    /**
     * Image editor container mode.
     * Three modes are available: transform mode, crop mode, and filter mode (default).
     * The default mode provides filters for image editing: brightness, contrast, saturation, sepia grayscale,
     * and others.
     */
    mode?:
      | (ImageEditorContainerMode | keyof typeof ImageEditorContainerMode)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Controls order and availability of the buttons in the header panel of ImageEditorContainer.
     */
    enabledButtons?:
      | Array<
          ImageEditorContainerButton | keyof typeof ImageEditorContainerButton
        >
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * {@link sap.suite.ui.commons.imageeditor.ImageEditor} containing the image for editing.
     * If no {@link sap.suite.ui.commons.imageeditor.ImageEditor} is specified, the `ImageEditorContainer` has
     * no effect.
     */
    imageEditor?: ImageEditor;

    /**
     * Custom controls that are added into the toolbar of `ImageEditorContainer`. Only use controls that are
     * supported by {@link sap.m.OverflowToolbar}.
     */
    customToolbarControls?:
      | Control[]
      | Control
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * Custom items for Resize option panel.
     */
    customResizeItems?:
      | CustomSizeItem[]
      | CustomSizeItem
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * Custom items for Rectangle Crop option panel.
     */
    customRectangleCropItems?:
      | CustomSizeItem[]
      | CustomSizeItem
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * Custom items for Ellipse Crop option panel.
     */
    customEllipseCropItems?:
      | CustomSizeItem[]
      | CustomSizeItem
      | AggregationBindingInfo
      | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/imageeditor/ImageEditorResponsiveContainer" {
  import {
    default as ImageEditorContainer,
    $ImageEditorContainerSettings,
  } from "sap/suite/ui/commons/imageeditor/ImageEditorContainer";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This control acts as a wrapper around the {@link sap.suite.ui.commons.imageeditor.ImageEditor} control.
   * It provides additional image editing capabilities for the convenience of your users.
   * This container is responsive and is rendered as a button on mobile devices.
   *
   * @since 1.68.0
   */
  export default class ImageEditorResponsiveContainer extends ImageEditorContainer {
    /**
     * Constructor for a new ImageEditorResponsiveContainer.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ImageEditorResponsiveContainerSettings
    );
    /**
     * Constructor for a new ImageEditorResponsiveContainer.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ImageEditorResponsiveContainerSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.ImageEditorResponsiveContainer with
     * name `sClassName` and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.imageeditor.ImageEditorContainer.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ImageEditorResponsiveContainer>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.ImageEditorResponsiveContainer.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the ImageEditorResponsiveContainer constructor.
   */
  export interface $ImageEditorResponsiveContainerSettings
    extends $ImageEditorContainerSettings {}
}

declare module "sap/suite/ui/commons/imageeditor/ResizeHistoryItem" {
  import HistoryItem from "sap/suite/ui/commons/imageeditor/HistoryItem";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Holds information about an {@link sap.suite.ui.commons.imageeditor.ImageEditor} action history item that
   * was created by one of the methods that change the size of the image, such as {@link sap.suite.ui.commons.imageeditor.ImageEditor#setSize},
   * {@link sap.suite.ui.commons.imageeditor.ImageEditor#setWidth}, and {@link sap.suite.ui.commons.imageeditor.ImageEditor#setHeight}.
   *
   * @since 1.66.0
   */
  export default class ResizeHistoryItem extends HistoryItem {
    /**
     * Constructor for a new ResizeHistoryItem.
     */
    constructor(
      /**
       * Property bag
       */
      mProperties: {
        /**
         * Width of the image
         */
        width: int;
        /**
         * Height of the image
         */
        height: int;
        /**
         * Previous width of the image
         */
        oldWidth: int;
        /**
         * Previous height of the image
         */
        oldHeight: int;
      }
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.ResizeHistoryItem with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.imageeditor.HistoryItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ResizeHistoryItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.ResizeHistoryItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
  }
}

declare module "sap/suite/ui/commons/imageeditor/RotateHistoryItem" {
  import HistoryItem from "sap/suite/ui/commons/imageeditor/HistoryItem";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * Holds information about an {@link sap.suite.ui.commons.imageeditor.ImageEditor} action history item that
   * was created by the {@link sap.suite.ui.commons.imageeditor.ImageEditor#rotate} method.
   *
   * @since 1.66.0
   */
  export default class RotateHistoryItem extends HistoryItem {
    /**
     * Constructor for a new RotateHistoryItem.
     */
    constructor(
      /**
       * Property bag
       */
      mProperties: {
        /**
         * Number of deegress
         */
        degrees: int;
      }
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.imageeditor.RotateHistoryItem with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.imageeditor.HistoryItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, RotateHistoryItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.imageeditor.RotateHistoryItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
  }
}

declare module "sap/suite/ui/commons/InfoTile" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { InfoTileSize, LoadState } from "sap/suite/ui/commons/library";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * The tile control that displays the title, description, footer, and customizable main area.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export default class InfoTile extends Control {
    /**
     * Constructor for a new InfoTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $InfoTileSettings
    );
    /**
     * Constructor for a new InfoTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $InfoTileSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.InfoTile with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, InfoTile>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.InfoTile.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.InfoTile`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.InfoTile` itself.
     *
     * The event is fired when the user chooses the tile.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.InfoTile` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.InfoTile`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.InfoTile` itself.
     *
     * The event is fired when the user chooses the tile.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.InfoTile` itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys the content in the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyContent(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.InfoTile`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets content of aggregation {@link #getContent content}.
     *
     * The switchable view that depends on the tile type.
     */
    getContent(): Control;
    /**
     * Gets current value of property {@link #getDescription description}.
     *
     * Shows the description of the selected tile.
     *
     *
     * @returns Value of property `description`
     */
    getDescription(): string;
    /**
     * Gets current value of property {@link #getFooter footer}.
     *
     * The footer text of the tile.
     *
     *
     * @returns Value of property `footer`
     */
    getFooter(): string;
    /**
     * Gets current value of property {@link #getSize size}.
     *
     * Updates the size of the tile. If not set then the default size is applied based on the device tile.
     *
     * Default value is `"Auto"`.
     *
     *
     * @returns Value of property `size`
     */
    getSize(): InfoTileSize | keyof typeof InfoTileSize;
    /**
     * Gets current value of property {@link #getState state}.
     *
     * Indicates the load status.
     *
     * Default value is `"Loading"`.
     *
     *
     * @returns Value of property `state`
     */
    getState(): LoadState | keyof typeof LoadState;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * The title of the tile.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets the aggregated {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setContent(
      /**
       * The content to set
       */
      oContent: Control
    ): this;
    /**
     * Sets a new value for property {@link #getDescription description}.
     *
     * Shows the description of the selected tile.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescription(
      /**
       * New value for property `description`
       */
      sDescription?: string
    ): this;
    /**
     * Sets a new value for property {@link #getFooter footer}.
     *
     * The footer text of the tile.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFooter(
      /**
       * New value for property `footer`
       */
      sFooter?: string
    ): this;
    /**
     * Sets a new value for property {@link #getSize size}.
     *
     * Updates the size of the tile. If not set then the default size is applied based on the device tile.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Auto"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSize(
      /**
       * New value for property `size`
       */
      sSize?: InfoTileSize | keyof typeof InfoTileSize
    ): this;
    /**
     * Sets a new value for property {@link #getState state}.
     *
     * Indicates the load status.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Loading"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setState(
      /**
       * New value for property `state`
       */
      sState?: LoadState | keyof typeof LoadState
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * The title of the tile.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the InfoTile constructor.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export interface $InfoTileSettings extends $ControlSettings {
    /**
     * Shows the description of the selected tile.
     */
    description?: string | PropertyBindingInfo;

    /**
     * The title of the tile.
     */
    title?: string | PropertyBindingInfo;

    /**
     * The footer text of the tile.
     */
    footer?: string | PropertyBindingInfo;

    /**
     * Updates the size of the tile. If not set then the default size is applied based on the device tile.
     */
    size?:
      | (InfoTileSize | keyof typeof InfoTileSize)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Indicates the load status.
     */
    state?:
      | (LoadState | keyof typeof LoadState)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * The switchable view that depends on the tile type.
     */
    content?: Control;

    /**
     * The event is fired when the user chooses the tile.
     */
    press?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the InfoTile#press event.
   */
  export interface InfoTile$PressEventParameters {}

  /**
   * Event object of the InfoTile#press event.
   */
  export type InfoTile$PressEvent = Event<
    InfoTile$PressEventParameters,
    InfoTile
  >;
}

declare module "sap/suite/ui/commons/JamContent" {
  import {
    default as FeedContent,
    $FeedContentSettings,
  } from "sap/m/FeedContent";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This control displays the jam content text, subheader, and numeric value in a tile.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.m.FeedContent.
   */
  export default class JamContent extends FeedContent {
    /**
     * Constructor for a new JamContent.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.m.FeedContent#constructor sap.m.FeedContent }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $JamContentSettings
    );
    /**
     * Constructor for a new JamContent.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.m.FeedContent#constructor sap.m.FeedContent }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $JamContentSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.JamContent with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.m.FeedContent.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, JamContent>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.JamContent.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the JamContent constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.m.FeedContent.
   */
  export interface $JamContentSettings extends $FeedContentSettings {}
}

declare module "sap/suite/ui/commons/KpiTile" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { ValueStatus } from "sap/suite/ui/commons/library";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control is used in UnifiedThingInspector to display object-related KPIs in a factsheet.
   *
   * @deprecated (since 1.32) - Deprecated. Numeric content or any other standard Fiori control should be
   * used instead.
   */
  export default class KpiTile extends Control {
    /**
     * Constructor for a new KpiTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $KpiTileSettings
    );
    /**
     * Constructor for a new KpiTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $KpiTileSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.KpiTile with name `sClassName` and enriches it with
     * the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, KpiTile>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.KpiTile.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getDescription description}.
     *
     * The Description field.
     *
     *
     * @returns Value of property `description`
     */
    getDescription(): string;
    /**
     * Gets current value of property {@link #getDoubleFontSize doubleFontSize}.
     *
     * If true, the value text will have 2 rem, if false - 1 rem.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `doubleFontSize`
     */
    getDoubleFontSize(): boolean;
    /**
     * Gets current value of property {@link #getValue value}.
     *
     * The Value field.
     *
     *
     * @returns Value of property `value`
     */
    getValue(): string;
    /**
     * Gets current value of property {@link #getValueScale valueScale}.
     *
     * The scale of a value.
     *
     *
     * @returns Value of property `valueScale`
     */
    getValueScale(): string;
    /**
     * Gets current value of property {@link #getValueStatus valueStatus}.
     *
     * The status color of the value. Depending on the status the tile is displayed in different colors.
     *
     * Default value is `"Neutral"`.
     *
     *
     * @returns Value of property `valueStatus`
     */
    getValueStatus(): ValueStatus | keyof typeof ValueStatus;
    /**
     * Gets current value of property {@link #getValueUnit valueUnit}.
     *
     * The percent sign, currency symbol, or unit for a value.
     *
     *
     * @returns Value of property `valueUnit`
     */
    getValueUnit(): string;
    /**
     * Sets a new value for property {@link #getDescription description}.
     *
     * The Description field.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescription(
      /**
       * New value for property `description`
       */
      sDescription?: string
    ): this;
    /**
     * Sets a new value for property {@link #getDoubleFontSize doubleFontSize}.
     *
     * If true, the value text will have 2 rem, if false - 1 rem.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDoubleFontSize(
      /**
       * New value for property `doubleFontSize`
       */
      bDoubleFontSize?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getValue value}.
     *
     * The Value field.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValue(
      /**
       * New value for property `value`
       */
      sValue?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValueScale valueScale}.
     *
     * The scale of a value.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValueScale(
      /**
       * New value for property `valueScale`
       */
      sValueScale?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValueStatus valueStatus}.
     *
     * The status color of the value. Depending on the status the tile is displayed in different colors.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Neutral"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValueStatus(
      /**
       * New value for property `valueStatus`
       */
      sValueStatus?: ValueStatus | keyof typeof ValueStatus
    ): this;
    /**
     * Sets a new value for property {@link #getValueUnit valueUnit}.
     *
     * The percent sign, currency symbol, or unit for a value.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValueUnit(
      /**
       * New value for property `valueUnit`
       */
      sValueUnit?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the KpiTile constructor.
   *
   * @deprecated (since 1.32) - Deprecated. Numeric content or any other standard Fiori control should be
   * used instead.
   */
  export interface $KpiTileSettings extends $ControlSettings {
    /**
     * The Value field.
     */
    value?: string | PropertyBindingInfo;

    /**
     * The Description field.
     */
    description?: string | PropertyBindingInfo;

    /**
     * If true, the value text will have 2 rem, if false - 1 rem.
     */
    doubleFontSize?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The percent sign, currency symbol, or unit for a value.
     */
    valueUnit?: string | PropertyBindingInfo;

    /**
     * The scale of a value.
     */
    valueScale?: string | PropertyBindingInfo;

    /**
     * The status color of the value. Depending on the status the tile is displayed in different colors.
     */
    valueStatus?:
      | (ValueStatus | keyof typeof ValueStatus)
      | PropertyBindingInfo
      | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/LaunchTile" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import { URI } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control launches a URL.
   *
   * @deprecated (since 1.25) - Deprecated.
   */
  export default class LaunchTile extends Control {
    /**
     * Constructor for a new LaunchTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $LaunchTileSettings
    );
    /**
     * Constructor for a new LaunchTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $LaunchTileSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.LaunchTile with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, LaunchTile>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.LaunchTile.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.LaunchTile`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.LaunchTile` itself.
     *
     * The event fired when the user clicks on the control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.LaunchTile` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.LaunchTile`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.LaunchTile` itself.
     *
     * The event fired when the user clicks on the control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.LaunchTile` itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.LaunchTile`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * Icon associated with the launch destination.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): URI;
    /**
     * Gets current value of property {@link #getLink link}.
     *
     * The launch destination.
     *
     *
     * @returns Value of property `link`
     */
    getLink(): URI;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * Descriptive title of the launch destination.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * Icon associated with the launch destination.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getLink link}.
     *
     * The launch destination.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLink(
      /**
       * New value for property `link`
       */
      sLink?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * Descriptive title of the launch destination.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the LaunchTile constructor.
   *
   * @deprecated (since 1.25) - Deprecated.
   */
  export interface $LaunchTileSettings extends $ControlSettings {
    /**
     * Descriptive title of the launch destination.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Icon associated with the launch destination.
     */
    icon?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * The launch destination.
     */
    link?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * The event fired when the user clicks on the control.
     */
    press?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the LaunchTile#press event.
   */
  export interface LaunchTile$PressEventParameters {}

  /**
   * Event object of the LaunchTile#press event.
   */
  export type LaunchTile$PressEvent = Event<
    LaunchTile$PressEventParameters,
    LaunchTile
  >;
}

declare module "sap/suite/ui/commons/LinkActionSheet" {
  import {
    default as ActionSheet,
    $ActionSheetSettings,
  } from "sap/m/ActionSheet";

  import Control from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { AggregationBindingInfo } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * This control contains one or more sap.m.Button controls or sap.ui.commons.Link controls. The LinkActionSheet
   * control is closed if the user chooses one of the buttons or links. It looks similar to sap.m.Dialog in
   * iPhone and Android, and to sap.m.Popover in iPad.
   *
   * @deprecated (since 1.32) - Deprecated. Object page should be used instead.
   */
  export default class LinkActionSheet extends ActionSheet {
    /**
     * Constructor for a new LinkActionSheet.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $LinkActionSheetSettings
    );
    /**
     * Constructor for a new LinkActionSheet.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $LinkActionSheetSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.LinkActionSheet with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.m.ActionSheet.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, LinkActionSheet>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.LinkActionSheet.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some item to the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addItem(
      /**
       * The item to add; if empty, nothing is inserted
       */
      oItem: Control
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:itemPress itemPress} event of this `sap.suite.ui.commons.LinkActionSheet`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.LinkActionSheet` itself.
     *
     * The event is fired when the user chooses any item.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachItemPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: LinkActionSheet$ItemPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.LinkActionSheet`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:itemPress itemPress} event of this `sap.suite.ui.commons.LinkActionSheet`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.LinkActionSheet` itself.
     *
     * The event is fired when the user chooses any item.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachItemPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: LinkActionSheet$ItemPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.LinkActionSheet`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys all the items in the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyItems(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:itemPress itemPress} event of this `sap.suite.ui.commons.LinkActionSheet`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachItemPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: LinkActionSheet$ItemPressEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:itemPress itemPress} to attached listeners.
     *
     * Listeners may prevent the default action of this event by calling the `preventDefault` method on the
     * event object. The return value of this method indicates whether the default action should be executed.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether or not to prevent the default action
     */
    fireItemPress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: LinkActionSheet$ItemPressEventParameters
    ): boolean;
    /**
     * Gets content of aggregation {@link #getItems items}.
     *
     * These buttons or links are added to the content area in the LinkActionSheet control. If the user chooses
     * a button or a link, the LinkActionSheet is closed before the event listener is called.
     */
    getItems(): Control[];
    /**
     * Checks for the provided `sap.ui.core.Control` in the aggregation {@link #getItems items}. and returns
     * its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfItem(
      /**
       * The item whose index is looked for
       */
      oItem: Control
    ): int;
    /**
     * Inserts a item into the aggregation {@link #getItems items}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertItem(
      /**
       * The item to insert; if empty, nothing is inserted
       */
      oItem: Control,
      /**
       * The `0`-based index the item should be inserted at; for a negative value of `iIndex`, the item is inserted
       * at position 0; for a value greater than the current size of the aggregation, the item is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getItems items}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllItems(): Control[];
    /**
     * Removes a item from the aggregation {@link #getItems items}.
     *
     *
     * @returns The removed item or `null`
     */
    removeItem(
      /**
       * The item to remove or its index or id
       */
      vItem: int | string | Control
    ): Control | null;
  }
  /**
   * Describes the settings that can be provided to the LinkActionSheet constructor.
   *
   * @deprecated (since 1.32) - Deprecated. Object page should be used instead.
   */
  export interface $LinkActionSheetSettings extends $ActionSheetSettings {
    /**
     * These buttons or links are added to the content area in the LinkActionSheet control. If the user chooses
     * a button or a link, the LinkActionSheet is closed before the event listener is called.
     */
    items?: Control[] | Control | AggregationBindingInfo | `{${string}}`;

    /**
     * The event is fired when the user chooses any item.
     */
    itemPress?: (oEvent: LinkActionSheet$ItemPressEvent) => void;
  }

  /**
   * Parameters of the LinkActionSheet#itemPress event.
   */
  export interface LinkActionSheet$ItemPressEventParameters {
    /**
     * The object that initiated the event.
     */
    item?: Control;
  }

  /**
   * Event object of the LinkActionSheet#itemPress event.
   */
  export type LinkActionSheet$ItemPressEvent = Event<
    LinkActionSheet$ItemPressEventParameters,
    LinkActionSheet
  >;
}

declare module "sap/suite/ui/commons/MicroAreaChart" {
  import {
    default as AreaMicroChart,
    $AreaMicroChartSettings,
  } from "sap/suite/ui/microchart/AreaMicroChart";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This control displays the history of values as a line mini chart or an area mini chart.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.AreaMicroChart.
   */
  export default class MicroAreaChart extends AreaMicroChart {
    /**
     * Constructor for a new MicroAreaChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.AreaMicroChart#constructor sap.suite.ui.microchart.AreaMicroChart }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $MicroAreaChartSettings
    );
    /**
     * Constructor for a new MicroAreaChart.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.AreaMicroChart#constructor sap.suite.ui.microchart.AreaMicroChart }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $MicroAreaChartSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.MicroAreaChart with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.AreaMicroChart.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, MicroAreaChart>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.MicroAreaChart.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the MicroAreaChart constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.AreaMicroChart.
   */
  export interface $MicroAreaChartSettings extends $AreaMicroChartSettings {}
}

declare module "sap/suite/ui/commons/MicroAreaChartItem" {
  import {
    default as AreaMicroChartItem,
    $AreaMicroChartItemSettings,
  } from "sap/suite/ui/microchart/AreaMicroChartItem";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * The configuration of the graphic element on the chart.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.AreaMicroChartItem.
   */
  export default class MicroAreaChartItem extends AreaMicroChartItem {
    /**
     * Constructor for a new MicroAreaChartItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.AreaMicroChartItem#constructor sap.suite.ui.microchart.AreaMicroChartItem }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $MicroAreaChartItemSettings
    );
    /**
     * Constructor for a new MicroAreaChartItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.AreaMicroChartItem#constructor sap.suite.ui.microchart.AreaMicroChartItem }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $MicroAreaChartItemSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.MicroAreaChartItem with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.AreaMicroChartItem.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, MicroAreaChartItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.MicroAreaChartItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the MicroAreaChartItem constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.AreaMicroChartItem.
   */
  export interface $MicroAreaChartItemSettings
    extends $AreaMicroChartItemSettings {}
}

declare module "sap/suite/ui/commons/MicroAreaChartLabel" {
  import {
    default as AreaMicroChartLabel,
    $AreaMicroChartLabelSettings,
  } from "sap/suite/ui/microchart/AreaMicroChartLabel";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This element contains data for a label in MicroAreaChart control.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.AreaMicroChartLabel.
   */
  export default class MicroAreaChartLabel extends AreaMicroChartLabel {
    /**
     * Constructor for a new MicroAreaChartLabel.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.AreaMicroChartLabel#constructor sap.suite.ui.microchart.AreaMicroChartLabel }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $MicroAreaChartLabelSettings
    );
    /**
     * Constructor for a new MicroAreaChartLabel.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.AreaMicroChartLabel#constructor sap.suite.ui.microchart.AreaMicroChartLabel }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $MicroAreaChartLabelSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.MicroAreaChartLabel with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.AreaMicroChartLabel.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, MicroAreaChartLabel>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.MicroAreaChartLabel.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the MicroAreaChartLabel constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.AreaMicroChartLabel.
   */
  export interface $MicroAreaChartLabelSettings
    extends $AreaMicroChartLabelSettings {}
}

declare module "sap/suite/ui/commons/MicroAreaChartPoint" {
  import {
    default as AreaMicroChartPoint,
    $AreaMicroChartPointSettings,
  } from "sap/suite/ui/microchart/AreaMicroChartPoint";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This control contains data for the point.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.AreaMicroChartPoint.
   */
  export default class MicroAreaChartPoint extends AreaMicroChartPoint {
    /**
     * Constructor for a new MicroAreaChartPoint.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.AreaMicroChartPoint#constructor sap.suite.ui.microchart.AreaMicroChartPoint }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $MicroAreaChartPointSettings
    );
    /**
     * Constructor for a new MicroAreaChartPoint.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.microchart.AreaMicroChartPoint#constructor sap.suite.ui.microchart.AreaMicroChartPoint }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $MicroAreaChartPointSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.MicroAreaChartPoint with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.microchart.AreaMicroChartPoint.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, MicroAreaChartPoint>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.MicroAreaChartPoint.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the MicroAreaChartPoint constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.suite.ui.microchart.AreaMicroChartPoint.
   */
  export interface $MicroAreaChartPointSettings
    extends $AreaMicroChartPointSettings {}
}

declare module "sap/suite/ui/commons/MicroProcessFlow" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import { IFormContent, CSSSize } from "sap/ui/core/library";

  import MicroProcessFlowItem from "sap/suite/ui/commons/MicroProcessFlowItem";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { MicroProcessFlowRenderType } from "sap/suite/ui/commons/library";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * The MicroProcessFlow control can be used to track the progress of a process.
   *  It is best used for linear processes, such as document, order, and approval flows, and can be embedded
   * in tiles, tables, and other containers.
   *
   *  To display more complex processes and workflows, use {@link sap.suite.ui.commons.ProcessFlow} instead.
   */
  export default class MicroProcessFlow
    extends Control
    implements IFormContent {
    __implements__sap_ui_core_IFormContent: boolean;
    /**
     * Constructor for a new MicroProcessFlow.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.MicroProcessFlow with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, MicroProcessFlow>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.MicroProcessFlow.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some content to the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addContent(
      /**
       * The content to add; if empty, nothing is inserted
       */
      oContent: MicroProcessFlowItem
    ): this;
    /**
     * Destroys all the content in the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyContent(): this;
    /**
     * Gets current value of property {@link #getAriaLabel ariaLabel}.
     *
     * ARIA label for this control to be used by screen reader software.
     *
     *
     * @returns Value of property `ariaLabel`
     */
    getAriaLabel(): string;
    /**
     * Gets content of aggregation {@link #getContent content}.
     *
     * Holds the nodes and other content diplayed in the micro process flow.
     */
    getContent(): MicroProcessFlowItem[];
    /**
     * Gets current value of property {@link #getRenderType renderType}.
     *
     * Determines how the micro process flow should be rendered inside its parent container.
     * When the width of the parent container does not allow for all nodes in the micro process flow to be displayed,
     * you can wrap it inside the parent container or add scrolling icons.
     *
     * Default value is `Wrap`.
     *
     *
     * @returns Value of property `renderType`
     */
    getRenderType():
      | MicroProcessFlowRenderType
      | keyof typeof MicroProcessFlowRenderType;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * The width of the micro process flow.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Checks for the provided `sap.suite.ui.commons.MicroProcessFlowItem` in the aggregation {@link #getContent content}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfContent(
      /**
       * The content whose index is looked for
       */
      oContent: MicroProcessFlowItem
    ): int;
    /**
     * Inserts a content into the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertContent(
      /**
       * The content to insert; if empty, nothing is inserted
       */
      oContent: MicroProcessFlowItem,
      /**
       * The `0`-based index the content should be inserted at; for a negative value of `iIndex`, the content
       * is inserted at position 0; for a value greater than the current size of the aggregation, the content
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getContent content}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllContent(): MicroProcessFlowItem[];
    /**
     * Removes a content from the aggregation {@link #getContent content}.
     *
     *
     * @returns The removed content or `null`
     */
    removeContent(
      /**
       * The content to remove or its index or id
       */
      vContent: int | string | MicroProcessFlowItem
    ): MicroProcessFlowItem | null;
    /**
     * Sets a new value for property {@link #getAriaLabel ariaLabel}.
     *
     * ARIA label for this control to be used by screen reader software.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAriaLabel(
      /**
       * New value for property `ariaLabel`
       */
      sAriaLabel?: string
    ): this;
    /**
     * Sets a new value for property {@link #getRenderType renderType}.
     *
     * Determines how the micro process flow should be rendered inside its parent container.
     * When the width of the parent container does not allow for all nodes in the micro process flow to be displayed,
     * you can wrap it inside the parent container or add scrolling icons.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Wrap`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setRenderType(
      /**
       * New value for property `renderType`
       */
      sRenderType?:
        | MicroProcessFlowRenderType
        | keyof typeof MicroProcessFlowRenderType
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * The width of the micro process flow.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
  }
  /**
   * Describes the settings that can be provided to the MicroProcessFlow constructor.
   */
  export interface $MicroProcessFlowSettings extends $ControlSettings {
    /**
     * ARIA label for this control to be used by screen reader software.
     */
    ariaLabel?: string | PropertyBindingInfo;

    /**
     * Determines how the micro process flow should be rendered inside its parent container.
     * When the width of the parent container does not allow for all nodes in the micro process flow to be displayed,
     * you can wrap it inside the parent container or add scrolling icons.
     */
    renderType?:
      | (MicroProcessFlowRenderType | keyof typeof MicroProcessFlowRenderType)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * The width of the micro process flow.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Holds the nodes and other content diplayed in the micro process flow.
     */
    content?:
      | MicroProcessFlowItem[]
      | MicroProcessFlowItem
      | AggregationBindingInfo
      | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/MicroProcessFlowItem" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { ValueState, CSSSize } from "sap/ui/core/library";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * Holds information about one node in the micro process flow.
   */
  export default class MicroProcessFlowItem extends Control {
    /**
     * Constructor for a new MicroProcessFlowItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $MicroProcessFlowItemSettings
    );
    /**
     * Constructor for a new MicroProcessFlowItem.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $MicroProcessFlowItemSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.MicroProcessFlowItem with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, MicroProcessFlowItem>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.MicroProcessFlowItem.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.MicroProcessFlowItem`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.MicroProcessFlowItem` itself.
     *
     * This event is fired when the user clicks or taps a node.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: MicroProcessFlowItem$PressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.MicroProcessFlowItem`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.MicroProcessFlowItem`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.MicroProcessFlowItem` itself.
     *
     * This event is fired when the user clicks or taps a node.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: MicroProcessFlowItem$PressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.MicroProcessFlowItem`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys the customControl in the aggregation {@link #getCustomControl customControl}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCustomControl(): this;
    /**
     * Destroys the intermediary in the aggregation {@link #getIntermediary intermediary}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyIntermediary(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.MicroProcessFlowItem`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: MicroProcessFlowItem$PressEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * Listeners may prevent the default action of this event by calling the `preventDefault` method on the
     * event object. The return value of this method indicates whether the default action should be executed.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Whether or not to prevent the default action
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: MicroProcessFlowItem$PressEventParameters
    ): boolean;
    /**
     * Gets content of aggregation {@link #getCustomControl customControl}.
     *
     * Holds custom controls to be used as process flow nodes.
     */
    getCustomControl(): Control;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * Icon that is displayed inside the node.
     * By default, an icon that corresponds to the node's `state` is used.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): string;
    /**
     * Gets content of aggregation {@link #getIntermediary intermediary}.
     *
     * Holds objects to be displayed between the process flow nodes.
     */
    getIntermediary(): Control;
    /**
     * Gets current value of property {@link #getKey key}.
     *
     * Key of the node.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Gets current value of property {@link #getShowIntermediary showIntermediary}.
     *
     * Defines whether an object should be displayed between this node and the following node.
     * When set to `true`, a vertical red bar is displayed by default. To define custom objects, use the `intermediary`
     * aggregation.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `showIntermediary`
     */
    getShowIntermediary(): boolean;
    /**
     * Gets current value of property {@link #getShowSeparator showSeparator}.
     *
     * Defines whether a connector line should be displayed between this node and the node that follows it.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showSeparator`
     */
    getShowSeparator(): boolean;
    /**
     * Gets current value of property {@link #getState state}.
     *
     * State associated with this node.
     * The state defines the semantic color applied to the node. Available states include `Standard` (neutral),
     * `Error` (negative), `Success` (positive), and `Warning` (critical).
     *
     * Default value is `Standard`.
     *
     *
     * @returns Value of property `state`
     */
    getState(): ValueState | keyof typeof ValueState;
    /**
     * Gets current value of property {@link #getStepWidth stepWidth}.
     *
     * Distance between this node and the following node.
     * When set to a percentage value, the distance is calculated based on the height of the parent container.
     *
     *
     * @returns Value of property `stepWidth`
     */
    getStepWidth(): CSSSize;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * Title associated with this node.
     * The title is displayed as a tooltip when the user hovers over the node. This title can also be used by
     * screen reader software.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets the aggregated {@link #getCustomControl customControl}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCustomControl(
      /**
       * The customControl to set
       */
      oCustomControl: Control
    ): this;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * Icon that is displayed inside the node.
     * By default, an icon that corresponds to the node's `state` is used.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: string
    ): this;
    /**
     * Sets the aggregated {@link #getIntermediary intermediary}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIntermediary(
      /**
       * The intermediary to set
       */
      oIntermediary: Control
    ): this;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * Key of the node.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
    /**
     * Sets a new value for property {@link #getShowIntermediary showIntermediary}.
     *
     * Defines whether an object should be displayed between this node and the following node.
     * When set to `true`, a vertical red bar is displayed by default. To define custom objects, use the `intermediary`
     * aggregation.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowIntermediary(
      /**
       * New value for property `showIntermediary`
       */
      bShowIntermediary?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowSeparator showSeparator}.
     *
     * Defines whether a connector line should be displayed between this node and the node that follows it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowSeparator(
      /**
       * New value for property `showSeparator`
       */
      bShowSeparator?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getState state}.
     *
     * State associated with this node.
     * The state defines the semantic color applied to the node. Available states include `Standard` (neutral),
     * `Error` (negative), `Success` (positive), and `Warning` (critical).
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Standard`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setState(
      /**
       * New value for property `state`
       */
      sState?: ValueState | keyof typeof ValueState
    ): this;
    /**
     * Sets a new value for property {@link #getStepWidth stepWidth}.
     *
     * Distance between this node and the following node.
     * When set to a percentage value, the distance is calculated based on the height of the parent container.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setStepWidth(
      /**
       * New value for property `stepWidth`
       */
      sStepWidth?: CSSSize
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * Title associated with this node.
     * The title is displayed as a tooltip when the user hovers over the node. This title can also be used by
     * screen reader software.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the MicroProcessFlowItem constructor.
   */
  export interface $MicroProcessFlowItemSettings extends $ControlSettings {
    /**
     * Key of the node.
     */
    key?: string | PropertyBindingInfo;

    /**
     * Icon that is displayed inside the node.
     * By default, an icon that corresponds to the node's `state` is used.
     */
    icon?: string | PropertyBindingInfo;

    /**
     * Title associated with this node.
     * The title is displayed as a tooltip when the user hovers over the node. This title can also be used by
     * screen reader software.
     */
    title?: string | PropertyBindingInfo;

    /**
     * State associated with this node.
     * The state defines the semantic color applied to the node. Available states include `Standard` (neutral),
     * `Error` (negative), `Success` (positive), and `Warning` (critical).
     */
    state?:
      | (ValueState | keyof typeof ValueState)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Defines whether an object should be displayed between this node and the following node.
     * When set to `true`, a vertical red bar is displayed by default. To define custom objects, use the `intermediary`
     * aggregation.
     */
    showIntermediary?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Distance between this node and the following node.
     * When set to a percentage value, the distance is calculated based on the height of the parent container.
     */
    stepWidth?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines whether a connector line should be displayed between this node and the node that follows it.
     */
    showSeparator?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Holds custom controls to be used as process flow nodes.
     */
    customControl?: Control;

    /**
     * Holds objects to be displayed between the process flow nodes.
     */
    intermediary?: Control;

    /**
     * This event is fired when the user clicks or taps a node.
     */
    press?: (oEvent: MicroProcessFlowItem$PressEvent) => void;
  }

  /**
   * Parameters of the MicroProcessFlowItem#press event.
   */
  export interface MicroProcessFlowItem$PressEventParameters {
    item?: object;
  }

  /**
   * Event object of the MicroProcessFlowItem#press event.
   */
  export type MicroProcessFlowItem$PressEvent = Event<
    MicroProcessFlowItem$PressEventParameters,
    MicroProcessFlowItem
  >;
}

declare module "sap/suite/ui/commons/MonitoringContent" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import Icon from "sap/ui/core/Icon";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { InfoTileSize, LoadState } from "sap/suite/ui/commons/library";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control is used in a tile or any other place to display numeric values and an icon.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.NumericContent.
   */
  export default class MonitoringContent extends Control {
    /**
     * Constructor for a new MonitoringContent.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $MonitoringContentSettings
    );
    /**
     * Constructor for a new MonitoringContent.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $MonitoringContentSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.MonitoringContent with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, MonitoringContent>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.MonitoringContent.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.MonitoringContent`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.MonitoringContent` itself.
     *
     * The event is fired when the user chooses the monitoring content.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.MonitoringContent`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.MonitoringContent`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.MonitoringContent` itself.
     *
     * The event is fired when the user chooses the monitoring content.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.MonitoringContent`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys the icon in the aggregation {@link #getIcon icon}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyIcon(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.MonitoringContent`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getAnimateTextChange animateTextChange}.
     *
     * If set to true, changing of the value is animated.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `animateTextChange`
     */
    getAnimateTextChange(): boolean;
    /**
     * Gets content of aggregation {@link #getIcon icon}.
     *
     * The icon that is displayed in the content.
     */
    getIcon(): Icon;
    /**
     * Gets current value of property {@link #getIconSrc iconSrc}.
     *
     * This property is set by the return value of sap.ui.core.IconPool.getIconURI that is called with an icon
     * name parameter and optional collection parameter. The collection parameter is required when the application
     * extended icons are used.
     *
     *
     * @returns Value of property `iconSrc`
     */
    getIconSrc(): string;
    /**
     * Gets current value of property {@link #getSize size}.
     *
     * Updates the size of the chart. If not set then the default size is applied based on the device tile.
     *
     * Default value is `"Auto"`.
     *
     *
     * @returns Value of property `size`
     */
    getSize(): InfoTileSize | keyof typeof InfoTileSize;
    /**
     * Gets current value of property {@link #getState state}.
     *
     * Indicates the load status.
     *
     * Default value is `"Loaded"`.
     *
     *
     * @returns Value of property `state`
     */
    getState(): LoadState | keyof typeof LoadState;
    /**
     * Gets current value of property {@link #getValue value}.
     *
     * The actual value.
     *
     *
     * @returns Value of property `value`
     */
    getValue(): string;
    /**
     * Sets a new value for property {@link #getAnimateTextChange animateTextChange}.
     *
     * If set to true, changing of the value is animated.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAnimateTextChange(
      /**
       * New value for property `animateTextChange`
       */
      bAnimateTextChange?: boolean
    ): this;
    /**
     * Sets the aggregated {@link #getIcon icon}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * The icon to set
       */
      oIcon: Icon
    ): this;
    /**
     * Sets a new value for property {@link #getIconSrc iconSrc}.
     *
     * This property is set by the return value of sap.ui.core.IconPool.getIconURI that is called with an icon
     * name parameter and optional collection parameter. The collection parameter is required when the application
     * extended icons are used.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIconSrc(
      /**
       * New value for property `iconSrc`
       */
      sIconSrc?: string
    ): this;
    /**
     * Sets a new value for property {@link #getSize size}.
     *
     * Updates the size of the chart. If not set then the default size is applied based on the device tile.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Auto"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSize(
      /**
       * New value for property `size`
       */
      sSize?: InfoTileSize | keyof typeof InfoTileSize
    ): this;
    /**
     * Sets a new value for property {@link #getState state}.
     *
     * Indicates the load status.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Loaded"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setState(
      /**
       * New value for property `state`
       */
      sState?: LoadState | keyof typeof LoadState
    ): this;
    /**
     * Sets a new value for property {@link #getValue value}.
     *
     * The actual value.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValue(
      /**
       * New value for property `value`
       */
      sValue?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the MonitoringContent constructor.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.NumericContent.
   */
  export interface $MonitoringContentSettings extends $ControlSettings {
    /**
     * The actual value.
     */
    value?: string | PropertyBindingInfo;

    /**
     * This property is set by the return value of sap.ui.core.IconPool.getIconURI that is called with an icon
     * name parameter and optional collection parameter. The collection parameter is required when the application
     * extended icons are used.
     */
    iconSrc?: string | PropertyBindingInfo;

    /**
     * Updates the size of the chart. If not set then the default size is applied based on the device tile.
     */
    size?:
      | (InfoTileSize | keyof typeof InfoTileSize)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Indicates the load status.
     */
    state?:
      | (LoadState | keyof typeof LoadState)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * If set to true, changing of the value is animated.
     */
    animateTextChange?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * The icon that is displayed in the content.
     */
    icon?: Icon;

    /**
     * The event is fired when the user chooses the monitoring content.
     */
    press?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the MonitoringContent#press event.
   */
  export interface MonitoringContent$PressEventParameters {}

  /**
   * Event object of the MonitoringContent#press event.
   */
  export type MonitoringContent$PressEvent = Event<
    MonitoringContent$PressEventParameters,
    MonitoringContent
  >;
}

declare module "sap/suite/ui/commons/MonitoringTile" {
  import {
    default as InfoTile,
    $InfoTileSettings,
  } from "sap/suite/ui/commons/InfoTile";

  import { InfoTileTextColor } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control is the implementation of the InfoTile to show a numeric value and an icon.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export default class MonitoringTile extends InfoTile {
    /**
     * Constructor for a new MonitoringTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $MonitoringTileSettings
    );
    /**
     * Constructor for a new MonitoringTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $MonitoringTileSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.MonitoringTile with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.InfoTile.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, MonitoringTile>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.MonitoringTile.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getFooterColor footerColor}.
     *
     * The color of the tile footer text.
     *
     * Default value is `"Positive"`.
     *
     *
     * @returns Value of property `footerColor`
     */
    getFooterColor(): InfoTileTextColor | keyof typeof InfoTileTextColor;
    /**
     * Gets current value of property {@link #getIconSrc iconSrc}.
     *
     * This property is set by the return value of sap.ui.core.IconPool.getIconURI that is called with an icon
     * name parameter and optional collection parameter. The collection parameter is required when the application
     * extended icons are used.
     *
     *
     * @returns Value of property `iconSrc`
     */
    getIconSrc(): string;
    /**
     * Gets current value of property {@link #getValue value}.
     *
     * The actual value.
     *
     *
     * @returns Value of property `value`
     */
    getValue(): string;
    /**
     * Sets a new value for property {@link #getFooterColor footerColor}.
     *
     * The color of the tile footer text.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Positive"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFooterColor(
      /**
       * New value for property `footerColor`
       */
      sFooterColor?: InfoTileTextColor | keyof typeof InfoTileTextColor
    ): this;
    /**
     * Sets a new value for property {@link #getIconSrc iconSrc}.
     *
     * This property is set by the return value of sap.ui.core.IconPool.getIconURI that is called with an icon
     * name parameter and optional collection parameter. The collection parameter is required when the application
     * extended icons are used.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIconSrc(
      /**
       * New value for property `iconSrc`
       */
      sIconSrc?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValue value}.
     *
     * The actual value.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValue(
      /**
       * New value for property `value`
       */
      sValue?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the MonitoringTile constructor.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export interface $MonitoringTileSettings extends $InfoTileSettings {
    /**
     * The actual value.
     */
    value?: string | PropertyBindingInfo;

    /**
     * This property is set by the return value of sap.ui.core.IconPool.getIconURI that is called with an icon
     * name parameter and optional collection parameter. The collection parameter is required when the application
     * extended icons are used.
     */
    iconSrc?: string | PropertyBindingInfo;

    /**
     * The color of the tile footer text.
     */
    footerColor?:
      | (InfoTileTextColor | keyof typeof InfoTileTextColor)
      | PropertyBindingInfo
      | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/networkgraph/ActionButton" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * Holds information about one custom action button.
   *
   * @since 1.50
   */
  export default class ActionButton extends UI5Element {
    /**
     * Constructor for a new ActionButton.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.ActionButton with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ActionButton>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.ActionButton.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.ActionButton`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.ActionButton` itself.
     *
     * This event is fired when the action button is clicked or tapped.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ActionButton$PressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.ActionButton`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.ActionButton`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.ActionButton` itself.
     *
     * This event is fired when the action button is clicked or tapped.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ActionButton$PressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.ActionButton`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.ActionButton`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: ActionButton$PressEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: ActionButton$PressEventParameters
    ): this;
    /**
     * Gets current value of property {@link #getEnabled enabled}.
     *
     * Indicates whether the action button is enabled.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `enabled`
     */
    getEnabled(): boolean;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * The icon to be used for the custom action button.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): string;
    /**
     * Gets current value of property {@link #getPosition position}.
     *
     * Position of the action button. Available only for nodes.
     *
     * Default value is `"Right"`.
     *
     *
     * @returns Value of property `position`
     */
    getPosition():
      | networkgraph.ActionButtonPosition
      | keyof typeof networkgraph.ActionButtonPosition;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * Tooltip title for custom action button.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets a new value for property {@link #getEnabled enabled}.
     *
     * Indicates whether the action button is enabled.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEnabled(
      /**
       * New value for property `enabled`
       */
      bEnabled?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * The icon to be used for the custom action button.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: string
    ): this;
    /**
     * Sets a new value for property {@link #getPosition position}.
     *
     * Position of the action button. Available only for nodes.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"Right"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setPosition(
      /**
       * New value for property `position`
       */
      sPosition?:
        | networkgraph.ActionButtonPosition
        | keyof typeof networkgraph.ActionButtonPosition
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * Tooltip title for custom action button.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ActionButton constructor.
   */
  export interface $ActionButtonSettings extends $ElementSettings {
    /**
     * The icon to be used for the custom action button.
     */
    icon?: string | PropertyBindingInfo;

    /**
     * Tooltip title for custom action button.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Indicates whether the action button is enabled.
     */
    enabled?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Position of the action button. Available only for nodes.
     */
    position?:
      | (
          | networkgraph.ActionButtonPosition
          | keyof typeof networkgraph.ActionButtonPosition
        )
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * This event is fired when the action button is clicked or tapped.
     */
    press?: (oEvent: ActionButton$PressEvent) => void;
  }

  /**
   * Parameters of the ActionButton#press event.
   */
  export interface ActionButton$PressEventParameters {
    buttonElement?: object;
  }

  /**
   * Event object of the ActionButton#press event.
   */
  export type ActionButton$PressEvent = Event<
    ActionButton$PressEventParameters,
    ActionButton
  >;
}

declare module "sap/suite/ui/commons/networkgraph/Coordinate" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Holds information about coordinates on a two-dimensional Cartesian plane.
   *
   * @since 1.50
   */
  export default class Coordinate extends UI5Element {
    /**
     * Constructor for a new Coordinate.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.Coordinate with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Coordinate>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.Coordinate.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getX x}.
     *
     * The value of the x coordinate.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Value of property `x`
     */
    getX(): float;
    /**
     * Gets current value of property {@link #getY y}.
     *
     * The value of the y coordinate.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Value of property `y`
     */
    getY(): float;
    /**
     * Sets a new value for property {@link #getX x}.
     *
     * The value of the x coordinate.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setX(
      /**
       * New value for property `x`
       */
      fX?: float
    ): this;
    /**
     * Sets a new value for property {@link #getY y}.
     *
     * The value of the y coordinate.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setY(
      /**
       * New value for property `y`
       */
      fY?: float
    ): this;
  }
  /**
   * Describes the settings that can be provided to the Coordinate constructor.
   */
  export interface $CoordinateSettings extends $ElementSettings {
    /**
     * The value of the x coordinate.
     */
    x?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * The value of the y coordinate.
     */
    y?: float | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/networkgraph/ElementAttribute" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Holds details of an attribute used in the graph.
   *
   * @since 1.50
   */
  export default class ElementAttribute extends UI5Element {
    /**
     * Constructor for a new ElementAttribute.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.ElementAttribute with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ElementAttribute>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.ElementAttribute.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * Icon assigned to the attribute.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): string;
    /**
     * Gets current value of property {@link #getLabel label}.
     *
     * Label of the attribute. If set to null, the label is not displayed.
     *
     *
     * @returns Value of property `label`
     */
    getLabel(): string;
    /**
     * Gets current value of property {@link #getLabelStatus labelStatus}.
     *
     * Name of custom status that can change the color of the label. Note that only contentColor, selectedContentColor
     * and hoverContentColor are relevant to this attribute.
     *
     *
     * @returns Value of property `labelStatus`
     */
    getLabelStatus(): string;
    /**
     * Gets current value of property {@link #getValue value}.
     *
     * Value of the attribute. If set to null, the value is not displayed.
     *
     *
     * @returns Value of property `value`
     */
    getValue(): string;
    /**
     * Gets current value of property {@link #getValueStatus valueStatus}.
     *
     * Name of custom status that can change the color of the value. Note that only contentColor, selectedContentColor
     * and hoverContentColor are relevant to this attribute.
     *
     *
     * @returns Value of property `valueStatus`
     */
    getValueStatus(): string;
    /**
     * Gets current value of property {@link #getVisible visible}.
     *
     * Defines whether the attribute is visible. This option is used for rectangular nodes. Visible attributes
     * are displayed right inside the rectangular node. The node's details popup shows all attributes, including
     * the invisible ones.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `visible`
     */
    getVisible(): boolean;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * Icon assigned to the attribute.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: string
    ): this;
    /**
     * Sets a new value for property {@link #getLabel label}.
     *
     * Label of the attribute. If set to null, the label is not displayed.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLabel(
      /**
       * New value for property `label`
       */
      sLabel?: string
    ): this;
    /**
     * Sets a new value for property {@link #getLabelStatus labelStatus}.
     *
     * Name of custom status that can change the color of the label. Note that only contentColor, selectedContentColor
     * and hoverContentColor are relevant to this attribute.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLabelStatus(
      /**
       * New value for property `labelStatus`
       */
      sLabelStatus?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValue value}.
     *
     * Value of the attribute. If set to null, the value is not displayed.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValue(
      /**
       * New value for property `value`
       */
      sValue?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValueStatus valueStatus}.
     *
     * Name of custom status that can change the color of the value. Note that only contentColor, selectedContentColor
     * and hoverContentColor are relevant to this attribute.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValueStatus(
      /**
       * New value for property `valueStatus`
       */
      sValueStatus?: string
    ): this;
    /**
     * Sets a new value for property {@link #getVisible visible}.
     *
     * Defines whether the attribute is visible. This option is used for rectangular nodes. Visible attributes
     * are displayed right inside the rectangular node. The node's details popup shows all attributes, including
     * the invisible ones.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setVisible(
      /**
       * New value for property `visible`
       */
      bVisible?: boolean
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ElementAttribute constructor.
   */
  export interface $ElementAttributeSettings extends $ElementSettings {
    /**
     * Label of the attribute. If set to null, the label is not displayed.
     */
    label?: string | PropertyBindingInfo;

    /**
     * Value of the attribute. If set to null, the value is not displayed.
     */
    value?: string | PropertyBindingInfo;

    /**
     * Name of custom status that can change the color of the label. Note that only contentColor, selectedContentColor
     * and hoverContentColor are relevant to this attribute.
     */
    labelStatus?: string | PropertyBindingInfo;

    /**
     * Name of custom status that can change the color of the value. Note that only contentColor, selectedContentColor
     * and hoverContentColor are relevant to this attribute.
     */
    valueStatus?: string | PropertyBindingInfo;

    /**
     * Defines whether the attribute is visible. This option is used for rectangular nodes. Visible attributes
     * are displayed right inside the rectangular node. The node's details popup shows all attributes, including
     * the invisible ones.
     */
    visible?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Icon assigned to the attribute.
     */
    icon?: string | PropertyBindingInfo;
  }
}

declare module "sap/suite/ui/commons/networkgraph/ElementBase" {
  import {
    default as SvgBase,
    $SvgBaseSettings,
  } from "sap/suite/ui/commons/networkgraph/SvgBase";

  import ElementAttribute from "sap/suite/ui/commons/networkgraph/ElementAttribute";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * ElementBase class
   *
   * @since 1.50
   */
  export default abstract class ElementBase extends SvgBase {
    /**
     * Constructor for a new ElementBase.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.ElementBase with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.networkgraph.SvgBase.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ElementBase>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.ElementBase.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some attribute to the aggregation {@link #getAttributes attributes}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addAttribute(
      /**
       * The attribute to add; if empty, nothing is inserted
       */
      oAttribute: ElementAttribute
    ): this;
    /**
     * Destroys all the attributes in the aggregation {@link #getAttributes attributes}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyAttributes(): this;
    /**
     * Gets content of aggregation {@link #getAttributes attributes}.
     *
     * Attributes associated with the element.
     */
    getAttributes(): ElementAttribute[];
    /**
     * Gets current value of property {@link #getDescription description}.
     *
     * Description.
     *
     *
     * @returns Value of property `description`
     */
    getDescription(): string;
    /**
     * Gets current value of property {@link #getStatus status}.
     *
     * Status associated with this element. You can use any of the custom statuses defined by the `statuses`
     * aggregation in the {@link sap.suite.ui.commons.networkgraph.Graph} or use the default statuses provided
     * by {@link sap.suite.ui.commons.networkgraph.ElementStatus}.
     *
     * Default value is `Status.Standard`.
     *
     *
     * @returns Value of property `status`
     */
    getStatus(): string;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * A title associated with the element.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Checks for the provided `sap.suite.ui.commons.networkgraph.ElementAttribute` in the aggregation {@link #getAttributes attributes}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfAttribute(
      /**
       * The attribute whose index is looked for
       */
      oAttribute: ElementAttribute
    ): int;
    /**
     * Inserts a attribute into the aggregation {@link #getAttributes attributes}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertAttribute(
      /**
       * The attribute to insert; if empty, nothing is inserted
       */
      oAttribute: ElementAttribute,
      /**
       * The `0`-based index the attribute should be inserted at; for a negative value of `iIndex`, the attribute
       * is inserted at position 0; for a value greater than the current size of the aggregation, the attribute
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getAttributes attributes}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllAttributes(): ElementAttribute[];
    /**
     * Removes a attribute from the aggregation {@link #getAttributes attributes}.
     *
     *
     * @returns The removed attribute or `null`
     */
    removeAttribute(
      /**
       * The attribute to remove or its index or id
       */
      vAttribute: int | string | ElementAttribute
    ): ElementAttribute | null;
    /**
     * Scrolls the view port to show this element
     */
    scrollToElement(): void;
    /**
     * Sets a new value for property {@link #getDescription description}.
     *
     * Description.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescription(
      /**
       * New value for property `description`
       */
      sDescription?: string
    ): this;
    /**
     * Sets a new value for property {@link #getStatus status}.
     *
     * Status associated with this element. You can use any of the custom statuses defined by the `statuses`
     * aggregation in the {@link sap.suite.ui.commons.networkgraph.Graph} or use the default statuses provided
     * by {@link sap.suite.ui.commons.networkgraph.ElementStatus}.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Status.Standard`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setStatus(
      /**
       * New value for property `status`
       */
      sStatus?: string
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * A title associated with the element.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ElementBase constructor.
   */
  export interface $ElementBaseSettings extends $SvgBaseSettings {
    /**
     * A title associated with the element.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Description.
     */
    description?: string | PropertyBindingInfo;

    /**
     * Status associated with this element. You can use any of the custom statuses defined by the `statuses`
     * aggregation in the {@link sap.suite.ui.commons.networkgraph.Graph} or use the default statuses provided
     * by {@link sap.suite.ui.commons.networkgraph.ElementStatus}.
     */
    status?: string | PropertyBindingInfo;

    /**
     * Attributes associated with the element.
     */
    attributes?:
      | ElementAttribute[]
      | ElementAttribute
      | AggregationBindingInfo
      | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/networkgraph/Graph" {
  import {
    default as SvgBase,
    $SvgBaseSettings,
  } from "sap/suite/ui/commons/networkgraph/SvgBase";

  import { ID, URI, CSSSize } from "sap/ui/core/library";

  import Control from "sap/ui/core/Control";

  import Group from "sap/suite/ui/commons/networkgraph/Group";

  import Line from "sap/suite/ui/commons/networkgraph/Line";

  import Node from "sap/suite/ui/commons/networkgraph/Node";

  import Status from "sap/suite/ui/commons/networkgraph/Status";

  import Event from "sap/ui/base/Event";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import LayoutAlgorithm from "sap/suite/ui/commons/networkgraph/layout/LayoutAlgorithm";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import OverflowToolbar from "sap/m/OverflowToolbar";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  import ElementBase from "sap/suite/ui/commons/networkgraph/ElementBase";

  /**
   * The network graph control allows you to display data as a network of nodes connected by lines. The nodes
   * can be circular or rectangular and can be joined into groups. You can define custom attributes for nodes
   * and groups of nodes as well as apply layout algorithms that define the graph's appearance.
   *
   * @since 1.50
   */
  export default class Graph extends SvgBase {
    /**
     * Constructor for a new Graph.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     * See:
     * 	{@link https://ui5.sap.com/#/topic/b5649c8de7f74739b66747dcc9356d0b Network Graph}
     * 	{@link fiori:https://experience.sap.com/fiori-design-web/network-graph/ Network Graph}
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.Graph with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.networkgraph.SvgBase.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Graph>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.Graph.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some ariaDescribedBy into the association {@link #getAriaDescribedBy ariaDescribedBy}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addAriaDescribedBy(
      /**
       * The ariaDescribedBy to add; if empty, nothing is inserted
       */
      vAriaDescribedBy: ID | Control
    ): this;
    /**
     * Adds some ariaLabelledBy into the association {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addAriaLabelledBy(
      /**
       * The ariaLabelledBy to add; if empty, nothing is inserted
       */
      vAriaLabelledBy: ID | Control
    ): this;
    /**
     * Adds some group to the aggregation {@link #getGroups groups}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addGroup(
      /**
       * The group to add; if empty, nothing is inserted
       */
      oGroup: Group
    ): this;
    /**
     * Adds some line to the aggregation {@link #getLines lines}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addLine(
      /**
       * The line to add; if empty, nothing is inserted
       */
      oLine: Line
    ): this;
    /**
     * Adds some node to the aggregation {@link #getNodes nodes}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addNode(
      /**
       * The node to add; if empty, nothing is inserted
       */
      oNode: Node
    ): this;
    /**
     * Adds some status to the aggregation {@link #getStatuses statuses}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addStatus(
      /**
       * The status to add; if empty, nothing is inserted
       */
      oStatus: Status
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:afterLayouting afterLayouting} event of this
     * `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when the layouting algorithm has finished arranging the graph and SVG rendering starts.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAfterLayouting(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:afterLayouting afterLayouting} event of this
     * `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when the layouting algorithm has finished arranging the graph and SVG rendering starts.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAfterLayouting(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:beforeLayouting beforeLayouting} event of this
     * `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired just before the layout computation begins.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeLayouting(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:beforeLayouting beforeLayouting} event of this
     * `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired just before the layout computation begins.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachBeforeLayouting(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:failure failure} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when an error has occured and the graph cannot be rendered properly.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachFailure(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Graph$FailureEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:failure failure} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when an error has occured and the graph cannot be rendered properly.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachFailure(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Graph$FailureEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:graphReady graphReady} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when the graph is fully rendered.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachGraphReady(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:graphReady graphReady} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when the graph is fully rendered.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachGraphReady(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:search search} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when the user enters a keyword into the search field.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSearch(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Graph$SearchEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:search search} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when the user enters a keyword into the search field.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSearch(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Graph$SearchEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:searchSuggest searchSuggest} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when the user runs a search and there is a matching term found among the suggestions.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSearchSuggest(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Graph$SearchSuggestEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:searchSuggest searchSuggest} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when the user runs a search and there is a matching term found among the suggestions.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSearchSuggest(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Graph$SearchSuggestEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:selectionChange selectionChange} event of this
     * `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when a selection of elements in the graph changes.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSelectionChange(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Graph$SelectionChangeEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:selectionChange selectionChange} event of this
     * `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when a selection of elements in the graph changes.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachSelectionChange(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Graph$SelectionChangeEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:zoomChanged zoomChanged} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when zooming in or out.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachZoomChanged(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:zoomChanged zoomChanged} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Graph` itself.
     *
     * This event is fired when zooming in or out.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachZoomChanged(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Graph`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Deselects all currently selected items.
     *
     *
     * @returns All items that have been deselected
     */
    deselect(
      /**
       * Indicates whether the `selecitonChange` event should be fired
       */
      bSuppressEvent: boolean
    ): any[];
    /**
     * This method removes all elements (lines, nodes, groups) without triggering invalidation. You should use
     * this method when changing graph data that uses data binding, for example with {@link sap.suite.ui.commons.networkgraph.Graph#setModel}.
     */
    destroyAllElements(): void;
    /**
     * Destroys all the groups in the aggregation {@link #getGroups groups}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyGroups(): this;
    /**
     * Destroys the layoutAlgorithm in the aggregation {@link #getLayoutAlgorithm layoutAlgorithm}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyLayoutAlgorithm(): this;
    /**
     * Destroys the legend in the aggregation {@link #getLegend legend}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyLegend(): this;
    /**
     * Destroys all the lines in the aggregation {@link #getLines lines}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyLines(): this;
    /**
     * Destroys all the nodes in the aggregation {@link #getNodes nodes}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyNodes(): this;
    /**
     * Destroys all the statuses in the aggregation {@link #getStatuses statuses}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyStatuses(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:afterLayouting afterLayouting} event of this
     * `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAfterLayouting(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:beforeLayouting beforeLayouting} event of
     * this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachBeforeLayouting(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:failure failure} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachFailure(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Graph$FailureEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:graphReady graphReady} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachGraphReady(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:search search} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachSearch(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Graph$SearchEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:searchSuggest searchSuggest} event of this
     * `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachSearchSuggest(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Graph$SearchSuggestEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:selectionChange selectionChange} event of
     * this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachSelectionChange(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Graph$SelectionChangeEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:zoomChanged zoomChanged} event of this `sap.suite.ui.commons.networkgraph.Graph`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachZoomChanged(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:afterLayouting afterLayouting} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAfterLayouting(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:beforeLayouting beforeLayouting} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireBeforeLayouting(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:failure failure} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireFailure(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: Graph$FailureEventParameters
    ): this;
    /**
     * Fires event {@link #event:graphReady graphReady} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireGraphReady(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:search search} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireSearch(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: Graph$SearchEventParameters
    ): this;
    /**
     * Fires event {@link #event:searchSuggest searchSuggest} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireSearchSuggest(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: Graph$SearchSuggestEventParameters
    ): this;
    /**
     * Fires event {@link #event:selectionChange selectionChange} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireSelectionChange(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: Graph$SelectionChangeEventParameters
    ): this;
    /**
     * Fires event {@link #event:zoomChanged zoomChanged} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireZoomChanged(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Returns array of IDs of the elements which are the current targets of the association {@link #getAriaDescribedBy ariaDescribedBy}.
     */
    getAriaDescribedBy(): ID[];
    /**
     * Returns array of IDs of the elements which are the current targets of the association {@link #getAriaLabelledBy ariaLabelledBy}.
     */
    getAriaLabelledBy(): ID[];
    /**
     * Gets current value of property {@link #getBackgroundColor backgroundColor}.
     *
     * Background color.
     *
     * Default value is `"White"`.
     *
     *
     * @returns Value of property `backgroundColor`
     */
    getBackgroundColor():
      | networkgraph.BackgroundColor
      | keyof typeof networkgraph.BackgroundColor;
    /**
     * Gets current value of property {@link #getBackgroundImage backgroundImage}.
     *
     * An image to be rendered on the background.
     *
     *
     * @returns Value of property `backgroundImage`
     */
    getBackgroundImage(): URI;
    /**
     * Returns current zoom level.
     */
    getCurrentZoomLevel(): number;
    /**
     * Gets current value of property {@link #getEnableWheelZoom enableWheelZoom}.
     *
     * If this property is set to `false`, zooming in or out using the mouse wheel is available only when the
     * Ctrl key is pressed.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `enableWheelZoom`
     */
    getEnableWheelZoom(): boolean;
    /**
     * Gets current value of property {@link #getEnableZoom enableZoom}.
     *
     * If this property is set to `false`, zoom buttons are hidden, and the mouse wheel can be used for scrolling
     * only.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `enableZoom`
     */
    getEnableZoom(): boolean;
    /**
     * Gets content of aggregation {@link #getGroups groups}.
     *
     * Holds a list of groups used in the graph.
     */
    getGroups(): Group[];
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * The height of the graph. If this property is set to 'auto', the network graph will be resized to fit
     * the height of its content, regardless of the height of the parent control.
     *
     * Default value is `"100%"`.
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): CSSSize;
    /**
     * Gets content of aggregation {@link #getLayoutAlgorithm layoutAlgorithm}.
     *
     * Defines the layout algorithm to be used. If not defined, {@link sap.suite.ui.commons.networkgraph.layout.LayeredLayout }
     * is used.
     */
    getLayoutAlgorithm(): LayoutAlgorithm;
    /**
     * Gets content of aggregation {@link #getLegend legend}.
     *
     * A custom legend to be rendered instead of the autogenerated one.
     */
    getLegend(): Control;
    /**
     * Gets content of aggregation {@link #getLines lines}.
     *
     * Holds the lines to be displayed in the graph.
     */
    getLines(): Line[];
    /**
     * Gets current value of property {@link #getNoData noData}.
     *
     * Set this property to `true` if you want to notify the user that no data has been loaded.
     *  There is no internal check for data received, so we recommend that you make sure you set it correctly
     * when there is no data received, not just when the application is waiting for data to be retrieved.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `noData`
     */
    getNoData(): boolean;
    /**
     * Gets current value of property {@link #getNoDataText noDataText}.
     *
     * Text displayed when no data is set.
     *  This property takes effect only when the `noData` property is set to `true`.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `noDataText`
     */
    getNoDataText(): string;
    /**
     * Returns a node by its key.
     *
     *
     * @returns The node, if such a node exists, or undefined
     */
    getNodeByKey(
      /**
       * Node's key
       */
      sKey: string
    ): Node;
    /**
     * Gets content of aggregation {@link #getNodes nodes}.
     *
     * Holds the nodes to be displayed in the graph.
     */
    getNodes(): Node[];
    /**
     * Gets current value of property {@link #getOrientation orientation}.
     *
     * Orientation of the graph flow. This property is used by layout algorithms.
     *
     * Default value is `LeftRight`.
     *
     *
     * @returns Value of property `orientation`
     */
    getOrientation():
      | networkgraph.Orientation
      | keyof typeof networkgraph.Orientation;
    /**
     * Gets current value of property {@link #getRenderType renderType}.
     *
     * Nodes rendering type. For optimal performance and usability, it is recommended that you use HTML whenever
     * possible.
     *
     * Default value is `Html`.
     *
     *
     * @returns Value of property `renderType`
     */
    getRenderType():
      | networkgraph.RenderType
      | keyof typeof networkgraph.RenderType;
    /**
     * Gets content of aggregation {@link #getStatuses statuses}.
     *
     * Holds a collection of custom statuses that can be used to assign custom colors to nodes, lines, and groups
     * of nodes, based on their status.
     */
    getStatuses(): Status[];
    /**
     * Returns a toolbar instance to allow its customization.
     *
     *
     * @returns Toolbar instance
     */
    getToolbar(): OverflowToolbar;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * The width of the graph. If this property is set to 'auto', the network graph will be resized to fit the
     * width of its content, regardless of the width of the parent control.
     *
     * Default value is `"100%"`.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Returns predefined zoom level steps used by the default zoom functionality.
     *
     *
     * @returns zoom level milestones
     */
    getZoomLevelMilestones(): number[];
    /**
     * Checks for the provided `sap.suite.ui.commons.networkgraph.Group` in the aggregation {@link #getGroups groups}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfGroup(
      /**
       * The group whose index is looked for
       */
      oGroup: Group
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.networkgraph.Line` in the aggregation {@link #getLines lines}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfLine(
      /**
       * The line whose index is looked for
       */
      oLine: Line
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.networkgraph.Node` in the aggregation {@link #getNodes nodes}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfNode(
      /**
       * The node whose index is looked for
       */
      oNode: Node
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.networkgraph.Status` in the aggregation {@link #getStatuses statuses}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfStatus(
      /**
       * The status whose index is looked for
       */
      oStatus: Status
    ): int;
    /**
     * Inserts a group into the aggregation {@link #getGroups groups}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertGroup(
      /**
       * The group to insert; if empty, nothing is inserted
       */
      oGroup: Group,
      /**
       * The `0`-based index the group should be inserted at; for a negative value of `iIndex`, the group is inserted
       * at position 0; for a value greater than the current size of the aggregation, the group is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a line into the aggregation {@link #getLines lines}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertLine(
      /**
       * The line to insert; if empty, nothing is inserted
       */
      oLine: Line,
      /**
       * The `0`-based index the line should be inserted at; for a negative value of `iIndex`, the line is inserted
       * at position 0; for a value greater than the current size of the aggregation, the line is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a node into the aggregation {@link #getNodes nodes}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertNode(
      /**
       * The node to insert; if empty, nothing is inserted
       */
      oNode: Node,
      /**
       * The `0`-based index the node should be inserted at; for a negative value of `iIndex`, the node is inserted
       * at position 0; for a value greater than the current size of the aggregation, the node is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a status into the aggregation {@link #getStatuses statuses}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertStatus(
      /**
       * The status to insert; if empty, nothing is inserted
       */
      oStatus: Status,
      /**
       * The `0`-based index the status should be inserted at; for a negative value of `iIndex`, the status is
       * inserted at position 0; for a value greater than the current size of the aggregation, the status is inserted
       * at the last position
       */
      iIndex: int
    ): this;
    /**
     * Returns `true` if the graph is in full screen mode.
     */
    isFullScreen(): void;
    /**
     * Removes all the controls in the association named {@link #getAriaDescribedBy ariaDescribedBy}.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllAriaDescribedBy(): ID[];
    /**
     * Removes all the controls in the association named {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllAriaLabelledBy(): ID[];
    /**
     * Removes all the controls from the aggregation {@link #getGroups groups}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllGroups(): Group[];
    /**
     * Removes all the controls from the aggregation {@link #getLines lines}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllLines(): Line[];
    /**
     * Removes all the controls from the aggregation {@link #getNodes nodes}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllNodes(): Node[];
    /**
     * Removes all the controls from the aggregation {@link #getStatuses statuses}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllStatuses(): Status[];
    /**
     * Removes an ariaDescribedBy from the association named {@link #getAriaDescribedBy ariaDescribedBy}.
     *
     *
     * @returns The removed ariaDescribedBy or `null`
     */
    removeAriaDescribedBy(
      /**
       * The ariaDescribedBy to be removed or its index or ID
       */
      vAriaDescribedBy: int | ID | Control
    ): ID | null;
    /**
     * Removes an ariaLabelledBy from the association named {@link #getAriaLabelledBy ariaLabelledBy}.
     *
     *
     * @returns The removed ariaLabelledBy or `null`
     */
    removeAriaLabelledBy(
      /**
       * The ariaLabelledBy to be removed or its index or ID
       */
      vAriaLabelledBy: int | ID | Control
    ): ID | null;
    /**
     * Removes a group from the aggregation {@link #getGroups groups}.
     *
     *
     * @returns The removed group or `null`
     */
    removeGroup(
      /**
       * The group to remove or its index or id
       */
      vGroup: int | string | Group
    ): Group | null;
    /**
     * Removes a line from the aggregation {@link #getLines lines}.
     *
     *
     * @returns The removed line or `null`
     */
    removeLine(
      /**
       * The line to remove or its index or id
       */
      vLine: int | string | Line
    ): Line | null;
    /**
     * Removes a node from the aggregation {@link #getNodes nodes}.
     *
     *
     * @returns The removed node or `null`
     */
    removeNode(
      /**
       * The node to remove or its index or id
       */
      vNode: int | string | Node
    ): Node | null;
    /**
     * Removes a status from the aggregation {@link #getStatuses statuses}.
     *
     *
     * @returns The removed status or `null`
     */
    removeStatus(
      /**
       * The status to remove or its index or id
       */
      vStatus: int | string | Status
    ): Status | null;
    /**
     * Scrolls to the element set in the parameter.
     */
    scrollToElement(
      /**
       * Element to scroll to
       */
      oElement: Object
    ): void;
    /**
     * Sets a new value for property {@link #getBackgroundColor backgroundColor}.
     *
     * Background color.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"White"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBackgroundColor(
      /**
       * New value for property `backgroundColor`
       */
      sBackgroundColor?:
        | networkgraph.BackgroundColor
        | keyof typeof networkgraph.BackgroundColor
    ): this;
    /**
     * Sets a new value for property {@link #getBackgroundImage backgroundImage}.
     *
     * An image to be rendered on the background.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBackgroundImage(
      /**
       * New value for property `backgroundImage`
       */
      sBackgroundImage?: URI
    ): this;
    /**
     * Sets current zoom level.
     */
    setCurrentZoomLevel(
      /**
       * New zoom level
       */
      zoomLevel: float
    ): void;
    /**
     * Sets a custom label for the legend.
     */
    setCustomLegendLabel(
      /**
       * Parameters for this method
       */
      mArguments: {
        /**
         * ]text for the legend label
         */
        label: string;
        /**
         * Status the custom text will be assigned to. Works only for the default legend, not when the Legend aggregation
         * is used
         */
        status:
          | networkgraph.ElementStatus
          | keyof typeof networkgraph.ElementStatus;
        /**
         * True for a legend describing nodes. True by default if 'isLine' and 'isGroup' are false.
         */
        isNode: boolean;
        /**
         * True for a legend describing lines. By default true if 'isNode' is false.
         */
        isLine: boolean;
        /**
         * True for a legend describing groups. False by default.
         */
        isGroup: boolean;
      }
    ): void;
    /**
     * Sets a new value for property {@link #getEnableWheelZoom enableWheelZoom}.
     *
     * If this property is set to `false`, zooming in or out using the mouse wheel is available only when the
     * Ctrl key is pressed.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEnableWheelZoom(
      /**
       * New value for property `enableWheelZoom`
       */
      bEnableWheelZoom?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getEnableZoom enableZoom}.
     *
     * If this property is set to `false`, zoom buttons are hidden, and the mouse wheel can be used for scrolling
     * only.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setEnableZoom(
      /**
       * New value for property `enableZoom`
       */
      bEnableZoom?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * The height of the graph. If this property is set to 'auto', the network graph will be resized to fit
     * the height of its content, regardless of the height of the parent control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"100%"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      sHeight?: CSSSize
    ): this;
    /**
     * Sets the aggregated {@link #getLayoutAlgorithm layoutAlgorithm}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLayoutAlgorithm(
      /**
       * The layoutAlgorithm to set
       */
      oLayoutAlgorithm: LayoutAlgorithm
    ): this;
    /**
     * Sets the aggregated {@link #getLegend legend}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLegend(
      /**
       * The legend to set
       */
      oLegend: Control
    ): this;
    /**
     * Sets a new value for property {@link #getNoData noData}.
     *
     * Set this property to `true` if you want to notify the user that no data has been loaded.
     *  There is no internal check for data received, so we recommend that you make sure you set it correctly
     * when there is no data received, not just when the application is waiting for data to be retrieved.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setNoData(
      /**
       * New value for property `noData`
       */
      bNoData?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getNoDataText noDataText}.
     *
     * Text displayed when no data is set.
     *  This property takes effect only when the `noData` property is set to `true`.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setNoDataText(
      /**
       * New value for property `noDataText`
       */
      sNoDataText?: string
    ): this;
    /**
     * Sets a new value for property {@link #getOrientation orientation}.
     *
     * Orientation of the graph flow. This property is used by layout algorithms.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `LeftRight`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setOrientation(
      /**
       * New value for property `orientation`
       */
      sOrientation?:
        | networkgraph.Orientation
        | keyof typeof networkgraph.Orientation
    ): this;
    /**
     * Sets a new value for property {@link #getRenderType renderType}.
     *
     * Nodes rendering type. For optimal performance and usability, it is recommended that you use HTML whenever
     * possible.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Html`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setRenderType(
      /**
       * New value for property `renderType`
       */
      sRenderType?:
        | networkgraph.RenderType
        | keyof typeof networkgraph.RenderType
    ): this;
    /**
     * Sets custom search suggestions.
     */
    setSearchSuggestionItems(
      /**
       * Array with suggestion items ({@link sap.m.SuggestionItem}) to display when the user runs a search.
       */
      aItems: any[]
    ): void;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * The width of the graph. If this property is set to 'auto', the network graph will be resized to fit the
     * width of its content, regardless of the width of the parent control.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"100%"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
    /**
     * Toggles full screen mode.
     */
    toggleFullScreen(): void;
    /**
     * Updates the legend.
     */
    updateLegend(): void;
    /**
     * Zooms in or out of the graph.
     */
    zoom(
      /**
       * Map which contains following parameters properties:
       */
      mParameters: {
        /**
         * x position from/to which the zoom should happen
         */
        x?: number;
        /**
         * y position from/to which the zoom should happen
         */
        y?: number;
        /**
         * whether to zoom in or out
         */
        zoomin?: boolean;
        /**
         * zoom level in percents, parameter zoomin ignored in this case
         */
        zoomLevel?: number;
      }
    ): void;
  }
  /**
   * Describes the settings that can be provided to the Graph constructor.
   */
  export interface $GraphSettings extends $SvgBaseSettings {
    /**
     * The height of the graph. If this property is set to 'auto', the network graph will be resized to fit
     * the height of its content, regardless of the height of the parent control.
     */
    height?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * The width of the graph. If this property is set to 'auto', the network graph will be resized to fit the
     * width of its content, regardless of the width of the parent control.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Orientation of the graph flow. This property is used by layout algorithms.
     */
    orientation?:
      | (networkgraph.Orientation | keyof typeof networkgraph.Orientation)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * If this property is set to `false`, zoom buttons are hidden, and the mouse wheel can be used for scrolling
     * only.
     */
    enableZoom?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * If this property is set to `false`, zooming in or out using the mouse wheel is available only when the
     * Ctrl key is pressed.
     */
    enableWheelZoom?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * An image to be rendered on the background.
     */
    backgroundImage?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * Background color.
     */
    backgroundColor?:
      | (
          | networkgraph.BackgroundColor
          | keyof typeof networkgraph.BackgroundColor
        )
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Nodes rendering type. For optimal performance and usability, it is recommended that you use HTML whenever
     * possible.
     */
    renderType?:
      | (networkgraph.RenderType | keyof typeof networkgraph.RenderType)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Set this property to `true` if you want to notify the user that no data has been loaded.
     *  There is no internal check for data received, so we recommend that you make sure you set it correctly
     * when there is no data received, not just when the application is waiting for data to be retrieved.
     */
    noData?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Text displayed when no data is set.
     *  This property takes effect only when the `noData` property is set to `true`.
     */
    noDataText?: string | PropertyBindingInfo;

    /**
     * Holds the lines to be displayed in the graph.
     */
    lines?: Line[] | Line | AggregationBindingInfo | `{${string}}`;

    /**
     * Holds the nodes to be displayed in the graph.
     */
    nodes?: Node[] | Node | AggregationBindingInfo | `{${string}}`;

    /**
     * Holds a list of groups used in the graph.
     */
    groups?: Group[] | Group | AggregationBindingInfo | `{${string}}`;

    /**
     * A custom legend to be rendered instead of the autogenerated one.
     */
    legend?: Control;

    /**
     * Defines the layout algorithm to be used. If not defined, {@link sap.suite.ui.commons.networkgraph.layout.LayeredLayout }
     * is used.
     */
    layoutAlgorithm?: LayoutAlgorithm;

    /**
     * Holds a collection of custom statuses that can be used to assign custom colors to nodes, lines, and groups
     * of nodes, based on their status.
     */
    statuses?: Status[] | Status | AggregationBindingInfo | `{${string}}`;

    /**
     * Controls or IDs that describe this control. This association is used by screen reader software.
     */
    ariaDescribedBy?: Array<Control | string>;

    /**
     * Controls or IDs that label this control. This association is used by screen reader software.
     */
    ariaLabelledBy?: Array<Control | string>;

    /**
     * This event is fired when the graph is fully rendered.
     */
    graphReady?: (oEvent: Event) => void;

    /**
     * This event is fired when the layouting algorithm has finished arranging the graph and SVG rendering starts.
     */
    afterLayouting?: (oEvent: Event) => void;

    /**
     * This event is fired just before the layout computation begins.
     */
    beforeLayouting?: (oEvent: Event) => void;

    /**
     * This event is fired when zooming in or out.
     */
    zoomChanged?: (oEvent: Event) => void;

    /**
     * This event is fired when an error has occured and the graph cannot be rendered properly.
     */
    failure?: (oEvent: Graph$FailureEvent) => void;

    /**
     * This event is fired when a selection of elements in the graph changes.
     */
    selectionChange?: (oEvent: Graph$SelectionChangeEvent) => void;

    /**
     * This event is fired when the user runs a search and there is a matching term found among the suggestions.
     */
    searchSuggest?: (oEvent: Graph$SearchSuggestEvent) => void;

    /**
     * This event is fired when the user enters a keyword into the search field.
     */
    search?: (oEvent: Graph$SearchEvent) => void;
  }

  /**
   * Parameters of the Graph#afterLayouting event.
   */
  export interface Graph$AfterLayoutingEventParameters {}

  /**
   * Event object of the Graph#afterLayouting event.
   */
  export type Graph$AfterLayoutingEvent = Event<
    Graph$AfterLayoutingEventParameters,
    Graph
  >;

  /**
   * Parameters of the Graph#beforeLayouting event.
   */
  export interface Graph$BeforeLayoutingEventParameters {}

  /**
   * Event object of the Graph#beforeLayouting event.
   */
  export type Graph$BeforeLayoutingEvent = Event<
    Graph$BeforeLayoutingEventParameters,
    Graph
  >;

  /**
   * Parameters of the Graph#failure event.
   */
  export interface Graph$FailureEventParameters {
    /**
     * Type of an error. This parameter can be used for decision making in the calling code.
     */
    type?: string;

    /**
     * A human readable message with a description of what went wrong.
     */
    message?: string;
  }

  /**
   * Event object of the Graph#failure event.
   */
  export type Graph$FailureEvent = Event<Graph$FailureEventParameters, Graph>;

  /**
   * Parameters of the Graph#graphReady event.
   */
  export interface Graph$GraphReadyEventParameters {}

  /**
   * Event object of the Graph#graphReady event.
   */
  export type Graph$GraphReadyEvent = Event<
    Graph$GraphReadyEventParameters,
    Graph
  >;

  /**
   * Parameters of the Graph#search event.
   */
  export interface Graph$SearchEventParameters {
    term?: string;

    key?: string;
  }

  /**
   * Event object of the Graph#search event.
   */
  export type Graph$SearchEvent = Event<Graph$SearchEventParameters, Graph>;

  /**
   * Parameters of the Graph#searchSuggest event.
   */
  export interface Graph$SearchSuggestEventParameters {
    term?: string;
  }

  /**
   * Event object of the Graph#searchSuggest event.
   */
  export type Graph$SearchSuggestEvent = Event<
    Graph$SearchSuggestEventParameters,
    Graph
  >;

  /**
   * Parameters of the Graph#selectionChange event.
   */
  export interface Graph$SelectionChangeEventParameters {
    /**
     * A list of elements that changed the state of the selection. To determine the new state, use the `getSelected()`
     * method.
     */
    items?: ElementBase[];
  }

  /**
   * Event object of the Graph#selectionChange event.
   */
  export type Graph$SelectionChangeEvent = Event<
    Graph$SelectionChangeEventParameters,
    Graph
  >;

  /**
   * Parameters of the Graph#zoomChanged event.
   */
  export interface Graph$ZoomChangedEventParameters {}

  /**
   * Event object of the Graph#zoomChanged event.
   */
  export type Graph$ZoomChangedEvent = Event<
    Graph$ZoomChangedEventParameters,
    Graph
  >;
}

declare module "sap/suite/ui/commons/networkgraph/GraphMap" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import { ID, CSSSize } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import Graph from "sap/suite/ui/commons/networkgraph/Graph";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * A component which displays an overview of the entire graph and allows users to quickly navigate in the
   * linked graph.
   *
   * @since 1.50
   */
  export default class GraphMap extends Control {
    /**
     * Constructor for a new GraphMap.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.GraphMap with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, GraphMap>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.GraphMap.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:mapReady mapReady} event of this `sap.suite.ui.commons.networkgraph.GraphMap`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.GraphMap` itself.
     *
     * This event is fired when the map is fully rendered.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachMapReady(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.GraphMap`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:mapReady mapReady} event of this `sap.suite.ui.commons.networkgraph.GraphMap`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.GraphMap` itself.
     *
     * This event is fired when the map is fully rendered.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachMapReady(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.GraphMap`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:mapReady mapReady} event of this `sap.suite.ui.commons.networkgraph.GraphMap`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachMapReady(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:mapReady mapReady} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireMapReady(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets current value of property {@link #getDirectRenderNodeLimit directRenderNodeLimit}.
     *
     * This property affects the threshold at which the renderer switches from link rendering to direct graph
     * rendering. Rendering using xlink is much faster, but with larger graphs it may slow down the browser
     * performance. Modify this property with care. Please also keep in mind that a graph map that is rendered
     * directly does not adapt to changes until it's fully rerendered. Available only for `SVG` rendering type.
     *
     * Default value is `250`.
     *
     *
     * @returns Value of property `directRenderNodeLimit`
     */
    getDirectRenderNodeLimit(): int;
    /**
     * ID of the element which is the current target of the association {@link #getGraph graph}, or `null`.
     */
    getGraph(): ID;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * The height of the graph map.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): CSSSize;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * Graph overview title
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * The width of the graph map.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): CSSSize;
    /**
     * Sets a new value for property {@link #getDirectRenderNodeLimit directRenderNodeLimit}.
     *
     * This property affects the threshold at which the renderer switches from link rendering to direct graph
     * rendering. Rendering using xlink is much faster, but with larger graphs it may slow down the browser
     * performance. Modify this property with care. Please also keep in mind that a graph map that is rendered
     * directly does not adapt to changes until it's fully rerendered. Available only for `SVG` rendering type.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `250`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDirectRenderNodeLimit(
      /**
       * New value for property `directRenderNodeLimit`
       */
      iDirectRenderNodeLimit?: int
    ): this;
    /**
     * Sets the associated {@link #getGraph graph}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setGraph(
      /**
       * ID of an element which becomes the new target of this graph association; alternatively, an element instance
       * may be given
       */
      oGraph: ID | Graph
    ): this;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * The height of the graph map.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      sHeight?: CSSSize
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * Graph overview title
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * The width of the graph map.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `empty string`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      sWidth?: CSSSize
    ): this;
  }
  /**
   * Describes the settings that can be provided to the GraphMap constructor.
   */
  export interface $GraphMapSettings extends $ControlSettings {
    /**
     * The height of the graph map.
     */
    height?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * The width of the graph map.
     */
    width?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * This property affects the threshold at which the renderer switches from link rendering to direct graph
     * rendering. Rendering using xlink is much faster, but with larger graphs it may slow down the browser
     * performance. Modify this property with care. Please also keep in mind that a graph map that is rendered
     * directly does not adapt to changes until it's fully rerendered. Available only for `SVG` rendering type.
     */
    directRenderNodeLimit?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Graph overview title
     */
    title?: string | PropertyBindingInfo;

    /**
     * An association to a graph displayed by this component.
     */
    graph?: Graph | string;

    /**
     * This event is fired when the map is fully rendered.
     */
    mapReady?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the GraphMap#mapReady event.
   */
  export interface GraphMap$MapReadyEventParameters {}

  /**
   * Event object of the GraphMap#mapReady event.
   */
  export type GraphMap$MapReadyEvent = Event<
    GraphMap$MapReadyEventParameters,
    GraphMap
  >;
}

declare module "sap/suite/ui/commons/networkgraph/Group" {
  import {
    default as ElementBase,
    $ElementBaseSettings,
  } from "sap/suite/ui/commons/networkgraph/ElementBase";

  import Event from "sap/ui/base/Event";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Holds details about a group.
   *
   * @since 1.50
   */
  export default class Group extends ElementBase {
    /**
     * Constructor for a new Group. Grouping is not supported for custom layouting.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.Group with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.networkgraph.ElementBase.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Group>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.Group.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:collapseExpand collapseExpand} event of this
     * `sap.suite.ui.commons.networkgraph.Group`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Group` itself.
     *
     * This event is fired when the user clicks or taps the collapse/expand button.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCollapseExpand(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Group`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:collapseExpand collapseExpand} event of this
     * `sap.suite.ui.commons.networkgraph.Group`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Group` itself.
     *
     * This event is fired when the user clicks or taps the collapse/expand button.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCollapseExpand(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Group`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:headerCheckBoxPress headerCheckBoxPress} event
     * of this `sap.suite.ui.commons.networkgraph.Group`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Group` itself.
     *
     * This event is fired when the user selects or clears the Header checkbox.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHeaderCheckBoxPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Group$HeaderCheckBoxPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Group`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:headerCheckBoxPress headerCheckBoxPress} event
     * of this `sap.suite.ui.commons.networkgraph.Group`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Group` itself.
     *
     * This event is fired when the user selects or clears the Header checkbox.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHeaderCheckBoxPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Group$HeaderCheckBoxPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Group`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:showDetail showDetail} event of this `sap.suite.ui.commons.networkgraph.Group`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Group` itself.
     *
     * This event is fired when the user clicks or taps the group's details button.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachShowDetail(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Group`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:showDetail showDetail} event of this `sap.suite.ui.commons.networkgraph.Group`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Group` itself.
     *
     * This event is fired when the user clicks or taps the group's details button.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachShowDetail(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Group`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:collapseExpand collapseExpand} event of this
     * `sap.suite.ui.commons.networkgraph.Group`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachCollapseExpand(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:headerCheckBoxPress headerCheckBoxPress} event
     * of this `sap.suite.ui.commons.networkgraph.Group`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachHeaderCheckBoxPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Group$HeaderCheckBoxPressEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:showDetail showDetail} event of this `sap.suite.ui.commons.networkgraph.Group`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachShowDetail(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:collapseExpand collapseExpand} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireCollapseExpand(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:headerCheckBoxPress headerCheckBoxPress} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireHeaderCheckBoxPress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: Group$HeaderCheckBoxPressEventParameters
    ): this;
    /**
     * Fires event {@link #event:showDetail showDetail} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireShowDetail(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     *
     * @returns Returns all lines connected to the group's child nodes.
     */
    getChildLines(): any[];
    /**
     *
     * @returns Returns all child nodes of the group.
     */
    getChildNodes(): any[];
    /**
     * Gets current value of property {@link #getCollapsed collapsed}.
     *
     * Defines whether the group is collapsed. By default, it is expanded.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `collapsed`
     */
    getCollapsed(): boolean;
    /**
     * Returns header checkbox object
     */
    getHeaderCheckbox(): void;
    /**
     * Gets current value of property {@link #getHeaderCheckBoxState headerCheckBoxState}.
     *
     * Defines if the Header checkbox should be displayed and whether it should be selected or not. By default,
     * the checkbox is hidden.
     *
     * Default value is `Hidden`.
     *
     *
     * @returns Value of property `headerCheckBoxState`
     */
    getHeaderCheckBoxState():
      | networkgraph.HeaderCheckboxState
      | keyof typeof networkgraph.HeaderCheckboxState;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * An icon associated with the group.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): string;
    /**
     * Gets current value of property {@link #getKey key}.
     *
     * The key of the group. This key is used to assign nodes to groups.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Gets current value of property {@link #getMinWidth minWidth}.
     *
     * Minimal width of the group (swim lane) for the `SwimLanes` type of {@link sap.suite.ui.commons.networkgraph.LayoutRenderType}.
     *  If not specified, the width of the group (swim lane) is defined automatically, based on the width of
     * the nodes included in the swim lane.
     *  If the width specified in this property is too small to fit the nodes included in this group (swim lane),
     * the width is increased accordingly.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Value of property `minWidth`
     */
    getMinWidth(): float;
    /**
     *
     * @returns Returns all nodes belonging to this group.
     */
    getNodes(): any[];
    /**
     * Gets current value of property {@link #getParentGroupKey parentGroupKey}.
     *
     * Key of the parent group. Can be used only with the `TwoColumns` layout.
     *
     *
     * @returns Value of property `parentGroupKey`
     */
    getParentGroupKey(): string;
    /**
     *
     * @returns Returns all lines connected to the parent nodes of the group.
     */
    getParentLines(): any[];
    /**
     *
     * @returns Returns all parent nodes of the group.
     */
    getParentNodes(): any[];
    /**
     * Sets a new value for property {@link #getCollapsed collapsed}.
     *
     * Defines whether the group is collapsed. By default, it is expanded.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCollapsed(
      /**
       * New value for property `collapsed`
       */
      bCollapsed?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getHeaderCheckBoxState headerCheckBoxState}.
     *
     * Defines if the Header checkbox should be displayed and whether it should be selected or not. By default,
     * the checkbox is hidden.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Hidden`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeaderCheckBoxState(
      /**
       * New value for property `headerCheckBoxState`
       */
      sHeaderCheckBoxState?:
        | networkgraph.HeaderCheckboxState
        | keyof typeof networkgraph.HeaderCheckboxState
    ): this;
    /**
     * Hides the group and the nodes included in it.
     */
    setHidden(): void;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * An icon associated with the group.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: string
    ): this;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * The key of the group. This key is used to assign nodes to groups.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
    /**
     * Sets a new value for property {@link #getMinWidth minWidth}.
     *
     * Minimal width of the group (swim lane) for the `SwimLanes` type of {@link sap.suite.ui.commons.networkgraph.LayoutRenderType}.
     *  If not specified, the width of the group (swim lane) is defined automatically, based on the width of
     * the nodes included in the swim lane.
     *  If the width specified in this property is too small to fit the nodes included in this group (swim lane),
     * the width is increased accordingly.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMinWidth(
      /**
       * New value for property `minWidth`
       */
      fMinWidth?: float
    ): this;
    /**
     * Sets a new value for property {@link #getParentGroupKey parentGroupKey}.
     *
     * Key of the parent group. Can be used only with the `TwoColumns` layout.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setParentGroupKey(
      /**
       * New value for property `parentGroupKey`
       */
      sParentGroupKey?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the Group constructor.
   */
  export interface $GroupSettings extends $ElementBaseSettings {
    /**
     * Defines if the Header checkbox should be displayed and whether it should be selected or not. By default,
     * the checkbox is hidden.
     */
    headerCheckBoxState?:
      | (
          | networkgraph.HeaderCheckboxState
          | keyof typeof networkgraph.HeaderCheckboxState
        )
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * The key of the group. This key is used to assign nodes to groups.
     */
    key?: string | PropertyBindingInfo;

    /**
     * Defines whether the group is collapsed. By default, it is expanded.
     */
    collapsed?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * An icon associated with the group.
     */
    icon?: string | PropertyBindingInfo;

    /**
     * Minimal width of the group (swim lane) for the `SwimLanes` type of {@link sap.suite.ui.commons.networkgraph.LayoutRenderType}.
     *  If not specified, the width of the group (swim lane) is defined automatically, based on the width of
     * the nodes included in the swim lane.
     *  If the width specified in this property is too small to fit the nodes included in this group (swim lane),
     * the width is increased accordingly.
     */
    minWidth?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Key of the parent group. Can be used only with the `TwoColumns` layout.
     */
    parentGroupKey?: string | PropertyBindingInfo;

    /**
     * This event is fired when the user selects or clears the Header checkbox.
     */
    headerCheckBoxPress?: (oEvent: Group$HeaderCheckBoxPressEvent) => void;

    /**
     * This event is fired when the user clicks or taps the collapse/expand button.
     */
    collapseExpand?: (oEvent: Event) => void;

    /**
     * This event is fired when the user clicks or taps the group's details button.
     */
    showDetail?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the Group#collapseExpand event.
   */
  export interface Group$CollapseExpandEventParameters {}

  /**
   * Event object of the Group#collapseExpand event.
   */
  export type Group$CollapseExpandEvent = Event<
    Group$CollapseExpandEventParameters,
    Group
  >;

  /**
   * Parameters of the Group#headerCheckBoxPress event.
   */
  export interface Group$HeaderCheckBoxPressEventParameters {
    /**
     * Indicates whether checkbox is selected.
     */
    checked?: boolean;
  }

  /**
   * Event object of the Group#headerCheckBoxPress event.
   */
  export type Group$HeaderCheckBoxPressEvent = Event<
    Group$HeaderCheckBoxPressEventParameters,
    Group
  >;

  /**
   * Parameters of the Group#showDetail event.
   */
  export interface Group$ShowDetailEventParameters {}

  /**
   * Event object of the Group#showDetail event.
   */
  export type Group$ShowDetailEvent = Event<
    Group$ShowDetailEventParameters,
    Group
  >;
}

declare module "sap/suite/ui/commons/networkgraph/layout/ForceBasedLayout" {
  import {
    default as LayoutAlgorithm,
    $LayoutAlgorithmSettings,
  } from "sap/suite/ui/commons/networkgraph/layout/LayoutAlgorithm";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import LayoutTask from "sap/suite/ui/commons/networkgraph/layout/LayoutTask";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This algorithm uses D3.force algorithm to layout the graph. It's good if the graph is too complicated
   * and LayeredLayout is not sufficient.
   *
   * @since 1.50
   */
  export default class ForceBasedLayout extends LayoutAlgorithm {
    /**
     * Constructor for a new ForceBasedLayout.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.layout.ForceBasedLayout with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.networkgraph.layout.LayoutAlgorithm.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ForceBasedLayout>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.layout.ForceBasedLayout.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getAlpha alpha}.
     *
     * See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#alpha}
     *
     * Default value is `0.3`.
     *
     *
     * @returns Value of property `alpha`
     */
    getAlpha(): float;
    /**
     * Gets current value of property {@link #getCharge charge}.
     *
     * See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#charge}
     *
     * Default value is `-30`.
     *
     *
     * @returns Value of property `charge`
     */
    getCharge(): float;
    /**
     * Gets current value of property {@link #getFriction friction}.
     *
     * Value in [0,1] range. See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#friction}
     *
     * Default value is `0.9`.
     *
     *
     * @returns Value of property `friction`
     */
    getFriction(): float;
    /**
     * Gets the type of the layout algorithm that defines the visual features and layout of the network graph.
     */
    getLayoutRenderType():
      | networkgraph.LayoutRenderType
      | keyof typeof networkgraph.LayoutRenderType;
    /**
     * Gets current value of property {@link #getMaximumDuration maximumDuration}.
     *
     * Specifies the maximal time in miliseconds the algorithm is allowed to run.
     *
     * Default value is `1000`.
     *
     *
     * @returns Value of property `maximumDuration`
     */
    getMaximumDuration(): int;
    /**
     * Specifies if this layouting algorithm distributes nodes into layers. Parent graph may change behaviour
     * based on this option.
     *
     *
     * @returns Always false
     */
    isLayered(): boolean;
    /**
     * Executes the layouting algorithm.
     *
     *
     * @returns Task to get the layout calculated.
     */
    layout(): LayoutTask;
    /**
     * Sets a new value for property {@link #getAlpha alpha}.
     *
     * See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#alpha}
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0.3`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAlpha(
      /**
       * New value for property `alpha`
       */
      fAlpha?: float
    ): this;
    /**
     * Sets a new value for property {@link #getCharge charge}.
     *
     * See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#charge}
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `-30`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCharge(
      /**
       * New value for property `charge`
       */
      fCharge?: float
    ): this;
    /**
     * Sets a new value for property {@link #getFriction friction}.
     *
     * Value in [0,1] range. See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#friction}
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0.9`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFriction(
      /**
       * New value for property `friction`
       */
      fFriction?: float
    ): this;
    /**
     * Sets a new value for property {@link #getMaximumDuration maximumDuration}.
     *
     * Specifies the maximal time in miliseconds the algorithm is allowed to run.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `1000`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMaximumDuration(
      /**
       * New value for property `maximumDuration`
       */
      iMaximumDuration?: int
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ForceBasedLayout constructor.
   */
  export interface $ForceBasedLayoutSettings extends $LayoutAlgorithmSettings {
    /**
     * See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#alpha}
     */
    alpha?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#charge}
     */
    charge?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Value in [0,1] range. See: {@link https://github.com/d3/d3-3.x-api-reference/blob/master/Force-Layout.md#friction}
     */
    friction?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Specifies the maximal time in miliseconds the algorithm is allowed to run.
     */
    maximumDuration?: int | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/networkgraph/layout/ForceDirectedLayout" {
  import {
    default as LayoutAlgorithm,
    $LayoutAlgorithmSettings,
  } from "sap/suite/ui/commons/networkgraph/layout/LayoutAlgorithm";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import LayoutTask from "sap/suite/ui/commons/networkgraph/layout/LayoutTask";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This algorithm uses the deterministic variant of the force-directed layout algorithm {@link sap.suite.ui.commons.networkgraph.layout.ForceBasedLayout }
   * to arrange the nodes included in the network graph. It can be useful for complex graphs, where {@link sap.suite.ui.commons.networkgraph.layout.LayeredLayout }
   * is not sufficient.
   *
   * @since 1.58
   */
  export default class ForceDirectedLayout extends LayoutAlgorithm {
    /**
     * Constructor for a new ForceDirectedLayout.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.layout.ForceDirectedLayout with name
     * `sClassName` and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.networkgraph.layout.LayoutAlgorithm.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ForceDirectedLayout>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.layout.ForceDirectedLayout.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getCoolDownStep coolDownStep}.
     *
     * Specifies how much the 'temperature' decreases after every iteration.
     *
     * Default value is `1`.
     *
     *
     * @returns Value of property `coolDownStep`
     */
    getCoolDownStep(): float;
    /**
     * Gets current value of property {@link #getInitialTemperature initialTemperature}.
     *
     * Initial 'temperature' of the system, that controls the step width of the nodes' movements and that decreases,
     * or 'cools down', after each iteration.
     *
     * Default value is `200`.
     *
     *
     * @returns Value of property `initialTemperature`
     */
    getInitialTemperature(): float;
    /**
     * Specifies the type of layout algorithm that defines the visual features and layout of the network graph.
     */
    getLayoutRenderType():
      | networkgraph.LayoutRenderType
      | keyof typeof networkgraph.LayoutRenderType;
    /**
     * Gets current value of property {@link #getMaxIterations maxIterations}.
     *
     * Maximum number of iterations (or cooldown steps) the layouting process will go through.
     *
     * Default value is `200`.
     *
     *
     * @returns Value of property `maxIterations`
     */
    getMaxIterations(): int;
    /**
     * Gets current value of property {@link #getMaxTime maxTime}.
     *
     * Maximum time in milliseconds the layouting process will run for.
     *
     * Default value is `2000`.
     *
     *
     * @returns Value of property `maxTime`
     */
    getMaxTime(): int;
    /**
     * Gets current value of property {@link #getOptimalDistanceConstant optimalDistanceConstant}.
     *
     * The "C" constant from the optimal distance equation used in the Fruchterman-Reingold Algorithm. See {@link https://www.researchgate.net/publication/309193795_Force-Directed_Graph_Drawing_Algorithm}
     *
     * Default value is `0.2`.
     *
     *
     * @returns Value of property `optimalDistanceConstant`
     */
    getOptimalDistanceConstant(): float;
    /**
     * Gets current value of property {@link #getStaticNodes staticNodes}.
     *
     * List of keys of nodes that should ignore cooldown iterations and keep their original positions (x and
     * y coordinates) in the graph.
     *
     * Default value is `[]`.
     *
     *
     * @returns Value of property `staticNodes`
     */
    getStaticNodes(): string[];
    /**
     * Specifies if this layout algorithm distributes nodes into layers. The parent graph may affect the behaviour
     * based on this option.
     *
     *
     * @returns Always false
     */
    isLayered(): boolean;
    /**
     * Runs the layout algorithm.
     *
     *
     * @returns Task to get the layout calculated.
     */
    layout(): LayoutTask;
    /**
     * Sets a new value for property {@link #getCoolDownStep coolDownStep}.
     *
     * Specifies how much the 'temperature' decreases after every iteration.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `1`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCoolDownStep(
      /**
       * New value for property `coolDownStep`
       */
      fCoolDownStep?: float
    ): this;
    /**
     * Sets a new value for property {@link #getInitialTemperature initialTemperature}.
     *
     * Initial 'temperature' of the system, that controls the step width of the nodes' movements and that decreases,
     * or 'cools down', after each iteration.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `200`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setInitialTemperature(
      /**
       * New value for property `initialTemperature`
       */
      fInitialTemperature?: float
    ): this;
    /**
     * Sets a new value for property {@link #getMaxIterations maxIterations}.
     *
     * Maximum number of iterations (or cooldown steps) the layouting process will go through.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `200`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMaxIterations(
      /**
       * New value for property `maxIterations`
       */
      iMaxIterations?: int
    ): this;
    /**
     * Sets a new value for property {@link #getMaxTime maxTime}.
     *
     * Maximum time in milliseconds the layouting process will run for.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `2000`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMaxTime(
      /**
       * New value for property `maxTime`
       */
      iMaxTime?: int
    ): this;
    /**
     * Sets a new value for property {@link #getOptimalDistanceConstant optimalDistanceConstant}.
     *
     * The "C" constant from the optimal distance equation used in the Fruchterman-Reingold Algorithm. See {@link https://www.researchgate.net/publication/309193795_Force-Directed_Graph_Drawing_Algorithm}
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0.2`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setOptimalDistanceConstant(
      /**
       * New value for property `optimalDistanceConstant`
       */
      fOptimalDistanceConstant?: float
    ): this;
    /**
     * Sets a new value for property {@link #getStaticNodes staticNodes}.
     *
     * List of keys of nodes that should ignore cooldown iterations and keep their original positions (x and
     * y coordinates) in the graph.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `[]`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setStaticNodes(
      /**
       * New value for property `staticNodes`
       */
      sStaticNodes?: string[]
    ): this;
  }
  /**
   * Describes the settings that can be provided to the ForceDirectedLayout constructor.
   */
  export interface $ForceDirectedLayoutSettings
    extends $LayoutAlgorithmSettings {
    /**
     * The "C" constant from the optimal distance equation used in the Fruchterman-Reingold Algorithm. See {@link https://www.researchgate.net/publication/309193795_Force-Directed_Graph_Drawing_Algorithm}
     */
    optimalDistanceConstant?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Maximum number of iterations (or cooldown steps) the layouting process will go through.
     */
    maxIterations?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Maximum time in milliseconds the layouting process will run for.
     */
    maxTime?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Initial 'temperature' of the system, that controls the step width of the nodes' movements and that decreases,
     * or 'cools down', after each iteration.
     */
    initialTemperature?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Specifies how much the 'temperature' decreases after every iteration.
     */
    coolDownStep?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * List of keys of nodes that should ignore cooldown iterations and keep their original positions (x and
     * y coordinates) in the graph.
     */
    staticNodes?: string[] | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/networkgraph/layout/LayeredLayout" {
  import {
    default as LayoutAlgorithm,
    $LayoutAlgorithmSettings,
  } from "sap/suite/ui/commons/networkgraph/layout/LayoutAlgorithm";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import LayoutTask from "sap/suite/ui/commons/networkgraph/layout/LayoutTask";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This algorithm uses the klay.js algorithm to rearrange the graph in grid form. It's suitable for process
   * flows and tree-like graphs. It can be used for almost any graph.
   *
   * @since 1.50
   */
  export default class LayeredLayout extends LayoutAlgorithm {
    /**
     * Constructor for a new LayeredLayout.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.layout.LayeredLayout with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.networkgraph.layout.LayoutAlgorithm.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, LayeredLayout>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.layout.LayeredLayout.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Specifies the type of layout algorithm that defines the visual features and layout of the network graph.
     */
    getLayoutRenderType():
      | networkgraph.LayoutRenderType
      | keyof typeof networkgraph.LayoutRenderType;
    /**
     * Gets current value of property {@link #getLineSpacingFactor lineSpacingFactor}.
     *
     * Default value is `0.25`.
     *
     *
     * @returns Value of property `lineSpacingFactor`
     */
    getLineSpacingFactor(): float;
    /**
     * Gets current value of property {@link #getMergeEdges mergeEdges}.
     *
     * Determines if all lines should lead to the same place in the node, or if each line should point to a
     * different place.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `mergeEdges`
     */
    getMergeEdges(): boolean;
    /**
     * Gets current value of property {@link #getNodePlacement nodePlacement}.
     *
     * A node placement strategy to use (see {@link sap.suite.ui.commons.networkgraph.NodePlacement}).
     *
     * Default value is `BrandesKoepf`.
     *
     *
     * @returns Value of property `nodePlacement`
     */
    getNodePlacement():
      | networkgraph.NodePlacement
      | keyof typeof networkgraph.NodePlacement;
    /**
     * Gets current value of property {@link #getNodeSpacing nodeSpacing}.
     *
     * Define a minimal distance on nodes the algorithm will try to keep. The default value is 55. Note that
     * values below 50 are incompatible with presence of groups due to insufficient space for group title bars
     * and space between nodes and their groups' borders.
     *
     * Default value is `55`.
     *
     *
     * @returns Value of property `nodeSpacing`
     */
    getNodeSpacing(): float;
    /**
     * Executes the layout algorithm.
     *
     *
     * @returns Task to get the layout calculated.
     */
    layout(): LayoutTask;
    /**
     * Sets a new value for property {@link #getLineSpacingFactor lineSpacingFactor}.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0.25`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLineSpacingFactor(
      /**
       * New value for property `lineSpacingFactor`
       */
      fLineSpacingFactor?: float
    ): this;
    /**
     * Sets a new value for property {@link #getMergeEdges mergeEdges}.
     *
     * Determines if all lines should lead to the same place in the node, or if each line should point to a
     * different place.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMergeEdges(
      /**
       * New value for property `mergeEdges`
       */
      bMergeEdges?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getNodePlacement nodePlacement}.
     *
     * A node placement strategy to use (see {@link sap.suite.ui.commons.networkgraph.NodePlacement}).
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `BrandesKoepf`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setNodePlacement(
      /**
       * New value for property `nodePlacement`
       */
      sNodePlacement?:
        | networkgraph.NodePlacement
        | keyof typeof networkgraph.NodePlacement
    ): this;
    /**
     * Sets a new value for property {@link #getNodeSpacing nodeSpacing}.
     *
     * Define a minimal distance on nodes the algorithm will try to keep. The default value is 55. Note that
     * values below 50 are incompatible with presence of groups due to insufficient space for group title bars
     * and space between nodes and their groups' borders.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `55`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setNodeSpacing(
      /**
       * New value for property `nodeSpacing`
       */
      fNodeSpacing?: float
    ): this;
  }
  /**
   * Describes the settings that can be provided to the LayeredLayout constructor.
   */
  export interface $LayeredLayoutSettings extends $LayoutAlgorithmSettings {
    /**
     * Define a minimal distance on nodes the algorithm will try to keep. The default value is 55. Note that
     * values below 50 are incompatible with presence of groups due to insufficient space for group title bars
     * and space between nodes and their groups' borders.
     */
    nodeSpacing?: float | PropertyBindingInfo | `{${string}}`;

    lineSpacingFactor?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * A node placement strategy to use (see {@link sap.suite.ui.commons.networkgraph.NodePlacement}).
     */
    nodePlacement?:
      | (networkgraph.NodePlacement | keyof typeof networkgraph.NodePlacement)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Determines if all lines should lead to the same place in the node, or if each line should point to a
     * different place.
     */
    mergeEdges?: boolean | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/networkgraph/layout/LayoutAlgorithm" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import LayoutTask from "sap/suite/ui/commons/networkgraph/layout/LayoutTask";

  /**
   * This is an abstract base class for Layout Algorithms.
   *
   * @since 1.50
   */
  export default abstract class LayoutAlgorithm extends UI5Element {
    /**
     * Constructor for a new LayoutingAlgorithm.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.ui.core.Element#constructor sap.ui.core.Element }
     * can be used.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.layout.LayoutAlgorithm with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, LayoutAlgorithm>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.layout.LayoutAlgorithm.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    _alignCoordinatesWithView(): void;
    /**
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns True if there are hierarchical groups, false otherwise.
     */
    _hasHierarchicalGroups(): boolean;
    /**
     * Specifies the type of layout algorithm that defines the visual features and layout of the network graph.
     */
    getType(): /* was: sap.suite.ui.commons.networkgraph.LayoutType */ any;
    /**
     * Specifies if this layouting algorithm distributes nodes into layers. Parent graph may change behaviour
     * based on this option.
     */
    isLayered(): boolean;
    /**
     * Executes the layouting algorithm.
     */
    layout(): LayoutTask;
  }
  /**
   * Describes the settings that can be provided to the LayoutAlgorithm constructor.
   */
  export interface $LayoutAlgorithmSettings extends $ElementSettings {}
}

declare module "sap/suite/ui/commons/networkgraph/layout/LayoutTask" {
  import BaseObject from "sap/ui/base/Object";

  import Metadata from "sap/ui/base/Metadata";

  /**
   * A layout task is a wrapper around Promise that allows the caller to terminate the task. A terminated
   * layout task should not modify the graph associated with it.
   *
   * @since 1.50
   */
  export default class LayoutTask extends BaseObject {
    /**
     * Constructor for a new LayoutTask.
     */
    constructor(
      /**
       * A function to be called from Promise main body.
       */
      fnPromiseCallback: (p1: Function, p2: Function, p3: LayoutTask) => void
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.layout.LayoutTask with name `sClassName`
     * and enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.base.Object.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, LayoutTask>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.layout.LayoutTask.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): Metadata;
  }
}

declare module "sap/suite/ui/commons/networkgraph/layout/NoopLayout" {
  import {
    default as LayoutAlgorithm,
    $LayoutAlgorithmSettings,
  } from "sap/suite/ui/commons/networkgraph/layout/LayoutAlgorithm";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import LayoutTask from "sap/suite/ui/commons/networkgraph/layout/LayoutTask";

  /**
   * This is a simple layout algorithm that expects the positions of nodes to be already present. It only
   * creates line coordinates (see {@link sap.suite.ui.commons.networkgraph.Line#setCoordinates}).
   *
   * @since 1.50
   */
  export default class NoopLayout extends LayoutAlgorithm {
    /**
     * Constructor for a new NoopLayout.
     */
    constructor();

    /**
     * Specifies the type of layout algorithm that defines the visual features and layout of the network graph.
     */
    getLayoutRenderType():
      | networkgraph.LayoutRenderType
      | keyof typeof networkgraph.LayoutRenderType;
    /**
     * Executes the layout algorithm.
     *
     *
     * @returns Task to get the layout calculated.
     */
    layout(): LayoutTask;
  }
  /**
   * Describes the settings that can be provided to the NoopLayout constructor.
   */
  export interface $NoopLayoutSettings extends $LayoutAlgorithmSettings {}
}

declare module "sap/suite/ui/commons/networkgraph/layout/SwimLaneChainLayout" {
  import {
    default as LayoutAlgorithm,
    $LayoutAlgorithmSettings,
  } from "sap/suite/ui/commons/networkgraph/layout/LayoutAlgorithm";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import LayoutTask from "sap/suite/ui/commons/networkgraph/layout/LayoutTask";

  /**
   * This algorithm uses the klay.js algorithm to rearrange the graph in grid form. It's suitable for process
   * flows and tree-like graphs. It can be used for almost any graph.
   *
   * @since 1.58
   */
  export default class SwimLaneChainLayout extends LayoutAlgorithm {
    /**
     * Constructor for a new SwimLaneChainLayout.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.suite.ui.commons.networkgraph.layout.LayoutAlgorithm#constructor sap.suite.ui.commons.networkgraph.layout.LayoutAlgorithm }
     * can be used.
     */
    constructor();

    /**
     * In this override of the generic method for all layouts we want to avoid global margin.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     */
    _alignCoordinatesWithView(): void;
    /**
     * Specifies the type of layout algorithm that defines the visual features and layout of the network graph.
     */
    getLayoutRenderType():
      | networkgraph.LayoutRenderType
      | keyof typeof networkgraph.LayoutRenderType;
    /**
     * Executes the layout algorithm.
     *
     *
     * @returns Task to get the layout calculated.
     */
    layout(): LayoutTask;
  }
  /**
   * Describes the settings that can be provided to the SwimLaneChainLayout constructor.
   */
  export interface $SwimLaneChainLayoutSettings
    extends $LayoutAlgorithmSettings {}
}

declare module "sap/suite/ui/commons/networkgraph/layout/TwoColumnsLayout" {
  import {
    default as LayoutAlgorithm,
    $LayoutAlgorithmSettings,
  } from "sap/suite/ui/commons/networkgraph/layout/LayoutAlgorithm";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import LayoutTask from "sap/suite/ui/commons/networkgraph/layout/LayoutTask";

  /**
   * This algorithm rearranges the graph into two columns.
   *
   * @since 1.63
   */
  export default class TwoColumnsLayout extends LayoutAlgorithm {
    /**
     * Constructor for a new TwoColumnsLayout.
     */
    constructor();

    /**
     * Specifies the type of layout algorithm that defines the visual features and layout of the network graph.
     */
    getLayoutRenderType():
      | networkgraph.LayoutRenderType
      | keyof typeof networkgraph.LayoutRenderType;
    /**
     * Executes the layout algorithm.
     *
     *
     * @returns Task to get the layout calculated.
     */
    layout(): LayoutTask;
  }
  /**
   * Describes the settings that can be provided to the TwoColumnsLayout constructor.
   */
  export interface $TwoColumnsLayoutSettings extends $LayoutAlgorithmSettings {}
}

declare module "sap/suite/ui/commons/networkgraph/Line" {
  import {
    default as ElementBase,
    $ElementBaseSettings,
  } from "sap/suite/ui/commons/networkgraph/ElementBase";

  import ActionButton from "sap/suite/ui/commons/networkgraph/ActionButton";

  import Coordinate from "sap/suite/ui/commons/networkgraph/Coordinate";

  import Event from "sap/ui/base/Event";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * Holds information about one connector line.
   *
   * @since 1.50
   */
  export default class Line extends ElementBase {
    /**
     * Constructor for a new Line.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.Line with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.networkgraph.ElementBase.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Line>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.Line.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some actionButton to the aggregation {@link #getActionButtons actionButtons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addActionButton(
      /**
       * The actionButton to add; if empty, nothing is inserted
       */
      oActionButton: ActionButton
    ): this;
    /**
     * Adds coordinates for points that should define the shape of the line between its start and end points.
     * This method doesn't call invalidate on the object.
     *
     *
     * @returns Newly added coordinates object
     */
    addBend(
      /**
       * X and Y coordinates
       */
      oPoint: {
        x: float;

        y: float;
      }
    ): Coordinate;
    /**
     * Adds some coordinate to the aggregation {@link #getCoordinates coordinates}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addCoordinate(
      /**
       * The coordinate to add; if empty, nothing is inserted
       */
      oCoordinate: Coordinate
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:hover hover} event of this `sap.suite.ui.commons.networkgraph.Line`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Line` itself.
     *
     * This event is fired when the user moves the mouse pointer over the line.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHover(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Line`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:hover hover} event of this `sap.suite.ui.commons.networkgraph.Line`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Line` itself.
     *
     * This event is fired when the user moves the mouse pointer over the line.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHover(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Line`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.Line`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Line` itself.
     *
     * This event is fired when the user clicks or taps the line.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Line$PressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Line`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.Line`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Line` itself.
     *
     * This event is fired when the user clicks or taps the line.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Line$PressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Line`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Removes all points that define the shape of the line between its start and end points. This method doesn't
     * call invalidate on the object.
     */
    clearBends(): void;
    /**
     * Destroys all the actionButtons in the aggregation {@link #getActionButtons actionButtons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyActionButtons(): this;
    /**
     * Destroys all the coordinates in the aggregation {@link #getCoordinates coordinates}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCoordinates(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:hover hover} event of this `sap.suite.ui.commons.networkgraph.Line`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachHover(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.Line`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Line$PressEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:hover hover} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireHover(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: Line$PressEventParameters
    ): this;
    /**
     * Gets content of aggregation {@link #getActionButtons actionButtons}.
     *
     * A list of custom action buttons.
     */
    getActionButtons(): ActionButton[];
    /**
     * Gets current value of property {@link #getArrowOrientation arrowOrientation}.
     *
     * Orientation of the line that defines the direction of the arrow.
     *
     * Default value is `ParentOf`.
     *
     *
     * @returns Value of property `arrowOrientation`
     */
    getArrowOrientation():
      | networkgraph.LineArrowOrientation
      | keyof typeof networkgraph.LineArrowOrientation;
    /**
     * Gets current value of property {@link #getArrowPosition arrowPosition}.
     *
     * Position of the arrow on the line. Can be set to End, Middle, or Start.
     *
     * Default value is `End`.
     *
     *
     * @returns Value of property `arrowPosition`
     */
    getArrowPosition():
      | networkgraph.LineArrowPosition
      | keyof typeof networkgraph.LineArrowPosition;
    /**
     * Returns the coordinates of all points that define the shape of the line between its start and end points.
     * This method doesn't call invalidate on the object.
     *
     *
     * @returns Coordinates of the points shaping the line
     */
    getBends(): Coordinate[];
    /**
     * Gets content of aggregation {@link #getCoordinates coordinates}.
     *
     * A list of points the line goes through. After the layouting algorithm has finished arranging the graph,
     * this aggregation contains the coordinates of at least two points: the starting point and the end point
     * of the line. The rest of the points making up the line are treated as break points.
     */
    getCoordinates(): Coordinate[];
    /**
     * Gets current value of property {@link #getFrom from}.
     *
     * Key of the node where the line begins.
     *
     *
     * @returns Value of property `from`
     */
    getFrom(): string;
    /**
     * Returns the node instance where the line starts. This method doesn't call invalidate on the object.
     *
     *
     * @returns Node instance where the line starts
     */
    getFromNode(): object;
    /**
     * Gets current value of property {@link #getLineType lineType}.
     *
     * Defines the appearance of the line. Can be set to solid, dashed, or dotted.
     *
     * Default value is `Solid`.
     *
     *
     * @returns Value of property `lineType`
     */
    getLineType(): networkgraph.LineType | keyof typeof networkgraph.LineType;
    /**
     * Gets current value of property {@link #getSelected selected}.
     *
     * Shows if the line is selected. Once the line is selected, its appearance changes slightly to distinguish
     * it from other lines.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `selected`
     */
    getSelected(): boolean;
    /**
     * Returns the coordinates of the line's starting point. This method doesn't call invalidate on the object.
     *
     *
     * @returns Coordinate object
     */
    getSource(): Coordinate;
    /**
     * Gets current value of property {@link #getStretchToCenter stretchToCenter}.
     *
     * Extends the line up to the node's horizontal or vertical axis to ensure that it meets the shape's outline
     * even when a fancy shape is used.
     *  Available for custom nodes only.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `stretchToCenter`
     */
    getStretchToCenter(): boolean;
    /**
     * Returns the coordinates of the line's end point. This method doesn't call invalidate on the object.
     *
     *
     * @returns Coordinate object
     */
    getTarget(): Coordinate;
    /**
     * Gets current value of property {@link #getTo to}.
     *
     * Key of the node the line leads to.
     *
     *
     * @returns Value of property `to`
     */
    getTo(): string;
    /**
     * Returns the node instance where the line leads to. This method doesn't call invalidate on the object.
     *
     *
     * @returns Node instance where the line ends
     */
    getToNode(): object;
    /**
     * Checks for the provided `sap.suite.ui.commons.networkgraph.ActionButton` in the aggregation {@link #getActionButtons actionButtons}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfActionButton(
      /**
       * The actionButton whose index is looked for
       */
      oActionButton: ActionButton
    ): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.networkgraph.Coordinate` in the aggregation {@link #getCoordinates coordinates}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfCoordinate(
      /**
       * The coordinate whose index is looked for
       */
      oCoordinate: Coordinate
    ): int;
    /**
     * Inserts a actionButton into the aggregation {@link #getActionButtons actionButtons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertActionButton(
      /**
       * The actionButton to insert; if empty, nothing is inserted
       */
      oActionButton: ActionButton,
      /**
       * The `0`-based index the actionButton should be inserted at; for a negative value of `iIndex`, the actionButton
       * is inserted at position 0; for a value greater than the current size of the aggregation, the actionButton
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a coordinate into the aggregation {@link #getCoordinates coordinates}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertCoordinate(
      /**
       * The coordinate to insert; if empty, nothing is inserted
       */
      oCoordinate: Coordinate,
      /**
       * The `0`-based index the coordinate should be inserted at; for a negative value of `iIndex`, the coordinate
       * is inserted at position 0; for a value greater than the current size of the aggregation, the coordinate
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes a actionButton from the aggregation {@link #getActionButtons actionButtons}.
     *
     *
     * @returns The removed actionButton or `null`
     */
    removeActionButton(
      /**
       * The actionButton to remove or its index or id
       */
      vActionButton: int | string | ActionButton
    ): ActionButton | null;
    /**
     * Removes all the controls from the aggregation {@link #getActionButtons actionButtons}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllActionButtons(): ActionButton[];
    /**
     * Removes all the controls from the aggregation {@link #getCoordinates coordinates}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllCoordinates(): Coordinate[];
    /**
     * Removes a coordinate from the aggregation {@link #getCoordinates coordinates}.
     *
     *
     * @returns The removed coordinate or `null`
     */
    removeCoordinate(
      /**
       * The coordinate to remove or its index or id
       */
      vCoordinate: int | string | Coordinate
    ): Coordinate | null;
    /**
     * Sets a new value for property {@link #getArrowOrientation arrowOrientation}.
     *
     * Orientation of the line that defines the direction of the arrow.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `ParentOf`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setArrowOrientation(
      /**
       * New value for property `arrowOrientation`
       */
      sArrowOrientation?:
        | networkgraph.LineArrowOrientation
        | keyof typeof networkgraph.LineArrowOrientation
    ): this;
    /**
     * Sets a new value for property {@link #getArrowPosition arrowPosition}.
     *
     * Position of the arrow on the line. Can be set to End, Middle, or Start.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `End`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setArrowPosition(
      /**
       * New value for property `arrowPosition`
       */
      sArrowPosition?:
        | networkgraph.LineArrowPosition
        | keyof typeof networkgraph.LineArrowPosition
    ): this;
    /**
     * Sets a new value for property {@link #getFrom from}.
     *
     * Key of the node where the line begins.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFrom(
      /**
       * New value for property `from`
       */
      sFrom?: string
    ): this;
    /**
     * Hides the line.
     */
    setHidden(): void;
    /**
     * Sets a new value for property {@link #getLineType lineType}.
     *
     * Defines the appearance of the line. Can be set to solid, dashed, or dotted.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Solid`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLineType(
      /**
       * New value for property `lineType`
       */
      sLineType?: networkgraph.LineType | keyof typeof networkgraph.LineType
    ): this;
    /**
     * Sets a new value for property {@link #getSelected selected}.
     *
     * Shows if the line is selected. Once the line is selected, its appearance changes slightly to distinguish
     * it from other lines.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSelected(
      /**
       * New value for property `selected`
       */
      bSelected?: boolean
    ): this;
    /**
     * Sets the starting point, or the source, for the line. This method doesn't call invalidate on the object.
     */
    setSource(
      /**
       * mArguments.x mArguments.y X and Y coordinates of the starting point
       */
      mArguments: object
    ): void;
    /**
     * Sets a new value for property {@link #getStretchToCenter stretchToCenter}.
     *
     * Extends the line up to the node's horizontal or vertical axis to ensure that it meets the shape's outline
     * even when a fancy shape is used.
     *  Available for custom nodes only.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setStretchToCenter(
      /**
       * New value for property `stretchToCenter`
       */
      bStretchToCenter?: boolean
    ): this;
    /**
     * Sets the end point, or the target, for the line. This method doesn't call invalidate on the object.
     */
    setTarget(
      /**
       * mArguments.x mArguments.y X and Y coordinates of the end point
       */
      mArguments: object
    ): void;
    /**
     * Sets a new value for property {@link #getTo to}.
     *
     * Key of the node the line leads to.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTo(
      /**
       * New value for property `to`
       */
      sTo?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the Line constructor.
   */
  export interface $LineSettings extends $ElementBaseSettings {
    /**
     * Shows if the line is selected. Once the line is selected, its appearance changes slightly to distinguish
     * it from other lines.
     */
    selected?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Key of the node where the line begins.
     */
    from?: string | PropertyBindingInfo;

    /**
     * Key of the node the line leads to.
     */
    to?: string | PropertyBindingInfo;

    /**
     * Defines the appearance of the line. Can be set to solid, dashed, or dotted.
     */
    lineType?:
      | (networkgraph.LineType | keyof typeof networkgraph.LineType)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Position of the arrow on the line. Can be set to End, Middle, or Start.
     */
    arrowPosition?:
      | (
          | networkgraph.LineArrowPosition
          | keyof typeof networkgraph.LineArrowPosition
        )
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Orientation of the line that defines the direction of the arrow.
     */
    arrowOrientation?:
      | (
          | networkgraph.LineArrowOrientation
          | keyof typeof networkgraph.LineArrowOrientation
        )
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Extends the line up to the node's horizontal or vertical axis to ensure that it meets the shape's outline
     * even when a fancy shape is used.
     *  Available for custom nodes only.
     */
    stretchToCenter?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * A list of points the line goes through. After the layouting algorithm has finished arranging the graph,
     * this aggregation contains the coordinates of at least two points: the starting point and the end point
     * of the line. The rest of the points making up the line are treated as break points.
     */
    coordinates?:
      | Coordinate[]
      | Coordinate
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * A list of custom action buttons.
     */
    actionButtons?:
      | ActionButton[]
      | ActionButton
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * This event is fired when the user moves the mouse pointer over the line.
     */
    hover?: (oEvent: Event) => void;

    /**
     * This event is fired when the user clicks or taps the line.
     */
    press?: (oEvent: Line$PressEvent) => void;
  }

  /**
   * Parameters of the Line#hover event.
   */
  export interface Line$HoverEventParameters {}

  /**
   * Event object of the Line#hover event.
   */
  export type Line$HoverEvent = Event<Line$HoverEventParameters, Line>;

  /**
   * Parameters of the Line#press event.
   */
  export interface Line$PressEventParameters {
    /**
     * Coordinates of the cursor when pressed.
     */
    point?: Object;

    /**
     * Object you can pass to 'openBy' method for custom tooltip. Its important for lines where you want to
     * display tooltip precisely where the cursor is.
     */
    opener?: Object;
  }

  /**
   * Event object of the Line#press event.
   */
  export type Line$PressEvent = Event<Line$PressEventParameters, Line>;
}

declare module "sap/suite/ui/commons/networkgraph/Node" {
  import {
    default as ElementBase,
    $ElementBaseSettings,
  } from "sap/suite/ui/commons/networkgraph/ElementBase";

  import ActionButton from "sap/suite/ui/commons/networkgraph/ActionButton";

  import Control from "sap/ui/core/Control";

  import Event from "sap/ui/base/Event";

  import { networkgraph } from "sap/suite/ui/commons/library";

  import NodeImage from "sap/suite/ui/commons/networkgraph/NodeImage";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  /**
   * Holds information about one node. When changing the size of the node after the graph is rendered, you
   * have to manually invalidate the graph, which causes the layout algorithm to be applied again.
   *
   * @since 1.50
   */
  export default class Node extends ElementBase {
    /**
     * Constructor for a new Node.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.Node with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.networkgraph.ElementBase.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Node>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.Node.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some actionButton to the aggregation {@link #getActionButtons actionButtons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addActionButton(
      /**
       * The actionButton to add; if empty, nothing is inserted
       */
      oActionButton: ActionButton
    ): this;
    /**
     * Adds some actionLink to the aggregation {@link #getActionLinks actionLinks}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addActionLink(
      /**
       * The actionLink to add; if empty, nothing is inserted
       */
      oActionLink: Control
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:collapseExpand collapseExpand} event of this
     * `sap.suite.ui.commons.networkgraph.Node`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Node` itself.
     *
     * This event is fired when the user clicks the node's collapse/expand button.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCollapseExpand(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Node`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:collapseExpand collapseExpand} event of this
     * `sap.suite.ui.commons.networkgraph.Node`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Node` itself.
     *
     * This event is fired when the user clicks the node's collapse/expand button.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachCollapseExpand(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Node`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:headerCheckBoxPress headerCheckBoxPress} event
     * of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Node` itself.
     *
     * This event is fired when the user selects or clears the Header checkbox.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHeaderCheckBoxPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Node$HeaderCheckBoxPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Node`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:headerCheckBoxPress headerCheckBoxPress} event
     * of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Node` itself.
     *
     * This event is fired when the user selects or clears the Header checkbox.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHeaderCheckBoxPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Node$HeaderCheckBoxPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Node`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:hover hover} event of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Node` itself.
     *
     * This event is fired when the user moves the mouse pointer over the node.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHover(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Node`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:hover hover} event of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Node` itself.
     *
     * This event is fired when the user moves the mouse pointer over the node.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHover(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Node`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Node` itself.
     *
     * This event is fired when the user clicks or taps the node.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Node`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.networkgraph.Node` itself.
     *
     * This event is fired when the user clicks or taps the node.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.networkgraph.Node`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Determines node sizes, applies text trimming, arranges the text into lines, and makes other necessary
     * adjustments.
     *
     *
     * This method can only be overridden when custom rendering is used. It must never be called directly.
     */
    calculateSizes(): any;
    /**
     * Returns an object that includes an array of text lines and the height of a single line.
     */
    computeTextDimensions(
      /**
       * Arguments passed to the method
       */
      mArguments?: {
        /**
         * Additional attributes appended to the text
         */
        attributes?: Object;
        /**
         * Width of the text is trimmed to
         */
        width?: int;
        /**
         * Text to trim
         */
        text?: string;
      }
    ): void;
    /**
     * Destroys all the actionButtons in the aggregation {@link #getActionButtons actionButtons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyActionButtons(): this;
    /**
     * Destroys all the actionLinks in the aggregation {@link #getActionLinks actionLinks}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyActionLinks(): this;
    /**
     * Destroys the content in the aggregation {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyContent(): this;
    /**
     * Destroys the image in the aggregation {@link #getImage image}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyImage(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:collapseExpand collapseExpand} event of this
     * `sap.suite.ui.commons.networkgraph.Node`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachCollapseExpand(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:headerCheckBoxPress headerCheckBoxPress} event
     * of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachHeaderCheckBoxPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Node$HeaderCheckBoxPressEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:hover hover} event of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachHover(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:press press} event of this `sap.suite.ui.commons.networkgraph.Node`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachPress(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: Event) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:collapseExpand collapseExpand} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireCollapseExpand(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:headerCheckBoxPress headerCheckBoxPress} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireHeaderCheckBoxPress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: Node$HeaderCheckBoxPressEventParameters
    ): this;
    /**
     * Fires event {@link #event:hover hover} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireHover(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Fires event {@link #event:press press} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    firePress(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: object
    ): this;
    /**
     * Gets content of aggregation {@link #getActionButtons actionButtons}.
     *
     * A list of custom action buttons. These buttons are displayed in the button area for each node. A node
     * may have up to 4 buttons. The default 3 buttons (collapse/expand, details, and links) have priority over
     * any other custom buttons that you add. If you want all 4 of your custom buttons to be displayed, set
     * the visibility of the default buttons to false.
     */
    getActionButtons(): ActionButton[];
    /**
     * Gets content of aggregation {@link #getActionLinks actionLinks}.
     *
     * A list of links to be shown in the links area. A link may point to any UI5 control. It's up to the caller
     * to set up all necessary callback functions.
     */
    getActionLinks(): Control[];
    /**
     * Gets current value of property {@link #getAltText altText}.
     *
     * In the absence of a title, alternate texts can be used to identify the nodes.
     *
     * @since 1.112
     *
     * @returns Value of property `altText`
     */
    getAltText(): string;
    /**
     * Returns center position of the node.
     *
     *
     * @returns Object with X and Y coordinates of the center of the node. For circular nodes, this method returns
     * the center of the circle.
     */
    getCenterPosition(): Object;
    /**
     * Returns all lines connected to the child nodes.
     *
     *
     * @returns Array of lines connecting this node with its child nodes
     */
    getChildLines(): any[];
    /**
     * Returns all child nodes.
     *
     *
     * @returns Array with child nodes
     */
    getChildNodes(): any[];
    /**
     * Gets current value of property {@link #getCollapsed collapsed}.
     *
     * Defines if the subtree of this node is collapsed. By default, it is expanded.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `collapsed`
     */
    getCollapsed(): boolean;
    /**
     * Gets content of aggregation {@link #getContent content}.
     *
     * Custom content rendered with the node. Available only for rectangular nodes with HTML rendering type.
     * When this aggregation is used, the description and attributes of {@link sap.suite.ui.commons.networkgraph.ElementBase }
     * are ignored.
     */
    getContent(): Control;
    /**
     * Gets current value of property {@link #getCoreNodeSize coreNodeSize}.
     *
     * Determines the size of a custom node, which can be useful for nodes that have text or other content outside
     * of its shape.
     * This property is available for custom nodes only.
     *
     * Default value is `0`.
     *
     *
     * @returns Value of property `coreNodeSize`
     */
    getCoreNodeSize(): int;
    /**
     * Gets current value of property {@link #getDescriptionLineSize descriptionLineSize}.
     *
     * Determines the maximum number of lines allowed to be displayed in the node's description. If you want
     * to hide the description, set this property to `-1` (default). To display an unlimited number of lines,
     * set this property to `0`. This property does not affect the description that appears in the tooltip,
     * which is always fully rendered.
     *
     * Default value is `-1`.
     *
     *
     * @returns Value of property `descriptionLineSize`
     */
    getDescriptionLineSize(): int;
    /**
     * Gets the node's action buttons that are enabled.
     *
     *
     * @returns Returns an array of enabled action buttons.
     */
    getEnabledActionButtons(): any[];
    /**
     * Gets current value of property {@link #getGroup group}.
     *
     * Key of the group where this node is included.
     *
     *
     * @returns Value of property `group`
     */
    getGroup(): string;
    /**
     * Returns header checkbox object
     */
    getHeaderCheckbox(): void;
    /**
     * Gets current value of property {@link #getHeaderCheckBoxState headerCheckBoxState}.
     *
     * Defines if the Header checkbox should be displayed and whether it should be selected or not. By default,
     * the checkbox is hidden. Available only for box nodes.
     *
     * Default value is `Hidden`.
     *
     *
     * @returns Value of property `headerCheckBoxState`
     */
    getHeaderCheckBoxState():
      | networkgraph.HeaderCheckboxState
      | keyof typeof networkgraph.HeaderCheckboxState;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * Height of the node. In circular nodes, it determines the circle diameter.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): int;
    /**
     * Gets current value of property {@link #getIcon icon}.
     *
     * An icon associated with the element.
     *
     *
     * @returns Value of property `icon`
     */
    getIcon(): string;
    /**
     * Gets current value of property {@link #getIconSize iconSize}.
     *
     * Determines the size of the node's icon. This property can be applied only to circular nodes.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Value of property `iconSize`
     */
    getIconSize(): int;
    /**
     * Gets content of aggregation {@link #getImage image}.
     *
     * The image that is displayed in the node's header instead of the icon.
     */
    getImage(): NodeImage;
    /**
     * Gets current value of property {@link #getKey key}.
     *
     * Key of the node. This key is used throughout the DOM to reference this node, mainly in the connector
     * line (Line) elements of the graph.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Gets current value of property {@link #getMaxWidth maxWidth}.
     *
     * Maximum width allowed. The auto grow algorithm stops increasing the width at this value.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Value of property `maxWidth`
     */
    getMaxWidth(): int;
    /**
     * Returns all lines connected to the parent nodes.
     *
     *
     * @returns Array with lines connecting this node with its parent nodes
     */
    getParentLines(): any[];
    /**
     * Returns all parent nodes.
     *
     *
     * @returns Array with all parent nodes
     */
    getParentNodes(): any[];
    /**
     * Gets current value of property {@link #getSelected selected}.
     *
     * Shows if the node is selected. Once the node is selected, its appearance changes slightly to distinguish
     * it from other nodes.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `selected`
     */
    getSelected(): boolean;
    /**
     * Gets current value of property {@link #getShape shape}.
     *
     * Shape of the node. The shape is round by default. To create a rectangular node, set this property to
     * Box.
     *
     * Default value is `Circle`.
     *
     *
     * @returns Value of property `shape`
     */
    getShape(): networkgraph.NodeShape | keyof typeof networkgraph.NodeShape;
    /**
     * Gets current value of property {@link #getShowActionLinksButton showActionLinksButton}.
     *
     * Determines if the links button is visible.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showActionLinksButton`
     */
    getShowActionLinksButton(): boolean;
    /**
     * Gets current value of property {@link #getShowDetailButton showDetailButton}.
     *
     * Determines if the details button is visible.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showDetailButton`
     */
    getShowDetailButton(): boolean;
    /**
     * Gets current value of property {@link #getShowExpandButton showExpandButton}.
     *
     * Determines if the expand button is visible.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `showExpandButton`
     */
    getShowExpandButton(): boolean;
    /**
     * Gets current value of property {@link #getStatusIcon statusIcon}.
     *
     * An additional status icon displayed when the node is collapsed.
     *
     *
     * @returns Value of property `statusIcon`
     */
    getStatusIcon(): string;
    /**
     * Gets current value of property {@link #getTitleLineSize titleLineSize}.
     *
     * Determines the maximum number of lines allowed for the node's label. If set to 0, the label may have
     * an unlimited number of lines.
     *
     * Default value is `1`.
     *
     *
     * @returns Value of property `titleLineSize`
     */
    getTitleLineSize(): int;
    /**
     * Gets current value of property {@link #getVisible visible}.
     *
     * Whether the control should be visible on the screen. Node is still used for layouter.
     *
     * Default value is `true`.
     *
     *
     * @returns Value of property `visible`
     */
    getVisible(): boolean;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Width of the node. If the width is not defined, the node expands, so it can fit the content.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): int;
    /**
     * Gets current value of property {@link #getX x}.
     *
     * The x coordinate of the node. This value must be set after the layout algorithm has finished arranging
     * the graph. It may come from the input data but is not required for most layout algorithms.
     * Works only for `CustomLayout` or `NoopLayout` layout algorithms. Other layout algorithms override this
     * property. However, you can still change it using an `afterLayouting` event.
     *
     * Default value is `0`.
     *
     *
     * @returns Value of property `x`
     */
    getX(): float;
    /**
     * Gets current value of property {@link #getY y}.
     *
     * The y coordinate of the node. This value must be set after the layout algorithm has finished arranging
     * the graph. It may come from the input data but is not required for most layout algorithms.
     * Works only for `CustomLayout` or `NoopLayout` layout algorithms. Other layout algorithms override this
     * property. However, you can still change it using an `afterLayouting` event.
     *
     * Default value is `0`.
     *
     *
     * @returns Value of property `y`
     */
    getY(): float;
    /**
     * Checks whether the node has visible action buttons
     *
     *
     * @returns Returns true if node has action buttons displayed.
     */
    hasVisibleActionButtons(): boolean;
    /**
     * Checks for the provided `sap.suite.ui.commons.networkgraph.ActionButton` in the aggregation {@link #getActionButtons actionButtons}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfActionButton(
      /**
       * The actionButton whose index is looked for
       */
      oActionButton: ActionButton
    ): int;
    /**
     * Checks for the provided `sap.ui.core.Control` in the aggregation {@link #getActionLinks actionLinks}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfActionLink(
      /**
       * The actionLink whose index is looked for
       */
      oActionLink: Control
    ): int;
    /**
     * Inserts a actionButton into the aggregation {@link #getActionButtons actionButtons}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertActionButton(
      /**
       * The actionButton to insert; if empty, nothing is inserted
       */
      oActionButton: ActionButton,
      /**
       * The `0`-based index the actionButton should be inserted at; for a negative value of `iIndex`, the actionButton
       * is inserted at position 0; for a value greater than the current size of the aggregation, the actionButton
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Inserts a actionLink into the aggregation {@link #getActionLinks actionLinks}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertActionLink(
      /**
       * The actionLink to insert; if empty, nothing is inserted
       */
      oActionLink: Control,
      /**
       * The `0`-based index the actionLink should be inserted at; for a negative value of `iIndex`, the actionLink
       * is inserted at position 0; for a value greater than the current size of the aggregation, the actionLink
       * is inserted at the last position
       */
      iIndex: int
    ): this;
    /**
     * Indicates whether the node is hidden by collapsing any of its parent nodes.
     *
     *
     * @returns `true` if the node is hidden
     */
    isHidden(): boolean | any;
    /**
     * Removes a actionButton from the aggregation {@link #getActionButtons actionButtons}.
     *
     *
     * @returns The removed actionButton or `null`
     */
    removeActionButton(
      /**
       * The actionButton to remove or its index or id
       */
      vActionButton: int | string | ActionButton
    ): ActionButton | null;
    /**
     * Removes a actionLink from the aggregation {@link #getActionLinks actionLinks}.
     *
     *
     * @returns The removed actionLink or `null`
     */
    removeActionLink(
      /**
       * The actionLink to remove or its index or id
       */
      vActionLink: int | string | Control
    ): Control | null;
    /**
     * Removes all the controls from the aggregation {@link #getActionButtons actionButtons}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllActionButtons(): ActionButton[];
    /**
     * Removes all the controls from the aggregation {@link #getActionLinks actionLinks}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllActionLinks(): Control[];
    /**
     * This method can only be overridden when custom rendering is used. It must never be called directly.
     */
    renderContent(
      /**
       * Options passed to the method
       */
      mOptions?: {
        /**
         * If `true`, the size is determined for the purposes of text trimming and other resizing operations that
         * are performed later, just before the final rendering of the graph, not when the method is called.
         *  If you don't plan to trim text or resize the node, skip the rendering if you set this property to `true`.
         */
        sizeDetermination?: boolean;
      }
    ): any;
    /**
     * Creates an SVG element string.
     *
     *
     * @returns SVG icon string
     */
    renderElement(
      /**
       * Name of the element
       */
      name?: string,
      /**
       * Attributes of the element.
       */
      mAttributes?: string,
      /**
       * Indicates whether to close the element. If set to `false`, the caller is responsible for adding a closing
       * tag.
       */
      bClose?: boolean
    ): string;
    /**
     * Renders wrappers for HTML action buttons. Used for custom rendering.
     */
    renderHtmlActionButtons(
      /**
       * Options passed to the method (optional).
       */
      mOptions?: {
        /**
         * Suffix (optional). When the suffix is specified, it is added to the IDs of the wrappers.
         */
        idSufix?: string;
      }
    ): string;
    /**
     * Creates HTML representation of an icon.
     *
     *
     * @returns HTML icon
     */
    renderHtmlIcon(
      /**
       * icon to render
       */
      sIcon: string,
      /**
       * Options passed to the method (optional).
       */
      mOptions?: Object
    ): string;
    /**
     * Renders custom info icon for HTML rendering.
     */
    renderHtmlInfoIcon(
      /**
       * Object with additional style appended to info icon
       */
      oStyle: object,
      /**
       * Options passed to the method (optional).
       */
      mOptions?: Object
    ): void;
    /**
     * Creates an SVG icon string.
     *
     *
     * @returns SVG icon string
     */
    renderIcon(
      /**
       * Arguments passed to the method
       */
      mArguments?: {
        /**
         * Icon name, based on the SAP icon font
         */
        icon?: string;
        /**
         * X coordinate that is added to node's top left x coordinate.
         */
        x?: int;
        /**
         * Y coordinate that is added to node's top left y coordinate.
         */
        y?: int;
        /**
         * Additional attributes rendered with the text
         */
        attributes?: Object;
      }
    ): string;
    /**
     * This method can only be overridden when custom rendering is used. It must never be called directly.
     */
    renderItemContent(
      /**
       * Options passed to the method
       */
      mOptions?: {
        /**
         * If `true`, the size is determined for the purposes of text trimming and other resizing operations that
         * are performed later, just before the final rendering of the graph, not when the method is called.
         *  If you don't plan to trim text or resize the node, skip the rendering if you set this property to `true`.
         */
        sizeDetermination?: boolean;
      }
    ): any;
    /**
     * Renders the status icon. Can be called directly or overridden when custom rendering is used.
     */
    renderStatusIcon(
      /**
       * Options passed to the method
       */
      mParameters?: Object
    ): void;
    /**
     * Renders the text of the node as SVG. Either text or lines of text must be passed to this method as an
     * argument.
     *
     *
     * @returns Text string rendered as SVG
     */
    renderText(
      /**
       * Arguments passed to the method
       */
      mArguments?: {
        /**
         * Text to render, available only for single-line text. Rendered as it is, cannot be used for trim or any
         * other similar operation
         */
        text?: string;
        /**
         * Array with lines of text. Words are spread across multiple lines
         */
        lines?: any[];
        /**
         * X coordinate that is added to the node's top left x coordinate
         */
        x?: int;
        /**
         * Y coordinate that is added to the node's top left y coordinate
         */
        y?: int;
        /**
         * Line size for the node, if not set, the default line size is used
         */
        lineSize?: int;
        /**
         * Additional attributes rendered with the text
         */
        attributes?: Object;
      }
    ): string;
    /**
     * Sets a new value for property {@link #getAltText altText}.
     *
     * In the absence of a title, alternate texts can be used to identify the nodes.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * @since 1.112
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAltText(
      /**
       * New value for property `altText`
       */
      sAltText?: string
    ): this;
    /**
     * Sets a new value for property {@link #getCollapsed collapsed}.
     *
     * Defines if the subtree of this node is collapsed. By default, it is expanded.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCollapsed(
      /**
       * New value for property `collapsed`
       */
      bCollapsed?: boolean
    ): this;
    /**
     * Sets the aggregated {@link #getContent content}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setContent(
      /**
       * The content to set
       */
      oContent: Control
    ): this;
    /**
     * Sets a new value for property {@link #getCoreNodeSize coreNodeSize}.
     *
     * Determines the size of a custom node, which can be useful for nodes that have text or other content outside
     * of its shape.
     * This property is available for custom nodes only.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCoreNodeSize(
      /**
       * New value for property `coreNodeSize`
       */
      iCoreNodeSize?: int
    ): this;
    /**
     * Sets a new value for property {@link #getDescriptionLineSize descriptionLineSize}.
     *
     * Determines the maximum number of lines allowed to be displayed in the node's description. If you want
     * to hide the description, set this property to `-1` (default). To display an unlimited number of lines,
     * set this property to `0`. This property does not affect the description that appears in the tooltip,
     * which is always fully rendered.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `-1`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescriptionLineSize(
      /**
       * New value for property `descriptionLineSize`
       */
      iDescriptionLineSize?: int
    ): this;
    /**
     * Sets a new value for property {@link #getGroup group}.
     *
     * Key of the group where this node is included.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setGroup(
      /**
       * New value for property `group`
       */
      sGroup?: string
    ): this;
    /**
     * Sets a new value for property {@link #getHeaderCheckBoxState headerCheckBoxState}.
     *
     * Defines if the Header checkbox should be displayed and whether it should be selected or not. By default,
     * the checkbox is hidden. Available only for box nodes.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Hidden`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeaderCheckBoxState(
      /**
       * New value for property `headerCheckBoxState`
       */
      sHeaderCheckBoxState?:
        | networkgraph.HeaderCheckboxState
        | keyof typeof networkgraph.HeaderCheckboxState
    ): this;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * Height of the node. In circular nodes, it determines the circle diameter.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      iHeight?: int
    ): this;
    /**
     * Hides the node and any lines that lead to it.
     */
    setHidden(
      /**
       * value
       */
      bValue: undefined
    ): void;
    /**
     * Sets a new value for property {@link #getIcon icon}.
     *
     * An icon associated with the element.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIcon(
      /**
       * New value for property `icon`
       */
      sIcon?: string
    ): this;
    /**
     * Sets a new value for property {@link #getIconSize iconSize}.
     *
     * Determines the size of the node's icon. This property can be applied only to circular nodes.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIconSize(
      /**
       * New value for property `iconSize`
       */
      iIconSize?: int
    ): this;
    /**
     * Sets the aggregated {@link #getImage image}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setImage(
      /**
       * The image to set
       */
      oImage: NodeImage
    ): this;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * Key of the node. This key is used throughout the DOM to reference this node, mainly in the connector
     * line (Line) elements of the graph.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
    /**
     * Sets a new value for property {@link #getMaxWidth maxWidth}.
     *
     * Maximum width allowed. The auto grow algorithm stops increasing the width at this value.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setMaxWidth(
      /**
       * New value for property `maxWidth`
       */
      iMaxWidth?: int
    ): this;
    /**
     * Sets a new value for property {@link #getSelected selected}.
     *
     * Shows if the node is selected. Once the node is selected, its appearance changes slightly to distinguish
     * it from other nodes.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSelected(
      /**
       * New value for property `selected`
       */
      bSelected?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShape shape}.
     *
     * Shape of the node. The shape is round by default. To create a rectangular node, set this property to
     * Box.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `Circle`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShape(
      /**
       * New value for property `shape`
       */
      sShape?: networkgraph.NodeShape | keyof typeof networkgraph.NodeShape
    ): this;
    /**
     * Sets a new value for property {@link #getShowActionLinksButton showActionLinksButton}.
     *
     * Determines if the links button is visible.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowActionLinksButton(
      /**
       * New value for property `showActionLinksButton`
       */
      bShowActionLinksButton?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowDetailButton showDetailButton}.
     *
     * Determines if the details button is visible.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowDetailButton(
      /**
       * New value for property `showDetailButton`
       */
      bShowDetailButton?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getShowExpandButton showExpandButton}.
     *
     * Determines if the expand button is visible.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setShowExpandButton(
      /**
       * New value for property `showExpandButton`
       */
      bShowExpandButton?: boolean
    ): this;
    /**
     * Sets the width and height of the node.
     *  If you don't want to change the `height` and `width` properties, use this method to change the internal
     * width and height.
     */
    setSize(
      /**
       * Arguments passed to the method
       */
      mArguments: {
        /**
         * The width of the node.
         */
        width?: number;
        /**
         * The height of the node.
         */
        height?: number;
        /**
         * The height of the title.
         */
        titleHeight?: number;
      }
    ): void;
    /**
     * Sets a new value for property {@link #getStatusIcon statusIcon}.
     *
     * An additional status icon displayed when the node is collapsed.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setStatusIcon(
      /**
       * New value for property `statusIcon`
       */
      sStatusIcon?: string
    ): this;
    /**
     * Sets a new value for property {@link #getTitleLineSize titleLineSize}.
     *
     * Determines the maximum number of lines allowed for the node's label. If set to 0, the label may have
     * an unlimited number of lines.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `1`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitleLineSize(
      /**
       * New value for property `titleLineSize`
       */
      iTitleLineSize?: int
    ): this;
    /**
     * Sets a new value for property {@link #getVisible visible}.
     *
     * Whether the control should be visible on the screen. Node is still used for layouter.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `true`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setVisible(
      /**
       * New value for property `visible`
       */
      bVisible?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Width of the node. If the width is not defined, the node expands, so it can fit the content.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `undefined`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      iWidth?: int
    ): this;
    /**
     * Sets a new value for property {@link #getX x}.
     *
     * The x coordinate of the node. This value must be set after the layout algorithm has finished arranging
     * the graph. It may come from the input data but is not required for most layout algorithms.
     * Works only for `CustomLayout` or `NoopLayout` layout algorithms. Other layout algorithms override this
     * property. However, you can still change it using an `afterLayouting` event.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setX(
      /**
       * New value for property `x`
       */
      fX?: float
    ): this;
    /**
     * Sets a new value for property {@link #getY y}.
     *
     * The y coordinate of the node. This value must be set after the layout algorithm has finished arranging
     * the graph. It may come from the input data but is not required for most layout algorithms.
     * Works only for `CustomLayout` or `NoopLayout` layout algorithms. Other layout algorithms override this
     * property. However, you can still change it using an `afterLayouting` event.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `0`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setY(
      /**
       * New value for property `y`
       */
      fY?: float
    ): this;
    /**
     * Shows or hides the node's action buttons.
     */
    showActionButtons(
      /**
       * Indicates whether to hide or to show buttons
       */
      bShow: boolean
    ): void;
  }
  /**
   * Describes the settings that can be provided to the Node constructor.
   */
  export interface $NodeSettings extends $ElementBaseSettings {
    /**
     * Defines if the Header checkbox should be displayed and whether it should be selected or not. By default,
     * the checkbox is hidden. Available only for box nodes.
     */
    headerCheckBoxState?:
      | (
          | networkgraph.HeaderCheckboxState
          | keyof typeof networkgraph.HeaderCheckboxState
        )
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * Defines if the subtree of this node is collapsed. By default, it is expanded.
     */
    collapsed?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Shows if the node is selected. Once the node is selected, its appearance changes slightly to distinguish
     * it from other nodes.
     */
    selected?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Key of the group where this node is included.
     */
    group?: string | PropertyBindingInfo;

    /**
     * Key of the node. This key is used throughout the DOM to reference this node, mainly in the connector
     * line (Line) elements of the graph.
     */
    key?: string | PropertyBindingInfo;

    /**
     * Shape of the node. The shape is round by default. To create a rectangular node, set this property to
     * Box.
     */
    shape?:
      | (networkgraph.NodeShape | keyof typeof networkgraph.NodeShape)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * An icon associated with the element.
     */
    icon?: string | PropertyBindingInfo;

    /**
     * An additional status icon displayed when the node is collapsed.
     */
    statusIcon?: string | PropertyBindingInfo;

    /**
     * Width of the node. If the width is not defined, the node expands, so it can fit the content.
     */
    width?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Height of the node. In circular nodes, it determines the circle diameter.
     */
    height?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Maximum width allowed. The auto grow algorithm stops increasing the width at this value.
     */
    maxWidth?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * The x coordinate of the node. This value must be set after the layout algorithm has finished arranging
     * the graph. It may come from the input data but is not required for most layout algorithms.
     * Works only for `CustomLayout` or `NoopLayout` layout algorithms. Other layout algorithms override this
     * property. However, you can still change it using an `afterLayouting` event.
     */
    x?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * The y coordinate of the node. This value must be set after the layout algorithm has finished arranging
     * the graph. It may come from the input data but is not required for most layout algorithms.
     * Works only for `CustomLayout` or `NoopLayout` layout algorithms. Other layout algorithms override this
     * property. However, you can still change it using an `afterLayouting` event.
     */
    y?: float | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines if the expand button is visible.
     */
    showExpandButton?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines if the links button is visible.
     */
    showActionLinksButton?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines if the details button is visible.
     */
    showDetailButton?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines the maximum number of lines allowed for the node's label. If set to 0, the label may have
     * an unlimited number of lines.
     */
    titleLineSize?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines the maximum number of lines allowed to be displayed in the node's description. If you want
     * to hide the description, set this property to `-1` (default). To display an unlimited number of lines,
     * set this property to `0`. This property does not affect the description that appears in the tooltip,
     * which is always fully rendered.
     */
    descriptionLineSize?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines the size of the node's icon. This property can be applied only to circular nodes.
     */
    iconSize?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Determines the size of a custom node, which can be useful for nodes that have text or other content outside
     * of its shape.
     * This property is available for custom nodes only.
     */
    coreNodeSize?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Whether the control should be visible on the screen. Node is still used for layouter.
     */
    visible?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * In the absence of a title, alternate texts can be used to identify the nodes.
     *
     * @since 1.112
     */
    altText?: string | PropertyBindingInfo;

    /**
     * A list of links to be shown in the links area. A link may point to any UI5 control. It's up to the caller
     * to set up all necessary callback functions.
     */
    actionLinks?: Control[] | Control | AggregationBindingInfo | `{${string}}`;

    /**
     * A list of custom action buttons. These buttons are displayed in the button area for each node. A node
     * may have up to 4 buttons. The default 3 buttons (collapse/expand, details, and links) have priority over
     * any other custom buttons that you add. If you want all 4 of your custom buttons to be displayed, set
     * the visibility of the default buttons to false.
     */
    actionButtons?:
      | ActionButton[]
      | ActionButton
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * The image that is displayed in the node's header instead of the icon.
     */
    image?: NodeImage;

    /**
     * Custom content rendered with the node. Available only for rectangular nodes with HTML rendering type.
     * When this aggregation is used, the description and attributes of {@link sap.suite.ui.commons.networkgraph.ElementBase }
     * are ignored.
     */
    content?: Control;

    /**
     * This event is fired when the user selects or clears the Header checkbox.
     */
    headerCheckBoxPress?: (oEvent: Node$HeaderCheckBoxPressEvent) => void;

    /**
     * This event is fired when the user clicks or taps the node.
     */
    press?: (oEvent: Event) => void;

    /**
     * This event is fired when the user moves the mouse pointer over the node.
     */
    hover?: (oEvent: Event) => void;

    /**
     * This event is fired when the user clicks the node's collapse/expand button.
     */
    collapseExpand?: (oEvent: Event) => void;
  }

  /**
   * Parameters of the Node#collapseExpand event.
   */
  export interface Node$CollapseExpandEventParameters {}

  /**
   * Event object of the Node#collapseExpand event.
   */
  export type Node$CollapseExpandEvent = Event<
    Node$CollapseExpandEventParameters,
    Node
  >;

  /**
   * Parameters of the Node#headerCheckBoxPress event.
   */
  export interface Node$HeaderCheckBoxPressEventParameters {
    /**
     * Indicates whether checkbox is selected.
     */
    checked?: boolean;
  }

  /**
   * Event object of the Node#headerCheckBoxPress event.
   */
  export type Node$HeaderCheckBoxPressEvent = Event<
    Node$HeaderCheckBoxPressEventParameters,
    Node
  >;

  /**
   * Parameters of the Node#hover event.
   */
  export interface Node$HoverEventParameters {}

  /**
   * Event object of the Node#hover event.
   */
  export type Node$HoverEvent = Event<Node$HoverEventParameters, Node>;

  /**
   * Parameters of the Node#press event.
   */
  export interface Node$PressEventParameters {}

  /**
   * Event object of the Node#press event.
   */
  export type Node$PressEvent = Event<Node$PressEventParameters, Node>;
}

declare module "sap/suite/ui/commons/networkgraph/NodeImage" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { URI } from "sap/ui/core/library";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Holds information about node image.
   *
   * @since 1.50
   */
  export default class NodeImage extends UI5Element {
    /**
     * Constructor for a new Node Image.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.NodeImage with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, NodeImage>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.NodeImage.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getHeight height}.
     *
     * Height of the image.
     *
     *
     * @returns Value of property `height`
     */
    getHeight(): int;
    /**
     * Gets current value of property {@link #getSrc src}.
     *
     * Relative or absolute path to the image file.
     *
     *
     * @returns Value of property `src`
     */
    getSrc(): URI;
    /**
     * Gets current value of property {@link #getWidth width}.
     *
     * Width of the image.
     *
     *
     * @returns Value of property `width`
     */
    getWidth(): int;
    /**
     * Sets a new value for property {@link #getHeight height}.
     *
     * Height of the image.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeight(
      /**
       * New value for property `height`
       */
      iHeight?: int
    ): this;
    /**
     * Sets a new value for property {@link #getSrc src}.
     *
     * Relative or absolute path to the image file.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSrc(
      /**
       * New value for property `src`
       */
      sSrc?: URI
    ): this;
    /**
     * Sets a new value for property {@link #getWidth width}.
     *
     * Width of the image.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setWidth(
      /**
       * New value for property `width`
       */
      iWidth?: int
    ): this;
  }
  /**
   * Describes the settings that can be provided to the NodeImage constructor.
   */
  export interface $NodeImageSettings extends $ElementSettings {
    /**
     * Relative or absolute path to the image file.
     */
    src?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * Width of the image.
     */
    width?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * Height of the image.
     */
    height?: int | PropertyBindingInfo | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/networkgraph/Status" {
  import { default as UI5Element, $ElementSettings } from "sap/ui/core/Element";

  import { CSSSize } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * The status element holds information about a custom status that can be applied to nodes, lines, and groups
   * of nodes.
   *
   *  You can create multiple custom statuses and assign them to your graph using the `statuses` aggregation
   * of the {@link sap.suite.ui.commons.networkgraph.Graph} control.
   *
   *  Alternatively, you can use default statuses provided by the {@link sap.suite.ui.commons.networkgraph.ElementStatus }
   * element.
   *
   * @since 1.50
   */
  export default class Status extends UI5Element {
    /**
     * Constructor for a new Status.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.Status with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Element.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, Status>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.Status.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getBackgroundColor backgroundColor}.
     *
     * Fill color.
     *  The fill color that is applied to elements and lines that are in this custom status.
     *
     *
     * @returns Value of property `backgroundColor`
     */
    getBackgroundColor(): string;
    /**
     * Gets current value of property {@link #getBorderColor borderColor}.
     *
     * Border color.
     *  The border color that is applied to elements in this custom status.
     *
     *
     * @returns Value of property `borderColor`
     */
    getBorderColor(): string;
    /**
     * Gets current value of property {@link #getBorderStyle borderStyle}.
     *
     * Style of the border. For {@link sap.suite.ui.commons.networkgraph.Line} elements use this value as parameter
     * for `stroke-dasharray`.
     *
     *
     * @returns Value of property `borderStyle`
     */
    getBorderStyle(): string;
    /**
     * Gets current value of property {@link #getBorderWidth borderWidth}.
     *
     * Border width of the element in specific units (for example 2px).
     *
     *
     * @returns Value of property `borderWidth`
     */
    getBorderWidth(): CSSSize;
    /**
     * Gets current value of property {@link #getContentColor contentColor}.
     *
     * Color of the content (text or icon) inside the node.
     *  The color is applied to a node's content when it enters this custom status.
     *
     *
     * @returns Value of property `contentColor`
     */
    getContentColor(): string;
    /**
     * Gets current value of property {@link #getHeaderContentColor headerContentColor}.
     *
     * Color of the header content (title and icon) of the node.
     *  Applicable only to the rectangular nodes.
     *
     *
     * @returns Value of property `headerContentColor`
     */
    getHeaderContentColor(): string;
    /**
     * Gets current value of property {@link #getHoverBackgroundColor hoverBackgroundColor}.
     *
     * Fill color on hover.
     *  The fill color of a custom status element, line, or group when you hover over it.
     *
     *
     * @returns Value of property `hoverBackgroundColor`
     */
    getHoverBackgroundColor(): string;
    /**
     * Gets current value of property {@link #getHoverBorderColor hoverBorderColor}.
     *
     * Border color on hover.
     *  The border color of a custom status element when you hover over it.
     *
     *
     * @returns Value of property `hoverBorderColor`
     */
    getHoverBorderColor(): string;
    /**
     * Gets current value of property {@link #getHoverContentColor hoverContentColor}.
     *
     * Content color on hover. Color of the content (text or icon) inside the custom status element when you
     * hover over it.
     *
     *
     * @returns Value of property `hoverContentColor`
     */
    getHoverContentColor(): string;
    /**
     * Gets current value of property {@link #getKey key}.
     *
     *
     * @returns Value of property `key`
     */
    getKey(): string;
    /**
     * Gets current value of property {@link #getLegendColor legendColor}.
     *
     * Color displayed in the graph's legend.
     *  If not set, the fill color defined in the `backgroundColor` property is used.
     *
     *
     * @returns Value of property `legendColor`
     */
    getLegendColor(): string;
    /**
     * Gets current value of property {@link #getSelectedBackgroundColor selectedBackgroundColor}.
     *
     * Fill color when selected.
     *  The fill color of a custom status element when you select it.
     *
     *
     * @returns Value of property `selectedBackgroundColor`
     */
    getSelectedBackgroundColor(): string;
    /**
     * Gets current value of property {@link #getSelectedBorderColor selectedBorderColor}.
     *
     * Border color when selected. The border color of a custom status element when you select it.
     *
     *
     * @returns Value of property `selectedBorderColor`
     */
    getSelectedBorderColor(): string;
    /**
     * Gets current value of property {@link #getSelectedContentColor selectedContentColor}.
     *
     * Content color when selected. Color of the content (text or icon) inside the custom status node when you
     * select it.
     *
     *
     * @returns Value of property `selectedContentColor`
     */
    getSelectedContentColor(): string;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * Title of the node.
     *  The title that is applied to elements that are in this custom status. The title is displayed in the
     * legend.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Gets current value of property {@link #getUseFocusColorAsContentColor useFocusColorAsContentColor}.
     *
     * Defines whether the focus border should match the color of the content in focus. If false, the focus
     * border has the default color.
     * This property can be used only for nodes and groups of nodes.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `useFocusColorAsContentColor`
     */
    getUseFocusColorAsContentColor(): boolean;
    /**
     * Sets a new value for property {@link #getBackgroundColor backgroundColor}.
     *
     * Fill color.
     *  The fill color that is applied to elements and lines that are in this custom status.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBackgroundColor(
      /**
       * New value for property `backgroundColor`
       */
      sBackgroundColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getBorderColor borderColor}.
     *
     * Border color.
     *  The border color that is applied to elements in this custom status.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBorderColor(
      /**
       * New value for property `borderColor`
       */
      sBorderColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getBorderStyle borderStyle}.
     *
     * Style of the border. For {@link sap.suite.ui.commons.networkgraph.Line} elements use this value as parameter
     * for `stroke-dasharray`.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBorderStyle(
      /**
       * New value for property `borderStyle`
       */
      sBorderStyle?: string
    ): this;
    /**
     * Sets a new value for property {@link #getBorderWidth borderWidth}.
     *
     * Border width of the element in specific units (for example 2px).
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBorderWidth(
      /**
       * New value for property `borderWidth`
       */
      sBorderWidth?: CSSSize
    ): this;
    /**
     * Sets a new value for property {@link #getContentColor contentColor}.
     *
     * Color of the content (text or icon) inside the node.
     *  The color is applied to a node's content when it enters this custom status.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setContentColor(
      /**
       * New value for property `contentColor`
       */
      sContentColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getHeaderContentColor headerContentColor}.
     *
     * Color of the header content (title and icon) of the node.
     *  Applicable only to the rectangular nodes.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeaderContentColor(
      /**
       * New value for property `headerContentColor`
       */
      sHeaderContentColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getHoverBackgroundColor hoverBackgroundColor}.
     *
     * Fill color on hover.
     *  The fill color of a custom status element, line, or group when you hover over it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHoverBackgroundColor(
      /**
       * New value for property `hoverBackgroundColor`
       */
      sHoverBackgroundColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getHoverBorderColor hoverBorderColor}.
     *
     * Border color on hover.
     *  The border color of a custom status element when you hover over it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHoverBorderColor(
      /**
       * New value for property `hoverBorderColor`
       */
      sHoverBorderColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getHoverContentColor hoverContentColor}.
     *
     * Content color on hover. Color of the content (text or icon) inside the custom status element when you
     * hover over it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHoverContentColor(
      /**
       * New value for property `hoverContentColor`
       */
      sHoverContentColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getKey key}.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setKey(
      /**
       * New value for property `key`
       */
      sKey?: string
    ): this;
    /**
     * Sets a new value for property {@link #getLegendColor legendColor}.
     *
     * Color displayed in the graph's legend.
     *  If not set, the fill color defined in the `backgroundColor` property is used.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setLegendColor(
      /**
       * New value for property `legendColor`
       */
      sLegendColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getSelectedBackgroundColor selectedBackgroundColor}.
     *
     * Fill color when selected.
     *  The fill color of a custom status element when you select it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSelectedBackgroundColor(
      /**
       * New value for property `selectedBackgroundColor`
       */
      sSelectedBackgroundColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getSelectedBorderColor selectedBorderColor}.
     *
     * Border color when selected. The border color of a custom status element when you select it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSelectedBorderColor(
      /**
       * New value for property `selectedBorderColor`
       */
      sSelectedBorderColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getSelectedContentColor selectedContentColor}.
     *
     * Content color when selected. Color of the content (text or icon) inside the custom status node when you
     * select it.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setSelectedContentColor(
      /**
       * New value for property `selectedContentColor`
       */
      sSelectedContentColor?: string
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * Title of the node.
     *  The title that is applied to elements that are in this custom status. The title is displayed in the
     * legend.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
    /**
     * Sets a new value for property {@link #getUseFocusColorAsContentColor useFocusColorAsContentColor}.
     *
     * Defines whether the focus border should match the color of the content in focus. If false, the focus
     * border has the default color.
     * This property can be used only for nodes and groups of nodes.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUseFocusColorAsContentColor(
      /**
       * New value for property `useFocusColorAsContentColor`
       */
      bUseFocusColorAsContentColor?: boolean
    ): this;
  }
  /**
   * Describes the settings that can be provided to the Status constructor.
   */
  export interface $StatusSettings extends $ElementSettings {
    key?: string | PropertyBindingInfo;

    /**
     * Title of the node.
     *  The title that is applied to elements that are in this custom status. The title is displayed in the
     * legend.
     */
    title?: string | PropertyBindingInfo;

    /**
     * Border color.
     *  The border color that is applied to elements in this custom status.
     */
    borderColor?: string | PropertyBindingInfo;

    /**
     * Fill color.
     *  The fill color that is applied to elements and lines that are in this custom status.
     */
    backgroundColor?: string | PropertyBindingInfo;

    /**
     * Color of the content (text or icon) inside the node.
     *  The color is applied to a node's content when it enters this custom status.
     */
    contentColor?: string | PropertyBindingInfo;

    /**
     * Color of the header content (title and icon) of the node.
     *  Applicable only to the rectangular nodes.
     */
    headerContentColor?: string | PropertyBindingInfo;

    /**
     * Fill color on hover.
     *  The fill color of a custom status element, line, or group when you hover over it.
     */
    hoverBackgroundColor?: string | PropertyBindingInfo;

    /**
     * Border color on hover.
     *  The border color of a custom status element when you hover over it.
     */
    hoverBorderColor?: string | PropertyBindingInfo;

    /**
     * Content color on hover. Color of the content (text or icon) inside the custom status element when you
     * hover over it.
     */
    hoverContentColor?: string | PropertyBindingInfo;

    /**
     * Fill color when selected.
     *  The fill color of a custom status element when you select it.
     */
    selectedBackgroundColor?: string | PropertyBindingInfo;

    /**
     * Border color when selected. The border color of a custom status element when you select it.
     */
    selectedBorderColor?: string | PropertyBindingInfo;

    /**
     * Content color when selected. Color of the content (text or icon) inside the custom status node when you
     * select it.
     */
    selectedContentColor?: string | PropertyBindingInfo;

    /**
     * Color displayed in the graph's legend.
     *  If not set, the fill color defined in the `backgroundColor` property is used.
     */
    legendColor?: string | PropertyBindingInfo;

    /**
     * Defines whether the focus border should match the color of the content in focus. If false, the focus
     * border has the default color.
     * This property can be used only for nodes and groups of nodes.
     */
    useFocusColorAsContentColor?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Border width of the element in specific units (for example 2px).
     */
    borderWidth?: CSSSize | PropertyBindingInfo | `{${string}}`;

    /**
     * Style of the border. For {@link sap.suite.ui.commons.networkgraph.Line} elements use this value as parameter
     * for `stroke-dasharray`.
     */
    borderStyle?: string | PropertyBindingInfo;
  }
}

declare module "sap/suite/ui/commons/networkgraph/SvgBase" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * SvgBase class
   *
   * @since 1.50
   */
  export default abstract class SvgBase extends Control {
    /**
     * Constructor for a new ElementBase.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor();

    /**
     * Creates a new subclass of class sap.suite.ui.commons.networkgraph.SvgBase with name `sClassName` and
     * enriches it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, SvgBase>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.networkgraph.SvgBase.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the SvgBase constructor.
   */
  export interface $SvgBaseSettings extends $ControlSettings {}
}

declare module "sap/suite/ui/commons/networkgraph/util/Dijkstra" {
  import Graph from "sap/suite/ui/commons/networkgraph/Graph";

  import Node from "sap/suite/ui/commons/networkgraph/Node";

  import Line from "sap/suite/ui/commons/networkgraph/Line";

  /**
   * This algorithm uses Dijkstra's algorithm to compute the shortest possible paths from the starting point
   * node to all other nodes in the graph and arrange them accordingly.
   *
   * @since 1.58
   */
  export default class Dijkstra {
    /**
     * Constructor for a new Dijkstra's algorithm. The constructor computes the shortest paths to all nodes
     * in the network graph from given starting point node.
     */
    constructor(
      /**
       * Graph to compute the paths in.
       */
      oGraph: Graph,
      /**
       * Node that serves as the starting point for computing the paths.
       */
      oSourceNode: Node,
      /**
       * Settings for the algorithm.
       */
      oSettings?: {
        /**
         * Evaluation function for determining the value of a line. The function will receive the line to be evaluated
         * as a parameter and must return a positive number representing its value (note that Dijkstra's algorithm
         * cannot be used on negative values of lines). If this function is not passed as a parameter, the algorithm
         * will consider the value of each line to be 1.
         */
        fnLineValue?: (p1: Line) => void;
        /**
         * Makes the algorithm ignore line directions and treat the graph as non-oriented.
         */
        bIgnoreDirections?: boolean;
        /**
         * Makes the algorithm ignore collapsed and hidden nodes.
         */
        bIgnoreCollapsed?: boolean;
      }
    );

    /**
     * Retrieves the shortest path to the given node. Due to the nature of the algorithm, the path is returned
     * in reversed order. In most cases, each line has to be processed anyway, so the order is not important.
     * However, if you need to retrieve lines in the direct order, you can use the `Array.reverse()` method.
     *
     *
     * @returns Lines representing the shortest path.
     */
    getShortestPathTo(
      /**
       * Node to find shortest path to.
       */
      oToNode: Node
    ): Line[];
  }
}

declare module "sap/suite/ui/commons/NewsContent" {
  import {
    default as NewsContent1,
    $NewsContentSettings as $NewsContentSettings1,
  } from "sap/m/NewsContent";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  /**
   * This control displays the news content text and subheader in a tile.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.m.NewsContent.
   */
  export default class NewsContent extends NewsContent1 {
    /**
     * Constructor for a new NewsContent.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.m.NewsContent#constructor sap.m.NewsContent }
     * can be used.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $NewsContentSettings
    );
    /**
     * Constructor for a new NewsContent.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     *
     * This class does not have its own settings, but all settings applicable to the base type {@link sap.m.NewsContent#constructor sap.m.NewsContent }
     * can be used.
     */
    constructor(
      /**
       * ID for the new control, automatically generated if no ID is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $NewsContentSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.NewsContent with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.m.NewsContent.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, NewsContent>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.NewsContent.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
  }
  /**
   * Describes the settings that can be provided to the NewsContent constructor.
   *
   * @deprecated (since 1.34) - this control is a mere wrapper for sap.m.NewsContent.
   */
  export interface $NewsContentSettings extends $NewsContentSettings1 {}
}

declare module "sap/suite/ui/commons/NoteTaker" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import NoteTakerCard from "sap/suite/ui/commons/NoteTakerCard";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * This control allows you to create and store your notes for further reference.
   *
   * @deprecated (since 1.32) - Deprecated. Standard Fiori technology should be used.
   */
  export default class NoteTaker extends Control {
    /**
     * Constructor for a new NoteTaker.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $NoteTakerSettings
    );
    /**
     * Constructor for a new NoteTaker.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $NoteTakerSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.NoteTaker with name `sClassName` and enriches it
     * with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, NoteTaker>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.NoteTaker.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some card to the aggregation {@link #getCards cards}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addCard(
      /**
       * The card to add; if empty, nothing is inserted
       */
      oCard: NoteTakerCard
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:addCard addCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when a new card is added if no data binding is available. If data binding is available,
     * the event is fired to notify a developer to add an element to the model.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAddCard(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AddCardEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:addCard addCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when a new card is added if no data binding is available. If data binding is available,
     * the event is fired to notify a developer to add an element to the model.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAddCard(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AddCardEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentClick attachmentClick} event of this
     * `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when a user presses the attachment link.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentClick(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentClickEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentClick attachmentClick} event of this
     * `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when a user presses the attachment link.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentClick(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentClickEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentDelete attachmentDelete} event of
     * this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired, when user deletes the attached file.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentDelete(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentDeleteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentDelete attachmentDelete} event of
     * this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired, when user deletes the attached file.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentDelete(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentDeleteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentSelect attachmentSelect} event of
     * this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when the value of attached file has been changed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentSelect(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentSelectEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentSelect attachmentSelect} event of
     * this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when the value of attached file has been changed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentSelect(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentSelectEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentUploadComplete attachmentUploadComplete }
     * event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * Fire event uploadComplete to attached listeners.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentUploadComplete(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentUploadCompleteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentUploadComplete attachmentUploadComplete }
     * event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * Fire event uploadComplete to attached listeners.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentUploadComplete(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentUploadCompleteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:deleteCard deleteCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when a new card is deleted if no data binding is available. If data binding is available,
     * the event is fired to notify a developer to delete the element from the model.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDeleteCard(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$DeleteCardEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:deleteCard deleteCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when a new card is deleted if no data binding is available. If data binding is available,
     * the event is fired to notify a developer to delete the element from the model.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDeleteCard(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$DeleteCardEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:editCard editCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when a new card has been edited.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachEditCard(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$EditCardEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:editCard editCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTaker` itself.
     *
     * The event is fired when a new card has been edited.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachEditCard(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTaker$EditCardEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTaker` itself
       */
      oListener?: object
    ): this;
    /**
     * Destroys all the cards in the aggregation {@link #getCards cards}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyCards(): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:addCard addCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAddCard(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTaker$AddCardEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentClick attachmentClick} event of
     * this `sap.suite.ui.commons.NoteTaker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentClick(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentClickEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentDelete attachmentDelete} event of
     * this `sap.suite.ui.commons.NoteTaker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentDelete(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentDeleteEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentSelect attachmentSelect} event of
     * this `sap.suite.ui.commons.NoteTaker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentSelect(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentSelectEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentUploadComplete attachmentUploadComplete }
     * event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentUploadComplete(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTaker$AttachmentUploadCompleteEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:deleteCard deleteCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachDeleteCard(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTaker$DeleteCardEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:editCard editCard} event of this `sap.suite.ui.commons.NoteTaker`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachEditCard(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTaker$EditCardEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:addCard addCard} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAddCard(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTaker$AddCardEventParameters
    ): this;
    /**
     * Fires event {@link #event:attachmentClick attachmentClick} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentClick(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTaker$AttachmentClickEventParameters
    ): this;
    /**
     * Fires event {@link #event:attachmentDelete attachmentDelete} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentDelete(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTaker$AttachmentDeleteEventParameters
    ): this;
    /**
     * Fires event {@link #event:attachmentSelect attachmentSelect} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentSelect(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTaker$AttachmentSelectEventParameters
    ): this;
    /**
     * Fires event {@link #event:attachmentUploadComplete attachmentUploadComplete} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentUploadComplete(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTaker$AttachmentUploadCompleteEventParameters
    ): this;
    /**
     * Fires event {@link #event:deleteCard deleteCard} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireDeleteCard(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTaker$DeleteCardEventParameters
    ): this;
    /**
     * Fires event {@link #event:editCard editCard} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireEditCard(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTaker$EditCardEventParameters
    ): this;
    /**
     * Gets current value of property {@link #getAttachmentName attachmentName}.
     *
     * The attachment property name for identification on the server side after sending data to the server.
     *
     * Default value is `'attachment'`.
     *
     *
     * @returns Value of property `attachmentName`
     */
    getAttachmentName(): string;
    /**
     * Gets current value of property {@link #getAttachmentUploadUrl attachmentUploadUrl}.
     *
     * Defines a path on the server where the attached files are uploaded.
     *
     *
     * @returns Value of property `attachmentUploadUrl`
     */
    getAttachmentUploadUrl(): string;
    /**
     * Gets content of aggregation {@link #getCards cards}.
     *
     * This aggregation allows you to add note cards to the Note Taker and remove them from the Note Taker.
     */
    getCards(): NoteTakerCard[];
    /**
     * Gets current value of property {@link #getCardViewAllTrigger cardViewAllTrigger}.
     *
     * The View All link appears in the Note Taker card when length of a card body text exceeds the specified
     * value. The updated value of this property is not applied to the existing cards.
     *
     * Default value is `1800`.
     *
     *
     * @returns Value of property `cardViewAllTrigger`
     */
    getCardViewAllTrigger(): int;
    /**
     * Gets current value of property {@link #getFilterCriteria filterCriteria}.
     *
     * A composite object containing criteria for filtering cards in the Note Taker.
     *
     *
     * @returns Value of property `filterCriteria`
     */
    getFilterCriteria(): object;
    /**
     * Gets current value of property {@link #getVisibleNotes visibleNotes}.
     *
     * Number of notes to display.
     *
     * Default value is `2`.
     *
     *
     * @returns Value of property `visibleNotes`
     */
    getVisibleNotes(): int;
    /**
     * Checks for the provided `sap.suite.ui.commons.NoteTakerCard` in the aggregation {@link #getCards cards}.
     * and returns its index if found or -1 otherwise.
     *
     *
     * @returns The index of the provided control in the aggregation if found, or -1 otherwise
     */
    indexOfCard(
      /**
       * The card whose index is looked for
       */
      oCard: NoteTakerCard
    ): int;
    /**
     * Inserts a card into the aggregation {@link #getCards cards}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    insertCard(
      /**
       * The card to insert; if empty, nothing is inserted
       */
      oCard: NoteTakerCard,
      /**
       * The `0`-based index the card should be inserted at; for a negative value of `iIndex`, the card is inserted
       * at position 0; for a value greater than the current size of the aggregation, the card is inserted at
       * the last position
       */
      iIndex: int
    ): this;
    /**
     * Removes all the controls from the aggregation {@link #getCards cards}.
     *
     * Additionally, it unregisters them from the hosting UIArea.
     *
     *
     * @returns An array of the removed elements (might be empty)
     */
    removeAllCards(): NoteTakerCard[];
    /**
     * Removes a card from the aggregation {@link #getCards cards}.
     *
     *
     * @returns The removed card or `null`
     */
    removeCard(
      /**
       * The card to remove or its index or id
       */
      vCard: int | string | NoteTakerCard
    ): NoteTakerCard | null;
    /**
     * Sets a new value for property {@link #getAttachmentName attachmentName}.
     *
     * The attachment property name for identification on the server side after sending data to the server.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'attachment'`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAttachmentName(
      /**
       * New value for property `attachmentName`
       */
      sAttachmentName?: string
    ): this;
    /**
     * Sets a new value for property {@link #getAttachmentUploadUrl attachmentUploadUrl}.
     *
     * Defines a path on the server where the attached files are uploaded.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAttachmentUploadUrl(
      /**
       * New value for property `attachmentUploadUrl`
       */
      sAttachmentUploadUrl?: string
    ): this;
    /**
     * Sets a new value for property {@link #getCardViewAllTrigger cardViewAllTrigger}.
     *
     * The View All link appears in the Note Taker card when length of a card body text exceeds the specified
     * value. The updated value of this property is not applied to the existing cards.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `1800`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setCardViewAllTrigger(
      /**
       * New value for property `cardViewAllTrigger`
       */
      iCardViewAllTrigger?: int
    ): this;
    /**
     * Sets a new value for property {@link #getFilterCriteria filterCriteria}.
     *
     * A composite object containing criteria for filtering cards in the Note Taker.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setFilterCriteria(
      /**
       * New value for property `filterCriteria`
       */
      oFilterCriteria?: object
    ): this;
    /**
     * Sets a new value for property {@link #getVisibleNotes visibleNotes}.
     *
     * Number of notes to display.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `2`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setVisibleNotes(
      /**
       * New value for property `visibleNotes`
       */
      iVisibleNotes?: int
    ): this;
  }
  /**
   * Describes the settings that can be provided to the NoteTaker constructor.
   *
   * @deprecated (since 1.32) - Deprecated. Standard Fiori technology should be used.
   */
  export interface $NoteTakerSettings extends $ControlSettings {
    /**
     * Number of notes to display.
     */
    visibleNotes?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * The View All link appears in the Note Taker card when length of a card body text exceeds the specified
     * value. The updated value of this property is not applied to the existing cards.
     */
    cardViewAllTrigger?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * A composite object containing criteria for filtering cards in the Note Taker.
     */
    filterCriteria?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Defines a path on the server where the attached files are uploaded.
     */
    attachmentUploadUrl?: string | PropertyBindingInfo;

    /**
     * The attachment property name for identification on the server side after sending data to the server.
     */
    attachmentName?: string | PropertyBindingInfo;

    /**
     * This aggregation allows you to add note cards to the Note Taker and remove them from the Note Taker.
     */
    cards?:
      | NoteTakerCard[]
      | NoteTakerCard
      | AggregationBindingInfo
      | `{${string}}`;

    /**
     * The event is fired when a new card is added if no data binding is available. If data binding is available,
     * the event is fired to notify a developer to add an element to the model.
     */
    addCard?: (oEvent: NoteTaker$AddCardEvent) => void;

    /**
     * The event is fired when a new card is deleted if no data binding is available. If data binding is available,
     * the event is fired to notify a developer to delete the element from the model.
     */
    deleteCard?: (oEvent: NoteTaker$DeleteCardEvent) => void;

    /**
     * The event is fired when a new card has been edited.
     */
    editCard?: (oEvent: NoteTaker$EditCardEvent) => void;

    /**
     * The event is fired when the value of attached file has been changed.
     */
    attachmentSelect?: (oEvent: NoteTaker$AttachmentSelectEvent) => void;

    /**
     * Fire event uploadComplete to attached listeners.
     */
    attachmentUploadComplete?: (
      oEvent: NoteTaker$AttachmentUploadCompleteEvent
    ) => void;

    /**
     * The event is fired, when user deletes the attached file.
     */
    attachmentDelete?: (oEvent: NoteTaker$AttachmentDeleteEvent) => void;

    /**
     * The event is fired when a user presses the attachment link.
     */
    attachmentClick?: (oEvent: NoteTaker$AttachmentClickEvent) => void;
  }

  /**
   * Parameters of the NoteTaker#addCard event.
   */
  export interface NoteTaker$AddCardEventParameters {
    /**
     * The title of the note to be added.
     */
    title?: string;

    /**
     * The text of the note to be added.
     */
    body?: string;

    /**
     * The timestamp of the note to be added.
     */
    timestamp?: object;

    /**
     * The View All link appears in the Note Taker card when length of a body text exceeds the specified value.
     */
    viewAllTrigger?: int;

    /**
     * Indicates positive information for a new card.
     */
    thumbUp?: boolean;

    /**
     * Indicates negative information for a new card.
     */
    thumbDown?: boolean;

    /**
     * Stores the name of the file attached to the card.
     */
    attachmentFilename?: string;

    /**
     * The unique ID if available.
     */
    uid?: string;

    /**
     * Stores the URL of the file attached to the card.
     */
    attachmentUrl?: string;

    /**
     * A newly created card if no data binding is available. If data binding is available, this parameter is
     * empty.
     */
    card?: NoteTakerCard;
  }

  /**
   * Event object of the NoteTaker#addCard event.
   */
  export type NoteTaker$AddCardEvent = Event<
    NoteTaker$AddCardEventParameters,
    NoteTaker
  >;

  /**
   * Parameters of the NoteTaker#attachmentClick event.
   */
  export interface NoteTaker$AttachmentClickEventParameters {
    /**
     * The unique ID of the card.
     */
    uid?: string;

    /**
     * If true, a user chooses the attachment in the Note card. If false, a user chooses the attachment in the
     * Feeder card.
     */
    isCardAttachment?: string;

    /**
     * A name of the attached file.
     */
    filename?: string;
  }

  /**
   * Event object of the NoteTaker#attachmentClick event.
   */
  export type NoteTaker$AttachmentClickEvent = Event<
    NoteTaker$AttachmentClickEventParameters,
    NoteTaker
  >;

  /**
   * Parameters of the NoteTaker#attachmentDelete event.
   */
  export interface NoteTaker$AttachmentDeleteEventParameters {
    /**
     * A name of the attached file.
     */
    filename?: string;

    /**
     * The unique ID of the card.
     */
    uid?: string;
  }

  /**
   * Event object of the NoteTaker#attachmentDelete event.
   */
  export type NoteTaker$AttachmentDeleteEvent = Event<
    NoteTaker$AttachmentDeleteEventParameters,
    NoteTaker
  >;

  /**
   * Parameters of the NoteTaker#attachmentSelect event.
   */
  export interface NoteTaker$AttachmentSelectEventParameters {
    /**
     * A name of the attached file.
     */
    filename?: string;
  }

  /**
   * Event object of the NoteTaker#attachmentSelect event.
   */
  export type NoteTaker$AttachmentSelectEvent = Event<
    NoteTaker$AttachmentSelectEventParameters,
    NoteTaker
  >;

  /**
   * Parameters of the NoteTaker#attachmentUploadComplete event.
   */
  export interface NoteTaker$AttachmentUploadCompleteEventParameters {
    /**
     * The response message of the String type that comes from the server. On the server side this response
     * has to be put within the "body" tags of the response document of the iFrame. It can consist of a return
     * code and an optional message. This does not work in cross-domain scenarios.
     */
    response?: string;

    /**
     * The unique ID of the card.
     */
    uid?: string;
  }

  /**
   * Event object of the NoteTaker#attachmentUploadComplete event.
   */
  export type NoteTaker$AttachmentUploadCompleteEvent = Event<
    NoteTaker$AttachmentUploadCompleteEventParameters,
    NoteTaker
  >;

  /**
   * Parameters of the NoteTaker#deleteCard event.
   */
  export interface NoteTaker$DeleteCardEventParameters {
    /**
     * The title of the card to be deleted.
     */
    title?: string;

    /**
     * The text of the card to be deleted.
     */
    body?: string;

    /**
     * The timestamp of the card to be deleted.
     */
    timestamp?: string;

    /**
     * The unique ID that was set by an application developer.
     */
    uid?: string;

    /**
     * Indicates positive information for the deleted card.
     */
    thumbUp?: boolean;

    /**
     * Indicates negative information for the deleted card.
     */
    thumbDown?: boolean;
  }

  /**
   * Event object of the NoteTaker#deleteCard event.
   */
  export type NoteTaker$DeleteCardEvent = Event<
    NoteTaker$DeleteCardEventParameters,
    NoteTaker
  >;

  /**
   * Parameters of the NoteTaker#editCard event.
   */
  export interface NoteTaker$EditCardEventParameters {
    /**
     * The title of the edited card.
     */
    title?: string;

    /**
     * A new text of the edited card.
     */
    body?: string;

    /**
     * A new timestamp of the edited card.
     */
    timestamp?: string;

    /**
     * A unique ID that was set by an application developer.
     */
    uid?: string;

    /**
     * Indicates positive information for the edited card.
     */
    thumbUp?: boolean;

    /**
     * Indicates negative information for the edited card.
     */
    thumbDown?: boolean;

    /**
     * Updated array of the String type tags applied to the card during editing.
     */
    tags?: object;
  }

  /**
   * Event object of the NoteTaker#editCard event.
   */
  export type NoteTaker$EditCardEvent = Event<
    NoteTaker$EditCardEventParameters,
    NoteTaker
  >;
}

declare module "sap/suite/ui/commons/NoteTakerCard" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * This control allows you to store Note Taker card header and body text.
   *
   * @deprecated (since 1.32) - Deprecated. Standard Fiori technology should be used.
   */
  export default class NoteTakerCard extends Control {
    /**
     * Constructor for a new NoteTakerCard.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $NoteTakerCardSettings
    );
    /**
     * Constructor for a new NoteTakerCard.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $NoteTakerCardSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.NoteTakerCard with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, NoteTakerCard>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.NoteTakerCard.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentClick attachmentClick} event of this
     * `sap.suite.ui.commons.NoteTakerCard`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerCard` itself.
     *
     * The event is fired when a user chooses the attachment download link.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentClick(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$AttachmentClickEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerCard`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentClick attachmentClick} event of this
     * `sap.suite.ui.commons.NoteTakerCard`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerCard` itself.
     *
     * The event is fired when a user chooses the attachment download link.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentClick(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$AttachmentClickEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerCard`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:deleteNote deleteNote} event of this `sap.suite.ui.commons.NoteTakerCard`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerCard` itself.
     *
     * The event is fired when a card needs to be deleted. This event is needed for the Note Taker control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDeleteNote(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$DeleteNoteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerCard`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:deleteNote deleteNote} event of this `sap.suite.ui.commons.NoteTakerCard`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerCard` itself.
     *
     * The event is fired when a card needs to be deleted. This event is needed for the Note Taker control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachDeleteNote(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$DeleteNoteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerCard`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:editNote editNote} event of this `sap.suite.ui.commons.NoteTakerCard`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerCard` itself.
     *
     * The event is fired when a user chooses the Edit button in the note card.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachEditNote(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$EditNoteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerCard`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:editNote editNote} event of this `sap.suite.ui.commons.NoteTakerCard`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerCard` itself.
     *
     * The event is fired when a user chooses the Edit button in the note card.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachEditNote(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$EditNoteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerCard`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentClick attachmentClick} event of
     * this `sap.suite.ui.commons.NoteTakerCard`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentClick(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$AttachmentClickEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:deleteNote deleteNote} event of this `sap.suite.ui.commons.NoteTakerCard`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachDeleteNote(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$DeleteNoteEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:editNote editNote} event of this `sap.suite.ui.commons.NoteTakerCard`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachEditNote(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTakerCard$EditNoteEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:attachmentClick attachmentClick} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentClick(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTakerCard$AttachmentClickEventParameters
    ): this;
    /**
     * Fires event {@link #event:deleteNote deleteNote} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireDeleteNote(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTakerCard$DeleteNoteEventParameters
    ): this;
    /**
     * Fires event {@link #event:editNote editNote} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireEditNote(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTakerCard$EditNoteEventParameters
    ): this;
    /**
     * Gets current value of property {@link #getAllTags allTags}.
     *
     * Contains an array of the String type tags available for selection during the card update.
     *
     * Default value is `[]`.
     *
     *
     * @returns Value of property `allTags`
     */
    getAllTags(): object;
    /**
     * Gets current value of property {@link #getAttachmentFilename attachmentFilename}.
     *
     * Stores the name of the file attached to the card.
     *
     *
     * @returns Value of property `attachmentFilename`
     */
    getAttachmentFilename(): string;
    /**
     * Gets current value of property {@link #getAttachmentUrl attachmentUrl}.
     *
     * Stores the address of the file attached to the card.
     *
     *
     * @returns Value of property `attachmentUrl`
     */
    getAttachmentUrl(): string;
    /**
     * Gets current value of property {@link #getBody body}.
     *
     * Stores the Note Taker card body text.
     *
     *
     * @returns Value of property `body`
     */
    getBody(): string;
    /**
     * Gets current value of property {@link #getHeader header}.
     *
     * Stores the Note Taker card header.
     *
     *
     * @returns Value of property `header`
     */
    getHeader(): string;
    /**
     * Gets current value of property {@link #getIsFiltered isFiltered}.
     *
     * Shows whether the note card is hidden by the applied filter.
     *
     * Default value is `false`.
     *
     *
     * @returns Value of property `isFiltered`
     */
    getIsFiltered(): boolean;
    /**
     * Gets current value of property {@link #getTags tags}.
     *
     * Contains an array of the String type tags applied to the current card.
     *
     * Default value is `[]`.
     *
     *
     * @returns Value of property `tags`
     */
    getTags(): object;
    /**
     * Gets current value of property {@link #getThumbDown thumbDown}.
     *
     * Indicates negative information for the card.
     *
     *
     * @returns Value of property `thumbDown`
     */
    getThumbDown(): boolean;
    /**
     * Gets current value of property {@link #getThumbUp thumbUp}.
     *
     * Indicates positive information for the card.
     *
     *
     * @returns Value of property `thumbUp`
     */
    getThumbUp(): boolean;
    /**
     * Gets current value of property {@link #getTimestamp timestamp}.
     *
     * Stores a timestamp of the Note Taker card.
     *
     * Default value is `...see text or source`.
     *
     *
     * @returns Value of property `timestamp`
     */
    getTimestamp(): object;
    /**
     * Gets current value of property {@link #getUid uid}.
     *
     * The card ID. This property should be set by an application developer.
     *
     *
     * @returns Value of property `uid`
     */
    getUid(): string;
    /**
     * Gets current value of property {@link #getViewAllTrigger viewAllTrigger}.
     *
     * The View All link appears in the Note Taker card if a body text length exceeds the specified value.
     *
     * Default value is `1800`.
     *
     *
     * @returns Value of property `viewAllTrigger`
     */
    getViewAllTrigger(): int;
    /**
     * Sets a new value for property {@link #getAllTags allTags}.
     *
     * Contains an array of the String type tags available for selection during the card update.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `[]`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAllTags(
      /**
       * New value for property `allTags`
       */
      oAllTags?: object
    ): this;
    /**
     * Sets a new value for property {@link #getAttachmentFilename attachmentFilename}.
     *
     * Stores the name of the file attached to the card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAttachmentFilename(
      /**
       * New value for property `attachmentFilename`
       */
      sAttachmentFilename?: string
    ): this;
    /**
     * Sets a new value for property {@link #getAttachmentUrl attachmentUrl}.
     *
     * Stores the address of the file attached to the card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAttachmentUrl(
      /**
       * New value for property `attachmentUrl`
       */
      sAttachmentUrl?: string
    ): this;
    /**
     * Sets a new value for property {@link #getBody body}.
     *
     * Stores the Note Taker card body text.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBody(
      /**
       * New value for property `body`
       */
      sBody?: string
    ): this;
    /**
     * Sets a new value for property {@link #getHeader header}.
     *
     * Stores the Note Taker card header.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setHeader(
      /**
       * New value for property `header`
       */
      sHeader?: string
    ): this;
    /**
     * Sets a new value for property {@link #getIsFiltered isFiltered}.
     *
     * Shows whether the note card is hidden by the applied filter.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `false`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIsFiltered(
      /**
       * New value for property `isFiltered`
       */
      bIsFiltered?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getTags tags}.
     *
     * Contains an array of the String type tags applied to the current card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `[]`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTags(
      /**
       * New value for property `tags`
       */
      oTags?: object
    ): this;
    /**
     * Sets a new value for property {@link #getThumbDown thumbDown}.
     *
     * Indicates negative information for the card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setThumbDown(
      /**
       * New value for property `thumbDown`
       */
      bThumbDown?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getThumbUp thumbUp}.
     *
     * Indicates positive information for the card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setThumbUp(
      /**
       * New value for property `thumbUp`
       */
      bThumbUp?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getTimestamp timestamp}.
     *
     * Stores a timestamp of the Note Taker card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `...see text or source`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTimestamp(
      /**
       * New value for property `timestamp`
       */
      oTimestamp?: object
    ): this;
    /**
     * Sets a new value for property {@link #getUid uid}.
     *
     * The card ID. This property should be set by an application developer.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUid(
      /**
       * New value for property `uid`
       */
      sUid?: string
    ): this;
    /**
     * Sets a new value for property {@link #getViewAllTrigger viewAllTrigger}.
     *
     * The View All link appears in the Note Taker card if a body text length exceeds the specified value.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `1800`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setViewAllTrigger(
      /**
       * New value for property `viewAllTrigger`
       */
      iViewAllTrigger?: int
    ): this;
  }
  /**
   * Describes the settings that can be provided to the NoteTakerCard constructor.
   *
   * @deprecated (since 1.32) - Deprecated. Standard Fiori technology should be used.
   */
  export interface $NoteTakerCardSettings extends $ControlSettings {
    /**
     * Stores the Note Taker card header.
     */
    header?: string | PropertyBindingInfo;

    /**
     * Stores the Note Taker card body text.
     */
    body?: string | PropertyBindingInfo;

    /**
     * Stores a timestamp of the Note Taker card.
     */
    timestamp?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Contains an array of the String type tags applied to the current card.
     */
    tags?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * The View All link appears in the Note Taker card if a body text length exceeds the specified value.
     */
    viewAllTrigger?: int | PropertyBindingInfo | `{${string}}`;

    /**
     * The card ID. This property should be set by an application developer.
     */
    uid?: string | PropertyBindingInfo;

    /**
     * Shows whether the note card is hidden by the applied filter.
     */
    isFiltered?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Indicates positive information for the card.
     */
    thumbUp?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Indicates negative information for the card.
     */
    thumbDown?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Contains an array of the String type tags available for selection during the card update.
     */
    allTags?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Stores the name of the file attached to the card.
     */
    attachmentFilename?: string | PropertyBindingInfo;

    /**
     * Stores the address of the file attached to the card.
     */
    attachmentUrl?: string | PropertyBindingInfo;

    /**
     * The event is fired when a user chooses the Edit button in the note card.
     */
    editNote?: (oEvent: NoteTakerCard$EditNoteEvent) => void;

    /**
     * The event is fired when a card needs to be deleted. This event is needed for the Note Taker control.
     */
    deleteNote?: (oEvent: NoteTakerCard$DeleteNoteEvent) => void;

    /**
     * The event is fired when a user chooses the attachment download link.
     */
    attachmentClick?: (oEvent: NoteTakerCard$AttachmentClickEvent) => void;
  }

  /**
   * Parameters of the NoteTakerCard#attachmentClick event.
   */
  export interface NoteTakerCard$AttachmentClickEventParameters {
    /**
     * A unique ID that was set by an application developer.
     */
    uid?: string;

    /**
     * The address of the file attached to the card.
     */
    url?: string;

    /**
     * The name of the file attached to the card.
     */
    filename?: string;
  }

  /**
   * Event object of the NoteTakerCard#attachmentClick event.
   */
  export type NoteTakerCard$AttachmentClickEvent = Event<
    NoteTakerCard$AttachmentClickEventParameters,
    NoteTakerCard
  >;

  /**
   * Parameters of the NoteTakerCard#deleteNote event.
   */
  export interface NoteTakerCard$DeleteNoteEventParameters {
    /**
     * The HTML ID of a card that needs to be deleted.
     */
    cardId?: string;

    /**
     * The title of the card to be deleted.
     */
    title?: string;

    /**
     * The text of the card to be deleted.
     */
    body?: string;

    /**
     * The timestamp of the card to be deleted.
     */
    timestamp?: string;

    /**
     * A unique ID that was set by an application developer.
     */
    uid?: string;

    /**
     * Indicates positive information for the deleted card.
     */
    thumbUp?: boolean;

    /**
     * Indicates negative information for the deleted card.
     */
    thumbDown?: boolean;
  }

  /**
   * Event object of the NoteTakerCard#deleteNote event.
   */
  export type NoteTakerCard$DeleteNoteEvent = Event<
    NoteTakerCard$DeleteNoteEventParameters,
    NoteTakerCard
  >;

  /**
   * Parameters of the NoteTakerCard#editNote event.
   */
  export interface NoteTakerCard$EditNoteEventParameters {
    /**
     * The title of edited card.
     */
    title?: string;

    /**
     * A new text of the edited card.
     */
    body?: string;

    /**
     * A new timestamp of the edited card.
     */
    timestamp?: string;

    /**
     * A unique ID that was set by an application developer.
     */
    uid?: string;

    /**
     * Indicates positive information for the edited card.
     */
    thumbUp?: boolean;

    /**
     * Indicates negative information for the edited card.
     */
    thumbDown?: boolean;

    /**
     * Updated array of the String type tags applied to the card during editing.
     */
    tags?: object;
  }

  /**
   * Event object of the NoteTakerCard#editNote event.
   */
  export type NoteTakerCard$EditNoteEvent = Event<
    NoteTakerCard$EditNoteEventParameters,
    NoteTakerCard
  >;
}

declare module "sap/suite/ui/commons/NoteTakerFeeder" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * This control allows you to enter a quick note and N note cards.
   *
   * @deprecated (since 1.32) - Deprecated. Standard Fiori technology should be used.
   */
  export default class NoteTakerFeeder extends Control {
    /**
     * Constructor for a new NoteTakerFeeder.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $NoteTakerFeederSettings
    );
    /**
     * Constructor for a new NoteTakerFeeder.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $NoteTakerFeederSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.NoteTakerFeeder with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, NoteTakerFeeder>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.NoteTakerFeeder.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:addNote addNote} event of this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when a user chooses the Add button in the control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAddNote(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AddNoteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:addNote addNote} event of this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when a user chooses the Add button in the control.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAddNote(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AddNoteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentClick attachmentClick} event of this
     * `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when a user presses the attachment link.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentClick(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentClickEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentClick attachmentClick} event of this
     * `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when a user presses the attachment link.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentClick(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentClickEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentDelete attachmentDelete} event of
     * this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when a user presses the Delete button in the Feeder card.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentDelete(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentDeleteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentDelete attachmentDelete} event of
     * this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when a user presses the Delete button in the Feeder card.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentDelete(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentDeleteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentSelect attachmentSelect} event of
     * this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when the value of attached file has been changed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentSelect(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentSelectEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentSelect attachmentSelect} event of
     * this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when the value of attached file has been changed.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentSelect(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentSelectEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentUploadComplete attachmentUploadComplete }
     * event of this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when the upload of the file is completed. However this covers only the client side
     * of the Upload process and does not give any success status from the server.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentUploadComplete(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentUploadCompleteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:attachmentUploadComplete attachmentUploadComplete }
     * event of this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.NoteTakerFeeder` itself.
     *
     * The event is fired when the upload of the file is completed. However this covers only the client side
     * of the Upload process and does not give any success status from the server.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachAttachmentUploadComplete(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentUploadCompleteEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.NoteTakerFeeder`
       * itself
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:addNote addNote} event of this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAddNote(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AddNoteEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentClick attachmentClick} event of
     * this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentClick(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentClickEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentDelete attachmentDelete} event of
     * this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentDelete(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentDeleteEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentSelect attachmentSelect} event of
     * this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentSelect(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentSelectEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Detaches event handler `fnFunction` from the {@link #event:attachmentUploadComplete attachmentUploadComplete }
     * event of this `sap.suite.ui.commons.NoteTakerFeeder`.
     *
     * The passed function and listener object must match the ones used for event registration.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    detachAttachmentUploadComplete(
      /**
       * The function to be called, when the event occurs
       */
      fnFunction: (p1: NoteTakerFeeder$AttachmentUploadCompleteEvent) => void,
      /**
       * Context object on which the given function had to be called
       */
      oListener?: object
    ): this;
    /**
     * Fires event {@link #event:addNote addNote} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAddNote(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTakerFeeder$AddNoteEventParameters
    ): this;
    /**
     * Fires event {@link #event:attachmentClick attachmentClick} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentClick(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTakerFeeder$AttachmentClickEventParameters
    ): this;
    /**
     * Fires event {@link #event:attachmentDelete attachmentDelete} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentDelete(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTakerFeeder$AttachmentDeleteEventParameters
    ): this;
    /**
     * Fires event {@link #event:attachmentSelect attachmentSelect} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentSelect(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTakerFeeder$AttachmentSelectEventParameters
    ): this;
    /**
     * Fires event {@link #event:attachmentUploadComplete attachmentUploadComplete} to attached listeners.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    fireAttachmentUploadComplete(
      /**
       * Parameters to pass along with the event
       */
      mParameters?: NoteTakerFeeder$AttachmentUploadCompleteEventParameters
    ): this;
    /**
     * Gets current value of property {@link #getAttachmentName attachmentName}.
     *
     * The attachment property name for identification on the server side after sending data to the server.
     *
     * Default value is `'attachment'`.
     *
     *
     * @returns Value of property `attachmentName`
     */
    getAttachmentName(): string;
    /**
     * Gets current value of property {@link #getAttachmentUploadUrl attachmentUploadUrl}.
     *
     * Sets the upload attachment URL for a new card.
     *
     *
     * @returns Value of property `attachmentUploadUrl`
     */
    getAttachmentUploadUrl(): string;
    /**
     * Gets current value of property {@link #getBody body}.
     *
     * The text inside the note card.
     *
     *
     * @returns Value of property `body`
     */
    getBody(): string;
    /**
     * Gets current value of property {@link #getTags tags}.
     *
     * The list of tags selected for addition to a new note card.
     *
     * Default value is `[]`.
     *
     *
     * @returns Value of property `tags`
     */
    getTags(): object;
    /**
     * Gets current value of property {@link #getThumbDown thumbDown}.
     *
     * Sets negative indication for a new note.
     *
     *
     * @returns Value of property `thumbDown`
     */
    getThumbDown(): boolean;
    /**
     * Gets current value of property {@link #getThumbUp thumbUp}.
     *
     * Sets positive indication for a new note.
     *
     *
     * @returns Value of property `thumbUp`
     */
    getThumbUp(): boolean;
    /**
     * Gets current value of property {@link #getTitle title}.
     *
     * This text is the header of a new note.
     *
     *
     * @returns Value of property `title`
     */
    getTitle(): string;
    /**
     * Sets a new value for property {@link #getAttachmentName attachmentName}.
     *
     * The attachment property name for identification on the server side after sending data to the server.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `'attachment'`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAttachmentName(
      /**
       * New value for property `attachmentName`
       */
      sAttachmentName?: string
    ): this;
    /**
     * Sets a new value for property {@link #getAttachmentUploadUrl attachmentUploadUrl}.
     *
     * Sets the upload attachment URL for a new card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setAttachmentUploadUrl(
      /**
       * New value for property `attachmentUploadUrl`
       */
      sAttachmentUploadUrl?: string
    ): this;
    /**
     * Sets a new value for property {@link #getBody body}.
     *
     * The text inside the note card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBody(
      /**
       * New value for property `body`
       */
      sBody?: string
    ): this;
    /**
     * Sets a new value for property {@link #getTags tags}.
     *
     * The list of tags selected for addition to a new note card.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `[]`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTags(
      /**
       * New value for property `tags`
       */
      oTags?: object
    ): this;
    /**
     * Sets a new value for property {@link #getThumbDown thumbDown}.
     *
     * Sets negative indication for a new note.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setThumbDown(
      /**
       * New value for property `thumbDown`
       */
      bThumbDown?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getThumbUp thumbUp}.
     *
     * Sets positive indication for a new note.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setThumbUp(
      /**
       * New value for property `thumbUp`
       */
      bThumbUp?: boolean
    ): this;
    /**
     * Sets a new value for property {@link #getTitle title}.
     *
     * This text is the header of a new note.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setTitle(
      /**
       * New value for property `title`
       */
      sTitle?: string
    ): this;
  }
  /**
   * Describes the settings that can be provided to the NoteTakerFeeder constructor.
   *
   * @deprecated (since 1.32) - Deprecated. Standard Fiori technology should be used.
   */
  export interface $NoteTakerFeederSettings extends $ControlSettings {
    /**
     * The text inside the note card.
     */
    body?: string | PropertyBindingInfo;

    /**
     * This text is the header of a new note.
     */
    title?: string | PropertyBindingInfo;

    /**
     * The list of tags selected for addition to a new note card.
     */
    tags?: object | PropertyBindingInfo | `{${string}}`;

    /**
     * Sets positive indication for a new note.
     */
    thumbUp?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Sets negative indication for a new note.
     */
    thumbDown?: boolean | PropertyBindingInfo | `{${string}}`;

    /**
     * Sets the upload attachment URL for a new card.
     */
    attachmentUploadUrl?: string | PropertyBindingInfo;

    /**
     * The attachment property name for identification on the server side after sending data to the server.
     */
    attachmentName?: string | PropertyBindingInfo;

    /**
     * The event is fired when a user chooses the Add button in the control.
     */
    addNote?: (oEvent: NoteTakerFeeder$AddNoteEvent) => void;

    /**
     * The event is fired when the value of attached file has been changed.
     */
    attachmentSelect?: (oEvent: NoteTakerFeeder$AttachmentSelectEvent) => void;

    /**
     * The event is fired when the upload of the file is completed. However this covers only the client side
     * of the Upload process and does not give any success status from the server.
     */
    attachmentUploadComplete?: (
      oEvent: NoteTakerFeeder$AttachmentUploadCompleteEvent
    ) => void;

    /**
     * The event is fired when a user presses the Delete button in the Feeder card.
     */
    attachmentDelete?: (oEvent: NoteTakerFeeder$AttachmentDeleteEvent) => void;

    /**
     * The event is fired when a user presses the attachment link.
     */
    attachmentClick?: (oEvent: NoteTakerFeeder$AttachmentClickEvent) => void;
  }

  /**
   * Parameters of the NoteTakerFeeder#addNote event.
   */
  export interface NoteTakerFeeder$AddNoteEventParameters {
    /**
     * The title of the note to be added.
     */
    title?: string;

    /**
     * The text of the note to be added.
     */
    body?: string;

    /**
     * The timestamp of the note to be added.
     */
    timestamp?: object;

    /**
     * If set to true, a new card should be marked as positive one.
     */
    thumbUp?: boolean;

    /**
     * If set to true, a new card should be marked as negative one.
     */
    thumbDown?: boolean;

    /**
     * Stores the attachment file name for a new card.
     */
    attachmentFilename?: string;
  }

  /**
   * Event object of the NoteTakerFeeder#addNote event.
   */
  export type NoteTakerFeeder$AddNoteEvent = Event<
    NoteTakerFeeder$AddNoteEventParameters,
    NoteTakerFeeder
  >;

  /**
   * Parameters of the NoteTakerFeeder#attachmentClick event.
   */
  export interface NoteTakerFeeder$AttachmentClickEventParameters {
    /**
     * A name of the attached file.
     */
    filename?: string;
  }

  /**
   * Event object of the NoteTakerFeeder#attachmentClick event.
   */
  export type NoteTakerFeeder$AttachmentClickEvent = Event<
    NoteTakerFeeder$AttachmentClickEventParameters,
    NoteTakerFeeder
  >;

  /**
   * Parameters of the NoteTakerFeeder#attachmentDelete event.
   */
  export interface NoteTakerFeeder$AttachmentDeleteEventParameters {
    /**
     * A name of the attached file.
     */
    filename?: string;
  }

  /**
   * Event object of the NoteTakerFeeder#attachmentDelete event.
   */
  export type NoteTakerFeeder$AttachmentDeleteEvent = Event<
    NoteTakerFeeder$AttachmentDeleteEventParameters,
    NoteTakerFeeder
  >;

  /**
   * Parameters of the NoteTakerFeeder#attachmentSelect event.
   */
  export interface NoteTakerFeeder$AttachmentSelectEventParameters {
    /**
     * A name of the attached file.
     */
    filename?: string;
  }

  /**
   * Event object of the NoteTakerFeeder#attachmentSelect event.
   */
  export type NoteTakerFeeder$AttachmentSelectEvent = Event<
    NoteTakerFeeder$AttachmentSelectEventParameters,
    NoteTakerFeeder
  >;

  /**
   * Parameters of the NoteTakerFeeder#attachmentUploadComplete event.
   */
  export interface NoteTakerFeeder$AttachmentUploadCompleteEventParameters {
    /**
     * The response message that comes from the server. On the server side this response has to be put within
     * the "body" tags of the response document of the iFrame. It can consist of a return code and an optional
     * message. This does not work in cross-domain scenarios.
     */
    response?: string;
  }

  /**
   * Event object of the NoteTakerFeeder#attachmentUploadComplete event.
   */
  export type NoteTakerFeeder$AttachmentUploadCompleteEvent = Event<
    NoteTakerFeeder$AttachmentUploadCompleteEventParameters,
    NoteTakerFeeder
  >;
}

declare module "sap/suite/ui/commons/NumericTile" {
  import {
    default as InfoTile,
    $InfoTileSettings,
  } from "sap/suite/ui/commons/InfoTile";

  import {
    DeviationIndicator,
    InfoTileValueColor,
  } from "sap/suite/ui/commons/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * This control is the implementation of the InfoTile to show a numeric value.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export default class NumericTile extends InfoTile {
    /**
     * Constructor for a new NumericTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $NumericTileSettings
    );
    /**
     * Constructor for a new NumericTile.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $NumericTileSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.NumericTile with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.suite.ui.commons.InfoTile.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, NumericTile>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.NumericTile.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Gets current value of property {@link #getIndicator indicator}.
     *
     * The indicator arrow that shows value deviation.
     *
     * Default value is `"None"`.
     *
     *
     * @returns Value of property `indicator`
     */
    getIndicator(): DeviationIndicator | keyof typeof DeviationIndicator;
    /**
     * Gets current value of property {@link #getScale scale}.
     *
     * The scaling prefix. Financial characters can be used for currencies and counters. The SI prefixes can
     * be used for units.
     *
     *
     * @returns Value of property `scale`
     */
    getScale(): string;
    /**
     * Gets current value of property {@link #getUnit unit}.
     *
     * The percent sign, the currency symbol, or the unit of measure.
     *
     *
     * @returns Value of property `unit`
     */
    getUnit(): string;
    /**
     * Gets current value of property {@link #getValue value}.
     *
     * The actual value.
     *
     *
     * @returns Value of property `value`
     */
    getValue(): string;
    /**
     * Gets current value of property {@link #getValueColor valueColor}.
     *
     * The semantic color of the value.
     *
     *
     * @returns Value of property `valueColor`
     */
    getValueColor(): InfoTileValueColor | keyof typeof InfoTileValueColor;
    /**
     * Sets a new value for property {@link #getIndicator indicator}.
     *
     * The indicator arrow that shows value deviation.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     * Default value is `"None"`.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setIndicator(
      /**
       * New value for property `indicator`
       */
      sIndicator?: DeviationIndicator | keyof typeof DeviationIndicator
    ): this;
    /**
     * Sets a new value for property {@link #getScale scale}.
     *
     * The scaling prefix. Financial characters can be used for currencies and counters. The SI prefixes can
     * be used for units.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setScale(
      /**
       * New value for property `scale`
       */
      sScale?: string
    ): this;
    /**
     * Sets a new value for property {@link #getUnit unit}.
     *
     * The percent sign, the currency symbol, or the unit of measure.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setUnit(
      /**
       * New value for property `unit`
       */
      sUnit?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValue value}.
     *
     * The actual value.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValue(
      /**
       * New value for property `value`
       */
      sValue?: string
    ): this;
    /**
     * Sets a new value for property {@link #getValueColor valueColor}.
     *
     * The semantic color of the value.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setValueColor(
      /**
       * New value for property `valueColor`
       */
      sValueColor?: InfoTileValueColor | keyof typeof InfoTileValueColor
    ): this;
  }
  /**
   * Describes the settings that can be provided to the NumericTile constructor.
   *
   * @deprecated (since 1.25) - This control has been deprecated in favor of new sap.suite.ui.commons.GenericTile.
   */
  export interface $NumericTileSettings extends $InfoTileSettings {
    /**
     * The actual value.
     */
    value?: string | PropertyBindingInfo;

    /**
     * The scaling prefix. Financial characters can be used for currencies and counters. The SI prefixes can
     * be used for units.
     */
    scale?: string | PropertyBindingInfo;

    /**
     * The percent sign, the currency symbol, or the unit of measure.
     */
    unit?: string | PropertyBindingInfo;

    /**
     * The semantic color of the value.
     */
    valueColor?:
      | (InfoTileValueColor | keyof typeof InfoTileValueColor)
      | PropertyBindingInfo
      | `{${string}}`;

    /**
     * The indicator arrow that shows value deviation.
     */
    indicator?:
      | (DeviationIndicator | keyof typeof DeviationIndicator)
      | PropertyBindingInfo
      | `{${string}}`;
  }
}

declare module "sap/suite/ui/commons/PictureZoomIn" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import { URI } from "sap/ui/core/library";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import { PropertyBindingInfo } from "sap/ui/base/ManagedObject";

  /**
   * Shows picture in fullscreen.
   *
   * @deprecated (since 1.34) - Deprecated. Not Fiori.
   */
  export default class PictureZoomIn extends Control {
    /**
     * Constructor for a new PictureZoomIn.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * initial settings for the new control
       */
      mSettings?: $PictureZoomInSettings
    );
    /**
     * Constructor for a new PictureZoomIn.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * id for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * initial settings for the new control
       */
      mSettings?: $PictureZoomInSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.PictureZoomIn with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, PictureZoomIn>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.PictureZoomIn.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Destroys the busyIndicator in the aggregation {@link #getBusyIndicator busyIndicator}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    destroyBusyIndicator(): this;
    /**
     * Gets content of aggregation {@link #getBusyIndicator busyIndicator}.
     *
     * Custom busy indicator.
     */
    getBusyIndicator(): Control;
    /**
     * Gets current value of property {@link #getDescription description}.
     *
     * Description is shown under image.
     *
     *
     * @returns Value of property `description`
     */
    getDescription(): string;
    /**
     * Gets current value of property {@link #getImageSrc imageSrc}.
     *
     * Source for image.
     *
     *
     * @returns Value of property `imageSrc`
     */
    getImageSrc(): URI;
    /**
     * Sets the aggregated {@link #getBusyIndicator busyIndicator}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setBusyIndicator(
      /**
       * The busyIndicator to set
       */
      oBusyIndicator: Control
    ): this;
    /**
     * Sets a new value for property {@link #getDescription description}.
     *
     * Description is shown under image.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setDescription(
      /**
       * New value for property `description`
       */
      sDescription?: string
    ): this;
    /**
     * Sets a new value for property {@link #getImageSrc imageSrc}.
     *
     * Source for image.
     *
     * When called with a value of `null` or `undefined`, the default value of the property will be restored.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    setImageSrc(
      /**
       * New value for property `imageSrc`
       */
      sImageSrc?: URI
    ): this;
  }
  /**
   * Describes the settings that can be provided to the PictureZoomIn constructor.
   *
   * @deprecated (since 1.34) - Deprecated. Not Fiori.
   */
  export interface $PictureZoomInSettings extends $ControlSettings {
    /**
     * Description is shown under image.
     */
    description?: string | PropertyBindingInfo;

    /**
     * Source for image.
     */
    imageSrc?: URI | PropertyBindingInfo | `{${string}}`;

    /**
     * Custom busy indicator.
     */
    busyIndicator?: Control;
  }
}

declare module "sap/suite/ui/commons/ProcessFlow" {
  import { default as Control, $ControlSettings } from "sap/ui/core/Control";

  import ProcessFlowLaneHeader from "sap/suite/ui/commons/ProcessFlowLaneHeader";

  import AriaProperties from "sap/suite/ui/commons/AriaProperties";

  import ElementMetadata from "sap/ui/core/ElementMetadata";

  import ProcessFlowNode from "sap/suite/ui/commons/ProcessFlowNode";

  import ProcessFlowConnectionLabel from "sap/suite/ui/commons/ProcessFlowConnectionLabel";

  import { ProcessFlowZoomLevel } from "sap/suite/ui/commons/library";

  import {
    PropertyBindingInfo,
    AggregationBindingInfo,
  } from "sap/ui/base/ManagedObject";

  import Event from "sap/ui/base/Event";

  /**
   * Complex control that enables you to display documents or other items in their flow.
   */
  export default class ProcessFlow extends Control {
    /**
     * Constructor for a new ProcessFlow.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * Initial settings for the new control
       */
      mSettings?: $ProcessFlowSettings
    );
    /**
     * Constructor for a new ProcessFlow.
     *
     * Accepts an object literal `mSettings` that defines initial property values, aggregated and associated
     * objects as well as event handlers. See {@link sap.ui.base.ManagedObject#constructor} for a general description
     * of the syntax of the settings object.
     */
    constructor(
      /**
       * ID for the new control, generated automatically if no id is given
       */
      sId?: string,
      /**
       * Initial settings for the new control
       */
      mSettings?: $ProcessFlowSettings
    );

    /**
     * Creates a new subclass of class sap.suite.ui.commons.ProcessFlow with name `sClassName` and enriches
     * it with the information contained in `oClassInfo`.
     *
     * `oClassInfo` might contain the same kind of information as described in {@link sap.ui.core.Control.extend}.
     *
     *
     * @returns Created class / constructor function
     */
    static extend<T extends Record<string, unknown>>(
      /**
       * Name of the class being created
       */
      sClassName: string,
      /**
       * Object literal with information about the class
       */
      oClassInfo?: sap.ClassInfo<T, ProcessFlow>,
      /**
       * Constructor function for the metadata object; if not given, it defaults to the metadata implementation
       * used by this class
       */
      FNMetaImpl?: Function
    ): Function;
    /**
     * Returns a metadata object for class sap.suite.ui.commons.ProcessFlow.
     *
     *
     * @returns Metadata object describing this class
     */
    static getMetadata(): ElementMetadata;
    /**
     * Adds some lane to the aggregation {@link #getLanes lanes}.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    addLane(
      /**
       * The lane to add; if empty, nothing is inserted
       */
      oLane: ProcessFlowLaneHeader
    ): this;
    /**
     * Function applies the changes to the display state based on the requirements. If any node is in the highlighted
     * state all others are set to the dimmed state.
     */
    applyNodeDisplayState(): void;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:headerPress headerPress} event of this `sap.suite.ui.commons.ProcessFlow`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ProcessFlow` itself.
     *
     * This event is fired when the header column is clicked. This event is available only in header mode, i.e.
     * when no nodes are defined.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHeaderPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ProcessFlow$HeaderPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ProcessFlow` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:headerPress headerPress} event of this `sap.suite.ui.commons.ProcessFlow`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ProcessFlow` itself.
     *
     * This event is fired when the header column is clicked. This event is available only in header mode, i.e.
     * when no nodes are defined.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachHeaderPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ProcessFlow$HeaderPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ProcessFlow` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:labelPress labelPress} event of this `sap.suite.ui.commons.ProcessFlow`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ProcessFlow` itself.
     *
     * This event is fired when a process flow connection label was clicked.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachLabelPress(
      /**
       * An application-specific payload object that will be passed to the event handler along with the event
       * object when firing the event
       */
      oData: object,
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ProcessFlow$LabelPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ProcessFlow` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:labelPress labelPress} event of this `sap.suite.ui.commons.ProcessFlow`.
     *
     * When called, the context of the event handler (its `this`) will be bound to `oListener` if specified,
     * otherwise it will be bound to this `sap.suite.ui.commons.ProcessFlow` itself.
     *
     * This event is fired when a process flow connection label was clicked.
     *
     *
     * @returns Reference to `this` in order to allow method chaining
     */
    attachLabelPress(
      /**
       * The function to be called when the event occurs
       */
      fnFunction: (p1: ProcessFlow$LabelPressEvent) => void,
      /**
       * Context object to call the event handler with. Defaults to this `sap.suite.ui.commons.ProcessFlow` itself
       */
      oListener?: object
    ): this;
    /**
     * Attaches event handler `fnFunction` to the {@link #event:nodePress nodePress} event of this `sap.suite.ui.commons.ProcessFlow