import path from "node:path/posix";
import { TraceMap } from "@jridgewell/trace-mapping";
import { resolveLinks } from "../../formatter/lib/resolveLinks.js";
import { MESSAGE } from "../messages.js";
import { getPositionsForNode } from "../../utils/nodePosition.js";
function isTsNode(node) {
    return !!node && "getSourceFile" in node && typeof node.getSourceFile === "function";
}
export default class SourceFileReporter {
    #context;
    #resourcePath;
    #originalResourcePath;
    #sourceFile;
    #traceMap;
    #rawMessages = [];
    #coverageInfo = [];
    constructor(context, resourcePath, sourceFile, sourceMap) {
        this.#context = context;
        this.#resourcePath = resourcePath;
        this.#sourceFile = sourceFile;
        if (sourceMap) {
            const parsedSourceMap = JSON.parse(sourceMap);
            if (parsedSourceMap.mappings !== "") {
                // Only create a trace map if there are mappings.
                // Otherwise, it will be useless and causes errors in some cases (Failed to map back to source).
                this.#traceMap = new TraceMap(parsedSourceMap);
            }
        }
        this.#originalResourcePath = this.#getOriginalResourcePath() ?? resourcePath;
        // Do not use messages from context yet, to allow local de-duplication
        this.#coverageInfo = context.getCoverageInfo(this.#originalResourcePath);
    }
    addMessage(id, argsOrNode, node) {
        if (!argsOrNode) {
            throw new Error("Invalid arguments: Missing second argument");
        }
        let args;
        if (isTsNode(argsOrNode)) {
            node = argsOrNode;
            args = null;
        }
        else if (!node) {
            throw new Error("Invalid arguments: Missing 'node'");
        }
        else {
            args = argsOrNode;
        }
        const position = {
            line: 1,
            column: 1,
        };
        if (node) {
            const { start } = this.#getPositionsForNode(node);
            // One-based to be aligned with most IDEs
            position.line = start.line;
            position.column = start.column;
            // endLine = end.line + 1;
            // endColumn = end.column + 1;
        }
        this.#rawMessages.push({ id, args, position });
    }
    addCoverageInfo({ node, message, messageDetails, category }) {
        const { start } = this.#getPositionsForNode(node);
        const coverageInfo = {
            category,
            // One-based to be aligned with most IDEs
            line: start.line,
            column: start.column,
            // endLine: end.line + 1,
            // endColumn: end.column + 1,
            message,
        };
        if (messageDetails) {
            coverageInfo.messageDetails = resolveLinks(messageDetails);
        }
        this.#coverageInfo.push(coverageInfo);
    }
    #getPositionsForNode(node) {
        return getPositionsForNode({
            node,
            sourceFile: this.#sourceFile,
            traceMap: this.#traceMap,
            resourcePath: this.#resourcePath,
        });
    }
    #getOriginalResourcePath() {
        if (this.#traceMap?.sources?.length && this.#traceMap.sources[0] &&
            this.#traceMap.sources[0] !== "UNKNOWN") {
            return path.join(path.dirname(this.#resourcePath), this.#traceMap.sources[0]);
        }
    }
    deduplicateMessages() {
        const lineColumnRawMessagesMap = new Map();
        const rawMessages = [];
        if (this.#rawMessages.length === 0) {
            return;
        }
        for (const message of this.#rawMessages) {
            // Group messages by line/column so that we can deduplicate them
            if (!message.position?.line || !message.position?.column) {
                // If there is no line or column, we cannot group/deduplicate
                rawMessages.push(message);
                continue;
            }
            const lineColumnKey = `${message.position.line}:${message.position.column}`;
            let lineColumnMessages = lineColumnRawMessagesMap.get(lineColumnKey);
            if (!lineColumnMessages) {
                lineColumnMessages = [];
                lineColumnRawMessagesMap.set(lineColumnKey, lineColumnMessages);
            }
            lineColumnMessages.push(message);
        }
        // Add deduplicated messages to the result
        for (const lineColumnMessages of lineColumnRawMessagesMap.values()) {
            // If there are multiple messages for the same line/column,
            // and at least one of them is NOT a "no-globals-js" message,
            // we can deduplicate the "no-globals-js" messages.
            const deduplicateGlobalMessages = lineColumnMessages.length > 1 &&
                lineColumnMessages.some((message) => message.id !== MESSAGE.NO_GLOBALS);
            lineColumnMessages.forEach((message) => {
                if (deduplicateGlobalMessages && message.id === MESSAGE.NO_GLOBALS) {
                    // Skip global messages if there are other messages for the same line/column
                    return;
                }
                rawMessages.push(message);
            });
        }
        this.#context.addLintingMessages(this.#originalResourcePath, rawMessages);
    }
}
//# sourceMappingURL=SourceFileReporter.js.map