import ts from "typescript";
import { PositionInfo } from "../../LinterContext.js";
import Fix, { GlobalAccessRequest, ModuleDependencyRequest } from "./Fix.js";
import { Ui5TypeInfo } from "../Ui5TypeInfo.js";
export interface BaseFixParams {
    /**
     * Name of the module to import for use in the fix
     */
    moduleName?: string;
    /**
     * The preferred identifier to use for the module import.
     * If not provided, the identifier will be derived from the module name
     */
    preferredIdentifier?: string;
    /**
     * Name of a global variable to use in the fix (e.g. "document")
     *
     * The fix will be provided with the identifier name or property access to use via
     * the setIdentifierForGlobal method.
     *
     * For example, if there is already a conflicting identifier within the same file,
     * the fix will be provided with an alternative like "globalThis.document"
     */
    globalName?: string;
    obsoleteModuleName?: string;
    inferObsoleteModuleName?: boolean;
}
export declare enum FixScope {
    /**
     * Replace the whole expression
     *
     * Call expression example: "sap.module.method()" for a call expression
     * Access expression example: "sap.module.property"
     */
    FullExpression = 0,
    /**
     * Replace the first child of the expression
     *
     * Call expression example: "sap.module.method"
     * Access expression example: "sap.module"
     */
    FirstChild = 1,
    /**
     * Replace the second child of the expression
     *
     * Call expression example: "sap.module"
     * Access expression example: "sap"
     */
    SecondChild = 2,
    /**
     * Replace the third child of the expression
     *
     * Call expression example: "sap"
     */
    ThirdChild = 3,
    /**
     * Replace the fourth child of the expression
     */
    FourthChild = 4
}
export default abstract class BaseFix extends Fix {
    protected params: BaseFixParams;
    protected startPos: number | undefined;
    protected endPos: number | undefined;
    protected moduleIdentifierNames: Map<string, string> | undefined;
    protected globalIdentifierNames: Map<string, string> | undefined;
    protected sourcePosition: PositionInfo | undefined;
    protected nodeTypes: ts.SyntaxKind[];
    protected requestsModuleOrGlobal: boolean;
    protected obsoleteModuleDependency?: string;
    constructor(params: BaseFixParams, ui5TypeInfo: Ui5TypeInfo);
    getAffectedSourceCodeRange(): {
        start: number;
        end: number;
    };
    getNodeSearchParameters(): {
        nodeTypes: ts.SyntaxKind[];
        position: PositionInfo;
    };
    setIdentifierForDependency(identifier: string, moduleName: string): void;
    setIdentifierForGlobal(identifier: string, globalName: string): void;
    protected getIdentifiersForSingleRequest(moduleName: string | undefined, globalName: string | undefined): string | undefined;
    /**
     * Helper method for fix classes that feature multiple imports/globals.
     *
     * Returns undefined if any of the requested identifiers could not be set, indicating that the
     * fix must not be applied
     */
    protected getIdentifiersForMultipleRequests(moduleNames: string[] | undefined, globalNames: string[] | undefined): string[] | undefined;
    getNewModuleDependencies(): ModuleDependencyRequest | ModuleDependencyRequest[] | undefined;
    getObsoleteModuleDependencies(): {
        moduleName: string;
        usagePosition: number;
    } | undefined;
    getNewGlobalAccess(): GlobalAccessRequest | GlobalAccessRequest[] | undefined;
}
