/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Subject, Subscription } from 'rxjs';
class EventEmitter_ extends Subject {
    constructor(isAsync = false) {
        super();
        this.__isAsync = isAsync;
    }
    emit(value) {
        super.next(value);
    }
    subscribe(observerOrNext, error, complete) {
        let nextFn = observerOrNext;
        let errorFn = error || (() => null);
        let completeFn = complete;
        if (observerOrNext && typeof observerOrNext === 'object') {
            const observer = observerOrNext;
            nextFn = observer.next?.bind(observer);
            errorFn = observer.error?.bind(observer);
            completeFn = observer.complete?.bind(observer);
        }
        if (this.__isAsync) {
            errorFn = _wrapInTimeout(errorFn);
            if (nextFn) {
                nextFn = _wrapInTimeout(nextFn);
            }
            if (completeFn) {
                completeFn = _wrapInTimeout(completeFn);
            }
        }
        const sink = super.subscribe({ next: nextFn, error: errorFn, complete: completeFn });
        if (observerOrNext instanceof Subscription) {
            observerOrNext.add(sink);
        }
        return sink;
    }
}
function _wrapInTimeout(fn) {
    return (value) => {
        setTimeout(fn, undefined, value);
    };
}
/**
 * @publicApi
 */
export const EventEmitter = EventEmitter_;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXZlbnRfZW1pdHRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvcmUvc3JjL2V2ZW50X2VtaXR0ZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HO0FBRUgsT0FBTyxFQUFrQixPQUFPLEVBQUUsWUFBWSxFQUFDLE1BQU0sTUFBTSxDQUFDO0FBOEY1RCxNQUFNLGFBQWMsU0FBUSxPQUFZO0lBR3RDLFlBQVksVUFBbUIsS0FBSztRQUNsQyxLQUFLLEVBQUUsQ0FBQztRQUNSLElBQUksQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDO0lBQzNCLENBQUM7SUFFRCxJQUFJLENBQUMsS0FBVztRQUNkLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDcEIsQ0FBQztJQUVRLFNBQVMsQ0FBQyxjQUFvQixFQUFFLEtBQVcsRUFBRSxRQUFjO1FBQ2xFLElBQUksTUFBTSxHQUFHLGNBQWMsQ0FBQztRQUM1QixJQUFJLE9BQU8sR0FBRyxLQUFLLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwQyxJQUFJLFVBQVUsR0FBRyxRQUFRLENBQUM7UUFFMUIsSUFBSSxjQUFjLElBQUksT0FBTyxjQUFjLEtBQUssUUFBUSxFQUFFLENBQUM7WUFDekQsTUFBTSxRQUFRLEdBQUcsY0FBMEMsQ0FBQztZQUM1RCxNQUFNLEdBQUcsUUFBUSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDdkMsT0FBTyxHQUFHLFFBQVEsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3pDLFVBQVUsR0FBRyxRQUFRLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNqRCxDQUFDO1FBRUQsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFLENBQUM7WUFDbkIsT0FBTyxHQUFHLGNBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUVsQyxJQUFJLE1BQU0sRUFBRSxDQUFDO2dCQUNYLE1BQU0sR0FBRyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDbEMsQ0FBQztZQUVELElBQUksVUFBVSxFQUFFLENBQUM7Z0JBQ2YsVUFBVSxHQUFHLGNBQWMsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUMxQyxDQUFDO1FBQ0gsQ0FBQztRQUVELE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsRUFBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLFVBQVUsRUFBQyxDQUFDLENBQUM7UUFFbkYsSUFBSSxjQUFjLFlBQVksWUFBWSxFQUFFLENBQUM7WUFDM0MsY0FBYyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMzQixDQUFDO1FBRUQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0NBQ0Y7QUFFRCxTQUFTLGNBQWMsQ0FBQyxFQUEyQjtJQUNqRCxPQUFPLENBQUMsS0FBYyxFQUFFLEVBQUU7UUFDeEIsVUFBVSxDQUFDLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDbkMsQ0FBQyxDQUFDO0FBQ0osQ0FBQztBQUVEOztHQUVHO0FBQ0gsTUFBTSxDQUFDLE1BQU0sWUFBWSxHQUdyQixhQUFvQixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCB7UGFydGlhbE9ic2VydmVyLCBTdWJqZWN0LCBTdWJzY3JpcHRpb259IGZyb20gJ3J4anMnO1xuXG4vKipcbiAqIFVzZSBpbiBjb21wb25lbnRzIHdpdGggdGhlIGBAT3V0cHV0YCBkaXJlY3RpdmUgdG8gZW1pdCBjdXN0b20gZXZlbnRzXG4gKiBzeW5jaHJvbm91c2x5IG9yIGFzeW5jaHJvbm91c2x5LCBhbmQgcmVnaXN0ZXIgaGFuZGxlcnMgZm9yIHRob3NlIGV2ZW50c1xuICogYnkgc3Vic2NyaWJpbmcgdG8gYW4gaW5zdGFuY2UuXG4gKlxuICogQHVzYWdlTm90ZXNcbiAqXG4gKiBFeHRlbmRzXG4gKiBbUnhKUyBgU3ViamVjdGBdKGh0dHBzOi8vcnhqcy5kZXYvYXBpL2luZGV4L2NsYXNzL1N1YmplY3QpXG4gKiBmb3IgQW5ndWxhciBieSBhZGRpbmcgdGhlIGBlbWl0KClgIG1ldGhvZC5cbiAqXG4gKiBJbiB0aGUgZm9sbG93aW5nIGV4YW1wbGUsIGEgY29tcG9uZW50IGRlZmluZXMgdHdvIG91dHB1dCBwcm9wZXJ0aWVzXG4gKiB0aGF0IGNyZWF0ZSBldmVudCBlbWl0dGVycy4gV2hlbiB0aGUgdGl0bGUgaXMgY2xpY2tlZCwgdGhlIGVtaXR0ZXJcbiAqIGVtaXRzIGFuIG9wZW4gb3IgY2xvc2UgZXZlbnQgdG8gdG9nZ2xlIHRoZSBjdXJyZW50IHZpc2liaWxpdHkgc3RhdGUuXG4gKlxuICogYGBgaHRtbFxuICogQENvbXBvbmVudCh7XG4gKiAgIHNlbGVjdG9yOiAnemlwcHknLFxuICogICB0ZW1wbGF0ZTogYFxuICogICA8ZGl2IGNsYXNzPVwiemlwcHlcIj5cbiAqICAgICA8ZGl2IChjbGljayk9XCJ0b2dnbGUoKVwiPlRvZ2dsZTwvZGl2PlxuICogICAgIDxkaXYgW2hpZGRlbl09XCIhdmlzaWJsZVwiPlxuICogICAgICAgPG5nLWNvbnRlbnQ+PC9uZy1jb250ZW50PlxuICogICAgIDwvZGl2PlxuICogIDwvZGl2PmB9KVxuICogZXhwb3J0IGNsYXNzIFppcHB5IHtcbiAqICAgdmlzaWJsZTogYm9vbGVhbiA9IHRydWU7XG4gKiAgIEBPdXRwdXQoKSBvcGVuOiBFdmVudEVtaXR0ZXI8YW55PiA9IG5ldyBFdmVudEVtaXR0ZXIoKTtcbiAqICAgQE91dHB1dCgpIGNsb3NlOiBFdmVudEVtaXR0ZXI8YW55PiA9IG5ldyBFdmVudEVtaXR0ZXIoKTtcbiAqXG4gKiAgIHRvZ2dsZSgpIHtcbiAqICAgICB0aGlzLnZpc2libGUgPSAhdGhpcy52aXNpYmxlO1xuICogICAgIGlmICh0aGlzLnZpc2libGUpIHtcbiAqICAgICAgIHRoaXMub3Blbi5lbWl0KG51bGwpO1xuICogICAgIH0gZWxzZSB7XG4gKiAgICAgICB0aGlzLmNsb3NlLmVtaXQobnVsbCk7XG4gKiAgICAgfVxuICogICB9XG4gKiB9XG4gKiBgYGBcbiAqXG4gKiBBY2Nlc3MgdGhlIGV2ZW50IG9iamVjdCB3aXRoIHRoZSBgJGV2ZW50YCBhcmd1bWVudCBwYXNzZWQgdG8gdGhlIG91dHB1dCBldmVudFxuICogaGFuZGxlcjpcbiAqXG4gKiBgYGBodG1sXG4gKiA8emlwcHkgKG9wZW4pPVwib25PcGVuKCRldmVudClcIiAoY2xvc2UpPVwib25DbG9zZSgkZXZlbnQpXCI+PC96aXBweT5cbiAqIGBgYFxuICpcbiAqIEBzZWUgW09ic2VydmFibGVzIGluIEFuZ3VsYXJdKGd1aWRlL29ic2VydmFibGVzLWluLWFuZ3VsYXIpXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRXZlbnRFbWl0dGVyPFQ+IGV4dGVuZHMgU3ViamVjdDxUPiB7XG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIF9faXNBc3luYzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBpbnN0YW5jZSBvZiB0aGlzIGNsYXNzIHRoYXQgY2FuXG4gICAqIGRlbGl2ZXIgZXZlbnRzIHN5bmNocm9ub3VzbHkgb3IgYXN5bmNocm9ub3VzbHkuXG4gICAqXG4gICAqIEBwYXJhbSBbaXNBc3luYz1mYWxzZV0gV2hlbiB0cnVlLCBkZWxpdmVyIGV2ZW50cyBhc3luY2hyb25vdXNseS5cbiAgICpcbiAgICovXG4gIG5ldyhpc0FzeW5jPzogYm9vbGVhbik6IEV2ZW50RW1pdHRlcjxUPjtcblxuICAvKipcbiAgICogRW1pdHMgYW4gZXZlbnQgY29udGFpbmluZyBhIGdpdmVuIHZhbHVlLlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlIHRvIGVtaXQuXG4gICAqL1xuICBlbWl0KHZhbHVlPzogVCk6IHZvaWQ7XG5cbiAgLyoqXG4gICAqIFJlZ2lzdGVycyBoYW5kbGVycyBmb3IgZXZlbnRzIGVtaXR0ZWQgYnkgdGhpcyBpbnN0YW5jZS5cbiAgICogQHBhcmFtIG5leHQgV2hlbiBzdXBwbGllZCwgYSBjdXN0b20gaGFuZGxlciBmb3IgZW1pdHRlZCBldmVudHMuXG4gICAqIEBwYXJhbSBlcnJvciBXaGVuIHN1cHBsaWVkLCBhIGN1c3RvbSBoYW5kbGVyIGZvciBhbiBlcnJvciBub3RpZmljYXRpb24gZnJvbSB0aGlzIGVtaXR0ZXIuXG4gICAqIEBwYXJhbSBjb21wbGV0ZSBXaGVuIHN1cHBsaWVkLCBhIGN1c3RvbSBoYW5kbGVyIGZvciBhIGNvbXBsZXRpb24gbm90aWZpY2F0aW9uIGZyb20gdGhpc1xuICAgKiAgICAgZW1pdHRlci5cbiAgICovXG4gIHN1YnNjcmliZShuZXh0PzogKHZhbHVlOiBUKSA9PiB2b2lkLCBlcnJvcj86IChlcnJvcjogYW55KSA9PiB2b2lkLCBjb21wbGV0ZT86ICgpID0+IHZvaWQpOlxuICAgICAgU3Vic2NyaXB0aW9uO1xuICAvKipcbiAgICogUmVnaXN0ZXJzIGhhbmRsZXJzIGZvciBldmVudHMgZW1pdHRlZCBieSB0aGlzIGluc3RhbmNlLlxuICAgKiBAcGFyYW0gb2JzZXJ2ZXJPck5leHQgV2hlbiBzdXBwbGllZCwgYSBjdXN0b20gaGFuZGxlciBmb3IgZW1pdHRlZCBldmVudHMsIG9yIGFuIG9ic2VydmVyXG4gICAqICAgICBvYmplY3QuXG4gICAqIEBwYXJhbSBlcnJvciBXaGVuIHN1cHBsaWVkLCBhIGN1c3RvbSBoYW5kbGVyIGZvciBhbiBlcnJvciBub3RpZmljYXRpb24gZnJvbSB0aGlzIGVtaXR0ZXIuXG4gICAqIEBwYXJhbSBjb21wbGV0ZSBXaGVuIHN1cHBsaWVkLCBhIGN1c3RvbSBoYW5kbGVyIGZvciBhIGNvbXBsZXRpb24gbm90aWZpY2F0aW9uIGZyb20gdGhpc1xuICAgKiAgICAgZW1pdHRlci5cbiAgICovXG4gIHN1YnNjcmliZShvYnNlcnZlck9yTmV4dD86IGFueSwgZXJyb3I/OiBhbnksIGNvbXBsZXRlPzogYW55KTogU3Vic2NyaXB0aW9uO1xufVxuXG5jbGFzcyBFdmVudEVtaXR0ZXJfIGV4dGVuZHMgU3ViamVjdDxhbnk+IHtcbiAgX19pc0FzeW5jOiBib29sZWFuOyAgLy8gdHNsaW50OmRpc2FibGUtbGluZVxuXG4gIGNvbnN0cnVjdG9yKGlzQXN5bmM6IGJvb2xlYW4gPSBmYWxzZSkge1xuICAgIHN1cGVyKCk7XG4gICAgdGhpcy5fX2lzQXN5bmMgPSBpc0FzeW5jO1xuICB9XG5cbiAgZW1pdCh2YWx1ZT86IGFueSkge1xuICAgIHN1cGVyLm5leHQodmFsdWUpO1xuICB9XG5cbiAgb3ZlcnJpZGUgc3Vic2NyaWJlKG9ic2VydmVyT3JOZXh0PzogYW55LCBlcnJvcj86IGFueSwgY29tcGxldGU/OiBhbnkpOiBTdWJzY3JpcHRpb24ge1xuICAgIGxldCBuZXh0Rm4gPSBvYnNlcnZlck9yTmV4dDtcbiAgICBsZXQgZXJyb3JGbiA9IGVycm9yIHx8ICgoKSA9PiBudWxsKTtcbiAgICBsZXQgY29tcGxldGVGbiA9IGNvbXBsZXRlO1xuXG4gICAgaWYgKG9ic2VydmVyT3JOZXh0ICYmIHR5cGVvZiBvYnNlcnZlck9yTmV4dCA9PT0gJ29iamVjdCcpIHtcbiAgICAgIGNvbnN0IG9ic2VydmVyID0gb2JzZXJ2ZXJPck5leHQgYXMgUGFydGlhbE9ic2VydmVyPHVua25vd24+O1xuICAgICAgbmV4dEZuID0gb2JzZXJ2ZXIubmV4dD8uYmluZChvYnNlcnZlcik7XG4gICAgICBlcnJvckZuID0gb2JzZXJ2ZXIuZXJyb3I/LmJpbmQob2JzZXJ2ZXIpO1xuICAgICAgY29tcGxldGVGbiA9IG9ic2VydmVyLmNvbXBsZXRlPy5iaW5kKG9ic2VydmVyKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5fX2lzQXN5bmMpIHtcbiAgICAgIGVycm9yRm4gPSBfd3JhcEluVGltZW91dChlcnJvckZuKTtcblxuICAgICAgaWYgKG5leHRGbikge1xuICAgICAgICBuZXh0Rm4gPSBfd3JhcEluVGltZW91dChuZXh0Rm4pO1xuICAgICAgfVxuXG4gICAgICBpZiAoY29tcGxldGVGbikge1xuICAgICAgICBjb21wbGV0ZUZuID0gX3dyYXBJblRpbWVvdXQoY29tcGxldGVGbik7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3Qgc2luayA9IHN1cGVyLnN1YnNjcmliZSh7bmV4dDogbmV4dEZuLCBlcnJvcjogZXJyb3JGbiwgY29tcGxldGU6IGNvbXBsZXRlRm59KTtcblxuICAgIGlmIChvYnNlcnZlck9yTmV4dCBpbnN0YW5jZW9mIFN1YnNjcmlwdGlvbikge1xuICAgICAgb2JzZXJ2ZXJPck5leHQuYWRkKHNpbmspO1xuICAgIH1cblxuICAgIHJldHVybiBzaW5rO1xuICB9XG59XG5cbmZ1bmN0aW9uIF93cmFwSW5UaW1lb3V0KGZuOiAodmFsdWU6IHVua25vd24pID0+IGFueSkge1xuICByZXR1cm4gKHZhbHVlOiB1bmtub3duKSA9PiB7XG4gICAgc2V0VGltZW91dChmbiwgdW5kZWZpbmVkLCB2YWx1ZSk7XG4gIH07XG59XG5cbi8qKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgY29uc3QgRXZlbnRFbWl0dGVyOiB7XG4gIG5ldyAoaXNBc3luYz86IGJvb2xlYW4pOiBFdmVudEVtaXR0ZXI8YW55PjsgbmV3PFQ+KGlzQXN5bmM/OiBib29sZWFuKTogRXZlbnRFbWl0dGVyPFQ+O1xuICByZWFkb25seSBwcm90b3R5cGU6IEV2ZW50RW1pdHRlcjxhbnk+O1xufSA9IEV2ZW50RW1pdHRlcl8gYXMgYW55O1xuIl19