/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { producerAccessed, SIGNAL, signalSetFn } from '@angular/core/primitives/signals';
import { RuntimeError } from '../../errors';
import { ɵINPUT_SIGNAL_BRAND_READ_TYPE, ɵINPUT_SIGNAL_BRAND_WRITE_TYPE } from '../input/input_signal';
import { INPUT_SIGNAL_NODE, REQUIRED_UNSET_VALUE } from '../input/input_signal_node';
/**
 * Creates a model signal.
 *
 * @param initialValue The initial value.
 *   Can be set to {@link REQUIRED_UNSET_VALUE} for required model signals.
 * @param options Additional options for the model.
 */
export function createModelSignal(initialValue) {
    const subscriptions = [];
    const node = Object.create(INPUT_SIGNAL_NODE);
    node.value = initialValue;
    function getter() {
        producerAccessed(node);
        assertModelSet(node.value);
        return node.value;
    }
    function notifySubscribers(value) {
        for (let i = 0; i < subscriptions.length; i++) {
            subscriptions[i](value);
        }
    }
    getter[SIGNAL] = node;
    getter.asReadonly = (() => getter());
    getter.set = (newValue) => {
        if (!node.equal(node.value, newValue)) {
            signalSetFn(node, newValue);
            notifySubscribers(newValue);
        }
    };
    getter.update = (updateFn) => {
        assertModelSet(node.value);
        getter.set(updateFn(node.value));
    };
    getter.subscribe = (callback) => {
        subscriptions.push(callback);
        return () => {
            const index = subscriptions.indexOf(callback);
            if (index > -1) {
                subscriptions.splice(index, 1);
            }
        };
    };
    if (ngDevMode) {
        getter.toString = () => `[Model Signal: ${getter()}]`;
    }
    return getter;
}
/** Asserts that a model's value is set. */
function assertModelSet(value) {
    if (value === REQUIRED_UNSET_VALUE) {
        throw new RuntimeError(-952 /* RuntimeErrorCode.REQUIRED_MODEL_NO_VALUE */, ngDevMode && 'Model is required but no value is available yet.');
    }
}
//# sourceMappingURL=data:application/json;base64,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