package org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.match.rev140714.nxm.of.tcp.src.grouping;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.PortNumber;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link NxmOfTcpSrcBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     NxmOfTcpSrcBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new NxmOfTcpSrcBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of NxmOfTcpSrcBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see NxmOfTcpSrcBuilder
 * @see Builder
 *
 */
public class NxmOfTcpSrcBuilder implements Builder<NxmOfTcpSrc> {

    private Integer _mask;
    private PortNumber _port;


    Map<Class<? extends Augmentation<NxmOfTcpSrc>>, Augmentation<NxmOfTcpSrc>> augmentation = Collections.emptyMap();

    public NxmOfTcpSrcBuilder() {
    }

    public NxmOfTcpSrcBuilder(NxmOfTcpSrc base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<NxmOfTcpSrc>>, Augmentation<NxmOfTcpSrc>> aug =((AugmentationHolder<NxmOfTcpSrc>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._mask = base.getMask();
        this._port = base.getPort();
    }


    public Integer getMask() {
        return _mask;
    }
    
    public PortNumber getPort() {
        return _port;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<NxmOfTcpSrc>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    private static void checkMaskRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxmOfTcpSrcBuilder setMask(final Integer value) {
    if (value != null) {
        checkMaskRange(value);
        
    }
        this._mask = value;
        return this;
    }
    
    public NxmOfTcpSrcBuilder setPort(final PortNumber value) {
        this._port = value;
        return this;
    }
    
    public NxmOfTcpSrcBuilder addAugmentation(Class<? extends Augmentation<NxmOfTcpSrc>> augmentationType, Augmentation<NxmOfTcpSrc> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public NxmOfTcpSrcBuilder removeAugmentation(Class<? extends Augmentation<NxmOfTcpSrc>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public NxmOfTcpSrc build() {
        return new NxmOfTcpSrcImpl(this);
    }

    private static final class NxmOfTcpSrcImpl
        extends AbstractAugmentable<NxmOfTcpSrc>
        implements NxmOfTcpSrc {
    
        private final Integer _mask;
        private final PortNumber _port;
    
        NxmOfTcpSrcImpl(NxmOfTcpSrcBuilder base) {
            super(base.augmentation);
            this._mask = base.getMask();
            this._port = base.getPort();
        }
    
        @Override
        public Integer getMask() {
            return _mask;
        }
        
        @Override
        public PortNumber getPort() {
            return _port;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_mask);
            result = prime * result + Objects.hashCode(_port);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!NxmOfTcpSrc.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            NxmOfTcpSrc other = (NxmOfTcpSrc)obj;
            if (!Objects.equals(_mask, other.getMask())) {
                return false;
            }
            if (!Objects.equals(_port, other.getPort())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                NxmOfTcpSrcImpl otherImpl = (NxmOfTcpSrcImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<NxmOfTcpSrc>>, Augmentation<NxmOfTcpSrc>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("NxmOfTcpSrc");
            CodeHelpers.appendValue(helper, "_mask", _mask);
            CodeHelpers.appendValue(helper, "_port", _port);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
