package org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.action.rev140714.nx.action.learn.grouping;
import com.google.common.base.MoreObjects;
import com.google.common.collect.Range;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.Short;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.lang.reflect.Array;
import java.math.BigInteger;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.action.rev140714.nx.action.learn.grouping.nx.learn.FlowMods;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link NxLearnBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     NxLearnBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new NxLearnBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of NxLearnBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see NxLearnBuilder
 * @see Builder
 *
 */
public class NxLearnBuilder implements Builder<NxLearn> {

    private BigInteger _cookie;
    private Integer _finHardTimeout;
    private Integer _finIdleTimeout;
    private Integer _flags;
    private List<FlowMods> _flowMods;
    private Integer _hardTimeout;
    private Integer _idleTimeout;
    private Integer _priority;
    private Short _tableId;


    Map<Class<? extends Augmentation<NxLearn>>, Augmentation<NxLearn>> augmentation = Collections.emptyMap();

    public NxLearnBuilder() {
    }

    public NxLearnBuilder(NxLearn base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<NxLearn>>, Augmentation<NxLearn>> aug =((AugmentationHolder<NxLearn>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._cookie = base.getCookie();
        this._finHardTimeout = base.getFinHardTimeout();
        this._finIdleTimeout = base.getFinIdleTimeout();
        this._flags = base.getFlags();
        this._flowMods = base.getFlowMods();
        this._hardTimeout = base.getHardTimeout();
        this._idleTimeout = base.getIdleTimeout();
        this._priority = base.getPriority();
        this._tableId = base.getTableId();
    }


    public BigInteger getCookie() {
        return _cookie;
    }
    
    public Integer getFinHardTimeout() {
        return _finHardTimeout;
    }
    
    public Integer getFinIdleTimeout() {
        return _finIdleTimeout;
    }
    
    public Integer getFlags() {
        return _flags;
    }
    
    public List<FlowMods> getFlowMods() {
        return _flowMods;
    }
    
    public Integer getHardTimeout() {
        return _hardTimeout;
    }
    
    public Integer getIdleTimeout() {
        return _idleTimeout;
    }
    
    public Integer getPriority() {
        return _priority;
    }
    
    public Short getTableId() {
        return _tableId;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<NxLearn>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    private static final Range<java.math.BigInteger>[] CHECKCOOKIERANGE_RANGES;
    static {
        @SuppressWarnings("unchecked")
        final Range<java.math.BigInteger>[] a = (Range<java.math.BigInteger>[]) Array.newInstance(Range.class, 1);
        a[0] = Range.closed(java.math.BigInteger.ZERO, new java.math.BigInteger("18446744073709551615"));
        CHECKCOOKIERANGE_RANGES = a;
    }
    private static void checkCookieRange(final java.math.BigInteger value) {
        for (Range<java.math.BigInteger> r : CHECKCOOKIERANGE_RANGES) {
            if (r.contains(value)) {
                return;
            }
        }
        CodeHelpers.throwInvalidRange(CHECKCOOKIERANGE_RANGES, value);
    }
    
    public NxLearnBuilder setCookie(final BigInteger value) {
    if (value != null) {
        checkCookieRange(value);
        
    }
        this._cookie = value;
        return this;
    }
    private static void checkFinHardTimeoutRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxLearnBuilder setFinHardTimeout(final Integer value) {
    if (value != null) {
        checkFinHardTimeoutRange(value);
        
    }
        this._finHardTimeout = value;
        return this;
    }
    private static void checkFinIdleTimeoutRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxLearnBuilder setFinIdleTimeout(final Integer value) {
    if (value != null) {
        checkFinIdleTimeoutRange(value);
        
    }
        this._finIdleTimeout = value;
        return this;
    }
    private static void checkFlagsRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxLearnBuilder setFlags(final Integer value) {
    if (value != null) {
        checkFlagsRange(value);
        
    }
        this._flags = value;
        return this;
    }
    public NxLearnBuilder setFlowMods(final List<FlowMods> values) {
        this._flowMods = values;
        return this;
    }
    
    private static void checkHardTimeoutRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxLearnBuilder setHardTimeout(final Integer value) {
    if (value != null) {
        checkHardTimeoutRange(value);
        
    }
        this._hardTimeout = value;
        return this;
    }
    private static void checkIdleTimeoutRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxLearnBuilder setIdleTimeout(final Integer value) {
    if (value != null) {
        checkIdleTimeoutRange(value);
        
    }
        this._idleTimeout = value;
        return this;
    }
    private static void checkPriorityRange(final int value) {
        if (value >= 0 && value <= 65535) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..65535]]", value);
    }
    
    public NxLearnBuilder setPriority(final Integer value) {
    if (value != null) {
        checkPriorityRange(value);
        
    }
        this._priority = value;
        return this;
    }
    private static void checkTableIdRange(final short value) {
        if (value >= (short)0 && value <= (short)255) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..255]]", value);
    }
    
    public NxLearnBuilder setTableId(final Short value) {
    if (value != null) {
        checkTableIdRange(value);
        
    }
        this._tableId = value;
        return this;
    }
    
    public NxLearnBuilder addAugmentation(Class<? extends Augmentation<NxLearn>> augmentationType, Augmentation<NxLearn> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public NxLearnBuilder removeAugmentation(Class<? extends Augmentation<NxLearn>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public NxLearn build() {
        return new NxLearnImpl(this);
    }

    private static final class NxLearnImpl
        extends AbstractAugmentable<NxLearn>
        implements NxLearn {
    
        private final BigInteger _cookie;
        private final Integer _finHardTimeout;
        private final Integer _finIdleTimeout;
        private final Integer _flags;
        private final List<FlowMods> _flowMods;
        private final Integer _hardTimeout;
        private final Integer _idleTimeout;
        private final Integer _priority;
        private final Short _tableId;
    
        NxLearnImpl(NxLearnBuilder base) {
            super(base.augmentation);
            this._cookie = base.getCookie();
            this._finHardTimeout = base.getFinHardTimeout();
            this._finIdleTimeout = base.getFinIdleTimeout();
            this._flags = base.getFlags();
            this._flowMods = base.getFlowMods();
            this._hardTimeout = base.getHardTimeout();
            this._idleTimeout = base.getIdleTimeout();
            this._priority = base.getPriority();
            this._tableId = base.getTableId();
        }
    
        @Override
        public BigInteger getCookie() {
            return _cookie;
        }
        
        @Override
        public Integer getFinHardTimeout() {
            return _finHardTimeout;
        }
        
        @Override
        public Integer getFinIdleTimeout() {
            return _finIdleTimeout;
        }
        
        @Override
        public Integer getFlags() {
            return _flags;
        }
        
        @Override
        public List<FlowMods> getFlowMods() {
            return _flowMods;
        }
        
        @Override
        public Integer getHardTimeout() {
            return _hardTimeout;
        }
        
        @Override
        public Integer getIdleTimeout() {
            return _idleTimeout;
        }
        
        @Override
        public Integer getPriority() {
            return _priority;
        }
        
        @Override
        public Short getTableId() {
            return _tableId;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_cookie);
            result = prime * result + Objects.hashCode(_finHardTimeout);
            result = prime * result + Objects.hashCode(_finIdleTimeout);
            result = prime * result + Objects.hashCode(_flags);
            result = prime * result + Objects.hashCode(_flowMods);
            result = prime * result + Objects.hashCode(_hardTimeout);
            result = prime * result + Objects.hashCode(_idleTimeout);
            result = prime * result + Objects.hashCode(_priority);
            result = prime * result + Objects.hashCode(_tableId);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!NxLearn.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            NxLearn other = (NxLearn)obj;
            if (!Objects.equals(_cookie, other.getCookie())) {
                return false;
            }
            if (!Objects.equals(_finHardTimeout, other.getFinHardTimeout())) {
                return false;
            }
            if (!Objects.equals(_finIdleTimeout, other.getFinIdleTimeout())) {
                return false;
            }
            if (!Objects.equals(_flags, other.getFlags())) {
                return false;
            }
            if (!Objects.equals(_flowMods, other.getFlowMods())) {
                return false;
            }
            if (!Objects.equals(_hardTimeout, other.getHardTimeout())) {
                return false;
            }
            if (!Objects.equals(_idleTimeout, other.getIdleTimeout())) {
                return false;
            }
            if (!Objects.equals(_priority, other.getPriority())) {
                return false;
            }
            if (!Objects.equals(_tableId, other.getTableId())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                NxLearnImpl otherImpl = (NxLearnImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<NxLearn>>, Augmentation<NxLearn>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("NxLearn");
            CodeHelpers.appendValue(helper, "_cookie", _cookie);
            CodeHelpers.appendValue(helper, "_finHardTimeout", _finHardTimeout);
            CodeHelpers.appendValue(helper, "_finIdleTimeout", _finIdleTimeout);
            CodeHelpers.appendValue(helper, "_flags", _flags);
            CodeHelpers.appendValue(helper, "_flowMods", _flowMods);
            CodeHelpers.appendValue(helper, "_hardTimeout", _hardTimeout);
            CodeHelpers.appendValue(helper, "_idleTimeout", _idleTimeout);
            CodeHelpers.appendValue(helper, "_priority", _priority);
            CodeHelpers.appendValue(helper, "_tableId", _tableId);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
