package org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.match.rev140714.nxm.nx.ct.mark.grouping;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint32;

/**
 * Class that builds {@link NxmNxCtMarkBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     NxmNxCtMarkBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new NxmNxCtMarkBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of NxmNxCtMarkBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see NxmNxCtMarkBuilder
 * @see Builder
 *
 */
public class NxmNxCtMarkBuilder implements Builder<NxmNxCtMark> {

    private Uint32 _ctMark;
    private Uint32 _mask;


    Map<Class<? extends Augmentation<NxmNxCtMark>>, Augmentation<NxmNxCtMark>> augmentation = Collections.emptyMap();

    public NxmNxCtMarkBuilder() {
    }

    public NxmNxCtMarkBuilder(NxmNxCtMark base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<NxmNxCtMark>>, Augmentation<NxmNxCtMark>> aug =((AugmentationHolder<NxmNxCtMark>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._ctMark = base.getCtMark();
        this._mask = base.getMask();
    }


    public Uint32 getCtMark() {
        return _ctMark;
    }
    
    public Uint32 getMask() {
        return _mask;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<NxmNxCtMark>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public NxmNxCtMarkBuilder setCtMark(final Uint32 value) {
        this._ctMark = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setCtMark(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public NxmNxCtMarkBuilder setCtMark(final Long value) {
        return setCtMark(CodeHelpers.compatUint(value));
    }
    
    public NxmNxCtMarkBuilder setMask(final Uint32 value) {
        this._mask = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setMask(Uint32)} instead.
     */
    @Deprecated(forRemoval = true)
    public NxmNxCtMarkBuilder setMask(final Long value) {
        return setMask(CodeHelpers.compatUint(value));
    }
    
    public NxmNxCtMarkBuilder addAugmentation(Class<? extends Augmentation<NxmNxCtMark>> augmentationType, Augmentation<NxmNxCtMark> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public NxmNxCtMarkBuilder removeAugmentation(Class<? extends Augmentation<NxmNxCtMark>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public NxmNxCtMark build() {
        return new NxmNxCtMarkImpl(this);
    }

    private static final class NxmNxCtMarkImpl
        extends AbstractAugmentable<NxmNxCtMark>
        implements NxmNxCtMark {
    
        private final Uint32 _ctMark;
        private final Uint32 _mask;
    
        NxmNxCtMarkImpl(NxmNxCtMarkBuilder base) {
            super(base.augmentation);
            this._ctMark = base.getCtMark();
            this._mask = base.getMask();
        }
    
        @Override
        public Uint32 getCtMark() {
            return _ctMark;
        }
        
        @Override
        public Uint32 getMask() {
            return _mask;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_ctMark);
            result = prime * result + Objects.hashCode(_mask);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!NxmNxCtMark.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            NxmNxCtMark other = (NxmNxCtMark)obj;
            if (!Objects.equals(_ctMark, other.getCtMark())) {
                return false;
            }
            if (!Objects.equals(_mask, other.getMask())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                NxmNxCtMarkImpl otherImpl = (NxmNxCtMarkImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<NxmNxCtMark>>, Augmentation<NxmNxCtMark>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("NxmNxCtMark");
            CodeHelpers.appendValue(helper, "_ctMark", _ctMark);
            CodeHelpers.appendValue(helper, "_mask", _mask);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
