package org.opendaylight.yang.gen.v1.urn.opendaylight.openflowplugin.extension.nicira.match.rev140714.nxm.of.udp.src.grouping;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.PortNumber;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint16;

/**
 * Class that builds {@link NxmOfUdpSrcBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     NxmOfUdpSrcBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new NxmOfUdpSrcBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of NxmOfUdpSrcBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see NxmOfUdpSrcBuilder
 * @see Builder
 *
 */
public class NxmOfUdpSrcBuilder implements Builder<NxmOfUdpSrc> {

    private Uint16 _mask;
    private PortNumber _port;


    Map<Class<? extends Augmentation<NxmOfUdpSrc>>, Augmentation<NxmOfUdpSrc>> augmentation = Collections.emptyMap();

    public NxmOfUdpSrcBuilder() {
    }

    public NxmOfUdpSrcBuilder(NxmOfUdpSrc base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<NxmOfUdpSrc>>, Augmentation<NxmOfUdpSrc>> aug =((AugmentationHolder<NxmOfUdpSrc>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._mask = base.getMask();
        this._port = base.getPort();
    }


    public Uint16 getMask() {
        return _mask;
    }
    
    public PortNumber getPort() {
        return _port;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<NxmOfUdpSrc>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public NxmOfUdpSrcBuilder setMask(final Uint16 value) {
        this._mask = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setMask(Uint16)} instead.
     */
    @Deprecated(forRemoval = true)
    public NxmOfUdpSrcBuilder setMask(final Integer value) {
        return setMask(CodeHelpers.compatUint(value));
    }
    
    public NxmOfUdpSrcBuilder setPort(final PortNumber value) {
        this._port = value;
        return this;
    }
    
    public NxmOfUdpSrcBuilder addAugmentation(Class<? extends Augmentation<NxmOfUdpSrc>> augmentationType, Augmentation<NxmOfUdpSrc> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public NxmOfUdpSrcBuilder removeAugmentation(Class<? extends Augmentation<NxmOfUdpSrc>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public NxmOfUdpSrc build() {
        return new NxmOfUdpSrcImpl(this);
    }

    private static final class NxmOfUdpSrcImpl
        extends AbstractAugmentable<NxmOfUdpSrc>
        implements NxmOfUdpSrc {
    
        private final Uint16 _mask;
        private final PortNumber _port;
    
        NxmOfUdpSrcImpl(NxmOfUdpSrcBuilder base) {
            super(base.augmentation);
            this._mask = base.getMask();
            this._port = base.getPort();
        }
    
        @Override
        public Uint16 getMask() {
            return _mask;
        }
        
        @Override
        public PortNumber getPort() {
            return _port;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_mask);
            result = prime * result + Objects.hashCode(_port);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!NxmOfUdpSrc.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            NxmOfUdpSrc other = (NxmOfUdpSrc)obj;
            if (!Objects.equals(_mask, other.getMask())) {
                return false;
            }
            if (!Objects.equals(_port, other.getPort())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                NxmOfUdpSrcImpl otherImpl = (NxmOfUdpSrcImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<NxmOfUdpSrc>>, Augmentation<NxmOfUdpSrc>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("NxmOfUdpSrc");
            CodeHelpers.appendValue(helper, "_mask", _mask);
            CodeHelpers.appendValue(helper, "_port", _port);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
