/*
 * Decompiled with CFR 0.152.
 */
package org.opencastproject.userdirectory.canvas;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.http.client.fluent.Content;
import org.apache.http.client.fluent.Request;
import org.opencastproject.security.api.JaxbOrganization;
import org.opencastproject.security.api.JaxbRole;
import org.opencastproject.security.api.JaxbUser;
import org.opencastproject.security.api.Organization;
import org.opencastproject.security.api.OrganizationDirectoryService;
import org.opencastproject.security.api.Role;
import org.opencastproject.security.api.RoleProvider;
import org.opencastproject.security.api.SecurityService;
import org.opencastproject.security.api.User;
import org.opencastproject.security.api.UserProvider;
import org.opencastproject.util.NotFoundException;
import org.opencastproject.util.OsgiUtil;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(property={"service.description=Provides for Canvas users and roles"}, immediate=true, service={UserProvider.class, RoleProvider.class})
public class CanvasUserRoleProvider
implements UserProvider,
RoleProvider {
    private static final Logger logger = LoggerFactory.getLogger(CanvasUserRoleProvider.class);
    private static final String LTI_LEARNER_ROLE = "Learner";
    private static final String LTI_INSTRUCTOR_ROLE = "Instructor";
    private static final String PROVIDER_NAME = "canvas";
    private static final String ORGANIZATION_KEY = "org.opencastproject.userdirectory.canvas.org";
    private static final String DEFAULT_ORGANIZATION_VALUE = "mh_default_org";
    private static final String CANVAS_URL_KEY = "org.opencastproject.userdirectory.canvas.url";
    private static final String CANVAS_USER_TOKEN_KEY = "org.opencastproject.userdirectory.canvas.token";
    private static final String CACHE_SIZE_KEY = "org.opencastproject.userdirectory.canvas.cache.size";
    private static final Integer DEFAULT_CACHE_SIZE_VALUE = 1000;
    private static final String CACHE_EXPIRATION_KEY = "org.opencastproject.userdirectory.canvas.cache.expiration";
    private static final Integer DEFAULT_CACHE_EXPIRATION_VALUE = 60;
    private static final String CANVAS_INSTRUCTOR_ROLES_KEY = "org.opencastproject.userdirectory.canvas.instructor.roles";
    private static final String DEFAULT_CANVAS_INSTRUCTOR_ROLES = "teacher,ta";
    private static final String IGNORED_USERNAMES_KEY = "org.opencastproject.userdirectory.canvas.ignored.usernames";
    private static final String DEFAULT_INGROED_USERNAMES = "admin,anonymous";
    private Organization org;
    private String url;
    private String token;
    private int cacheSize;
    private int cacheExpiration;
    private Set<String> instructorRoles;
    private Set<String> ignoredUsernames;
    private LoadingCache<String, Object> cache = null;
    private final Object nullToken = new Object();
    private OrganizationDirectoryService orgDirectory;
    private SecurityService securityService;

    @Activate
    public void activate(ComponentContext cc) throws ConfigurationException {
        this.loadConfig(cc);
        logger.info("Activating CanvasUserRoleProvider(url={}, cacheSize={}, cacheExpiration={}, instructorRoles={}, ignoredUserNames={}", new Object[]{this.url, this.cacheSize, this.cacheExpiration, this.instructorRoles, this.ignoredUsernames});
        this.cache = CacheBuilder.newBuilder().maximumSize((long)this.cacheSize).expireAfterWrite((long)this.cacheExpiration, TimeUnit.MINUTES).build((CacheLoader)new CacheLoader<String, Object>(){

            public Object load(String id) {
                User user = CanvasUserRoleProvider.this.loadUserFromCanvas(id);
                return user == null ? CanvasUserRoleProvider.this.nullToken : user;
            }
        });
    }

    @Reference
    public void setOrgDirectory(OrganizationDirectoryService orgDirectory) {
        this.orgDirectory = orgDirectory;
    }

    @Reference
    public void setSecurityService(SecurityService securityService) {
        this.securityService = securityService;
    }

    private void loadConfig(ComponentContext cc) throws ConfigurationException {
        String orgStr = OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)ORGANIZATION_KEY, (String)DEFAULT_ORGANIZATION_VALUE);
        try {
            this.org = this.orgDirectory.getOrganization(orgStr);
        }
        catch (NotFoundException e) {
            logger.warn("Organization {} not found!", (Object)orgStr);
            throw new ConfigurationException(ORGANIZATION_KEY, "not found");
        }
        this.url = OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)CANVAS_URL_KEY);
        if (this.url.endsWith("/")) {
            this.url = StringUtils.chop((String)this.url);
        }
        logger.debug("Canvas URL: {}", (Object)this.url);
        this.token = OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)CANVAS_USER_TOKEN_KEY);
        String cacheSizeStr = OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)CACHE_SIZE_KEY, (String)DEFAULT_CACHE_SIZE_VALUE.toString());
        this.cacheSize = NumberUtils.toInt((String)cacheSizeStr);
        String cacheExpireStr = OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)CACHE_EXPIRATION_KEY, (String)DEFAULT_CACHE_EXPIRATION_VALUE.toString());
        this.cacheExpiration = NumberUtils.toInt((String)cacheExpireStr);
        String rolesStr = OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)CANVAS_INSTRUCTOR_ROLES_KEY, (String)DEFAULT_CANVAS_INSTRUCTOR_ROLES);
        this.instructorRoles = this.parsePropertyLineAsSet(rolesStr);
        logger.debug("Canvas instructor roles: {}", this.instructorRoles);
        String ignoredUsersStr = OsgiUtil.getComponentContextProperty((ComponentContext)cc, (String)IGNORED_USERNAMES_KEY, (String)DEFAULT_INGROED_USERNAMES);
        this.ignoredUsernames = this.parsePropertyLineAsSet(ignoredUsersStr);
        logger.debug("Ignored users: {}", this.ignoredUsernames);
    }

    public List<Role> getRolesForUser(String userName) {
        logger.debug("getRolesForUser({})", (Object)userName);
        ArrayList<Role> roles = new ArrayList<Role>();
        if (this.ignoredUsernames.contains(userName)) {
            logger.debug("We don't answer for: {}", (Object)userName);
            return roles;
        }
        User user = this.loadUser(userName);
        if (user != null) {
            logger.debug("Returning cached rolset for {}", (Object)userName);
            return new ArrayList<Role>(user.getRoles());
        }
        logger.debug("Return empty roleset for {} - not found in Canvas", (Object)userName);
        return Collections.emptyList();
    }

    public Iterator<Role> findRoles(String query, Role.Target target, int offset, int limit) {
        logger.debug("findRoles(query={} offset={} limit={})", new Object[]{query, offset, limit});
        if (target == Role.Target.USER) {
            return Collections.emptyIterator();
        }
        boolean exact = true;
        if (query.endsWith("%")) {
            exact = false;
            query = StringUtils.chop((String)query);
        }
        if (query.isEmpty()) {
            return Collections.emptyIterator();
        }
        if (exact && !query.endsWith("_Learner") && !query.endsWith("_Instructor")) {
            return Collections.emptyIterator();
        }
        ArrayList<JaxbRole> roles = new ArrayList<JaxbRole>();
        JaxbOrganization jaxbOrganization = JaxbOrganization.fromOrganization((Organization)this.org);
        for (String siteRole : this.getCanvasSiteRolesByCurrentUser(query, exact)) {
            roles.add(new JaxbRole(siteRole, jaxbOrganization, "Canvas Site Role", Role.Type.EXTERNAL));
        }
        return roles.iterator();
    }

    public String getName() {
        return PROVIDER_NAME;
    }

    public Iterator<User> getUsers() {
        return Collections.emptyIterator();
    }

    public User loadUser(String userName) {
        logger.debug("loadUser({})", (Object)userName);
        Object user = this.cache.getUnchecked((Object)userName);
        if (user == this.nullToken) {
            logger.debug("Returning null user from cache");
            return null;
        }
        logger.debug("Returning user {} from cache", (Object)userName);
        return (JaxbUser)user;
    }

    public long countUsers() {
        return 0L;
    }

    public String getOrganization() {
        return this.org.getId();
    }

    public Iterator<User> findUsers(String query, int offset, int limit) {
        if (query == null) {
            throw new IllegalArgumentException("Query must be set");
        }
        if (query.endsWith("%")) {
            query = query.substring(0, query.length() - 1);
        }
        if (query.isEmpty()) {
            return Collections.emptyIterator();
        }
        if (!this.verifyCanvasUser(query)) {
            return Collections.emptyIterator();
        }
        ArrayList<JaxbUser> users = new ArrayList<JaxbUser>();
        JaxbOrganization jaxbOrganization = JaxbOrganization.fromOrganization((Organization)this.org);
        JaxbUser queryUser = new JaxbUser(query, PROVIDER_NAME, jaxbOrganization, new HashSet());
        users.add(queryUser);
        return users.iterator();
    }

    public void invalidate(String userName) {
        this.cache.invalidate((Object)userName);
    }

    private User loadUserFromCanvas(String userName) {
        if (this.cache == null) {
            throw new IllegalArgumentException("The Canvas user detail service has not yet been configured");
        }
        if (this.ignoredUsernames.contains(userName)) {
            this.cache.put((Object)userName, this.nullToken);
            logger.debug("We don't answer for: {}", (Object)userName);
            return null;
        }
        logger.debug("In loadUserFromCanvas, currently processing user: {}", (Object)userName);
        JaxbOrganization jaxbOrganization = JaxbOrganization.fromOrganization((Organization)this.org);
        String[] canvasUserInfo = this.getCanvasUserInfo(userName);
        if (canvasUserInfo == null) {
            this.cache.put((Object)userName, this.nullToken);
            return null;
        }
        String email = canvasUserInfo[0];
        String displayName = canvasUserInfo[1];
        List<String> canvasRoles = this.getRolesFromCanvas(userName);
        if (canvasRoles == null) {
            this.cache.put((Object)userName, this.nullToken);
            return null;
        }
        logger.debug("Canvas roles for {}: {}", (Object)userName, canvasRoles);
        HashSet<JaxbRole> roles = new HashSet<JaxbRole>();
        boolean isInstructor = false;
        for (String roleStr : canvasRoles) {
            roles.add(new JaxbRole(roleStr, jaxbOrganization, "Canvas external role", Role.Type.EXTERNAL));
            if (!roleStr.endsWith(LTI_INSTRUCTOR_ROLE)) continue;
            isInstructor = true;
        }
        roles.add(new JaxbRole("ROLE_GROUP_CANVAS", jaxbOrganization, "Canvas User", Role.Type.EXTERNAL_GROUP));
        if (isInstructor) {
            roles.add(new JaxbRole("ROLE_GROUP_CANVAS_INSTRUCTOR", jaxbOrganization, "Canvas Instructor", Role.Type.EXTERNAL_GROUP));
        }
        logger.debug("Returning JaxbRoles: {}", roles);
        JaxbUser user = new JaxbUser(userName, null, displayName, email, PROVIDER_NAME, jaxbOrganization, roles);
        this.cache.put((Object)userName, (Object)user);
        logger.debug("Returning user {}", (Object)userName);
        return user;
    }

    private String[] getCanvasUserInfo(String userName) {
        try {
            userName = URLEncoder.encode(userName, "UTF-8");
        }
        catch (UnsupportedEncodingException unsupportedEncodingException) {
            // empty catch block
        }
        String urlString = String.format("%s/api/v1/users/sis_login_id:%s", this.url, userName);
        try {
            JsonNode node = this.getRequestJson(urlString);
            String email = node.path("email").asText();
            String displayName = node.path("name").asText();
            return new String[]{email, displayName};
        }
        catch (IOException e) {
            logger.warn("Exception getting Canvas user information for user {} at {}: {}", new Object[]{userName, urlString, e});
            return null;
        }
    }

    private List<String> getRolesFromCanvas(String userName) {
        logger.debug("getRolesFromCanvas({})", (Object)userName);
        try {
            userName = URLEncoder.encode(userName, "UTF-8");
        }
        catch (UnsupportedEncodingException unsupportedEncodingException) {
            // empty catch block
        }
        String urlString = String.format("%s/api/v1/users/sis_login_id:%s/courses.json?per_page=500&enrollment_state=active&state[]=unpublished&state[]=available", this.url, userName);
        try {
            ArrayList<String> roleList = new ArrayList<String>();
            JsonNode nodes = this.getRequestJson(urlString);
            for (JsonNode node : nodes) {
                String courseId = node.path("id").asText();
                JsonNode enrollmentNodes = node.path("enrollments");
                for (JsonNode enrollmentNode : enrollmentNodes) {
                    String canvasRole = enrollmentNode.path("type").asText();
                    String ltiRole = this.instructorRoles.contains(canvasRole) ? LTI_INSTRUCTOR_ROLE : LTI_LEARNER_ROLE;
                    String opencastRole = String.format("%s_%s", courseId, ltiRole);
                    roleList.add(opencastRole);
                }
            }
            return roleList;
        }
        catch (IOException e) {
            logger.warn("Exception getting site/role membership for Canvas user {} at {}: {}", new Object[]{userName, urlString, e});
            return null;
        }
    }

    private JsonNode getRequestJson(String urlString) throws IOException {
        Content content = Request.Get((String)urlString).addHeader("Authorization", "Bearer " + this.token).execute().returnContent();
        ObjectMapper mapper = new ObjectMapper();
        return mapper.readTree(content.asStream());
    }

    private boolean verifyCanvasUser(String userName) {
        logger.debug("verifyCanvasUser({})", (Object)userName);
        try {
            userName = URLEncoder.encode(userName, "UTF-8");
        }
        catch (UnsupportedEncodingException unsupportedEncodingException) {
            // empty catch block
        }
        String urlString = String.format("%s/api/v1/users/sis_login_id:%s", this.url, userName);
        try {
            this.getRequestJson(urlString);
        }
        catch (IOException e) {
            return false;
        }
        return true;
    }

    private Set<String> parsePropertyLineAsSet(String configLine) {
        String[] configs;
        HashSet<String> set = new HashSet<String>();
        for (String config : configs = configLine.split(",")) {
            set.add(config.trim());
        }
        return set;
    }

    private List<String> getCanvasSiteRolesByCurrentUser(String query, boolean exact) {
        User user = this.securityService.getUser();
        if (exact) {
            Set roles = user.getRoles().stream().map(Role::getName).collect(Collectors.toSet());
            if (roles.contains(query)) {
                return Collections.singletonList(query);
            }
            return Collections.emptyList();
        }
        String finalQuery = StringUtils.chop((String)query);
        return user.getRoles().stream().map(Role::getName).filter(roleName -> roleName.endsWith("_Instructor") || roleName.endsWith("_Learner")).map(roleName -> StringUtils.substringBeforeLast((String)roleName, (String)"_")).distinct().map(site -> Arrays.asList(site + "_Learner", site + "_Instructor")).flatMap(siteRoles -> siteRoles.stream()).filter(roleName -> roleName.startsWith(finalQuery)).collect(Collectors.toList());
    }
}

