/*
 * Decompiled with CFR 0.152.
 */
package org.opencastproject.playlists;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.module.jaxb.JaxbAnnotationModule;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.opencastproject.elasticsearch.api.SearchIndexException;
import org.opencastproject.elasticsearch.api.SearchResult;
import org.opencastproject.elasticsearch.index.ElasticsearchIndex;
import org.opencastproject.elasticsearch.index.objects.event.Event;
import org.opencastproject.elasticsearch.index.objects.event.EventSearchQuery;
import org.opencastproject.playlists.Playlist;
import org.opencastproject.playlists.PlaylistAccessControlEntry;
import org.opencastproject.playlists.PlaylistEntry;
import org.opencastproject.playlists.PlaylistEntryType;
import org.opencastproject.playlists.persistence.PlaylistDatabaseException;
import org.opencastproject.playlists.persistence.PlaylistDatabaseService;
import org.opencastproject.playlists.serialization.JaxbPlaylist;
import org.opencastproject.playlists.serialization.JaxbPlaylistEntry;
import org.opencastproject.security.api.AccessControlEntry;
import org.opencastproject.security.api.AccessControlList;
import org.opencastproject.security.api.AuthorizationService;
import org.opencastproject.security.api.Organization;
import org.opencastproject.security.api.Permissions;
import org.opencastproject.security.api.SecurityService;
import org.opencastproject.security.api.UnauthorizedException;
import org.opencastproject.security.api.User;
import org.opencastproject.util.NotFoundException;
import org.opencastproject.util.requests.SortCriterion;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(property={"service.description=Playlist Service", "service.pid=org.opencastproject.playlists.PlaylistService"}, immediate=true, service={PlaylistService.class})
public class PlaylistService {
    private static final Logger logger = LoggerFactory.getLogger(PlaylistService.class);
    protected PlaylistDatabaseService persistence;
    protected SecurityService securityService;
    protected AuthorizationService authorizationService = null;
    private ElasticsearchIndex elasticsearchIndex;

    @Reference(name="playlist-persistence")
    public void setPersistence(PlaylistDatabaseService persistence) {
        this.persistence = persistence;
    }

    @Reference(name="security-service")
    public void setSecurityService(SecurityService securityService) {
        this.securityService = securityService;
    }

    @Reference
    public void setAuthorizationService(AuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    @Reference
    void setElasticsearchIndex(ElasticsearchIndex elasticsearchIndex) {
        this.elasticsearchIndex = elasticsearchIndex;
    }

    @Activate
    public void activate(ComponentContext cc) throws Exception {
        logger.info("Activating Playlist Service");
    }

    public Playlist getPlaylistById(String id) throws NotFoundException, IllegalStateException, UnauthorizedException {
        try {
            Playlist playlist = this.persistence.getPlaylist(id);
            if (!this.checkPermission(playlist, Permissions.Action.READ)) {
                throw new UnauthorizedException("User does not have read permissions");
            }
            return playlist;
        }
        catch (PlaylistDatabaseException e) {
            throw new IllegalStateException("Could not get playlist from database with id ");
        }
    }

    public List<Playlist> getPlaylists(int limit, int offset) throws IllegalStateException {
        return this.getPlaylists(limit, offset, new SortCriterion("", SortCriterion.Order.None));
    }

    public List<Playlist> getPlaylists(int limit, int offset, SortCriterion sortCriterion) throws IllegalStateException {
        try {
            List<Playlist> playlists = this.persistence.getPlaylists(limit, offset, sortCriterion);
            playlists.removeIf(playlist -> !this.checkPermission((Playlist)playlist, Permissions.Action.READ));
            return playlists;
        }
        catch (PlaylistDatabaseException e) {
            throw new IllegalStateException("Could not get playlist from database with id ");
        }
    }

    public List<Playlist> getAllForAdministrativeRead(Date from, Date to, int limit) throws IllegalStateException, UnauthorizedException {
        User user = this.securityService.getUser();
        String orgAdminRole = this.securityService.getOrganization().getAdminRole();
        if (!user.hasRole("ROLE_ADMIN") && !user.hasRole(orgAdminRole)) {
            throw new UnauthorizedException("Only (org-)admins can call this method");
        }
        try {
            return this.persistence.getAllForAdministrativeRead(from, to, limit);
        }
        catch (PlaylistDatabaseException e) {
            throw new IllegalStateException("Could not get playlist from database", e);
        }
    }

    public Playlist update(Playlist playlist) throws IllegalStateException, UnauthorizedException, IllegalArgumentException {
        try {
            Playlist existingPlaylist = this.persistence.getPlaylist(playlist.getId());
            if (!this.checkPermission(existingPlaylist, Permissions.Action.WRITE)) {
                throw new UnauthorizedException("User does not have write permissions");
            }
            for (PlaylistEntry playlistEntry : playlist.getEntries()) {
                if (!existingPlaylist.getEntries().stream().noneMatch(e -> playlistEntry.getId() == e.getId()) || playlistEntry.getId() == 0L) continue;
                throw new IllegalArgumentException("When updating a playlist, entries should either have the id of an existing entry, or no id at all.");
            }
            for (PlaylistAccessControlEntry playlistAccessControlEntry : playlist.getAccessControlEntries()) {
                if (!existingPlaylist.getAccessControlEntries().stream().noneMatch(e -> playlistAccessControlEntry.getId() == e.getId()) || playlistAccessControlEntry.getId() == 0L) continue;
                throw new IllegalArgumentException("When updating a playlist, ACL entries should either have the id of an existing entry, or no id at all.");
            }
        }
        catch (NotFoundException e2) {
            for (PlaylistEntry playlistEntry : playlist.getEntries()) {
                if (playlistEntry.getId() == 0L) continue;
                throw new IllegalArgumentException("Entries for new playlists should not have identifiers set");
            }
            for (PlaylistAccessControlEntry playlistAccessControlEntry : playlist.getAccessControlEntries()) {
                if (playlistAccessControlEntry.getId() == 0L) continue;
                throw new IllegalArgumentException("ACL Entries for new playlists should not have identifiers set");
            }
        }
        catch (PlaylistDatabaseException e3) {
            throw new IllegalStateException("Could not get playlist from database with id ");
        }
        if (playlist.getOrganization() == null) {
            playlist.setOrganization(this.securityService.getOrganization().getId());
        }
        for (PlaylistEntry playlistEntry : playlist.getEntries()) {
            playlistEntry.setPlaylist(playlist);
        }
        for (PlaylistAccessControlEntry playlistAccessControlEntry : playlist.getAccessControlEntries()) {
            playlistAccessControlEntry.setPlaylist(playlist);
        }
        try {
            playlist = this.persistence.updatePlaylist(playlist, this.securityService.getOrganization().getId());
            return playlist;
        }
        catch (PlaylistDatabaseException e4) {
            throw new IllegalStateException("Could not update playlist from database with id ");
        }
    }

    public Playlist updateWithJson(String id, String json) throws JsonProcessingException, UnauthorizedException {
        try {
            Playlist existingPlaylist = this.persistence.getPlaylist(id);
            if (!this.checkPermission(existingPlaylist, Permissions.Action.WRITE)) {
                throw new UnauthorizedException("User does not have write permissions");
            }
            JaxbAnnotationModule module = new JaxbAnnotationModule();
            ObjectMapper objectMapper = new ObjectMapper();
            objectMapper.registerModule((Module)module);
            objectMapper.enable(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY);
            ObjectReader updater = objectMapper.readerForUpdating((Object)new JaxbPlaylist(existingPlaylist));
            JaxbPlaylist merged = (JaxbPlaylist)updater.readValue(json);
            return this.update(merged.toPlaylist());
        }
        catch (PlaylistDatabaseException | NotFoundException e) {
            throw new IllegalStateException("Could not get playlist from database with id " + id);
        }
    }

    public Playlist remove(String playlistId) throws NotFoundException, IllegalStateException, UnauthorizedException {
        try {
            Playlist playlist = this.persistence.getPlaylist(playlistId);
            if (!this.checkPermission(playlist, Permissions.Action.WRITE)) {
                throw new UnauthorizedException("User does not have write permissions");
            }
            playlist = this.persistence.deletePlaylist(playlist, this.securityService.getOrganization().getId());
            return playlist;
        }
        catch (PlaylistDatabaseException e) {
            throw new IllegalStateException("Could not delete playlist from database with id " + playlistId);
        }
    }

    public Playlist updateEntries(String playlistId, List<PlaylistEntry> playlistEntries) throws NotFoundException, IllegalStateException, UnauthorizedException {
        Playlist playlist;
        try {
            playlist = this.persistence.getPlaylist(playlistId);
        }
        catch (PlaylistDatabaseException e) {
            throw new IllegalStateException(e);
        }
        if (!this.checkPermission(playlist, Permissions.Action.WRITE)) {
            throw new UnauthorizedException("User does not have write permissions");
        }
        playlist.setEntries(playlistEntries);
        try {
            playlist = this.persistence.updatePlaylist(playlist, this.securityService.getOrganization().getId());
            return playlist;
        }
        catch (PlaylistDatabaseException e) {
            throw new IllegalStateException("Could not delete playlist from database with id " + playlistId);
        }
    }

    public Playlist addEntry(String playlistId, String contentId, PlaylistEntryType type) throws NotFoundException, IllegalStateException, UnauthorizedException {
        Playlist playlist;
        try {
            playlist = this.persistence.getPlaylist(playlistId);
        }
        catch (PlaylistDatabaseException e) {
            throw new IllegalStateException(e);
        }
        if (!this.checkPermission(playlist, Permissions.Action.WRITE)) {
            throw new UnauthorizedException("User does not have write permissions");
        }
        PlaylistEntry playlistEntry = new PlaylistEntry();
        playlistEntry.setContentId(contentId);
        playlistEntry.setType(type);
        playlist.addEntry(playlistEntry);
        try {
            playlist = this.persistence.updatePlaylist(playlist, this.securityService.getOrganization().getId());
            return playlist;
        }
        catch (PlaylistDatabaseException e) {
            throw new IllegalStateException("Could not delete playlist from database with id " + playlistId);
        }
    }

    public Playlist removeEntry(String playlistId, long entryId) throws NotFoundException, IllegalStateException, UnauthorizedException {
        Playlist playlist;
        try {
            playlist = this.persistence.getPlaylist(playlistId);
            if (!this.checkPermission(playlist, Permissions.Action.WRITE)) {
                throw new UnauthorizedException("User does not have write permissions");
            }
        }
        catch (PlaylistDatabaseException e2) {
            throw new IllegalStateException(e2);
        }
        playlist.removeEntry(playlist.getEntries().stream().filter(e -> e.getId() == entryId).findFirst().get());
        try {
            playlist = this.persistence.updatePlaylist(playlist, this.securityService.getOrganization().getId());
            return playlist;
        }
        catch (PlaylistDatabaseException e3) {
            throw new IllegalStateException("Could not delete playlist from database with id " + playlistId);
        }
    }

    public JaxbPlaylist enrich(Playlist playlist) {
        JaxbPlaylist jaxbPlaylist = new JaxbPlaylist(playlist);
        String org = this.securityService.getOrganization().getId();
        User user = this.securityService.getUser();
        List<JaxbPlaylistEntry> jaxbPlaylistEntries = jaxbPlaylist.getEntries();
        for (JaxbPlaylistEntry entry : jaxbPlaylistEntries) {
            String contentId = entry.getContentId();
            if (contentId == null || contentId.isEmpty()) {
                entry.setType(PlaylistEntryType.INACCESSIBLE);
                logger.warn("Entry {} has no content, marking as inaccessible", (Object)entry.getId());
                continue;
            }
            try {
                if (entry.getType() != PlaylistEntryType.EVENT) continue;
                SearchResult result = this.elasticsearchIndex.getByQuery(new EventSearchQuery(org, user).withIdentifier(contentId));
                if (result.getPageSize() != 0L) {
                    Event event = (Event)result.getItems()[0].getSource();
                    entry.setPublications(event.getPublications());
                    continue;
                }
                entry.setType(PlaylistEntryType.INACCESSIBLE);
            }
            catch (SearchIndexException e) {
                throw new RuntimeException(e);
            }
        }
        jaxbPlaylist.setEntries(jaxbPlaylistEntries);
        return jaxbPlaylist;
    }

    private boolean checkPermission(Playlist playlist, Permissions.Action action) {
        User currentUser = this.securityService.getUser();
        Organization currentOrg = this.securityService.getOrganization();
        String currentOrgAdminRole = currentOrg.getAdminRole();
        String currentOrgId = currentOrg.getId();
        return currentUser.hasRole("ROLE_ADMIN") || currentUser.hasRole(currentOrgAdminRole) && currentOrgId.equals(playlist.getOrganization()) || this.authorizationService.hasPermission(this.getAccessControlList(playlist), action.toString());
    }

    private AccessControlList getAccessControlList(Playlist playlist) {
        ArrayList<AccessControlEntry> accessControlEntries = new ArrayList<AccessControlEntry>();
        for (PlaylistAccessControlEntry entry : playlist.getAccessControlEntries()) {
            accessControlEntries.add(entry.toAccessControlEntry());
        }
        return new AccessControlList(accessControlEntries);
    }
}

