/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.storemigration;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;
import org.neo4j.common.ProgressReporter;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.exceptions.KernelException;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.pagecache.tracing.PageCacheTracer;
import org.neo4j.io.pagecache.tracing.cursor.PageCursorTracer;
import org.neo4j.kernel.impl.storemigration.LogsUpgrader;
import org.neo4j.kernel.impl.storemigration.MigrationStatus;
import org.neo4j.kernel.internal.Version;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;
import org.neo4j.storageengine.api.IndexCapabilities;
import org.neo4j.storageengine.api.StoreVersion;
import org.neo4j.storageengine.api.StoreVersionCheck;
import org.neo4j.storageengine.api.format.Capability;
import org.neo4j.storageengine.migration.MigrationProgressMonitor;
import org.neo4j.storageengine.migration.StoreMigrationParticipant;
import org.neo4j.storageengine.migration.UpgradeNotAllowedException;
import org.neo4j.util.Preconditions;

public class StoreUpgrader {
    private static final String STORE_UPGRADE_TAG = "storeUpgrade";
    public static final String MIGRATION_DIRECTORY = "upgrade";
    public static final String MIGRATION_LEFT_OVERS_DIRECTORY = "upgrade_backup";
    private static final String MIGRATION_STATUS_FILE = "_status";
    private static final Pattern MIGRATION_LEFTOVERS_PATTERN = Pattern.compile("upgrade_backup(_\\d*)?");
    private final StoreVersionCheck storeVersionCheck;
    private final MigrationProgressMonitor progressMonitor;
    private final LinkedHashMap<String, StoreMigrationParticipant> participants = new LinkedHashMap();
    private final Config config;
    private final FileSystemAbstraction fileSystem;
    private final Log log;
    private final LogsUpgrader logsUpgrader;
    private final String configuredFormat;
    private final PageCacheTracer pageCacheTracer;

    public StoreUpgrader(StoreVersionCheck storeVersionCheck, MigrationProgressMonitor progressMonitor, Config config, FileSystemAbstraction fileSystem, LogProvider logProvider, LogsUpgrader logsUpgrader, PageCacheTracer pageCacheTracer) {
        this.storeVersionCheck = storeVersionCheck;
        this.progressMonitor = progressMonitor;
        this.fileSystem = fileSystem;
        this.config = config;
        this.logsUpgrader = logsUpgrader;
        this.log = logProvider.getLog(this.getClass());
        this.configuredFormat = storeVersionCheck.configuredVersion();
        this.pageCacheTracer = pageCacheTracer;
    }

    public void addParticipant(StoreMigrationParticipant participant) {
        assert (participant != null);
        if (!StoreMigrationParticipant.NOT_PARTICIPATING.equals(participant)) {
            String newParticipantName = participant.getName();
            Preconditions.checkState((!this.participants.containsKey(newParticipantName) ? 1 : 0) != 0, (String)"Migration participants should have unique names. Participant with name: `%s` is already registered.", (Object[])new Object[]{newParticipantName});
            this.participants.put(newParticipantName, participant);
        }
    }

    public void migrateIfNeeded(DatabaseLayout layout) {
        if (!layout.databaseDirectory().exists()) {
            return;
        }
        this.logsUpgrader.assertLogVersionIsCurrent(layout);
        if (layout.getDatabaseName().equals("system")) {
            return;
        }
        try (PageCursorTracer cursorTracer = this.pageCacheTracer.createPageCursorTracer(STORE_UPGRADE_TAG);){
            DatabaseLayout migrationStructure = DatabaseLayout.ofFlat((File)layout.file(MIGRATION_DIRECTORY));
            this.cleanupLegacyLeftOverDirsIn(layout.databaseDirectory());
            File migrationStateFile = migrationStructure.file(MIGRATION_STATUS_FILE);
            if (this.hasCurrentVersion(this.storeVersionCheck, cursorTracer) && !this.fileSystem.fileExists(migrationStateFile)) {
                return;
            }
            if (this.isUpgradeAllowed()) {
                this.migrate(layout, migrationStructure, migrationStateFile, cursorTracer);
            } else {
                Optional storeVersion = this.storeVersionCheck.storeVersion(cursorTracer);
                if (storeVersion.isPresent()) {
                    StoreVersion version = this.storeVersionCheck.versionInformation((String)storeVersion.get());
                    if (version.hasCapability((Capability)IndexCapabilities.LuceneCapability.LUCENE_5)) {
                        throw new UpgradeNotAllowedException("Upgrade is required to migrate store to new major version.");
                    }
                    String configuredVersion = this.storeVersionCheck.configuredVersion();
                    if (configuredVersion != null && !version.isCompatibleWith(this.storeVersionCheck.versionInformation(configuredVersion))) {
                        throw new UpgradeNotAllowedException();
                    }
                }
            }
        }
    }

    private boolean hasCurrentVersion(StoreVersionCheck storeVersionCheck, PageCursorTracer cursorTracer) {
        String configuredVersion = storeVersionCheck.configuredVersion();
        StoreVersionCheck.Result versionResult = storeVersionCheck.checkUpgrade(configuredVersion, cursorTracer);
        if (versionResult.outcome == StoreVersionCheck.Outcome.missingStoreFile) {
            return true;
        }
        return versionResult.outcome.isSuccessful() && versionResult.actualVersion.equals(configuredVersion);
    }

    private void migrate(DatabaseLayout dbDirectoryLayout, DatabaseLayout migrationLayout, File migrationStateFile, PageCursorTracer cursorTracer) {
        this.progressMonitor.started(this.participants.size());
        MigrationStatus migrationStatus = MigrationStatus.readMigrationStatus(this.fileSystem, migrationStateFile);
        String versionToMigrateFrom = null;
        if (MigrationStatus.migrating.isNeededFor(migrationStatus)) {
            StoreVersionCheck.Result upgradeCheck = this.storeVersionCheck.checkUpgrade(this.storeVersionCheck.configuredVersion(), cursorTracer);
            versionToMigrateFrom = this.getVersionFromResult(upgradeCheck);
            this.logsUpgrader.assertCleanlyShutDown(dbDirectoryLayout);
            this.cleanMigrationDirectory(migrationLayout.databaseDirectory());
            MigrationStatus.migrating.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.migrateToIsolatedDirectory(dbDirectoryLayout, migrationLayout, versionToMigrateFrom);
            MigrationStatus.moving.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        if (MigrationStatus.moving.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = MigrationStatus.moving.maybeReadInfo(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            String versionToMigrateTo = this.storeVersionCheck.configuredVersion();
            StoreUpgrader.moveMigratedFilesToStoreDirectory(this.participants.values(), migrationLayout, dbDirectoryLayout, versionToMigrateFrom, versionToMigrateTo);
        }
        this.progressMonitor.startTransactionLogsMigration();
        this.logsUpgrader.upgrade(dbDirectoryLayout);
        this.progressMonitor.completeTransactionLogsMigration();
        StoreUpgrader.cleanup(this.participants.values(), migrationLayout);
        this.progressMonitor.completed();
    }

    private String getVersionFromResult(StoreVersionCheck.Result result) {
        switch (result.outcome) {
            case ok: {
                return result.actualVersion;
            }
            case missingStoreFile: {
                throw new UpgradeMissingStoreFilesException(result.storeFilename);
            }
            case storeVersionNotFound: {
                throw new UpgradingStoreVersionNotFoundException(result.storeFilename);
            }
            case attemptedStoreDowngrade: {
                throw new AttemptedDowngradeException();
            }
            case unexpectedStoreVersion: {
                throw new UnexpectedUpgradingStoreVersionException(result.actualVersion, this.configuredFormat);
            }
            case storeNotCleanlyShutDown: {
                throw new DatabaseNotCleanlyShutDownException();
            }
            case unexpectedUpgradingVersion: {
                throw new UnexpectedUpgradingStoreFormatException();
            }
        }
        throw new IllegalArgumentException("Unexpected outcome: " + result.outcome.name());
    }

    List<StoreMigrationParticipant> getParticipants() {
        return List.copyOf(this.participants.values());
    }

    private boolean isUpgradeAllowed() {
        return (Boolean)this.config.get(GraphDatabaseSettings.allow_upgrade);
    }

    private void cleanupLegacyLeftOverDirsIn(File databaseDirectory) {
        File[] leftOverDirs = databaseDirectory.listFiles((file, name) -> file.isDirectory() && MIGRATION_LEFTOVERS_PATTERN.matcher(name).matches());
        if (leftOverDirs != null) {
            for (File leftOverDir : leftOverDirs) {
                this.deleteSilently(leftOverDir);
            }
        }
    }

    private static void cleanup(Iterable<StoreMigrationParticipant> participants, DatabaseLayout migrationStructure) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.cleanup(migrationStructure);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Failure cleaning up after migration", e);
        }
    }

    private static void moveMigratedFilesToStoreDirectory(Iterable<StoreMigrationParticipant> participants, DatabaseLayout migrationLayout, DatabaseLayout directoryLayout, String versionToMigrateFrom, String versionToMigrateTo) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.moveMigratedFiles(migrationLayout, directoryLayout, versionToMigrateFrom, versionToMigrateTo);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Unable to move migrated files into place", e);
        }
    }

    private void migrateToIsolatedDirectory(DatabaseLayout directoryLayout, DatabaseLayout migrationLayout, String versionToMigrateFrom) {
        try {
            for (Map.Entry<String, StoreMigrationParticipant> participantEntry : this.participants.entrySet()) {
                ProgressReporter progressReporter = this.progressMonitor.startSection(participantEntry.getKey());
                String versionToMigrateTo = this.storeVersionCheck.configuredVersion();
                participantEntry.getValue().migrate(directoryLayout, migrationLayout, progressReporter, versionToMigrateFrom, versionToMigrateTo);
                progressReporter.completed();
            }
        }
        catch (IOException | UncheckedIOException | KernelException e) {
            throw new UnableToUpgradeException("Failure doing migration", e);
        }
    }

    private void cleanMigrationDirectory(File migrationDirectory) {
        try {
            if (this.fileSystem.fileExists(migrationDirectory)) {
                this.fileSystem.deleteRecursively(migrationDirectory);
            }
        }
        catch (IOException | UncheckedIOException e) {
            throw new UnableToUpgradeException("Failure deleting upgrade directory " + migrationDirectory, e);
        }
        this.fileSystem.mkdir(migrationDirectory);
    }

    private void deleteSilently(File dir) {
        try {
            this.fileSystem.deleteRecursively(dir);
        }
        catch (IOException e) {
            this.log.error("Unable to delete directory: " + dir, (Throwable)e);
        }
    }

    public static class DatabaseNotCleanlyShutDownException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "The database is not cleanly shutdown. The database needs recovery, in order to recover the database, please run the old version of the database on this store.";

        DatabaseNotCleanlyShutDownException() {
            super(MESSAGE);
        }
    }

    public static class UnexpectedUpgradingStoreFormatException
    extends UnableToUpgradeException {
        static final String MESSAGE = "This is an enterprise-only store. Please configure '%s' to open.";

        UnexpectedUpgradingStoreFormatException() {
            super(String.format(MESSAGE, GraphDatabaseSettings.record_format.name()));
        }
    }

    public static class AttemptedDowngradeException
    extends UnableToUpgradeException {
        static final String MESSAGE = "Downgrading stores are not supported.";

        AttemptedDowngradeException() {
            super(MESSAGE);
        }
    }

    public static class UnexpectedUpgradingStoreVersionException
    extends UnableToUpgradeException {
        static final String MESSAGE = "Not possible to upgrade a store with version '%s' to current store version `%s` (Neo4j %s).";

        UnexpectedUpgradingStoreVersionException(String fileVersion, String currentVersion) {
            super(String.format(MESSAGE, fileVersion, currentVersion, Version.getNeo4jVersion()));
        }
    }

    static class UpgradingStoreVersionNotFoundException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "'%s' does not contain a store version, please ensure that the original database was shut down in a clean state.";

        UpgradingStoreVersionNotFoundException(String filenameWithoutStoreVersion) {
            super(String.format(MESSAGE, filenameWithoutStoreVersion));
        }
    }

    static class UpgradeMissingStoreFilesException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "Missing required store file '%s'.";

        UpgradeMissingStoreFilesException(String filenameExpectedToExist) {
            super(String.format(MESSAGE, filenameExpectedToExist));
        }
    }

    public static class UnableToUpgradeException
    extends RuntimeException {
        public UnableToUpgradeException(String message, Throwable cause) {
            super(message, cause);
        }

        UnableToUpgradeException(String message) {
            super(message);
        }
    }

    static class TransactionLogsRelocationException
    extends RuntimeException {
        TransactionLogsRelocationException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

