/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import org.neo4j.function.Function;
import org.neo4j.helpers.Clock;
import org.neo4j.helpers.Format;
import org.neo4j.helpers.Listeners;
import org.neo4j.helpers.collection.Iterables;

public class AvailabilityGuard {
    private Iterable<AvailabilityListener> listeners = Listeners.newListeners();
    private final AtomicInteger available;
    private final List<AvailabilityRequirement> blockingComponents = new CopyOnWriteArrayList<AvailabilityRequirement>();
    private final Clock clock;
    public static final Function<AvailabilityRequirement, String> DESCRIPTION = new Function<AvailabilityRequirement, String>(){

        public String apply(AvailabilityRequirement availabilityRequirement) {
            return availabilityRequirement.description();
        }
    };

    public static AvailabilityRequirement availabilityRequirement(final String descriptionWhenBlocking) {
        return new AvailabilityRequirement(){

            @Override
            public String description() {
                return descriptionWhenBlocking;
            }
        };
    }

    public AvailabilityGuard(Clock clock) {
        this(clock, 0);
    }

    public AvailabilityGuard(Clock clock, int conditionCount) {
        this.clock = clock;
        this.available = new AtomicInteger(conditionCount);
    }

    public void deny(AvailabilityRequirement requirementNotMet) {
        int val;
        do {
            if ((val = this.available.get()) != -1) continue;
            return;
        } while (!this.available.compareAndSet(val, val + 1));
        this.blockingComponents.add(requirementNotMet);
        if (val == 0) {
            Listeners.notifyListeners(this.listeners, new Listeners.Notification<AvailabilityListener>(){

                @Override
                public void notify(AvailabilityListener listener) {
                    listener.unavailable();
                }
            });
        }
    }

    public void grant(AvailabilityRequirement requirementNowMet) {
        int val;
        do {
            if ((val = this.available.get()) != -1) continue;
            return;
        } while (!this.available.compareAndSet(val, val - 1));
        assert (this.available.get() >= 0);
        this.blockingComponents.remove(requirementNowMet);
        if (val == 1) {
            Listeners.notifyListeners(this.listeners, new Listeners.Notification<AvailabilityListener>(){

                @Override
                public void notify(AvailabilityListener listener) {
                    listener.available();
                }
            });
        }
    }

    public void shutdown() {
        int val = this.available.getAndSet(-1);
        if (val == 0) {
            Listeners.notifyListeners(this.listeners, new Listeners.Notification<AvailabilityListener>(){

                @Override
                public void notify(AvailabilityListener listener) {
                    listener.unavailable();
                }
            });
        }
    }

    public boolean isAvailable(long millis) {
        return this.availability(millis).available;
    }

    private Availability availability(long millis) {
        int val = this.available.get();
        if (val == 0) {
            return Availability.AVAILABLE;
        }
        if (val == -1) {
            return Availability.UNAVAILABLE;
        }
        long start = this.clock.currentTimeMillis();
        while (this.clock.currentTimeMillis() < start + millis) {
            val = this.available.get();
            if (val == 0) {
                return Availability.AVAILABLE;
            }
            if (val == -1) {
                return Availability.UNAVAILABLE;
            }
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                break;
            }
            Thread.yield();
        }
        return Availability.TEMPORARILY_UNAVAILABLE;
    }

    public <EXCEPTION extends Throwable> void checkAvailability(long millis, Class<EXCEPTION> cls) throws EXCEPTION {
        Availability availability = this.availability(millis);
        if (!availability.available) {
            Throwable exception;
            try {
                String description = availability.temporarily ? "Timeout waiting for database to become available and allow new transactions. Waited " + Format.duration(millis) + ". " + this.describeWhoIsBlocking() : "Database not available because it's shutting down";
                exception = (Throwable)cls.getConstructor(String.class).newInstance(description);
            }
            catch (NoSuchMethodException e) {
                throw new Error("Bad exception class given to this method, it doesn't have a (String) constructor", e);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
            throw exception;
        }
    }

    public void addListener(AvailabilityListener listener) {
        this.listeners = Listeners.addListener(listener, this.listeners);
    }

    public void removeListener(AvailabilityListener listener) {
        this.listeners = Listeners.removeListener(listener, this.listeners);
    }

    public String describeWhoIsBlocking() {
        if (this.blockingComponents.size() > 0 || this.available.get() > 0) {
            String causes = Iterables.join(", ", Iterables.map(DESCRIPTION, this.blockingComponents));
            return this.available.get() + " reasons for blocking: " + causes + ".";
        }
        return "No blocking components";
    }

    private static enum Availability {
        AVAILABLE(true, true),
        TEMPORARILY_UNAVAILABLE(false, true),
        UNAVAILABLE(false, false);

        private final boolean available;
        private final boolean temporarily;

        private Availability(boolean available, boolean temporarily) {
            this.available = available;
            this.temporarily = temporarily;
        }
    }

    public static interface AvailabilityRequirement {
        public String description();
    }

    public static interface AvailabilityListener {
        public void available();

        public void unavailable();
    }
}

