/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;
import org.neo4j.graphdb.TransactionFailureException;
import org.neo4j.helpers.Listeners;
import org.neo4j.helpers.Predicate;
import org.neo4j.helpers.UTF8;
import org.neo4j.helpers.collection.MapUtil;
import org.neo4j.kernel.impl.nioneo.xa.NeoStoreXaDataSource;
import org.neo4j.kernel.impl.transaction.DataSourceRegistrationListener;
import org.neo4j.kernel.impl.transaction.TxLog;
import org.neo4j.kernel.impl.transaction.XidImpl;
import org.neo4j.kernel.impl.transaction.xaframework.XaDataSource;
import org.neo4j.kernel.impl.transaction.xaframework.XaResource;
import org.neo4j.kernel.impl.util.StringLogger;
import org.neo4j.kernel.lifecycle.LifeSupport;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.kernel.lifecycle.LifecycleStatus;

public class XaDataSourceManager
implements Lifecycle {
    private final Map<String, XaDataSource> dataSources = new HashMap<String, XaDataSource>();
    private final Map<String, XaDataSource> branchIdMapping = new HashMap<String, XaDataSource>();
    private final Map<String, byte[]> sourceIdMapping = new HashMap<String, byte[]>();
    private Iterable<DataSourceRegistrationListener> dsRegistrationListeners = Listeners.newListeners();
    private LifeSupport life = new LifeSupport();
    private final StringLogger msgLog;
    private boolean isShutdown = false;

    public XaDataSourceManager(StringLogger msgLog) {
        this.msgLog = msgLog;
    }

    public static DataSourceRegistrationListener filterListener(final DataSourceRegistrationListener listener, final Predicate<XaDataSource> filter) {
        return new DataSourceRegistrationListener(){

            @Override
            public void registeredDataSource(XaDataSource ds) {
                if (filter.accept(ds)) {
                    listener.registeredDataSource(ds);
                }
            }

            @Override
            public void unregisteredDataSource(XaDataSource ds) {
                if (filter.accept(ds)) {
                    listener.unregisteredDataSource(ds);
                }
            }
        };
    }

    public static DataSourceRegistrationListener neoStoreListener(DataSourceRegistrationListener listener) {
        return XaDataSourceManager.filterListener(listener, new Predicate<XaDataSource>(){

            @Override
            public boolean accept(XaDataSource item) {
                return item.getName().equals("nioneodb");
            }
        });
    }

    public void addDataSourceRegistrationListener(DataSourceRegistrationListener listener) {
        if (this.life.getStatus().equals((Object)LifecycleStatus.STARTED)) {
            try {
                for (XaDataSource ds : this.dataSources.values()) {
                    listener.registeredDataSource(ds);
                }
            }
            catch (Throwable t) {
                this.msgLog.logMessage("Failed when notifying registering listener", t);
            }
        }
        this.dsRegistrationListeners = Listeners.addListener(listener, this.dsRegistrationListeners);
    }

    public void removeDataSourceRegistrationListener(DataSourceRegistrationListener dataSourceRegistrationListener) {
        this.dsRegistrationListeners = Listeners.removeListener(dataSourceRegistrationListener, this.dsRegistrationListeners);
    }

    @Override
    public void init() throws Throwable {
        if (this.dsRegistrationListeners == null) {
            this.dsRegistrationListeners = Listeners.newListeners();
        }
    }

    @Override
    public void start() throws Throwable {
        this.life = new LifeSupport();
        for (XaDataSource ds : this.dataSources.values()) {
            this.life.add(ds);
        }
        this.life.start();
        for (DataSourceRegistrationListener listener : this.dsRegistrationListeners) {
            try {
                for (XaDataSource ds : this.dataSources.values()) {
                    listener.registeredDataSource(ds);
                }
            }
            catch (Throwable t) {
                this.msgLog.logMessage("Failed when notifying registering listener", t);
            }
        }
    }

    @Override
    public void stop() throws Throwable {
        this.life.stop();
    }

    @Override
    public void shutdown() throws Throwable {
        this.dsRegistrationListeners = null;
        this.life.shutdown();
        this.dataSources.clear();
        this.branchIdMapping.clear();
        this.sourceIdMapping.clear();
        this.isShutdown = true;
    }

    public XaDataSource getXaDataSource(String name) {
        if (this.isShutdown) {
            throw new IllegalStateException("XaDataSourceManager has been shut down.");
        }
        return this.dataSources.get(name);
    }

    @Deprecated
    public NeoStoreXaDataSource getNeoStoreDataSource() {
        return (NeoStoreXaDataSource)this.getXaDataSource("nioneodb");
    }

    public synchronized void registerDataSource(final XaDataSource dataSource) {
        this.dataSources.put(dataSource.getName(), dataSource);
        this.branchIdMapping.put(UTF8.decode(dataSource.getBranchId()), dataSource);
        this.sourceIdMapping.put(dataSource.getName(), dataSource.getBranchId());
        this.life.add(dataSource);
        if (this.life.getStatus().equals((Object)LifecycleStatus.STARTED)) {
            Listeners.notifyListeners(this.dsRegistrationListeners, new Listeners.Notification<DataSourceRegistrationListener>(){

                @Override
                public void notify(DataSourceRegistrationListener listener) {
                    listener.registeredDataSource(dataSource);
                }
            });
        }
    }

    public synchronized void unregisterDataSource(String name) {
        final XaDataSource dataSource = this.dataSources.get(name);
        if (dataSource == null) {
            return;
        }
        this.dataSources.remove(name);
        this.branchIdMapping.remove(UTF8.decode(dataSource.getBranchId()));
        this.sourceIdMapping.remove(name);
        Listeners.notifyListeners(this.dsRegistrationListeners, new Listeners.Notification<DataSourceRegistrationListener>(){

            @Override
            public void notify(DataSourceRegistrationListener listener) {
                listener.unregisteredDataSource(dataSource);
            }
        });
        this.life.remove(dataSource);
    }

    synchronized byte[] getBranchId(XAResource xaResource) {
        byte[] branchId;
        if (xaResource instanceof XaResource && (branchId = ((XaResource)xaResource).getBranchId()) != null) {
            return branchId;
        }
        for (Map.Entry<String, XaDataSource> entry : this.dataSources.entrySet()) {
            XaDataSource dataSource = entry.getValue();
            XAResource resource = dataSource.getXaConnection().getXaResource();
            try {
                if (!resource.isSameRM(xaResource)) continue;
                String name = entry.getKey();
                return this.sourceIdMapping.get(name);
            }
            catch (XAException e) {
                throw new TransactionFailureException("Unable to check is same resource", e);
            }
        }
        throw new TransactionFailureException("Unable to find mapping for XAResource[" + xaResource + "]");
    }

    private XaDataSource getDataSource(byte[] branchId) {
        XaDataSource dataSource = this.branchIdMapping.get(UTF8.decode(branchId));
        if (dataSource == null) {
            throw new TransactionFailureException("No mapping found for branchId[0x" + UTF8.decode(branchId) + "]");
        }
        return dataSource;
    }

    public Collection<XaDataSource> getAllRegisteredDataSources() {
        return this.dataSources.values();
    }

    public void recover(Iterator<List<TxLog.Record>> knownDanglingRecordList) {
        ArrayList<NonCompletedTransaction> commitList = new ArrayList<NonCompletedTransaction>();
        LinkedList<Xid> rollbackList = new LinkedList<Xid>();
        HashMap<Resource, XaDataSource> resourceMap = new HashMap<Resource, XaDataSource>();
        this.buildRecoveryInfo(commitList, rollbackList, resourceMap, knownDanglingRecordList);
        LinkedList<Xid> recoveredXidsList = new LinkedList<Xid>();
        try {
            Resource resource;
            Xid[] xids;
            for (XaDataSource xaDataSource : this.dataSources.values()) {
                XAResource xaRes = xaDataSource.getXaConnection().getXaResource();
                for (Xid xid : xids = xaRes.recover(0)) {
                    if (XidImpl.isThisTm(xid.getGlobalTransactionId())) {
                        if (rollbackList.contains(xid)) {
                            this.msgLog.logMessage("TM: Found pre commit " + xid + " rolling back ... ", true);
                            rollbackList.remove(xid);
                            xaRes.rollback(xid);
                            continue;
                        }
                        resource = new Resource(xid.getBranchQualifier());
                        if (!resourceMap.containsKey(resource)) {
                            resourceMap.put(resource, xaDataSource);
                        }
                        recoveredXidsList.add(xid);
                        continue;
                    }
                    this.msgLog.warn("Unknown xid: " + xid);
                }
            }
            Collections.sort(commitList);
            for (NonCompletedTransaction nct : commitList) {
                int seq = nct.getSequenceNumber();
                xids = nct.getXids();
                this.msgLog.debug("Marked as commit tx-seq[" + seq + "] branch length: " + xids.length);
                for (Xid xid : xids) {
                    if (!recoveredXidsList.contains(xid)) {
                        this.msgLog.debug("Tx-seq[" + seq + "][" + xid + "] not found in recovered xid list, " + "assuming already committed");
                        continue;
                    }
                    recoveredXidsList.remove(xid);
                    resource = new Resource(xid.getBranchQualifier());
                    if (!resourceMap.containsKey(resource)) {
                        TransactionFailureException ex = new TransactionFailureException("Couldn't find XAResource for " + xid);
                        throw this.logAndReturn("TM: recovery error", ex);
                    }
                    this.msgLog.debug("TM: Committing tx " + xid);
                    ((XaDataSource)resourceMap.get(resource)).getXaConnection().getXaResource().commit(xid, false);
                }
            }
            for (Xid xid : recoveredXidsList) {
                Resource resource2 = new Resource(xid.getBranchQualifier());
                if (!resourceMap.containsKey(resource2)) {
                    TransactionFailureException ex = new TransactionFailureException("Couldn't find XAResource for " + xid);
                    throw this.logAndReturn("TM: recovery error", ex);
                }
                this.msgLog.debug("TM: no match found for " + xid + " removing");
                ((XaDataSource)resourceMap.get(resource2)).getXaConnection().getXaResource().rollback(xid);
            }
            if (rollbackList.size() > 0) {
                this.msgLog.debug("TxLog contained unresolved xids that needed rollback. They couldn't be matched to any of the XAResources recover list. Assuming " + rollbackList.size() + " transactions already rolled back.");
            }
            for (XaDataSource participant : MapUtil.reverse(resourceMap).keySet()) {
                participant.recoveryCompleted();
                participant.rotateLogicalLog();
            }
            for (XaDataSource ds : this.allOtherDataSources(resourceMap.values())) {
                ds.recoveryCompleted();
            }
        }
        catch (IOException | XAException e) {
            throw this.logAndReturn("TM: recovery failed", new TransactionFailureException("Recovery failed.", e));
        }
    }

    private Collection<XaDataSource> allOtherDataSources(Collection<XaDataSource> recoveredDataSources) {
        HashSet<XaDataSource> dataSources = new HashSet<XaDataSource>(this.dataSources.values());
        dataSources.removeAll(recoveredDataSources);
        return dataSources;
    }

    private void buildRecoveryInfo(List<NonCompletedTransaction> commitList, List<Xid> rollbackList, Map<Resource, XaDataSource> resourceMap, Iterator<List<TxLog.Record>> danglingRecordList) {
        while (danglingRecordList.hasNext()) {
            Iterator<TxLog.Record> dListItr = danglingRecordList.next().iterator();
            TxLog.Record startRecord = dListItr.next();
            if (startRecord.getType() != 1) {
                throw this.logAndReturn("TM error building recovery info", new TransactionFailureException("First record not a start record, type=" + startRecord.getType()));
            }
            HashSet<Resource> branchSet = new HashSet<Resource>();
            int markedCommit = -1;
            while (dListItr.hasNext()) {
                TxLog.Record record = dListItr.next();
                if (record.getType() == 2) {
                    if (markedCommit != -1) {
                        throw this.logAndReturn("TM error building recovery info", new TransactionFailureException("Already marked commit " + startRecord));
                    }
                    branchSet.add(new Resource(record.getBranchId()));
                    continue;
                }
                if (record.getType() == 3) {
                    if (markedCommit != -1) {
                        throw this.logAndReturn("TM error building recovery info", new TransactionFailureException("Already marked commit " + startRecord));
                    }
                    markedCommit = record.getSequenceNumber();
                    continue;
                }
                throw this.logAndReturn("TM error building recovery info", new TransactionFailureException("Illegal record type[" + record.getType() + "]"));
            }
            Iterator resourceItr = branchSet.iterator();
            LinkedList<Xid> xids = new LinkedList<Xid>();
            while (resourceItr.hasNext()) {
                Resource resource = (Resource)resourceItr.next();
                if (!resourceMap.containsKey(resource)) {
                    resourceMap.put(resource, this.getDataSource(resource.getResourceId()));
                }
                xids.add(new XidImpl(startRecord.getGlobalId(), resource.getResourceId()));
            }
            if (markedCommit != -1) {
                commitList.add(new NonCompletedTransaction(markedCommit, xids));
                continue;
            }
            rollbackList.addAll(xids);
        }
    }

    private <E extends Exception> E logAndReturn(String msg, E exception) {
        try {
            this.msgLog.logMessage(msg, exception, true);
            return exception;
        }
        catch (Throwable t) {
            return exception;
        }
    }

    public void rotateLogicalLogs() {
        for (XaDataSource dataSource : this.dataSources.values()) {
            try {
                dataSource.rotateLogicalLog();
            }
            catch (IOException e) {
                this.msgLog.logMessage("Couldn't rotate logical log for " + dataSource.getName(), e);
            }
        }
    }

    private static class Resource {
        private byte[] resourceId = null;
        private volatile int hashCode = 0;

        Resource(byte[] resourceId) {
            if (resourceId == null || resourceId.length == 0) {
                throw new IllegalArgumentException("Illegal resourceId");
            }
            this.resourceId = resourceId;
        }

        byte[] getResourceId() {
            return this.resourceId;
        }

        public boolean equals(Object o) {
            if (!(o instanceof Resource)) {
                return false;
            }
            byte[] otherResourceId = ((Resource)o).getResourceId();
            if (this.resourceId.length != otherResourceId.length) {
                return false;
            }
            for (int i = 0; i < this.resourceId.length; ++i) {
                if (this.resourceId[i] == otherResourceId[i]) continue;
                return false;
            }
            return true;
        }

        public int hashCode() {
            if (this.hashCode == 0) {
                int calcHash = 0;
                for (int i = 0; i < this.resourceId.length; ++i) {
                    calcHash += this.resourceId[i] << i * 8;
                }
                this.hashCode = 3217 * calcHash;
            }
            return this.hashCode;
        }
    }

    private static class NonCompletedTransaction
    implements Comparable<NonCompletedTransaction> {
        private int seqNr = -1;
        private List<Xid> xidList = null;

        NonCompletedTransaction(int seqNr, List<Xid> xidList) {
            this.seqNr = seqNr;
            this.xidList = xidList;
        }

        int getSequenceNumber() {
            return this.seqNr;
        }

        Xid[] getXids() {
            return this.xidList.toArray(new Xid[this.xidList.size()]);
        }

        public String toString() {
            return "NonCompletedTx[" + this.seqNr + "," + this.xidList + "]";
        }

        @Override
        public int compareTo(NonCompletedTransaction nct) {
            return this.getSequenceNumber() - nct.getSequenceNumber();
        }
    }
}

