/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.nioneo.store;

import java.io.File;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.neo4j.helpers.Pair;
import org.neo4j.kernel.impl.nioneo.store.BrickElement;
import org.neo4j.kernel.impl.nioneo.store.BrickElementFactory;
import org.neo4j.kernel.impl.nioneo.store.LockableWindow;
import org.neo4j.kernel.impl.nioneo.store.MappedMemException;
import org.neo4j.kernel.impl.nioneo.store.MappedPersistenceWindow;
import org.neo4j.kernel.impl.nioneo.store.OperationType;
import org.neo4j.kernel.impl.nioneo.store.PersistenceRow;
import org.neo4j.kernel.impl.nioneo.store.PersistenceWindow;
import org.neo4j.kernel.impl.nioneo.store.PlainPersistenceWindow;
import org.neo4j.kernel.impl.nioneo.store.StoreChannel;
import org.neo4j.kernel.impl.nioneo.store.UnderlyingStorageException;
import org.neo4j.kernel.impl.nioneo.store.WindowPoolStats;
import org.neo4j.kernel.impl.nioneo.store.windowpool.WindowPool;
import org.neo4j.kernel.impl.util.StringLogger;

public class PersistenceWindowPool
implements WindowPool {
    private static final int MAX_BRICK_COUNT = 100000;
    private final File storeName;
    private final int blockSize;
    private StoreChannel fileChannel;
    private final ConcurrentMap<Long, PersistenceRow> activeRowWindows;
    private long availableMem = 0L;
    private long memUsed = 0L;
    private int brickCount = 0;
    private int brickSize = 0;
    private BrickElement[] brickArray = new BrickElement[0];
    private int brickMiss = 0;
    static final int REFRESH_BRICK_COUNT = 50000;
    private final FileChannel.MapMode mapMode;
    private int hit = 0;
    private int miss = 0;
    private int switches = 0;
    private int ooe = 0;
    private boolean useMemoryMapped = true;
    private final boolean readOnly;
    private final AtomicBoolean refreshing = new AtomicBoolean();
    private final AtomicInteger avertedRefreshes = new AtomicInteger();
    private final AtomicLong refreshTime = new AtomicLong();
    private final AtomicInteger refreshes = new AtomicInteger();
    private final StringLogger log;
    private final BrickElementFactory brickFactory;
    private static final Comparator<BrickElement> BRICK_SORTER = new Comparator<BrickElement>(){

        @Override
        public int compare(BrickElement o1, BrickElement o2) {
            return o1.getHitCountSnapshot() - o2.getHitCountSnapshot();
        }
    };

    public PersistenceWindowPool(File storeName, int blockSize, StoreChannel fileChannel, long mappedMem, boolean useMemoryMappedBuffers, boolean readOnly, ConcurrentMap<Long, PersistenceRow> activeRowWindows, BrickElementFactory brickFactory, StringLogger log) {
        this.storeName = storeName;
        this.blockSize = blockSize;
        this.fileChannel = fileChannel;
        this.availableMem = mappedMem;
        this.useMemoryMapped = useMemoryMappedBuffers;
        this.readOnly = readOnly;
        this.activeRowWindows = activeRowWindows;
        this.brickFactory = brickFactory;
        this.mapMode = readOnly ? FileChannel.MapMode.READ_ONLY : FileChannel.MapMode.READ_WRITE;
        this.log = log;
        this.setupBricks();
        this.dumpStatus();
    }

    @Override
    public PersistenceWindow acquire(long position, OperationType operationType) {
        LockableWindow window = null;
        if (this.brickMiss >= 50000) {
            this.refreshBricks();
        }
        BrickElement theBrick = null;
        while (window == null) {
            if (this.brickSize > 0) {
                int brickIndex = this.positionToBrickIndex(position);
                if (brickIndex >= this.brickArray.length) {
                    this.expandBricks(brickIndex + 1);
                }
                theBrick = this.brickArray[brickIndex];
                window = theBrick.getAndMarkWindow();
            }
            if (window == null) {
                ++this.miss;
                ++this.brickMiss;
                PersistenceRow dpw = (PersistenceRow)this.activeRowWindows.get(position);
                if (dpw != null && dpw.markAsInUse()) {
                    window = dpw;
                    break;
                }
                dpw = new PersistenceRow(position, this.blockSize, this.fileChannel);
                PersistenceRow existing = this.activeRowWindows.putIfAbsent(position, dpw);
                if (existing == null) {
                    window = dpw;
                    continue;
                }
                dpw.close();
                if (theBrick == null) continue;
                theBrick.unLock();
                continue;
            }
            ++this.hit;
        }
        window.lock(operationType);
        return window;
    }

    private int positionToBrickIndex(long position) {
        return (int)(position * (long)this.blockSize / (long)this.brickSize);
    }

    private long brickIndexToPosition(int brickIndex) {
        return (long)brickIndex * (long)this.brickSize / (long)this.blockSize;
    }

    void dumpStatistics() {
        this.log.info(this.storeName + " hit=" + this.hit + " miss=" + this.miss + " switches=" + this.switches + " ooe=" + this.ooe);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void release(PersistenceWindow window) {
        block10: {
            try {
                if (!(window instanceof PersistenceRow)) break block10;
                PersistenceRow dpw = (PersistenceRow)window;
                try {
                    if (this.brickSize > 0 && dpw.isDirty()) {
                        this.applyChangesToWindowIfNecessary(dpw);
                    }
                    if (dpw.writeOutAndCloseIfFree(this.readOnly)) {
                        this.activeRowWindows.remove(dpw.position(), dpw);
                    } else {
                        dpw.reset();
                    }
                }
                finally {
                    if (this.brickSize > 0) {
                        int brickIndex = this.positionToBrickIndex(dpw.position());
                        BrickElement theBrick = this.brickArray[brickIndex];
                        theBrick.unLock();
                    }
                }
            }
            finally {
                ((LockableWindow)window).unLock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void applyChangesToWindowIfNecessary(PersistenceRow dpw) {
        LockableWindow existingBrickWindow;
        int brickIndex = this.positionToBrickIndex(dpw.position());
        LockableWindow lockableWindow = existingBrickWindow = brickIndex < this.brickArray.length ? this.brickArray[brickIndex].getWindow() : null;
        if (existingBrickWindow != null && !(existingBrickWindow instanceof MappedPersistenceWindow) && existingBrickWindow.markAsInUse()) {
            existingBrickWindow.lock(OperationType.WRITE);
            try {
                existingBrickWindow.acceptContents(dpw);
            }
            finally {
                existingBrickWindow.unLock();
            }
        }
    }

    @Override
    public synchronized void close() {
        this.flushAll();
        for (BrickElement element : this.brickArray) {
            if (element.getWindow() == null) continue;
            element.getWindow().close();
            element.setWindow(null);
        }
        this.fileChannel = null;
        this.activeRowWindows.clear();
        this.dumpStatistics();
    }

    @Override
    public void flushAll() {
        if (this.readOnly) {
            return;
        }
        for (BrickElement element : this.brickArray) {
            LockableWindow window = element.getWindow();
            if (window == null) continue;
            window.force();
        }
        try {
            this.fileChannel.force(false);
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Failed to flush file channel " + this.storeName, e);
        }
    }

    private void setupBricks() {
        long fileSize = -1L;
        try {
            fileSize = this.fileChannel.size();
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to get file size for " + this.storeName, e);
        }
        if (this.blockSize == 0) {
            return;
        }
        if (this.availableMem > 0L && this.availableMem < (long)this.blockSize * 10L) {
            this.logWarn("Unable to use " + this.availableMem + "b as memory mapped windows, need at least " + this.blockSize * 10 + "b (block size * 10)");
            this.logWarn("Memory mapped windows have been turned off");
            this.availableMem = 0L;
            this.brickCount = 0;
            this.brickSize = 0;
            return;
        }
        if (this.availableMem > 0L && fileSize > 0L) {
            double ratio = ((double)this.availableMem + 0.0) / (double)fileSize;
            if (ratio >= 1.0) {
                this.brickSize = (int)(this.availableMem / 1000L);
                if (this.brickSize < 0) {
                    this.brickSize = Integer.MAX_VALUE;
                }
                this.brickSize = this.brickSize / this.blockSize * this.blockSize;
                this.brickCount = (int)(fileSize / (long)this.brickSize);
            } else {
                this.brickCount = (int)(1000.0 / ratio);
                if (this.brickCount > 100000) {
                    this.brickCount = 100000;
                }
                if (fileSize / (long)this.brickCount > this.availableMem) {
                    this.logWarn("Unable to use " + this.availableMem / 1024L + "kb as memory mapped windows, need at least " + fileSize / (long)this.brickCount / 1024L + "kb");
                    this.logWarn("Memory mapped windows have been turned off");
                    this.availableMem = 0L;
                    this.brickCount = 0;
                    this.brickSize = 0;
                    return;
                }
                this.brickSize = (int)(fileSize / (long)this.brickCount);
                if (this.brickSize < 0) {
                    this.brickSize = Integer.MAX_VALUE;
                    this.brickSize = this.brickSize / this.blockSize * this.blockSize;
                    this.brickCount = (int)(fileSize / (long)this.brickSize);
                } else {
                    this.brickSize = this.brickSize / this.blockSize * this.blockSize;
                }
                assert (this.brickSize > this.blockSize);
            }
        } else if (this.availableMem > 0L) {
            this.brickSize = (int)(this.availableMem / 100L);
            if (this.brickSize < 0) {
                this.brickSize = Integer.MAX_VALUE;
            }
            this.brickSize = this.brickSize / this.blockSize * this.blockSize;
        }
        this.brickArray = new BrickElement[this.brickCount];
        for (int i = 0; i < this.brickCount; ++i) {
            BrickElement element;
            this.brickArray[i] = element = this.brickFactory.create(i);
        }
    }

    private void freeWindows(int nr) {
        int i;
        if (this.brickSize <= 0) {
            return;
        }
        ArrayList<BrickElement> mappedBricks = new ArrayList<BrickElement>();
        for (i = 0; i < this.brickCount; ++i) {
            BrickElement be = this.brickArray[i];
            if (be.getWindow() == null) continue;
            be.snapshotHitCount();
            mappedBricks.add(be);
        }
        Collections.sort(mappedBricks, BRICK_SORTER);
        for (i = 0; i < nr && i < mappedBricks.size(); ++i) {
            BrickElement mappedBrick = (BrickElement)mappedBricks.get(i);
            LockableWindow window = mappedBrick.getWindow();
            if (!window.writeOutAndCloseIfFree(this.readOnly)) continue;
            mappedBrick.setWindow(null);
            this.memUsed -= (long)this.brickSize;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refreshBricks() {
        if (this.brickMiss < 50000 || this.brickSize <= 0) {
            return;
        }
        if (this.refreshing.compareAndSet(false, true)) {
            try {
                long t = System.currentTimeMillis();
                this.doRefreshBricks();
                this.refreshes.incrementAndGet();
                this.refreshTime.addAndGet(System.currentTimeMillis() - t);
            }
            finally {
                this.refreshing.set(false);
            }
        } else {
            this.avertedRefreshes.incrementAndGet();
        }
    }

    private synchronized void doRefreshBricks() {
        this.brickMiss = 0;
        Pair<List<BrickElement>, List<BrickElement>> currentMappings = this.gatherMappedVersusUnmappedWindows();
        List<BrickElement> mappedBricks = currentMappings.first();
        List<BrickElement> unmappedBricks = currentMappings.other();
        int unmappedIndex = unmappedBricks.size() - 1;
        while (this.memUsed + (long)this.brickSize <= this.availableMem && unmappedIndex >= 0) {
            BrickElement unmappedBrick;
            if ((unmappedBrick = unmappedBricks.get(unmappedIndex--)).getHit() == 0) {
                return;
            }
            this.allocateNewWindow(unmappedBrick);
        }
        int mappedIndex = 0;
        while (unmappedIndex >= 0 && mappedIndex < mappedBricks.size()) {
            BrickElement mappedBrick = mappedBricks.get(mappedIndex++);
            BrickElement unmappedBrick = unmappedBricks.get(unmappedIndex--);
            if (mappedBrick.getHit() >= unmappedBrick.getHit()) break;
            LockableWindow window = mappedBrick.getWindow();
            if (!window.writeOutAndCloseIfFree(this.readOnly)) continue;
            mappedBrick.setWindow(null);
            this.memUsed -= (long)this.brickSize;
            if (!this.allocateNewWindow(unmappedBrick)) continue;
            ++this.switches;
        }
    }

    private Pair<List<BrickElement>, List<BrickElement>> gatherMappedVersusUnmappedWindows() {
        ArrayList<BrickElement> mappedBricks = new ArrayList<BrickElement>();
        ArrayList<BrickElement> unmappedBricks = new ArrayList<BrickElement>();
        for (int i = 0; i < this.brickCount; ++i) {
            BrickElement be = this.brickArray[i];
            be.snapshotHitCount();
            if (be.getWindow() != null) {
                mappedBricks.add(be);
            } else {
                unmappedBricks.add(be);
            }
            be.refresh();
        }
        Collections.sort(unmappedBricks, BRICK_SORTER);
        Collections.sort(mappedBricks, BRICK_SORTER);
        return Pair.of(mappedBricks, unmappedBricks);
    }

    private synchronized void expandBricks(int newBrickCount) {
        if (newBrickCount > this.brickCount) {
            BrickElement[] tmpArray = new BrickElement[newBrickCount];
            System.arraycopy(this.brickArray, 0, tmpArray, 0, this.brickArray.length);
            if (this.memUsed + (long)this.brickSize >= this.availableMem) {
                this.freeWindows(1);
            }
            for (int i = this.brickArray.length; i < tmpArray.length; ++i) {
                BrickElement be;
                tmpArray[i] = be = this.brickFactory.create(i);
                if (this.memUsed + (long)this.brickSize > this.availableMem) continue;
                this.allocateNewWindow(be);
            }
            this.brickArray = tmpArray;
            this.brickCount = tmpArray.length;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean allocateNewWindow(BrickElement brick) {
        try {
            LockableWindow window = null;
            if (this.useMemoryMapped) {
                window = new MappedPersistenceWindow(this.brickIndexToPosition(brick.index()), this.blockSize, this.brickSize, this.fileChannel, this.mapMode);
            } else {
                PlainPersistenceWindow dpw = new PlainPersistenceWindow(this.brickIndexToPosition(brick.index()), this.blockSize, this.brickSize, this.fileChannel);
                dpw.readFullWindow();
                window = dpw;
            }
            while (true) {
                BrickElement brickElement = brick;
                synchronized (brickElement) {
                    if (brick.lockCount.get() == 0) {
                        brick.setWindow(window);
                        break;
                    }
                }
            }
            this.memUsed += (long)this.brickSize;
            return true;
        }
        catch (MappedMemException e) {
            ++this.ooe;
            this.logWarn("Unable to memory map", e);
        }
        catch (OutOfMemoryError e) {
            ++this.ooe;
            this.logWarn("Unable to allocate direct buffer", e);
        }
        return false;
    }

    private void dumpStatus() {
        try {
            this.log.info("[" + this.storeName + "] brickCount=" + this.brickCount + " brickSize=" + this.brickSize + "b mappedMem=" + this.availableMem + "b (storeSize=" + this.fileChannel.size() + "b)");
        }
        catch (IOException e) {
            throw new UnderlyingStorageException("Unable to get file size for " + this.storeName, e);
        }
    }

    private void logWarn(String logMessage) {
        this.log.warn("[" + this.storeName + "] " + logMessage);
    }

    private void logWarn(String logMessage, Throwable cause) {
        this.log.warn("[" + this.storeName + "] " + logMessage, cause);
    }

    @Override
    public WindowPoolStats getStats() {
        int avgRefreshTime = this.refreshes.get() == 0 ? 0 : (int)(this.refreshTime.get() / (long)this.refreshes.get());
        return new WindowPoolStats(this.storeName, this.availableMem, this.memUsed, this.brickCount, this.brickSize, this.hit, this.miss, this.ooe, this.switches, avgRefreshTime, this.refreshes.get(), this.avertedRefreshes.get());
    }
}

