/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import org.neo4j.cli.AbstractCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.commandline.dbms.LockChecker;
import org.neo4j.commandline.dbms.storeutil.StoreCopy;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.ConfigUtils;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.helpers.NormalizedDatabaseName;
import org.neo4j.internal.helpers.Strings;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.kernel.impl.store.format.RecordFormatSelector;
import org.neo4j.kernel.impl.transaction.log.files.TransactionLogFilesHelper;
import org.neo4j.kernel.impl.util.Validators;
import org.neo4j.kernel.internal.locker.FileLockException;
import org.neo4j.kernel.recovery.Recovery;
import picocli.CommandLine;

@CommandLine.Command(name="copy", header={"Copy a database and optionally apply filters."}, description={"This command will create a copy of a database."})
public class StoreCopyCommand
extends AbstractCommand {
    @CommandLine.ArgGroup(multiplicity="1")
    private SourceOption source = new SourceOption();
    @CommandLine.Option(names={"--from-path-tx"}, description={"Path to the transaction files, if they are not in the same folder as '--from-path'."}, paramLabel="<path>")
    private Path sourceTxLogs;
    @CommandLine.Option(names={"--to-database"}, description={"Name of database to copy to."}, required=true, converter={Converters.DatabaseNameConverter.class})
    private NormalizedDatabaseName database;
    @CommandLine.Option(names={"--force"}, description={"Force the command to run even if the integrity of the database can not be verified."})
    private boolean force;
    @CommandLine.Option(completionCandidates=StoreFormatCandidates.class, names={"--to-format"}, defaultValue="same", converter={FormatNameConverter.class}, description={"Set the format for the new database. Must be one of ${COMPLETION-CANDIDATES}. 'same' will use the same format as the source. WARNING: 'high_limit' format is only available in enterprise edition. If you go from 'high_limit' to 'standard' there is no validation that the data will actually fit."})
    private StoreCopy.FormatEnum format;
    @CommandLine.Option(names={"--delete-nodes-with-labels"}, description={"A comma separated list of labels. All nodes that have ANY of the specified labels will be deleted."}, split=",", paramLabel="<label>", showDefaultValue=CommandLine.Help.Visibility.NEVER)
    private List<String> deleteNodesWithLabels = new ArrayList<String>();
    @CommandLine.Option(names={"--skip-labels"}, description={"A comma separated list of labels to ignore."}, split=",", paramLabel="<label>", showDefaultValue=CommandLine.Help.Visibility.NEVER)
    private List<String> skipLabels = new ArrayList<String>();
    @CommandLine.Option(names={"--skip-properties"}, description={"A comma separated list of property keys to ignore."}, split=",", paramLabel="<property>", showDefaultValue=CommandLine.Help.Visibility.NEVER)
    private List<String> skipProperties = new ArrayList<String>();
    @CommandLine.Option(names={"--skip-relationships"}, description={"A comma separated list of relationships to ignore."}, split=",", paramLabel="<relationship>", showDefaultValue=CommandLine.Help.Visibility.NEVER)
    private List<String> skipRelationships = new ArrayList<String>();
    @CommandLine.Option(names={"--from-pagecache"}, paramLabel="<size>", defaultValue="8m", description={"The size of the page cache to use for reading."})
    private String fromPageCacheMemory;
    @CommandLine.Option(names={"--to-pagecache"}, paramLabel="<size>", defaultValue="8m", description={"The size of the page cache to use for writing."})
    private String toPageCacheMemory;

    public StoreCopyCommand(ExecutionContext ctx) {
        super(ctx);
    }

    public void execute() throws Exception {
        Config config = this.buildConfig();
        DatabaseLayout fromDatabaseLayout = this.getFromDatabaseLayout(config);
        StoreCopyCommand.validateSource(fromDatabaseLayout);
        DatabaseLayout toDatabaseLayout = Neo4jLayout.of((Config)config).databaseLayout(this.database.name());
        StoreCopyCommand.validateTarget(toDatabaseLayout);
        try (Closeable ignored = LockChecker.checkDatabaseLock(fromDatabaseLayout);){
            if (!this.force) {
                StoreCopyCommand.checkDbState(fromDatabaseLayout, config);
            }
            try (Closeable ignored2 = LockChecker.checkDatabaseLock(toDatabaseLayout);){
                StoreCopy copy = new StoreCopy(fromDatabaseLayout, config, this.format, this.deleteNodesWithLabels, this.skipLabels, this.skipProperties, this.skipRelationships, this.verbose, this.ctx.out());
                try {
                    copy.copyTo(toDatabaseLayout, this.fromPageCacheMemory, this.toPageCacheMemory);
                }
                catch (Exception e) {
                    throw new CommandFailedException("There was a problem during copy.", (Throwable)e);
                }
            }
            catch (FileLockException e) {
                throw new CommandFailedException("Unable to lock destination.", (Throwable)e);
            }
        }
        catch (FileLockException e) {
            throw new CommandFailedException("The database is in use. Stop database '" + fromDatabaseLayout.getDatabaseName() + "' and try again.", (Throwable)e);
        }
    }

    private DatabaseLayout getFromDatabaseLayout(Config config) {
        if (this.source.path != null) {
            this.source.path = this.source.path.toAbsolutePath();
            if (!Files.isDirectory(this.source.path, new LinkOption[0])) {
                throw new CommandFailedException("The path doesn't exist or not a directory: " + this.source.path);
            }
            if (new TransactionLogFilesHelper(this.ctx.fs(), this.source.path.toFile()).getLogFiles().length > 0) {
                return DatabaseLayout.ofFlat((File)this.source.path.toFile());
            }
            if (this.sourceTxLogs == null) {
                throw new CommandFailedException("Unable to find transaction logs, please specify the location with '--from-path-tx'.");
            }
            Path databaseName = this.source.path.getFileName();
            if (!databaseName.equals(this.sourceTxLogs.getFileName())) {
                throw new CommandFailedException("The directory with data and the directory with transaction logs need to have the same name.");
            }
            this.sourceTxLogs = this.sourceTxLogs.toAbsolutePath();
            Config cfg = Config.newBuilder().set(GraphDatabaseSettings.default_database, (Object)databaseName.toString()).set(GraphDatabaseSettings.neo4j_home, (Object)this.source.path.getParent()).set(GraphDatabaseSettings.databases_root_path, (Object)this.source.path.getParent()).set(GraphDatabaseSettings.transaction_logs_root_path, (Object)this.sourceTxLogs.getParent()).build();
            return DatabaseLayout.of((Config)cfg);
        }
        return Neo4jLayout.of((Config)config).databaseLayout(this.source.database.name());
    }

    private static void validateSource(DatabaseLayout fromDatabaseLayout) {
        try {
            Validators.CONTAINS_EXISTING_DATABASE.validate((Object)fromDatabaseLayout.databaseDirectory());
        }
        catch (IllegalArgumentException e) {
            throw new CommandFailedException("Database does not exist: " + fromDatabaseLayout.getDatabaseName(), (Throwable)e);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static void validateTarget(DatabaseLayout toDatabaseLayout) {
        File targetFile = toDatabaseLayout.databaseDirectory();
        if (targetFile.exists()) {
            if (!targetFile.isDirectory()) throw new CommandFailedException("Specified path is a file: " + targetFile.getAbsolutePath());
            String[] files = targetFile.list();
            if (files != null && files.length <= 0) return;
            throw new CommandFailedException("The directory is not empty: " + targetFile.getAbsolutePath());
        }
        try {
            Files.createDirectories(targetFile.toPath(), new FileAttribute[0]);
            return;
        }
        catch (IOException e) {
            throw new CommandFailedException("Unable to create directory: " + targetFile.getAbsolutePath());
        }
    }

    private static void checkDbState(DatabaseLayout databaseLayout, Config additionalConfiguration) {
        if (StoreCopyCommand.checkRecoveryState(databaseLayout, additionalConfiguration)) {
            throw new CommandFailedException(Strings.joinAsLines((String[])new String[]{"The database " + databaseLayout.getDatabaseName() + "  was not shut down properly.", "Please perform a recovery by starting and stopping the database.", "If recovery is not possible, you can force the command to continue with the '--force' flag."}));
        }
    }

    private static boolean checkRecoveryState(DatabaseLayout databaseLayout, Config additionalConfiguration) {
        try {
            return Recovery.isRecoveryRequired((DatabaseLayout)databaseLayout, (Config)additionalConfiguration);
        }
        catch (Exception e) {
            throw new CommandFailedException("Failure when checking for recovery state: '%s'." + e.getMessage(), (Throwable)e);
        }
    }

    private Config buildConfig() {
        Config cfg = Config.newBuilder().fromFileNoThrow(this.ctx.confDir().resolve("neo4j.conf")).set(GraphDatabaseSettings.neo4j_home, (Object)this.ctx.homeDir()).build();
        ConfigUtils.disableAllConnectors((Config)cfg);
        cfg.set(GraphDatabaseSettings.record_format, (Object)"");
        return cfg;
    }

    private static boolean isHighLimitFormatMissing() {
        return Iterables.stream((Iterable)RecordFormatSelector.allFormats()).noneMatch(format -> "high_limit".equals(format.name()));
    }

    public static class StoreFormatCandidates
    implements Iterable<String> {
        @Override
        public Iterator<String> iterator() {
            ArrayList storeFormats = new ArrayList(Arrays.stream(StoreCopy.FormatEnum.values()).map(Enum::name).collect(Collectors.toList()));
            if (StoreCopyCommand.isHighLimitFormatMissing()) {
                storeFormats.remove(StoreCopy.FormatEnum.high_limit.name());
            }
            return storeFormats.iterator();
        }
    }

    public static class FormatNameConverter
    implements CommandLine.ITypeConverter<StoreCopy.FormatEnum> {
        public StoreCopy.FormatEnum convert(String name) {
            try {
                StoreCopy.FormatEnum format = StoreCopy.FormatEnum.valueOf(name);
                if (format == StoreCopy.FormatEnum.high_limit && StoreCopyCommand.isHighLimitFormatMissing()) {
                    throw new CommandLine.TypeConversionException("High limit format available only in enterprise edition.");
                }
                return format;
            }
            catch (Exception e) {
                throw new CommandLine.TypeConversionException(String.format("Invalid database format name '%s'. (%s)", name, e));
            }
        }
    }

    private static class SourceOption {
        @CommandLine.Option(names={"--from-database"}, description={"Name of database to copy from."}, required=true, converter={Converters.DatabaseNameConverter.class})
        private NormalizedDatabaseName database;
        @CommandLine.Option(names={"--from-path"}, description={"Path to the database to copy from."}, required=true)
        private Path path;

        private SourceOption() {
        }
    }
}

