/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.fs;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.nio.file.NotDirectoryException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.time.Clock;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import org.apache.commons.lang3.ArrayUtils;
import org.neo4j.graphdb.Resource;
import org.neo4j.internal.helpers.collection.CombiningIterator;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.EphemeralFileChannel;
import org.neo4j.io.fs.EphemeralFileData;
import org.neo4j.io.fs.EphemeralFileStillOpenException;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.fs.StoreFileChannel;
import org.neo4j.io.fs.watcher.FileWatcher;
import org.neo4j.io.memory.ByteBuffers;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.test.impl.ChannelInputStream;
import org.neo4j.test.impl.ChannelOutputStream;

public class EphemeralFileSystemAbstraction
implements FileSystemAbstraction {
    private static final AtomicLong UNIQUE_TEMP_FILE = new AtomicLong();
    private final Clock clock;
    private final AtomicInteger keepFiles = new AtomicInteger();
    private final Set<Path> directories = ConcurrentHashMap.newKeySet();
    private final Map<Path, EphemeralFileData> files;
    private final String tempDirectory = System.getProperty("java.io.tmpdir");
    private volatile boolean closed;

    public EphemeralFileSystemAbstraction() {
        this(Clock.systemUTC());
    }

    public EphemeralFileSystemAbstraction(Clock clock) {
        this.clock = clock;
        this.files = new ConcurrentHashMap<Path, EphemeralFileData>();
        this.initCurrentWorkingDirectory();
    }

    private void initCurrentWorkingDirectory() {
        try {
            this.mkdirs(Path.of(".", new String[0]).toAbsolutePath().normalize());
        }
        catch (IOException e) {
            throw new UncheckedIOException("EphemeralFileSystemAbstraction could not initialise current working directory", e);
        }
    }

    private EphemeralFileSystemAbstraction(Set<Path> directories, Map<Path, EphemeralFileData> files, Clock clock) {
        this.clock = clock;
        this.files = new ConcurrentHashMap<Path, EphemeralFileData>(files);
        this.directories.addAll(directories);
        this.initCurrentWorkingDirectory();
    }

    public void clear() {
        this.closeFiles();
    }

    public void crash() {
        this.files.values().forEach(EphemeralFileData::crash);
    }

    public Resource keepFiles() {
        this.keepFiles.getAndIncrement();
        return this.keepFiles::decrementAndGet;
    }

    public synchronized void close() throws IOException {
        if (this.keepFiles.get() > 0) {
            return;
        }
        this.closeFiles();
        this.closed = true;
    }

    public boolean isClosed() {
        return this.closed;
    }

    private void closeFiles() {
        for (EphemeralFileData file : this.files.values()) {
            file.free();
        }
        this.files.clear();
    }

    public void assertNoOpenFiles() throws Exception {
        Throwable exception = null;
        for (EphemeralFileData file : this.files.values()) {
            Iterator<EphemeralFileChannel> channels = file.getOpenChannels();
            while (channels.hasNext()) {
                EphemeralFileChannel channel = channels.next();
                if (exception == null) {
                    exception = new IOException("Expected no open files. The stack traces of the currently open files are attached as suppressed exceptions.");
                }
                exception.addSuppressed(channel.openedAt);
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    public FileWatcher fileWatcher() {
        return FileWatcher.SILENT_WATCHER;
    }

    public synchronized StoreChannel open(Path fileName, Set<OpenOption> options) throws IOException {
        return this.getStoreChannel(fileName);
    }

    public OutputStream openAsOutputStream(Path fileName, boolean append) throws IOException {
        return new ChannelOutputStream(this.write(fileName), append, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
    }

    public InputStream openAsInputStream(Path fileName) throws IOException {
        return new ChannelInputStream(this.read(fileName), (MemoryTracker)EmptyMemoryTracker.INSTANCE);
    }

    public synchronized StoreChannel write(Path fileName) throws IOException {
        Path parentFile = fileName.getParent();
        if (parentFile != null && !this.fileExists(parentFile)) {
            throw new NoSuchFileException("'" + String.valueOf(fileName) + "' (The system cannot find the path specified)");
        }
        EphemeralFileData data = this.files.computeIfAbsent(EphemeralFileSystemAbstraction.canonicalFile(fileName), key -> new EphemeralFileData((Path)key, this.clock));
        return new StoreFileChannel((FileChannel)new EphemeralFileChannel(data, () -> new EphemeralFileStillOpenException(fileName.toString())));
    }

    public synchronized StoreChannel read(Path fileName) throws IOException {
        return this.getStoreChannel(fileName);
    }

    public long getFileSize(Path fileName) {
        EphemeralFileData file = this.files.get(EphemeralFileSystemAbstraction.canonicalFile(fileName));
        return file == null ? 0L : file.size();
    }

    public long getBlockSize(Path file) {
        return 512L;
    }

    public boolean fileExists(Path file) {
        return this.directories.contains(file = EphemeralFileSystemAbstraction.canonicalFile(file)) || this.files.containsKey(file);
    }

    private static Path canonicalFile(Path path) {
        return path.toAbsolutePath().normalize();
    }

    public boolean isDirectory(Path file) {
        return this.directories.contains(EphemeralFileSystemAbstraction.canonicalFile(file));
    }

    public void mkdir(Path directory) {
        this.directories.add(EphemeralFileSystemAbstraction.canonicalFile(directory));
    }

    public void mkdirs(Path directory) throws IOException {
        for (Path currentDirectory = EphemeralFileSystemAbstraction.canonicalFile(directory); currentDirectory != null; currentDirectory = currentDirectory.getParent()) {
            if (this.files.containsKey(currentDirectory)) {
                throw new IOException(String.format("Unable to write directory path [%s] for Neo4j store.", currentDirectory));
            }
            this.mkdir(currentDirectory);
        }
    }

    public void deleteFile(Path fileName) throws IOException {
        EphemeralFileData removed = this.files.remove(fileName = EphemeralFileSystemAbstraction.canonicalFile(fileName));
        if (removed != null) {
            removed.free();
        } else {
            if (!this.fileExists(fileName)) {
                return;
            }
            Path[] fileList = this.listFiles(fileName);
            if (fileList.length > 0) {
                throw new DirectoryNotEmptyException(fileName.toString());
            }
            if (!this.directories.remove(fileName)) {
                throw new NoSuchFileException(fileName.toString());
            }
        }
    }

    public void deleteRecursively(Path directory) throws IOException {
        if (!this.fileExists(directory)) {
            return;
        }
        if (!this.isDirectory(directory)) {
            throw new NotDirectoryException(directory.toString());
        }
        directory = EphemeralFileSystemAbstraction.canonicalFile(directory);
        for (Map.Entry<Path, EphemeralFileData> file : this.files.entrySet()) {
            Path fileName = file.getKey();
            if (!fileName.startsWith(directory) || fileName.equals(directory)) continue;
            this.deleteFile(fileName);
        }
        Path finalDirectory = directory;
        List<Path> subDirectories = this.directories.stream().filter(p -> p.startsWith(finalDirectory) && !p.equals(finalDirectory)).sorted(Comparator.reverseOrder()).toList();
        for (Path subDirectory : subDirectories) {
            this.deleteFile(subDirectory);
        }
        this.deleteFile(directory);
    }

    public void deleteRecursively(Path directory, Predicate<Path> removeFilePredicate) throws IOException {
        if (!this.fileExists(directory)) {
            return;
        }
        if (!this.isDirectory(directory)) {
            throw new NotDirectoryException(directory.toString());
        }
        directory = EphemeralFileSystemAbstraction.canonicalFile(directory);
        for (Map.Entry<Path, EphemeralFileData> file : this.files.entrySet()) {
            Path fileName = file.getKey();
            if (!fileName.startsWith(directory) || fileName.equals(directory) || !removeFilePredicate.test(fileName)) continue;
            this.deleteFile(fileName);
        }
        Path finalDirectory = directory;
        List<Path> subDirectories = this.directories.stream().filter(p -> p.startsWith(finalDirectory) && !p.equals(finalDirectory) && removeFilePredicate.test((Path)p)).sorted(Comparator.reverseOrder()).toList();
        for (Path subDirectory : subDirectories) {
            this.tryDeleteDirectoryIgnoreNotEmpty(subDirectory);
        }
        this.tryDeleteDirectoryIgnoreNotEmpty(directory);
    }

    private void tryDeleteDirectoryIgnoreNotEmpty(Path directory) throws IOException {
        try {
            this.deleteFile(directory);
        }
        catch (DirectoryNotEmptyException directoryNotEmptyException) {
            // empty catch block
        }
    }

    public void renameFile(Path from, Path to, CopyOption ... copyOptions) throws IOException {
        from = EphemeralFileSystemAbstraction.canonicalFile(from);
        to = EphemeralFileSystemAbstraction.canonicalFile(to);
        if (this.directories.contains(from)) {
            if (!this.isDirectory(to.getParent())) {
                throw new NoSuchFileException("Target directory[" + String.valueOf(to.getParent()) + "] does not exists");
            }
            this.directories.add(to);
            for (Path child : this.listFiles(from)) {
                if (this.isDirectory(child)) {
                    this.internalRenameDirectory(to, child);
                    continue;
                }
                this.internalRenameFile(child, to.resolve(child.getFileName().toString()), copyOptions);
            }
        } else {
            this.internalRenameFile(from, to, copyOptions);
        }
    }

    private void internalRenameDirectory(Path to, Path child) {
        this.directories.remove(child);
        this.directories.add(to.resolve(child.getFileName().toString()));
    }

    private void internalRenameFile(Path from, Path to, CopyOption[] copyOptions) throws NoSuchFileException, FileAlreadyExistsException {
        if (!this.files.containsKey(from)) {
            throw new NoSuchFileException("'" + String.valueOf(from) + "' doesn't exist");
        }
        boolean replaceExisting = false;
        for (CopyOption copyOption : copyOptions) {
            replaceExisting |= copyOption == StandardCopyOption.REPLACE_EXISTING;
        }
        if (this.files.containsKey(to) && !replaceExisting) {
            throw new FileAlreadyExistsException("'" + String.valueOf(to) + "' already exists");
        }
        if (!this.isDirectory(to.getParent())) {
            throw new NoSuchFileException("Target directory[" + String.valueOf(to.getParent()) + "] does not exists");
        }
        this.files.put(to, this.files.remove(from));
    }

    public Path[] listFiles(Path directory) throws IOException {
        if (this.files.containsKey(directory = EphemeralFileSystemAbstraction.canonicalFile(directory))) {
            throw new NotDirectoryException(directory.toString());
        }
        if (!this.directories.contains(directory)) {
            throw new NoSuchFileException(directory.toString());
        }
        HashSet<Path> found = new HashSet<Path>();
        CombiningIterator filesAndFolders = new CombiningIterator(Arrays.asList(this.files.keySet().iterator(), this.directories.iterator()));
        while (filesAndFolders.hasNext()) {
            Path file = (Path)filesAndFolders.next();
            if (!directory.equals(file.getParent())) continue;
            found.add(file);
        }
        return found.toArray(new Path[0]);
    }

    public Path[] listFiles(Path directory, DirectoryStream.Filter<Path> filter) {
        if (this.files.containsKey(directory = EphemeralFileSystemAbstraction.canonicalFile(directory))) {
            return null;
        }
        HashSet<Path> found = new HashSet<Path>();
        CombiningIterator files = new CombiningIterator(Arrays.asList(this.files.keySet().iterator(), this.directories.iterator()));
        while (files.hasNext()) {
            Path path = (Path)files.next();
            if (!directory.equals(path.getParent())) continue;
            try {
                if (!filter.accept(path)) continue;
                found.add(path);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return found.toArray(new Path[0]);
    }

    private StoreChannel getStoreChannel(Path fileName) throws IOException {
        EphemeralFileData data = this.files.get(EphemeralFileSystemAbstraction.canonicalFile(fileName));
        if (data != null) {
            return new StoreFileChannel((FileChannel)new EphemeralFileChannel(data, () -> new EphemeralFileStillOpenException(fileName.toAbsolutePath().toString())));
        }
        return this.write(fileName);
    }

    public void moveToDirectory(Path file, Path toDirectory) throws IOException {
        Path destinationFile = toDirectory.resolve(file.getFileName());
        if (this.isDirectory(file)) {
            this.mkdir(destinationFile);
            for (Path f : this.listFiles(file)) {
                this.moveToDirectory(f, destinationFile);
            }
            this.deleteFile(file);
        } else {
            EphemeralFileData fileToMove = this.files.remove(EphemeralFileSystemAbstraction.canonicalFile(file));
            if (fileToMove == null) {
                throw new NoSuchFileException(file.toAbsolutePath().toString());
            }
            this.files.put(EphemeralFileSystemAbstraction.canonicalFile(destinationFile), fileToMove);
        }
    }

    public void copyToDirectory(Path file, Path toDirectory) throws IOException {
        Path targetFile = toDirectory.resolve(file.getFileName());
        this.copyFile(file, targetFile);
    }

    public void copyFile(Path from, Path to, CopyOption ... copyOptions) throws IOException {
        EphemeralFileData data = this.files.get(EphemeralFileSystemAbstraction.canonicalFile(from));
        if (data == null) {
            throw new NoSuchFileException("File " + String.valueOf(from) + " not found");
        }
        if (!ArrayUtils.contains((Object[])copyOptions, (Object)StandardCopyOption.REPLACE_EXISTING) && this.files.get(EphemeralFileSystemAbstraction.canonicalFile(from)) != null) {
            throw new FileAlreadyExistsException(to.toAbsolutePath().toString());
        }
        this.copyFile(from, this, to, EphemeralFileSystemAbstraction.newCopyBuffer());
    }

    public void copyRecursively(Path fromDirectory, Path toDirectory) throws IOException {
        this.copyRecursivelyFromOtherFs(fromDirectory, this, toDirectory, EphemeralFileSystemAbstraction.newCopyBuffer());
    }

    public synchronized EphemeralFileSystemAbstraction snapshot() {
        HashMap<Path, EphemeralFileData> copiedFiles = new HashMap<Path, EphemeralFileData>();
        for (Map.Entry<Path, EphemeralFileData> file : this.files.entrySet()) {
            copiedFiles.put(file.getKey(), file.getValue().copy());
        }
        return new EphemeralFileSystemAbstraction(this.directories, copiedFiles, this.clock);
    }

    private void copyRecursivelyFromOtherFs(Path from, FileSystemAbstraction fromFs, Path to) throws IOException {
        this.copyRecursivelyFromOtherFs(from, fromFs, to, EphemeralFileSystemAbstraction.newCopyBuffer());
    }

    private static ByteBuffer newCopyBuffer() {
        return ByteBuffers.allocate((int)Math.toIntExact(ByteUnit.MebiByte.toBytes(1L)), (ByteOrder)ByteOrder.LITTLE_ENDIAN, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
    }

    private void copyRecursivelyFromOtherFs(Path from, FileSystemAbstraction fromFs, Path to, ByteBuffer buffer) throws IOException {
        this.mkdirs(to);
        for (Path fromFile : fromFs.listFiles(from)) {
            Path toFile = to.resolve(fromFile.getFileName());
            if (fromFs.isDirectory(fromFile)) {
                this.copyRecursivelyFromOtherFs(fromFile, fromFs, toFile);
                continue;
            }
            this.copyFile(fromFile, fromFs, toFile, buffer);
        }
    }

    private void copyFile(Path from, FileSystemAbstraction fromFs, Path to, ByteBuffer buffer) throws IOException {
        try (StoreChannel source = fromFs.read(from);
             StoreChannel sink = this.write(to);){
            int available;
            sink.truncate(0L);
            long sourceSize = source.size();
            while ((available = (int)(sourceSize - source.position())) > 0) {
                buffer.clear();
                buffer.limit(Math.min(available, buffer.capacity()));
                source.read(buffer);
                buffer.flip();
                sink.write(buffer);
            }
        }
    }

    public void truncate(Path file, long size) throws IOException {
        EphemeralFileData data = this.files.get(EphemeralFileSystemAbstraction.canonicalFile(file));
        if (data == null) {
            throw new NoSuchFileException("File " + String.valueOf(file) + " not found");
        }
        data.truncate(size);
    }

    public long lastModifiedTime(Path file) {
        EphemeralFileData data = this.files.get(EphemeralFileSystemAbstraction.canonicalFile(file));
        if (data == null) {
            return 0L;
        }
        return data.getLastModified();
    }

    public void deleteFileOrThrow(Path file) throws IOException {
        if (!this.fileExists(file = EphemeralFileSystemAbstraction.canonicalFile(file))) {
            throw new NoSuchFileException(file.toAbsolutePath().toString());
        }
        this.deleteFile(file);
    }

    public int getFileDescriptor(StoreChannel channel) {
        return -1;
    }

    public boolean isPersistent() {
        return false;
    }

    public Path createTempFile(String prefix, String suffix) throws IOException {
        return this.createTempFile(Path.of(this.tempDirectory, new String[0]), prefix, suffix);
    }

    public Path createTempFile(Path dir, String prefix, String suffix) throws IOException {
        Path tmp;
        EphemeralFileData prev;
        Path parent = EphemeralFileSystemAbstraction.canonicalFile(dir);
        this.mkdirs(parent);
        while ((prev = this.files.putIfAbsent(tmp = parent.resolve(prefix + Long.toUnsignedString(UNIQUE_TEMP_FILE.getAndIncrement()) + suffix), new EphemeralFileData(tmp, this.clock))) != null) {
        }
        return tmp;
    }

    public Path createTempDirectory(String prefix) throws IOException {
        return this.createTempDirectory(Path.of(this.tempDirectory, new String[0]), prefix);
    }

    public Path createTempDirectory(Path dir, String prefix) throws IOException {
        Path tmp;
        Path parent = EphemeralFileSystemAbstraction.canonicalFile(dir);
        this.mkdirs(parent);
        while (!this.directories.add(tmp = parent.resolve(prefix + Long.toUnsignedString(UNIQUE_TEMP_FILE.getAndIncrement())))) {
        }
        return tmp;
    }
}

