/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.retry;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.neo4j.driver.Logger;
import org.neo4j.driver.Logging;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.exceptions.SessionExpiredException;
import org.neo4j.driver.exceptions.TransientException;
import org.neo4j.driver.internal.retry.RetryLogic;
import org.neo4j.driver.internal.retry.RetrySettings;
import org.neo4j.driver.internal.shaded.io.netty.util.concurrent.EventExecutor;
import org.neo4j.driver.internal.shaded.io.netty.util.concurrent.EventExecutorGroup;
import org.neo4j.driver.internal.shaded.reactor.core.publisher.Flux;
import org.neo4j.driver.internal.shaded.reactor.core.publisher.Mono;
import org.neo4j.driver.internal.shaded.reactor.core.scheduler.Schedulers;
import org.neo4j.driver.internal.shaded.reactor.util.context.Context;
import org.neo4j.driver.internal.shaded.reactor.util.retry.Retry;
import org.neo4j.driver.internal.util.Clock;
import org.neo4j.driver.internal.util.Futures;
import org.neo4j.driver.util.Experimental;
import org.reactivestreams.Publisher;

public class ExponentialBackoffRetryLogic
implements RetryLogic {
    private static final String RETRY_LOGIC_LOG_NAME = "RetryLogic";
    static final long DEFAULT_MAX_RETRY_TIME_MS = TimeUnit.SECONDS.toMillis(30L);
    private static final long INITIAL_RETRY_DELAY_MS = TimeUnit.SECONDS.toMillis(1L);
    private static final double RETRY_DELAY_MULTIPLIER = 2.0;
    private static final double RETRY_DELAY_JITTER_FACTOR = 0.2;
    private static final long MAX_RETRY_DELAY = 0x3FFFFFFFFFFFFFFFL;
    private final long maxRetryTimeMs;
    private final long initialRetryDelayMs;
    private final double multiplier;
    private final double jitterFactor;
    private final EventExecutorGroup eventExecutorGroup;
    private final Clock clock;
    private final Logger log;

    public ExponentialBackoffRetryLogic(RetrySettings settings, EventExecutorGroup eventExecutorGroup, Clock clock, Logging logging) {
        this(settings.maxRetryTimeMs(), INITIAL_RETRY_DELAY_MS, 2.0, 0.2, eventExecutorGroup, clock, logging);
    }

    ExponentialBackoffRetryLogic(long maxRetryTimeMs, long initialRetryDelayMs, double multiplier, double jitterFactor, EventExecutorGroup eventExecutorGroup, Clock clock, Logging logging) {
        this.maxRetryTimeMs = maxRetryTimeMs;
        this.initialRetryDelayMs = initialRetryDelayMs;
        this.multiplier = multiplier;
        this.jitterFactor = jitterFactor;
        this.eventExecutorGroup = eventExecutorGroup;
        this.clock = clock;
        this.log = logging.getLog(RETRY_LOGIC_LOG_NAME);
        this.verifyAfterConstruction();
    }

    @Override
    public <T> T retry(Supplier<T> work) {
        List<Throwable> errors = null;
        long startTime = -1L;
        long nextDelayMs = this.initialRetryDelayMs;
        while (true) {
            try {
                return work.get();
            }
            catch (Throwable throwable) {
                Throwable error = ExponentialBackoffRetryLogic.extractPossibleTerminationCause(throwable);
                if (this.canRetryOn(error)) {
                    long elapsedTime;
                    long currentTime = this.clock.millis();
                    if (startTime == -1L) {
                        startTime = currentTime;
                    }
                    if ((elapsedTime = currentTime - startTime) < this.maxRetryTimeMs) {
                        long delayWithJitterMs = this.computeDelayWithJitter(nextDelayMs);
                        this.log.warn("Transaction failed and will be retried in " + delayWithJitterMs + "ms", error);
                        this.sleep(delayWithJitterMs);
                        nextDelayMs = (long)((double)nextDelayMs * this.multiplier);
                        errors = ExponentialBackoffRetryLogic.recordError(error, errors);
                        continue;
                    }
                }
                ExponentialBackoffRetryLogic.addSuppressed(throwable, errors);
                throw throwable;
            }
            break;
        }
    }

    @Override
    public <T> CompletionStage<T> retryAsync(Supplier<CompletionStage<T>> work) {
        CompletableFuture resultFuture = new CompletableFuture();
        this.executeWorkInEventLoop(resultFuture, work);
        return resultFuture;
    }

    @Override
    public <T> Publisher<T> retryRx(Publisher<T> work) {
        return Flux.from(work).retryWhen(this.exponentialBackoffRetryRx());
    }

    protected boolean canRetryOn(Throwable error) {
        return ExponentialBackoffRetryLogic.isRetryable(error);
    }

    @Experimental
    public static boolean isRetryable(Throwable error) {
        return error instanceof SessionExpiredException || error instanceof ServiceUnavailableException || ExponentialBackoffRetryLogic.isTransientError(error);
    }

    private static Throwable extractPossibleTerminationCause(Throwable error) {
        if (error instanceof ClientException && error.getCause() != null) {
            return error.getCause();
        }
        return error;
    }

    private Retry exponentialBackoffRetryRx() {
        return Retry.from(retrySignals -> retrySignals.flatMap(retrySignal -> Mono.deferContextual(contextView -> {
            Throwable throwable = retrySignal.failure();
            Throwable error = ExponentialBackoffRetryLogic.extractPossibleTerminationCause(throwable);
            List<Throwable> errors = contextView.getOrDefault("errors", null);
            if (this.canRetryOn(error)) {
                long currentTime = this.clock.millis();
                long startTime = contextView.getOrDefault("startTime", currentTime);
                long nextDelayMs = contextView.getOrDefault("nextDelayMs", this.initialRetryDelayMs);
                long elapsedTime = currentTime - startTime;
                if (elapsedTime < this.maxRetryTimeMs) {
                    long delayWithJitterMs = this.computeDelayWithJitter(nextDelayMs);
                    this.log.warn("Reactive transaction failed and is scheduled to retry in " + delayWithJitterMs + "ms", error);
                    nextDelayMs = (long)((double)nextDelayMs * this.multiplier);
                    errors = ExponentialBackoffRetryLogic.recordError(error, errors);
                    EventExecutor eventExecutor = this.eventExecutorGroup.next();
                    Context context = Context.of("errors", errors, "startTime", startTime, "nextDelayMs", nextDelayMs);
                    return Mono.just(context).delayElement(Duration.ofMillis(delayWithJitterMs), Schedulers.fromExecutorService(eventExecutor));
                }
            }
            ExponentialBackoffRetryLogic.addSuppressed(throwable, errors);
            return Mono.error(throwable);
        })));
    }

    private <T> void executeWorkInEventLoop(CompletableFuture<T> resultFuture, Supplier<CompletionStage<T>> work) {
        EventExecutor eventExecutor = this.eventExecutorGroup.next();
        eventExecutor.execute(() -> this.executeWork(resultFuture, work, -1L, this.initialRetryDelayMs, null));
    }

    private <T> void retryWorkInEventLoop(CompletableFuture<T> resultFuture, Supplier<CompletionStage<T>> work, Throwable error, long startTime, long delayMs, List<Throwable> errors) {
        EventExecutor eventExecutor = this.eventExecutorGroup.next();
        long delayWithJitterMs = this.computeDelayWithJitter(delayMs);
        this.log.warn("Async transaction failed and is scheduled to retry in " + delayWithJitterMs + "ms", error);
        eventExecutor.schedule(() -> {
            long newRetryDelayMs = (long)((double)delayMs * this.multiplier);
            this.executeWork(resultFuture, work, startTime, newRetryDelayMs, errors);
        }, delayWithJitterMs, TimeUnit.MILLISECONDS);
    }

    private <T> void executeWork(CompletableFuture<T> resultFuture, Supplier<CompletionStage<T>> work, long startTime, long retryDelayMs, List<Throwable> errors) {
        CompletionStage<Object> workStage;
        try {
            workStage = work.get();
        }
        catch (Throwable error) {
            this.retryOnError(resultFuture, work, startTime, retryDelayMs, error, errors);
            return;
        }
        workStage.whenComplete((result, completionError) -> {
            Throwable error = Futures.completionExceptionCause(completionError);
            if (error != null) {
                this.retryOnError(resultFuture, work, startTime, retryDelayMs, error, errors);
            } else {
                resultFuture.complete(result);
            }
        });
    }

    private <T> void retryOnError(CompletableFuture<T> resultFuture, Supplier<CompletionStage<T>> work, long startTime, long retryDelayMs, Throwable throwable, List<Throwable> errors) {
        Throwable error = ExponentialBackoffRetryLogic.extractPossibleTerminationCause(throwable);
        if (this.canRetryOn(error)) {
            long elapsedTime;
            long currentTime = this.clock.millis();
            if (startTime == -1L) {
                startTime = currentTime;
            }
            if ((elapsedTime = currentTime - startTime) < this.maxRetryTimeMs) {
                errors = ExponentialBackoffRetryLogic.recordError(error, errors);
                this.retryWorkInEventLoop(resultFuture, work, error, startTime, retryDelayMs, errors);
                return;
            }
        }
        ExponentialBackoffRetryLogic.addSuppressed(throwable, errors);
        resultFuture.completeExceptionally(throwable);
    }

    private long computeDelayWithJitter(long delayMs) {
        if (delayMs > 0x3FFFFFFFFFFFFFFFL) {
            delayMs = 0x3FFFFFFFFFFFFFFFL;
        }
        long jitter = (long)((double)delayMs * this.jitterFactor);
        long min = delayMs - jitter;
        long max = delayMs + jitter;
        return ThreadLocalRandom.current().nextLong(min, max + 1L);
    }

    private void sleep(long delayMs) {
        try {
            this.clock.sleep(delayMs);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Retries interrupted", e);
        }
    }

    private void verifyAfterConstruction() {
        if (this.maxRetryTimeMs < 0L) {
            throw new IllegalArgumentException("Max retry time should be >= 0: " + this.maxRetryTimeMs);
        }
        if (this.initialRetryDelayMs < 0L) {
            throw new IllegalArgumentException("Initial retry delay should >= 0: " + this.initialRetryDelayMs);
        }
        if (this.multiplier < 1.0) {
            throw new IllegalArgumentException("Multiplier should be >= 1.0: " + this.multiplier);
        }
        if (this.jitterFactor < 0.0 || this.jitterFactor > 1.0) {
            throw new IllegalArgumentException("Jitter factor should be in [0.0, 1.0]: " + this.jitterFactor);
        }
        if (this.clock == null) {
            throw new IllegalArgumentException("Clock should not be null");
        }
    }

    private static boolean isTransientError(Throwable error) {
        if (error instanceof TransientException) {
            String code = ((TransientException)error).code();
            return !"Neo.TransientError.Transaction.Terminated".equals(code) && !"Neo.TransientError.Transaction.LockClientStopped".equals(code);
        }
        return false;
    }

    private static List<Throwable> recordError(Throwable error, List<Throwable> errors) {
        if (errors == null) {
            errors = new ArrayList<Throwable>();
        }
        errors.add(error);
        return errors;
    }

    private static void addSuppressed(Throwable error, List<Throwable> suppressedErrors) {
        if (suppressedErrors != null) {
            for (Throwable suppressedError : suppressedErrors) {
                if (error == suppressedError) continue;
                error.addSuppressed(suppressedError);
            }
        }
    }
}

