/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.cluster;

import java.net.UnknownHostException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Logger;
import org.neo4j.driver.exceptions.DiscoveryException;
import org.neo4j.driver.exceptions.FatalDiscoveryException;
import org.neo4j.driver.exceptions.SecurityException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.internal.BoltServerAddress;
import org.neo4j.driver.internal.cluster.ClusterComposition;
import org.neo4j.driver.internal.cluster.ClusterCompositionProvider;
import org.neo4j.driver.internal.cluster.Rediscovery;
import org.neo4j.driver.internal.cluster.RoutingSettings;
import org.neo4j.driver.internal.cluster.RoutingTable;
import org.neo4j.driver.internal.shaded.io.netty.util.concurrent.EventExecutorGroup;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.spi.ConnectionPool;
import org.neo4j.driver.internal.util.Futures;
import org.neo4j.driver.net.ServerAddressResolver;

public class RediscoveryImpl
implements Rediscovery {
    private static final String NO_ROUTERS_AVAILABLE = "Could not perform discovery for database '%s'. No routing server available.";
    private static final String RECOVERABLE_ROUTING_ERROR = "Failed to update routing table with server '%s'.";
    private static final String RECOVERABLE_DISCOVERY_ERROR_WITH_SERVER = "Received a recoverable discovery error with server '%s', will continue discovery with other routing servers if available. Complete failure is reported separately from this entry.";
    private final BoltServerAddress initialRouter;
    private final RoutingSettings settings;
    private final Logger logger;
    private final ClusterCompositionProvider provider;
    private final ServerAddressResolver resolver;
    private final EventExecutorGroup eventExecutorGroup;

    public RediscoveryImpl(BoltServerAddress initialRouter, RoutingSettings settings, ClusterCompositionProvider provider, EventExecutorGroup eventExecutorGroup, ServerAddressResolver resolver, Logger logger) {
        this.initialRouter = initialRouter;
        this.settings = settings;
        this.logger = logger;
        this.provider = provider;
        this.resolver = resolver;
        this.eventExecutorGroup = eventExecutorGroup;
    }

    @Override
    public CompletionStage<ClusterComposition> lookupClusterComposition(RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark) {
        CompletableFuture<ClusterComposition> result = new CompletableFuture<ClusterComposition>();
        ServiceUnavailableException baseError = new ServiceUnavailableException(String.format(NO_ROUTERS_AVAILABLE, routingTable.database().description()));
        this.lookupClusterComposition(routingTable, connectionPool, 0, 0L, result, bookmark, baseError);
        return result;
    }

    private void lookupClusterComposition(RoutingTable routingTable, ConnectionPool pool, int failures, long previousDelay, CompletableFuture<ClusterComposition> result, Bookmark bookmark, Throwable baseError) {
        this.lookup(routingTable, pool, bookmark, baseError).whenComplete((composition, completionError) -> {
            Throwable error = Futures.completionExceptionCause(completionError);
            if (error != null) {
                result.completeExceptionally(error);
            } else if (composition != null) {
                result.complete((ClusterComposition)composition);
            } else {
                int newFailures = failures + 1;
                if (newFailures >= this.settings.maxRoutingFailures()) {
                    result.completeExceptionally(baseError);
                } else {
                    long nextDelay = Math.max(this.settings.retryTimeoutDelay(), previousDelay * 2L);
                    this.logger.info("Unable to fetch new routing table, will try again in " + nextDelay + "ms", new Object[0]);
                    this.eventExecutorGroup.next().schedule(() -> this.lookupClusterComposition(routingTable, pool, newFailures, nextDelay, result, bookmark, baseError), nextDelay, TimeUnit.MILLISECONDS);
                }
            }
        });
    }

    private CompletionStage<ClusterComposition> lookup(RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark, Throwable baseError) {
        CompletionStage<ClusterComposition> compositionStage = routingTable.preferInitialRouter() ? this.lookupOnInitialRouterThenOnKnownRouters(routingTable, connectionPool, bookmark, baseError) : this.lookupOnKnownRoutersThenOnInitialRouter(routingTable, connectionPool, bookmark, baseError);
        return compositionStage;
    }

    private CompletionStage<ClusterComposition> lookupOnKnownRoutersThenOnInitialRouter(RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark, Throwable baseError) {
        HashSet<BoltServerAddress> seenServers = new HashSet<BoltServerAddress>();
        return this.lookupOnKnownRouters(routingTable, connectionPool, seenServers, bookmark, baseError).thenCompose(composition -> {
            if (composition != null) {
                return CompletableFuture.completedFuture(composition);
            }
            return this.lookupOnInitialRouter(routingTable, connectionPool, seenServers, bookmark, baseError);
        });
    }

    private CompletionStage<ClusterComposition> lookupOnInitialRouterThenOnKnownRouters(RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark, Throwable baseError) {
        Set<BoltServerAddress> seenServers = Collections.emptySet();
        return this.lookupOnInitialRouter(routingTable, connectionPool, seenServers, bookmark, baseError).thenCompose(composition -> {
            if (composition != null) {
                return CompletableFuture.completedFuture(composition);
            }
            return this.lookupOnKnownRouters(routingTable, connectionPool, new HashSet<BoltServerAddress>(), bookmark, baseError);
        });
    }

    private CompletionStage<ClusterComposition> lookupOnKnownRouters(RoutingTable routingTable, ConnectionPool connectionPool, Set<BoltServerAddress> seenServers, Bookmark bookmark, Throwable baseError) {
        BoltServerAddress[] addresses = routingTable.routers().toArray();
        CompletionStage<ClusterComposition> result = Futures.completedWithNull();
        for (BoltServerAddress address : addresses) {
            result = result.thenCompose(composition -> {
                if (composition != null) {
                    return CompletableFuture.completedFuture(composition);
                }
                return this.lookupOnRouter(address, routingTable, connectionPool, bookmark, baseError).whenComplete((ignore, error) -> seenServers.add(address));
            });
        }
        return result;
    }

    private CompletionStage<ClusterComposition> lookupOnInitialRouter(RoutingTable routingTable, ConnectionPool connectionPool, Set<BoltServerAddress> seenServers, Bookmark bookmark, Throwable baseError) {
        List<BoltServerAddress> addresses;
        try {
            addresses = this.resolve();
        }
        catch (Throwable error) {
            return Futures.failedFuture(error);
        }
        addresses.removeAll(seenServers);
        CompletionStage<ClusterComposition> result = Futures.completedWithNull();
        for (BoltServerAddress address : addresses) {
            result = result.thenCompose(composition -> {
                if (composition != null) {
                    return CompletableFuture.completedFuture(composition);
                }
                return this.lookupOnRouter(address, routingTable, connectionPool, bookmark, baseError);
            });
        }
        return result;
    }

    private CompletionStage<ClusterComposition> lookupOnRouter(BoltServerAddress routerAddress, RoutingTable routingTable, ConnectionPool connectionPool, Bookmark bookmark, Throwable baseError) {
        CompletionStage<Connection> connectionStage = connectionPool.acquire(routerAddress);
        return connectionStage.thenCompose(connection -> this.provider.getClusterComposition((Connection)connection, routingTable.database(), bookmark)).handle((response, error) -> {
            Throwable cause = Futures.completionExceptionCause(error);
            if (cause != null) {
                return this.handleRoutingProcedureError(cause, routingTable, routerAddress, baseError);
            }
            return response;
        });
    }

    private ClusterComposition handleRoutingProcedureError(Throwable error, RoutingTable routingTable, BoltServerAddress routerAddress, Throwable baseError) {
        if (error instanceof SecurityException || error instanceof FatalDiscoveryException) {
            throw new CompletionException(error);
        }
        DiscoveryException discoveryError = new DiscoveryException(String.format(RECOVERABLE_ROUTING_ERROR, routerAddress), error);
        Futures.combineErrors(baseError, discoveryError);
        String warningMessage = String.format(RECOVERABLE_DISCOVERY_ERROR_WITH_SERVER, routerAddress);
        this.logger.warn(warningMessage, new Object[0]);
        this.logger.debug(warningMessage, discoveryError);
        routingTable.forget(routerAddress);
        return null;
    }

    @Override
    public List<BoltServerAddress> resolve() {
        return this.resolver.resolve(this.initialRouter).stream().flatMap(resolved -> this.resolveAll(BoltServerAddress.from(resolved))).collect(Collectors.toList());
    }

    private Stream<BoltServerAddress> resolveAll(BoltServerAddress address) {
        try {
            return address.resolveAll().stream();
        }
        catch (UnknownHostException e) {
            this.logger.error("Failed to resolve address `" + address + "` to IPs due to error: " + e.getMessage(), e);
            return Stream.of(address);
        }
    }
}

