package org.mule.weave.v2.runtime.utils

import org.mule.weave.v2.core.io.FileHelper
import org.mule.weave.v2.debugger.util.FileElement
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.wrappers.WrapperValue
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.SimpleLocation
import org.mule.weave.v2.parser.location.UnknownLocation

import java.io.StringReader
import java.util.Properties
import scala.collection.convert.ImplicitConversions.`set asScala`

object WeaveRuntimeUtils {

  private val CONFIG_FILENAME_PREFIX = "-config.properties"

  private def isConfigFile(p: FileElement): Boolean = {
    p.name.endsWith(CONFIG_FILENAME_PREFIX)
  }

  def toContext(inputs: Array[FileElement]): Map[String, Value[_]] = {
    if (inputs == null) {
      Map()
    } else {
      val valuesEntries = inputs
        .filter(p => !isConfigFile(p))
        .map(file => {
          val name = FileHelper.baseName(file.name)
          val value = if (file.isNode) {
            ObjectValue(ObjectSeq(toContext(file.children)))
          } else {
            new FileValue(file, inputs)
          }
          (name, value)
        })
      valuesEntries.toMap
    }
  }

  class FileValue(file: FileElement, inputs: Array[FileElement]) extends WrapperValue {
    override def value(implicit ctx: EvaluationContext): Value[Any] = {
      val name = FileHelper.baseName(file.name)
      val extension = FileHelper.getExtension(file.name)
      val mayBeDataFormat = DataFormatManager.byExtension(extension.getOrElse(".txt"))
      mayBeDataFormat match {
        case Some(dataFormat) =>
          val sourceProvider = if (file.bytes != null) {
            SourceProvider(file.bytes.get)
          } else {
            SourceProvider(file.content.get)
          }
          val reader = dataFormat.reader(sourceProvider)
          val maybeElement: Option[FileElement] = inputs.find(fe => fe.name.equals(name + CONFIG_FILENAME_PREFIX))
          if (maybeElement.isDefined && maybeElement.get.content.isDefined) {
            val settings: Properties = new Properties()
            settings.load(new StringReader(maybeElement.get.content.get))
            settings
              .entrySet()
              .foreach(property => {
                reader.setOption(UnknownLocation, property.getKey.toString, property.getValue)
              })
          }
          reader.read(name)
        case None =>
          NullValue
      }
    }

    override def location(): Location = SimpleLocation(file.name)
  }
}
