package org.mule.weave.v2.runtime.core.functions.stringops

import org.mule.weave.v2.core.functions.UnaryFunctionValue
import org.mule.weave.v2.core.exception.InvalidUrlEncodingException
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value

import java.net.URLDecoder

object StringUrlEncodeFunctionValue extends UnaryFunctionValue {
  override val R = StringType

  val ALLOWED_CHARS: Array[Char] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789;,/?:&$-_.!~*'()=@#".toCharArray

  def getHex(buf: Seq[Byte]): String = {
    var output = ""

    buf.foreach(value => {
      val n = value & 0xff

      output = output + "%"

      if (n < 0x10) {
        output = output + "0"
      }

      output = output + n.toHexString.toUpperCase()
    })

    output
  }

  def encodeURIComponent(input: String, charset: String, CHARS: Array[Char]): String = {
    if (input.isEmpty) {
      return input
    }

    var output = ""

    input.toCharArray.foreach((e: Char) => {
      val char = e.toString
      if (CHARS.contains(e)) {
        output = output + char
      } else {
        val b = char.getBytes(charset)
        output = output + getHex(b)
      }
    })

    output
  }

  override def doExecute(v: R.V)(implicit executionContext: EvaluationContext): Value[_] = {
    StringValue(encodeURIComponent(v.evaluate.toString, "UTF-8", ALLOWED_CHARS), UnknownLocationCapable)
  }
}

object UrlEncodeFunctionValue {
  val value = Seq(StringUrlEncodeFunctionValue)
}

object StringUrlEncodeComponentFunctionValue extends UnaryFunctionValue {
  override val R = StringType

  val ALLOWED_CHARS: Array[Char] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_.!~*'()".toCharArray

  override def doExecute(v: R.V)(implicit executionContext: EvaluationContext): Value[_] = {
    StringValue(StringUrlEncodeFunctionValue.encodeURIComponent(v.evaluate.toString, "UTF-8", ALLOWED_CHARS), UnknownLocationCapable)
  }
}

object UrlEncodeComponentFunctionValue {
  val value = Seq(StringUrlEncodeComponentFunctionValue)
}

trait StringUrlDecodeFunction extends UnaryFunctionValue {
  override val R = StringType

  override def doExecute(v: R.V)(implicit executionContext: EvaluationContext): Value[_] = {
    val url = v.evaluate.toString
    StringValue(
      try {
        URLDecoder.decode(
          url
            .replaceAll("\\(", "%28")
            .replaceAll("\\)", "%29")
            .replaceAll("\\%20", "+")
            .replaceAll("'", "\\%27")
            .replaceAll("!", "\\%21")
            .replaceAll("~", "\\%7E"),
          "UTF-8")
      } catch {
        case _: IllegalArgumentException =>
          throw new InvalidUrlEncodingException(UnknownLocationCapable.location(), url)
      },
      UnknownLocationCapable)
  }
}

object StringUrlDecodeFunctionValue extends StringUrlDecodeFunction {}

object UrlDecodeFunctionValue {
  val value = Seq(StringUrlDecodeFunctionValue)
}

object StringUrlDecodeComponentFunctionValue extends StringUrlDecodeFunction {}

object UrlDecodeComponentFunctionValue {
  val value = Seq(StringUrlDecodeComponentFunctionValue)
}
