package org.mule.weave.v2.runtime

import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.NameIdentifierHelper
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceFactory
import org.mule.weave.v2.utils.DataWeaveVersion

import java.io.File
import java.net.URL

class CompilationConfig(componentsFactory: ModuleComponentsFactory) {

  private var _resource: Option[WeaveResource] = None
  private var _script: Option[String] = None
  private var _identifier: Option[NameIdentifier] = None
  private var _inputs: Option[Array[InputType]] = None
  private var _defaultOutputMimeType: Option[String] = None
  private var _defaultWriterProperties: Option[Map[String, Any]] = None
  private var _maxTime: Option[Long] = None
  private var _dataWeaveVersion: Option[DataWeaveVersion] = None

  def withResource(resource: WeaveResource): CompilationConfig = {
    _resource = Some(resource)
    this
  }

  def getVersion(): Option[DataWeaveVersion] = {
    _dataWeaveVersion
  }

  def withLanguageVersion(dataWeaveVersion: DataWeaveVersion): CompilationConfig = {
    _dataWeaveVersion = Option(dataWeaveVersion)
    this
  }

  def withScript(script: String): CompilationConfig = {
    _script = Some(script)
    this
  }

  def withUrl(url: URL): CompilationConfig = {
    _resource = Some(WeaveResourceFactory.fromUrl(url))
    _identifier = Some(NameIdentifier.fromPath(url.getPath))
    this
  }

  def withFile(file: File): CompilationConfig = {
    _resource = Some(WeaveResourceFactory.fromFile(file))
    _identifier = Some(ParsingContextFactory.nameIdentifier(file))
    this
  }

  def withNameIdentifier(nameIdentifier: NameIdentifier): CompilationConfig = {
    _identifier = Some(nameIdentifier)
    this
  }

  def withNameIdentifier(nameIdentifier: String): CompilationConfig = {
    _identifier = Some(NameIdentifier.apply(nameIdentifier))
    this
  }

  def withInputs(implicitInputs: Array[String]): CompilationConfig = {
    _inputs = Some(implicitInputs.map(new InputType(_, None)))
    this
  }

  def withInputs(implicitInputs: Array[InputType]): CompilationConfig = {
    _inputs = Some(implicitInputs)
    this
  }

  def withDefaultOutputType(mimeType: String): CompilationConfig = {
    _defaultOutputMimeType = Some(mimeType)
    this
  }

  def withDefaultWriterProperties(props: Map[String, Any]): CompilationConfig = {
    _defaultWriterProperties = Some(props)
    this
  }

  def withMaxTime(maxTime: Long): CompilationConfig = {
    _maxTime = Some(maxTime)
    this
  }

  def getResource(): WeaveResource = {
    if (_resource.isDefined) {
      _resource.get
    } else if (_script.isDefined && _identifier.isDefined) {
      WeaveResource(NameIdentifierHelper.toWeaveFilePath(_identifier.get), _script.get)
    } else if (_script.isDefined) {
      WeaveResource.anonymous(_script.get)
    } else if (_identifier.isDefined) {
      componentsFactory.createComponents().resourceResolver.resolve(_identifier.get).getOrElse(throw new ScriptingEngineSetupException("Unable to resolve resource for " + _identifier.get.name))
    } else {
      throw new ScriptingEngineSetupException("Unable to resolve resource")
    }
  }

  def getIdentifier(): NameIdentifier = _identifier.getOrElse(NameIdentifier.anonymous)

  def getInputs(): Array[InputType] = _inputs.getOrElse(Array.empty[InputType])

  def getDefaultOutputMimeType(): String = _defaultOutputMimeType.getOrElse("application/dw")

  def getDefaultWriterProperties(): Map[String, Any] = _defaultWriterProperties.getOrElse(Map())

  def getMaxTime(): Long = _maxTime.getOrElse(-1L)
}
