package org.mule.weave.v2.interpreted.node.executors

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.ast.WeaveLocationCapable
import org.mule.weave.v2.parser.location.WeaveLocation

/**
  * Function executors implements the logic of dynamic function dispatching.
  */
sealed trait FunctionExecutor extends Product with WeaveLocationCapable {
  /**
    * Executes the functions with the given arguments
    *
    * @param arguments The arguments
    * @param ctx       The context
    * @return The result of the function invocation
    */
  def execute(arguments: Array[Value[_]])(implicit ctx: ExecutionContext): Value[Any]

  /**
    * The name of the function
    *
    * @param ctx The context
    * @return The name of the function
    */
  def name()(implicit ctx: ExecutionContext): String

  /**
    * The location of the functions being called
    *
    * @return The location
    */
  def location(): WeaveLocation

  /**
    * If this function call should be added to stacktrace if it fails of if it should be ignored
    *
    * @return true if it should be added
    */
  def showInStacktrace(): Boolean

  /**
    * The node that represents the functions
    *
    * @return The node
    */
  def node(): ValueNode[_]

  /**
    * Product function
    */
  override def canEqual(that: Any): Boolean = that.isInstanceOf[this.type]

}

trait EmptyExecutor extends FunctionExecutor {

  def executeEmpty()(implicit ctx: ExecutionContext): Value[Any]
}

trait UnaryExecutor extends FunctionExecutor {

  def executeUnary(fv: Value[_])(implicit ctx: ExecutionContext): Value[Any]
}

trait BinaryExecutor extends FunctionExecutor {

  def executeBinary(fv: Value[_], sv: Value[_])(implicit ctx: ExecutionContext): Value[Any]
}

trait TernaryExecutor extends FunctionExecutor {

  def executeTernary(fv: Value[_], sv: Value[_], tv: Value[_])(implicit ctx: ExecutionContext): Value[Any]
}

trait DefaultExecutor extends FunctionExecutor

